/* gt_cmd.c - simple command line parser
 *
 * Copyright (C) 1997, 1998, 1999 Free Software Foundation
 * Copyright (C) 1994,95,96 Eric M. Ludlam
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 *
 * Purpose:
 *   This file contains the main read loop, and subordinate functions
 * used to parse input based on descriptor type in the object lists.
 * Commands are stored in structures which are searched for the
 * matching functionality.
 *
 * $Log: gt_cmd.c,v $
 * Revision 1.44  1999/11/29 17:02:33  zappo
 * Converted old crypt code into generic filter code.
 *
 * Revision 1.43  1999/08/26 11:56:34  zappo
 * Added new SET EIGHTBIT command for toggling the 8bit flag.
 *
 * Revision 1.42  1998/09/22 13:25:50  zappo
 * Added a route setting command, and a show route command.
 *
 * Revision 1.41  1998/01/09 22:32:33  zappo
 * The test command can now take a host name to test.
 *
 * Revision 1.40  1997/12/14 19:14:57  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.39  1997/11/05 03:44:10  zappo
 * Removed message size numbers from status message
 *
 * Revision 1.38  1997/10/25 02:47:31  zappo
 * Removed the test for socket-only commands.  This makes the window the
 * correct size.
 *
 * Revision 1.37  1997/10/17 02:32:15  zappo
 * CRYPT now uses crypt_index to find the index to use.
 * added defaultcrypt to set the default encryption method.
 * Changed status to display the default encryption method.
 *
 * Revision 1.36  1997/10/08 18:54:49  zappo
 * Fixed bug when setting the editcharacters.
 *
 * Revision 1.35  1997/10/07 00:07:25  zappo
 * Don't allow the setting of editchars if someone is already connected
 *
 * Revision 1.34  1997/03/21 12:08:13  zappo
 * Fixed size of device popup.
 *
 * Revision 1.33  1997/03/12 01:33:10  zappo
 * changed GTC_test to return Success/Fail instead of what a shell expects
 *
 * Revision 1.32  1997/03/01 13:15:11  zappo
 * Gave each command a label as to its usefulness outside of socket mode.
 * If etalk is not is socket mode, those commands are not printed in help.
 *
 * Revision 1.31  1997/02/25 02:47:30  zappo
 * Fixed DISP_message calls w/ /ns at the end.
 *
 * Revision 1.30  1997/02/23 03:21:11  zappo
 * Added GTC_setclean and changed GTC_status to display cleanup status
 *
 * Revision 1.29  1997/02/01 14:28:04  zappo
 * Added checks for some dialogwindows not to appear when there is
 * nothing to display in them.
 *
 * Revision 1.28  1997/01/29  02:59:57  zappo
 * Fixed warnings, and re-ordered so there aren't so many silly
 * prototypes
 *
 * Revision 1.27  1997/01/18  02:39:59  zappo
 * Added comments for script files
 *
 * Revision 1.26  1996/07/27  21:26:14  zappo
 * Added in commands for auxilliary ports, and encryption, plus short
 * string checks
 *
 * Revision 1.25  1996/03/03  15:45:53  zappo
 * Updated HANGUP, START, SHARED to accept login names as well as uid
 * Fixed c2str for DELETE, moved sendfile for grouping, and fixed
 * warnings
 *
 * Revision 1.24  1996/02/01  01:13:03  zappo
 * Added commands to manage (start, print, etc) shared apps.  Also fixes
 * to some parsing and ringer related changes in other parts.
 *
 * Revision 1.23  1995/12/09  23:30:29  zappo
 * Ringer now toggles with no parameter to SET RINGER <ON|OFF>
 * Fixed SET NAME command output printing methods
 * Added SET ACCEPTTCP < ALWAYS | QUERY | NEVER >
 * SHOW USERS now can creates a bigger popup window
 * SHOW HOSTS now can create a popup window
 * SHOW STATUS now uses multiple columns so it fits on 25x80 col display
 * SHOW COPYWRITE now can create a popup window
 * SHOW DEVICES now can create a popup window
 *
 * Revision 1.22  1995/11/21  03:38:03  zappo
 * Renamed READ command to LOAD.  READ command now reads in an etalk
 * script.  SENDFILE sends a file to remote, SET ASSOC sets a file suffix
 * association, SET DOWNLOAD sets the download path for transferred
 * files, SHOW ASSOC lists associations, SHOW JOS lists running
 * subprocesses.
 *
 * Revision 1.21  1995/09/29  08:26:02  zappo
 * Removed SET CLIENT command as being unnecessary, and now allow Xt
 * interface to print list of devices
 *
 * Revision 1.20  1995/09/22  13:42:09  zappo
 * Added codes needed to optionally have the X toolkit, or curses, or
 * both interfaces removed from the binary, as well as fixed more of the
 * SHOW STATUS command, and the SET RINGER command
 *
 * Revision 1.19  1995/09/20  23:25:02  zappo
 * Updated to use new structure elements
 *
 * Revision 1.18  1995/09/14  11:36:14  zappo
 * Added everything you need to handle the new curses interface with help
 * windows and the like
 *
 * Revision 1.17  1995/07/13  01:19:56  zappo
 * Changed a %s printf to a %d to fix bug
 *
 * Revision 1.16  1995/05/09  23:45:44  zappo
 * Added command to get copyright information
 *
 * Revision 1.15  1995/04/08  20:04:50  zappo
 * Replace some malloc/strcpy with strdup
 *
 * Revision 1.14  1995/04/01  16:21:45  zappo
 * Statisized a initialized variable.
 *
 * Revision 1.13  1995/03/30  02:37:02  zappo
 * Moved GTC_scmp --> GTL_scmp in gtl_util.c
 *
 * Revision 1.12  1995/03/25  03:30:41  zappo
 * Update copyright.
 *
 * Revision 1.11  1995/03/12  20:24:05  zappo
 * Fixed a help message
 *
 * Revision 1.10  1995/03/03  02:50:08  zappo
 * Added new APPLICATION command to set the name of the running
 * application, and also modified verbose command to accept a numeric
 * perameter.
 *
 * Revision 1.9  1995/02/28  03:35:38  zappo
 * Modified the show status, and read commands.  Status has more stuff,
 * read takes optional parameter to read in a specified file.
 *
 * Revision 1.8  1995/02/11  17:49:22  zappo
 * Modified quit to call the same procedure used by the sig handler.
 *
 * Revision 1.7  1995/02/01  03:35:58  zappo
 * Added a parameter to the ring activate command
 *
 * Revision 1.6  1995/01/29  14:23:29  zappo
 * Fixed some -Wall warnings
 *
 * Revision 1.5  1995/01/28  16:03:10  zappo
 * Worked on the ringer command some, then made SET and SHOW commands,
 * and moved many other featurers (verbose, name, editchar) under SET,
 * and displays (device, user, host) under the show command.
 *
 * Revision 1.4  1994/11/24  02:45:27  zappo
 * Added inclusion of sitecnfg.h which is where site configurable
 * parameters are now kept.
 *
 * Revision 1.3  1994/11/19  17:04:11  zappo
 * Added reply and query commands, and made blank command suggest help
 *
 * Revision 1.2  1994/11/16  23:13:10  zappo
 * Changed some comments, and help wordings. Also changed some verbosity
 * issues
 *
 * Revision 1.1  1994/08/29  23:26:16  zappo
 * Initial revision
 *
 * History:
 * eml Aug 10, 1994 Modified cmd_test to return error status.
 *
 * ::Header:: gtalkc.h
 */

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtproc.h"
#include "otalk.h"
#include "talk.h"
#include "gtalk.h"
#include "gtl_union.h"

#include "sitecnfg.h"

#define BUFSIZE 300

struct command_associate {
  char *command;		/* the command string                   */
  char *description;		/* verbose description of this command  */
#ifdef PROTOTYPES
  int (*parse)(struct TalkContext *Ctxt, char *cmdline);
#else
  int (*parse)();		/* the function to call                 */
#endif
  unsigned char socket_only;	/* flag: available for socket mode only */
};


/*
 * Function: GTC_c2str
 *
 * Takes a character and returns a string.  The string will place ^
 * for control characters.
 *
 * Returns:     static char * - A local static string value.
 * Parameters:  c - the character to convert.
 *
 * History:
 * eml Jul 12, 1994 Created 
 */
char *GTC_c2str(c)
     char c;
{
  static char cs[3] = { 0, 0, 0 };

  if(c < ' ')
    {
      cs[0] = '^';
      cs[1] = c + '@';
      return cs;
    }
  else if(c == 127)
    {
      cs[0] = '^';
      cs[1] = '?';
      return cs;
    }
  else
    {
      cs[0] = c;
      cs[1] = 0;
      return cs;
    }
}

/*
 * Function: GTC_findmatch
 *
 *   Locally defined function which matches MATCH into a list of
 * possibilityes and finds the best unique index, and returns that.
 *
 * Returns:     static int  - 
 * Parameters:  list  - Pointer to list of options
 *              num   - number of elts in list.
 *              match - Pointer to match string
 * History:
 * zappo   12/7/94    Created
 */
static int GTC_findmatch(list, num, match)
     char *list[];
     int   num;
     char *match;
{
  int i;
  int found = -1;

  for(i = 0; i < num; i++)
    {
      if(GTL_scmp(list[i], match, strlen(match)) == Success)
	{
	  if(found == -1)
	    found = i;
	  else
	    {
	      found = -2;	/* not unique */
	      break;
	    }
	}
    }
  return found;
}

/*
 * Function: GTC_call
 *
 * Initiate a call using the BSD talk daemons.
 * 
 * Parameters: Ctxt - context
 *             cmd  - end of command line
 * History:
 * 4/18/93
 */
static int GTC_call(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char buffer[100];
  char *idstr = NULL;
  int   id;
  char *user = NULL;		/* user name             */
  char *host;			/* host name             */
  char *tty;			/* tty of user           */

  if(!Ctxt->myname)
    {
      DISP_message(Ctxt, "You must use the NAME command before making a call.");
      return Fail;
    }

  if(cmd)
    {
      idstr = strtok(cmd, " ");
      user  = strtok(NULL, "!@");
    }
  if(!idstr || !user)
    {
      DISP_message(Ctxt, "Usage: CALL UID USER[<@ | !>HOSTNAME[ TTY]]");
      return Fail;
    }
  host  = strtok(NULL, " \t\n\r");
  id    = atoi(idstr);

  if(host) {
    tty = strtok(NULL, " \t\n\r");
  } else {
    tty = NULL;
  }
  
  sprintf(buffer, "\03Calling %s on %s id %d", user, host?host:"<null>", id);
  DISP_message(Ctxt, buffer);
  
  PROTOCOL_attach(Ctxt, id, user, host, tty);

  return Success;
}


/*
 * Function: GTC_connect
 *
 * Complete a call by connecting to a known socket.
 * 
 * Parameters: Ctxt - context
 *             cmd  - end of command line
 * History:
 * 4/18/93
 */
static int GTC_connect(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *idstr = NULL;		/* user id (name/id #) */
  int   id;
  char *sockstr = NULL;		/* socket number in ascii */
  int   sock;
  char *user = NULL;		/* user name             */
  char *host;			/* host name             */
  char *tty = NULL;		/* tty of user           */

  if(cmd)
    {
      idstr = strtok(cmd, " ");
      sockstr = strtok(NULL, " ");
      user  = strtok(NULL, "!@");
    }
  if(!idstr || !sockstr)
    {
      DISP_message(Ctxt, "Usage: CONNECT UID SOCK USER[[@]HOSTNAME[ TTY]]");
      return Fail;
    }
  id    = atoi(idstr);
  sock  = atoi(sockstr);
  host  = strtok(NULL, " \t\n\r");
  if(host) {
    tty = strtok(NULL, " \t\n\r");
  }

  if(verbose)
    {
      char buffer[100];
      
      sprintf(buffer, "\03Connecting %s on socket %d host %s id %d",
	      user, sock, host, id);
      DISP_message(Ctxt, buffer);
    }

  PROTOCOL_connect(Ctxt, id, sock, user, host, tty);

  return Success;
}

/*
 * Function: GTC_wait
 *
 * Wait for someone to connect to a know socket ID.
 * 
 * Parameters: Ctxt - context
 *             cmd  - end of command line
 * History:
 * 4/18/93
 */
static int GTC_wait(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *idstr = NULL;
  int   id;
  char *user = NULL;		/* user name             */
  char *host;			/* host name             */
  char *tty=NULL;		/* tty of user           */

  if(cmd)
    {
      idstr = strtok(cmd, " ");
      user  = strtok(NULL, "!@");
    }
  if(!idstr)
    {
      DISP_message(Ctxt, "Usage: WAIT UID USER[[@]HOSTNAME[ TTY]]");
      return Fail;
    }
  id    = atoi(idstr);
  host  = strtok(NULL, " \t\n\r");
  if(host) {
    tty = strtok(NULL, " \t\n\r");
  }

  /* This is non-verbose because nothing happens after this. */
  {
    char buffer[100];
    sprintf(buffer, "Waiting for %s on host %s id %d",
	    user, host, id);
    DISP_message(Ctxt, buffer);
  }

  PROTOCOL_wait(Ctxt, id, user, host, tty);

  return Success;
}


/*
 * Function: GTC_reply
 *
 *   Locally defined function which will query local daemon for name
 * of last caller, then reply to that answer.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Rest of the command line
 * History:
 * zappo   11/18/94   Created
 */
static int GTC_reply(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  PROTOCOL_reply(Ctxt);

  return Success;
}


/*
 * Function: GTC_ringer
 *
 *   Locally defined function which allows the user to toggle the
 * auto-ringer on and off.  Autoringer allows gtalk to intercept
 * announcements before they are sent to a terminal under the GNU talk
 * daemon protocol.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   11/23/94   Created
 */
static int GTC_ringer(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  static char *tog[] = { "OFF", "ON" };
  char *toks = NULL;
  int   togv = 0;

  if(cmd)
    toks = strtok(cmd, " \n\t\r");

  if(!toks)
    {
      /* Toggle whatever it is currently set to */
      if(Ctxt->ringerflag == TRUE)
	{
	  return RING_deactivate(Ctxt->myname);  
	}
      else
	{
	  return RING_activate(Ctxt->myname, &Ctxt->ringerflag, Ctxt->udp_ring, ETR_read);
	}
    }

  togv = GTC_findmatch(tog, sizeof(tog)/sizeof(char *), toks);
  
  if(togv < 0)
    {
      DISP_message(Ctxt, "Usage: SET RINGER < ON | OFF >");
      return Fail;
    }
  
  if(verbose)
    printf("Ringer set to %s\n", tog[togv]);

  if(Ctxt->ringerflag == togv)
    return Success;

  if(togv)
    {
      return RING_activate(Ctxt->myname, &Ctxt->ringerflag, Ctxt->udp_ring, ETR_read);
    }
  else
    {
      return RING_deactivate(Ctxt->myname);
    }
}
/*
 * Function: GTC_eightbit
 *
 *   Locally defined function which allows the user to toggle the
 * the use of eight-bit characters on and off.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   11/23/94   Created
 */
static int GTC_eightbit(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  static char *tog[] = { "OFF", "ON" };
  static char *msg[] = { "Meta commands", "displayed" };
  char *toks = NULL;
  int   togv = 0;

  if(cmd)
    toks = strtok(cmd, " \n\t\r");

  if(!toks)
    {
      /* Toggle whatever it is currently set to */
      if(Ctxt->eight_bit == FALSE)
	{
	  Ctxt->eight_bit = 1;
	}
      else
	{
	  Ctxt->eight_bit = 0;
	}
    } else {

      togv = GTC_findmatch(tog, sizeof(tog)/sizeof(char *), toks);
  
      if(togv < 0)
	{
	  DISP_message(Ctxt, "Usage: SET EIGHTBIT < ON | OFF >");
	  return Fail;
	}

      Ctxt->eight_bit = togv;
    }

  if(Ctxt->eight_bit)
    {
      DISP_message(Ctxt, "Eight bit characters will be displayed.\n");
    } else {
      DISP_message(Ctxt, "Eight bit characters will be META commands.\n");
    }

  return Success;
}


/*
 * Function: GTC_route
 *
 *   Apply or remove a new route.  This means that if we are to send a
 * request a given host that matches DESTINATION, then we will use ADDRESS
 * as the response andress.
 *
 * Returns:     static in - 
 * Parameters:  Ctxt - Context
 *              cmd  - Routing command text
 * History:
 * zappo   9/8/98     Created
 */
static int GTC_route(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *destination, *address;

  if(cmd)
    {
      destination = strtok(cmd, " ");
      address = strtok(NULL, " \t\n\r");
    }
  if(!destination || !address)
    {
      DISP_message
	(Ctxt, "Usage: ROUTE DESTINATION ADDRESS.");
      return Fail;
    }
  return ROUTE_update(Ctxt, destination, address);
}

/*
 * Function: GTC_showroute
 *
 *   Display a list of active routes
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - rest of the command line
 * History:
 * zappo   11/9/95    Created
 */
static int GTC_showroute(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char *cmd;
{
  if(ROUTE_number() == 0)
    {
      DISP_message(Ctxt, "There are no active routes. Use HELP");
    }
  else
    {
#ifndef NO_IFACE
      DISP_create_popup(Ctxt, 75, ROUTE_number() + 3);
#endif
      ROUTE_print(Ctxt);
    }

  return Success;
}


/*
 * Function: GTC_editc
 *
 * Read in three characters, (with a preceeding ^ indicating control)
 * and set the editcharacters for the system.  These characters are sent
 * first thing whenever a new connection is created.
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 4/14/94
 */
static int GTC_editc(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *s;
  int   curkey = 0;

  if(!cmd)
    {
      DISP_message(Ctxt, "Usage: SET EDITCHAR <del char><del line><del word>");
      return Fail;
    }

  if(USER_OnlyOne() == (struct UserObject *)0) {

    DISP_message(Ctxt, "You cannot change your edit characters when connected");

  } else {

    s = cmd;

    while(*s && (curkey < 3))
      {
	if(*s == '^')
	  {
	    s++;
	    if(*s == '?')
	      Ctxt->editkeys[curkey] = 127; /* delete is special somehow. */
	    else
	      Ctxt->editkeys[curkey] = *s - '@'; /* convert to control */
	  }
	else
	  Ctxt->editkeys[curkey] = *s;
	s++;
	curkey++;
      }
  }

  if(verbose)
    {
      printf("Edit characters set to Delchar(%s)",
	     GTC_c2str(Ctxt->editkeys[0]));
      printf(" DelLine(%s)",
	     GTC_c2str(Ctxt->editkeys[1]));
      printf(" DelWord(%s)\n", 
	     GTC_c2str(Ctxt->editkeys[2]));
    }
  return Success;
}

/*
 * Function: GTC_hangup
 *
 * Cleanly hangup on the userid specified on the command line
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 4/14/94
 */
static int GTC_hangup(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  struct UserObject *uo;
  char *ustr;

  if(!cmd)
    {
      DISP_message(Ctxt, "Usage: HANGUP <user id>");
      return Fail;
    }

  ustr = strtok(cmd, " \t\n\r");
  
  uo = USER_finduser(ustr);

  if(uo == NULL)
    {
      char buffer[100];

      sprintf(buffer, "HANGUP: id %s is an unknown user identifier.", ustr);
      DISP_message(Ctxt, buffer);
      return Fail;
    }

  USER_hangup(Ctxt, uo->id);

  DISP_update_user(Ctxt, uo);

  return Success;
}


/*
 * Function: GTC_filter
 *
 *   Locally defined function which initiates a filter of
 * outbound data for a given user.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - String Pointer  cmd
 * History:
 * zappo   5/18/96    Created
 */
static int GTC_filter(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *uid;
  struct UserObject *uo;
  char *type;
  int idx;

  if(!cmd)
    {
      DISP_message(Ctxt, "Usage: FILTER [ HELP | <UID> <TYPE> ]");
      return Fail;
    }

  uid = strtok(cmd, " \t\n\r");

  if(uid && GTL_scmp("HELP", uid, 4))
    {
      FILTER_describe(Ctxt);
      return Fail;
    }

  uo = USER_finduser(uid);

  if(uo == NULL)
    {
      char buffer[100];

      sprintf(buffer, "FILTER: Unknown user id %s", uid?uid:"<none>");
      DISP_message(Ctxt, buffer);
      return Fail;
    }

  type = strtok(NULL, " \t\n\r");
  
  if(!type)
    {
      FILTER_describe(Ctxt);
      return Fail;
    }

  idx = FILTER_index(type);

  if(idx == -1) {
      char buff[100];
      sprintf(buff, "\03Unsupported filter type %s.", type);
      DISP_message(Ctxt, buff);
      FILTER_describe(Ctxt);
      return Fail;
    } else {
      return FILTER_init(Ctxt, uo, idx);
    }
}

/*
 * Function: GTC_aux
 *
 *   Locally defined function which initiates the creation of an
 * auxiliry connection between the parent process (emacs) and the
 * parent process of a remote etalk session.  Requires and existing UID
 * to have been created.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - The rest of the command
 * History:
 * zappo   3/29/96    Created
 */
static int GTC_aux(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  struct UserObject *uo, *ruo;
  char *uid;
  char *request;

  if(!cmd)
    {
      DISP_message(Ctxt, "Usage: AUXILIRY <UID> <RUID> <REQUEST>");
      return Fail;
    }

  uid = strtok(cmd, " \t\n\r");
  
  uo = USER_finduser(uid);

  if(uo == NULL)
    {
      char buffer[100];

      sprintf(buffer, "AUXILIARY: Unknown user id %s", uid);
      DISP_message(Ctxt, buffer);
      return Fail;
    }

  uid = strtok(NULL, " \t\n\r");
  
  ruo = USER_finduser(uid);

  if(ruo == NULL)
    {
      char buffer[100];

      sprintf(buffer, "AUXILIARY: Unknown existing id %s", uid);
      DISP_message(Ctxt, buffer);
      return Fail;
    }

  request = strtok(NULL, "\n\r");

  if(!request)
    {
      DISP_message(Ctxt, "You cannot create an AUX port without a purpose!");
      return Fail;
    }
  
  /* Make tha aux link */
  return DATA_aux_connection(Ctxt, ruo, uo, request);
}

/*
 * Function: GTC_sendfile
 *
 *   Locally defined function which sends a file to the specified
 * remote user.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   11/7/95    Created
 */
static int GTC_sendfile(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *idstr = NULL;
  struct UserObject *uo;
  char *file = NULL;		/* file name to send */

  if(cmd)
    {
      idstr = strtok(cmd, " ");
      file  = strtok(NULL, " \n\r");
    }
  if(!idstr || !file)
    {
      DISP_message(Ctxt, "Usage: sendfile UID FILE");
      return Fail;
    }

  uo = USER_finduser(idstr);

  if(!uo || (uo->state != USER_CONNECTED))
    {
      DISP_message(Ctxt, "You can only send files to connected users.");
      return Fail;
    }

  if(verbose)
    printf("Attempting to send file %s to user %s\n", file, uo->name);

  return DATA_send_file(Ctxt, uo, file);
}

/*
 * Function: GTC_start
 *
 *   Locally defined function which will start a shared application
 * with a person specified in UID.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer to Character of cmd
 * History:
 * zappo   1/6/96     Created
 */
static int GTC_start(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *uid = NULL, *alias;
  struct UserObject *uo;

  if(cmd)
    uid = strtok(cmd, " \t\n\r");

  if(uid)
    uo = USER_finduser(uid);
  else
    {
      DISP_message(Ctxt, "Usage: START UID ALIAS_OF_SHARED_APP");
      return Fail;
    }
  if(!uo)
    {
      DISP_message(Ctxt, "Usage: Unknown or non-unique user specified");
      return Fail;
    }
  alias = strtok(NULL, "\n\r");

  if(!alias)
    {
      DISP_message(Ctxt, "Usage: START UID ALIAS_OF_SHARED_APP");
      return Fail;
    }

  SHAPP_fork_shared(Ctxt, alias, uo);

  return Success;
}


/*
 * Function: GTC_name
 *
 * Locally defined function which provides sets the user name, or the
 * name which is used to announce to other users
 *
 * Parameters:  Ctxt - Context
 *              cmd  - tail of command line
 * History:
 * eml	May 27, 1994	Created
 */
static int GTC_name(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *name = NULL;
  char buff[100];

  if(cmd)
    name = strtok(cmd, " \n\r");

  if(name)
    name = strdup(name);
  else
    DISP_message(Ctxt, "Usage: SET NAME <new login name>");

  if(Ctxt->myname) {
    sprintf(buff, "Replacing old name %s with %s", Ctxt->myname, name);
    DISP_message(Ctxt, buff);
    free(Ctxt->myname);
  } else {
    sprintf(buff, "Setting name to %s", name);
    DISP_message(Ctxt, buff);
  }

  Ctxt->myname = name;

  return Success;
}

/*
 * Function: GTC_setdownload
 *
 *   Locally defined function which sets the download path to a new string.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   11/7/95    Created
 */
static int GTC_setdownload(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *path;

  if(!cmd)
    {
      DISP_message(Ctxt, "Usage: SET DOWNPATH <new path to file storage>");
      return Fail;
    }

  path = strdup(strtok(cmd, " \n"));

  if(Ctxt->downloadpath) {
    if(verbose)
      printf("Replacing old path %s with ", Ctxt->downloadpath);
    free(Ctxt->downloadpath);
  } else {
    if(verbose)
      printf("Setting path to ");
  }

  Ctxt->downloadpath = path;
  if(verbose)
    printf("%s\n", path);

  return Success;
}

/*
 * Function: GTC_setaccept
 *
 *   Locally defined function which sets how new tcp connections are
 * recieved.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   11/28/95   Created
 */
static int GTC_setaccept(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  static char *methv[] = { "NEVER", "ALWAYS", "QUERY" };
  char *method = NULL;
  int   meth = -1;

  if(cmd)
    {
      method = strdup(strtok(cmd, " \n\r"));

      meth = GTC_findmatch(methv, sizeof(methv)/sizeof(char *), method);
    }

  if(meth < 0)
    {
      DISP_message(Ctxt, "Usage: SET ACCEPTTCP < NEVER | QUERY | ALWAYS>");
      return Fail;
    }

  if(verbose)
    printf("TCP accept method set to %s\n", methv[meth]);

  Ctxt->querynewtcp = meth;

  return Success;
}


/*
 * Function: GTC_setclean
 *
 *   Locally defined function lets the user define how clean up is managed
 *
 * Returns:     static int - 
 * Parameters:  Ctxt - Context
 *              cmd  - String Pointer  cmd
 * History:
 * zappo   2/22/97    Created
 */
static int GTC_setclean(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  static char *methv[] = { "MANUAL", "AUTO" };
  char *method = NULL;
  int   meth = -1;

  if(cmd)
    {
      method = strdup(strtok(cmd, " \n\r"));

      meth = GTC_findmatch(methv, sizeof(methv)/sizeof(char *), method);
    }

  if(meth < 0)
    {
      DISP_message(Ctxt, "Usage: SET CLEANUP < MANUAL | AUTO >");
      return Fail;
    }

  if(verbose)
    printf("Cleanup method set to %s\n", methv[meth]);

  Ctxt->cleanup = meth;

  return Success;
}

/*
 * Function: GTC_app
 *
 *   Locally defined function which sets the application name for this
 * session of gtalk.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer to Character of cmd
 * History:
 * zappo   2/28/95    Created
 */
static int GTC_app(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *name;

  if(!cmd)
    {
      DISP_message(Ctxt, "Usage: SET APPLICTION <this application name>");
      return Fail;
    }

  name = strdup(strtok(cmd, " "));

  if(Ctxt->myappname) {
    if(verbose)
      printf("Replacing old application name %s with ", Ctxt->myappname);
    free(Ctxt->myappname);
  } else {
    if(verbose)
      printf("Setting application name to ");
  }

  Ctxt->myappname = name;
  if(verbose)
    printf("%s\n", name);

  return Success;
}

/*
 * Function: GTC_setassoc
 *
 *   Locally defined function which sets an associate between a suffix
 * and an application.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - rest of command line
 * History:
 * zappo   11/9/95    Created
 */
static int GTC_setassoc(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *suffix = NULL;
  char *app = NULL;

  if(cmd)
    {
      suffix = strtok(cmd, " \t\n\r");
      app = strtok(NULL, "\n\r");
    }

  if(!suffix || !app)
    {
      DISP_message(Ctxt, "Usage: SET ASSOCIATION <suffix> <application>");
      return Fail;
    }
  /* Add the suffix to the list */
  ASSOC_add(Ctxt, suffix, app);

  return Success;
}

/*
 * Function: GTC_defaultfilter
 *
 *   Locally defined function which initiates a filter of
 * outbound data for a given user.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - String Pointer  cmd
 * History:
 * zappo   10/15/97   Created
 */
static int GTC_defaultfilter(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *type;
  int   idx;

  if(!cmd)
    {
      DISP_message(Ctxt, "Usage: SET FILTERDEFAULT [ HELP | <TYPE> ]");
      return Fail;
    }

  type = strtok(cmd, " \t\n\r");
  
  if(!type || GTL_scmp("HELP", type, 4))
    {
      FILTER_describe(Ctxt);
      return Fail;
    }

  idx = FILTER_index(type);

  if(idx == -1)
    {
      char buff[100];
      sprintf(buff, "\03Unsupported filter type %s.", type);
      DISP_message(Ctxt, buff);
      return Fail;
    } else {
      char buff[100];

      Ctxt->default_filter = idx;

      sprintf(buff, "\03Default Filter type changed to %s",
	      FILTER_name(idx));
      DISP_message(Ctxt, buff);

      return Success;
    }
}

/*
 * Function: GTC_showassoc
 *
 *   Locally defined function which displays a list of active associations
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - rest of the command line
 * History:
 * zappo   11/9/95    Created
 */
static int GTC_showassoc(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char *cmd;
{
  if(ASSOC_number() == 0)
    {
      DISP_message(Ctxt, "There are no active file associations. Use SET HELP");
    }
  else
    {
#ifndef NO_IFACE
    DISP_create_popup(Ctxt, 75, ASSOC_number() + 3);
#endif
    ASSOC_print(Ctxt);
    }

  return Success;
}

/*
 * Function: GTC_setshapp
 *
 *   Locally defined function which reads in one shared app command line
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - rest of command line
 * History:
 * zappo   12/21/95   Created
 */
static int GTC_setshapp(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  static char *proxyname[] = { "NONE", "TO_ONE", "TO_N" };
  static char *dispreqname[] = { "NONE", "KEYBOARD", "TTY", "X" };
  char *alias = NULL, *tmp, *command;
  int proxy=0, dispreq=0;

  /* Load alias */
  if(cmd)
    alias = strtok(cmd, " \t\n\r");
  if(!alias)
    {
      DISP_message(Ctxt, "Usage: SET SHARED <alias> <proxy> <dispreq> <commandline>");
      return Fail;
    }
  /* Load proxy */
  tmp = strtok(NULL, " \t\n\r");
  if(tmp)
    {
      proxy = GTC_findmatch(proxyname, sizeof(proxyname)/sizeof(char *), tmp);
      if(proxy < 0)
	{
	  DISP_message(Ctxt, "PROXY is one of NONE, TO_ONE, TO_N");
	  return Fail;
	}
    }
  /* Load display requirements */
  tmp = strtok(NULL, " \t\n\r");
  if(tmp)
    {
      dispreq = GTC_findmatch(dispreqname, sizeof(dispreqname)/sizeof(char *), tmp);
      if(dispreq < 0)
	{
	  DISP_message(Ctxt, "DISPLAY REQUIREMENT is one of NONE, KEYBOARD, TTY, X");
	  return Fail;
	}
    }
  /* Load command line */
  command = strtok(NULL, "\n\r");
  if(!command)
    {
      DISP_message(Ctxt, "Usage: SET SHARED <alias> <proxy> <dispreq> <commandline>");
      return Fail;
    }

  /* Add the suffix to the list */
  SHAPP_add(Ctxt, alias, command, proxy, dispreq);

  return Success;
}

/*
 * Function: GTC_showshapp
 *
 *   Locally defined function which prints out the list of shared
 * applications currently known.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   12/21/95   Created
 */
static int GTC_showshapp(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char *cmd;
{
  if(SHAPP_number() == 0)
    {
      DISP_message(Ctxt, "There are no available shared apps.  Use SET HELP");
    }
  else
    {
#ifndef NO_IFACE
      DISP_create_popup(Ctxt, 75, SHAPP_number() + 3);
#endif
      SHAPP_print(Ctxt);
    }

  return Success;
}

/*
 * Function: GTC_showjobs
 *
 *   Locally defined function which will display a list of active jobs
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   11/19/95   Created
 */
static int GTC_showjobs(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char *cmd;
{
  if(FORK_pid_number() == 0)
    {
      DISP_message(Ctxt, "There are no jobs currently running.");
    }
  else
    {
#ifndef NO_IFACE
      DISP_create_popup(Ctxt, 75, FORK_pid_number() + 3);
#endif
      FORK_pid_print(Ctxt);
    }

  return Success;
}


/*
 * Function: GTC_clean
 *
 * Call a clean routine for users structures and devices to free up
 * any dead structures.  Mabee link to garbage collecting or something
 * equally goofy later.
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 4/15/94
 */
static int GTC_clean(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  GT_clean();			/* clean up devices. */
#ifndef NO_IFACE
  DISP_clean();			/* clean up windows  */
#endif
  /* DISP must be cleaned before users are cleaned, or pointers are hosed! */
  USER_clean();			/* clean up users    */

  if(verbose)
    printf("Cleaning complete.\n");

  return Success;
}

/*
 * Function: GTC_abort
 *
 * Abort any currently active call outwards (waits or otherwise)
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 4/19/94
 */
static int GTC_abort(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  /* shut off any active call
   */
  PROTOCOL_abort(Ctxt);

  return Success;
}
/*
 * Function: GTC_users
 *
 * Print a list of all users.
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 4/14/94
 */
static int GTC_users(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
#ifndef NO_IFACE
  if(USER_number())
    DISP_create_popup(Ctxt, 75, USER_number() + 3);
#endif
  USER_print(Ctxt);

  return Success;
}
/*
 * Function: GTC_host
 *
 * Print a list of all hosts.
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 4/21/94
 */
static int GTC_host(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  /* There should never be 0 hosts */
#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 75, HOST_number() + 3);
#endif
  HOST_print();

  return Success;
}

/*
 * Function: GTC_status
 *
 * Locally defined function which displays compile time, and run time
 * statistics about the program.
 *
 * Parameters:  Ctxt - Context of program
 *              cmd  - command line
 * History:
 * eml	Jul 11, 1994	Created
 * zappo   2/27/95    Added some new things which needed to be displayed
 */
static int GTC_status(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  static char *onoff[] = { "OFF", "ON" };
  static char *qmeth[] = { "NEVER", "ALWAYS", "QUERY" };
  static char *cmeth[] = { "MANUAL", "AUTO" };
  char buffer[100];

#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 75, 16);
#endif
  DISP_message(Ctxt, "Compile time parameters:");
#if (TALKDTEST == 1) && (OTALK_ONLY == 1)
  DISP_message(Ctxt, "TALKDTEST    : TRUE\t\t\tOTALK_ONLY   : TRUE");
#else
# if (TALKDTEST == 0) && (OTALK_ONLY == 1)
  DISP_message(Ctxt, "TALKDTEST    : FALSE\t\t\tOTALK_ONLY   : TRUE");
# else
#  if (TALKDTEST == 1) && (OTALK_ONLY == 0)
  DISP_message(Ctxt, "TALKDTEST    : TRUE\t\t\tOTALK_ONLY   : FALSE");
#  else
  DISP_message(Ctxt, "TALKDTEST    : FALSE\t\t\tOTALK_ONLY   : FALSE");
#  endif
# endif
#endif

#if defined(SYSTEM_RC) && defined(RINGER_RC)
  sprintf(buffer, "SYSTEM_RC    : %s\t\tRINGER_RC    : %s",
	  SYSTEM_RC, RINGER_RC);
#else
# if defined(SYSTEM_RC)
  sprintf(buffer, "SYSTEM_RC    : %s\t\tRINGER_RC    : NONE",
	  SYSTEM_RC);
# else
#  if defined(RINGER_RC)
  sprintf(buffer, "SYSTEM_RC    : NONE\t\tRINGER_RC    : %s",
	  RINGER_RC);
#  else
  sprintf(buffer, "SYSTEM_RC    : NONE\t\tRINGER_RC    : NONE");
#  endif
# endif
#endif
  DISP_message(Ctxt, buffer);

#if defined(LOCAL_RC) && defined (LOCAL_RC_ENV)
  sprintf(buffer, "LOCAL_RC     : %s\t\tLOCAL_RC_ENV : %s = %s",
	  LOCAL_RC, LOCAL_RC_ENV, 
	  getenv(LOCAL_RC_ENV)?getenv(LOCAL_RC_ENV):"<none>");
#else
# if defined(LOCAL_RC)
  sprintf(buffer, "LOCAL_RC     : %s\t\tLOCAL_RC_ENV : NONE", LOCAL_RC);
# else
#  if defined(LOCAL_RC_ENV)
  sprintf(buffer, "LOCAL_RC     : NONE\t\tLOCAL_RC_ENV : %s = %s",
	  LOCAL_RC_ENV, getenv(LOCAL_RC_ENV)?getenv(LOCAL_RC_ENV):"<none>");
#  else
  sprintf(buffer, "LOCAL_RC     : NONE\t\tLOCAL_RC_ENV : NONE");
#  endif
# endif
#endif
  DISP_message(Ctxt, buffer);

#if defined(DEFAULT_PAGER) && defined(DEFAULT_PAGER_ENV)
  sprintf(buffer, "DEFAULT_PAGER: %s\t\t\tPAGER_ENV    : %s = %s",
	  DEFAULT_PAGER, DEFAULT_PAGER_ENV,
	  getenv(DEFAULT_PAGER_ENV)?getenv(DEFAULT_PAGER_ENV):"<none>");
#else
# if defined(DEFAULT_PAGER)
  sprintf(buffer, "DEFAULT_PAGER: %s\t\t\tPAGER_ENV    : NONE",
	  DEFAULT_PAGER);
# else
#  if defined(DEFAULT_PAGER_ENV)
  sprintf(buffer, "DEFAULT_PAGER: NONE\t\t\tPAGER_ENV    : %s = %s",
	  DEFAULT_PAGER_ENV,
	  getenv(DEFAULT_PAGER_ENV)?getenv(DEFAULT_PAGER_ENV):"<none>");
#  else
  sprintf(buffer, "DEFAULT_PAGER: NONE\t\t\tPAGER_ENV    : NONE");
#  endif
# endif
#endif
  DISP_message(Ctxt, buffer);

#ifdef X_TOOLKIT
#ifdef USE_CURSES
  sprintf(buffer, "INTERFACES   : SOCKET %s %s", USE_CURSES, X_TOOLKIT);
#else /* USE_CURSES */
  sprintf(buffer, "INTERFACES   : SOCKET %s", X_TOOLKIT);
#endif /* USE_CURSES */
#else /* X_TOOLKIT */
#ifdef USE_CURSES
  sprintf(buffer, "INTERFACES   : SOCKET %s", USE_CURSES);
#else /* USE_CURSES */
  sprintf(buffer, "INTERFACES   : SOCKET");
#endif /* USE_CURSES */
#endif /* X_TOOLKIT */
  DISP_message(Ctxt, buffer);

#ifdef DO_USERS_REALLY_CARE_ABOUT_THESE_THINGS
  DISP_message(Ctxt, 
	       "Compile time message sizes header(Cntrl/Respns) sizeof(Cntrl/Respns):");
  sprintf(buffer, "OTALK        : %d/%d  %d/%d\t\tNTALK        : %d/%d  %d/%d",
	  OTALK_CM_SIZE, OTALK_CMR_SIZE, sizeof(CTL_MSG_OLD), sizeof(CTL_RESPONSE_OLD),
	  NTALK_CM_SIZE, NTALK_CMR_SIZE, sizeof(CTL_MSG), sizeof(CTL_RESPONSE)
	  );
  DISP_message(Ctxt, buffer);
  sprintf(buffer, "GTALK        : %d/%d  %d/%d", GTALK_CM_SIZE, GTALK_CMR_SIZE,
	  sizeof(CTL_MSG_GNU), sizeof(CTL_RESPONSE_GNU));
  DISP_message(Ctxt, buffer);
#endif

  DISP_message(Ctxt, "Run time parameters:");
  sprintf(buffer, "Local Host   : %s Talk Version=%d", Ctxt->me->name, Ctxt->me->type);
  DISP_message(Ctxt, buffer);
  sprintf(buffer, "Local Name   : %s\t\tApp name     : %s", 
	  Ctxt->myname?Ctxt->myname:"Undefined",
	  Ctxt->myappname?Ctxt->myappname:"None");
  DISP_message(Ctxt, buffer);
  sprintf(buffer, "Accept Method: %s\t\tVerbosity    : %d",
	  qmeth[Ctxt->querynewtcp], verbose);
  DISP_message(Ctxt, buffer);
  sprintf(buffer, "Ringer       : %s\t\tPID          : %ld", 
	  onoff[Ctxt->ringerflag], (long)Ctxt->pid);
  DISP_message(Ctxt, buffer);
  sprintf(buffer, "Clean Method : %s\t\tCall state   : %s", 
	  cmeth[Ctxt->cleanup], PROTOCOL_status(Ctxt));
  DISP_message(Ctxt, buffer);
  sprintf(buffer, "Deflt Filter : %s\t\tDownload Path: %s",
	  FILTER_name(Ctxt->default_filter),
	  Ctxt->downloadpath?Ctxt->downloadpath:"/tmp");
  DISP_message(Ctxt, buffer);
  sprintf(buffer, "Editkeys     : CHAR(%s)", GTC_c2str(Ctxt->editkeys[0]));
  strcat(buffer, " LINE(");
  strcat(buffer, GTC_c2str(Ctxt->editkeys[1]));
  strcat(buffer, ") WORD(");
  strcat(buffer, GTC_c2str(Ctxt->editkeys[2]));
  strcat(buffer, ")");
  DISP_message(Ctxt, buffer);

  return Success;
}

/*
 * Function: GTC_copyright
 *
 *   Locally defined function which will display copyright information.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Rest of command line
 * History:
 * zappo   5/9/95     Created
 */
static int GTC_copyright(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  DISP_message(Ctxt, GTALK_);
#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 75, 23);
#endif
  CW_display_copywrite();

  return Success;
}

/*
 * Function: GTC_device
 *
 * Print out a list of all devices.
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 4/15/94
 */
static int GTC_device(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  /* There should never be 0 active devices. */
#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 75, GT_dev_num() + 3);
#endif
  GT_print_q_list();

  return Success;
}
/*
 * Function: GTC_quit
 *
 * Quit gtalk binary
 * 
 * Parameters: Ctxt - context
 *             cmd - tail of command line
 * History:
 * eml 3/1/94
 */
static int GTC_quit(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  /* shut off protocol things before closing sockets. ;)
   */
  gtalk_shutdown(NULL);
  return 0;
}

/*
 * Function: GTC_load
 *
 * Locally defined function which re-reads the system RC files for
 * hosts and their daemon types.
 *
 * Parameters:  Ctxt - Context of talk program
 *              cmd  - command line
 * History:
 * eml	Jul 11, 1994	Created
 */
static int GTC_load(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *newf = NULL;

  if(cmd)
    newf = strtok(cmd, " \t\n");

  if(newf && newf[0])
    RC_load_file(Ctxt, newf);
  else
    RC_load_all_hosts(Ctxt);

  return Success;
}

/*
 * Function: GTC_read
 *
 *   Locally defined function which reads from the script specified on
 * the command line.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   11/19/95   Created
 */
static int GTC_read(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *newf=NULL;

  if(cmd)
    newf = strtok(cmd, " \t\n");

  if(newf && newf[0])
    return RC_read_script(Ctxt, newf);
  else
    return RC_read_script(Ctxt, NULL);
}

/*
 * Function: GTC_verbose
 *
 * Locally defined function which toggles the verbosity variable.
 *
 * Parameters:  Ctxt - Context of talk program
 *              cmd  - the command line
 * History:
 * eml	Jul 11, 1994	Created
 */
static int GTC_verbose(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *val = NULL;

  if(Ctxt->runstate == Curses)
    {
      DISP_message(Ctxt, "Cannot change verbosity in CURSES interface.");
      return Success;
    }

  if(cmd)
    val = strtok(cmd, " \n\t");

  if(!val)
    {
      /* Toggle the verbosity
       */
      verbose = !verbose;
    }
  else
    {
      /* we have specific setting. */
      verbose = atoi(val);
    }
      
  if(verbose)
    {
      char buff[100];
      sprintf(buff, "Verbosity is now set to %d.", verbose);
      DISP_message(Ctxt, buff);
    }
  else
    DISP_message(Ctxt, "Verbosity is now off.");

  return Success;
}

/*
 * Function: GTC_test
 *
 * Tests the active gtalk system as best it can.
 *
 * Parameters:  Ctxt - Context
 *              cmd  -  tail of command line
 * History:
 * eml	May 27, 1994  Created
 * eml  Aug 10, 1994  Returns error code on failure.
 * zappo   9/24/94    Removed it's staticity for command line args.
 */
int GTC_test(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *host = NULL;

  if(verbose ||
     (cmd && (strlen(cmd) > 0) && 
      (GTL_scmp(cmd, "VERBOSE", strlen(cmd)) == Success)))
    {
      if(verbose == FALSE)
	{
	  verbose = 0;
	  GTC_verbose(Ctxt, "");
	}
      GTC_status(Ctxt, "");
    } else {
      if (cmd && *cmd != ' ')
	{
	  host = cmd;
	}
    }

  if(PROTOCOL_test(Ctxt, host) != Success)
    {
      /* Return error code for makefile */
      return Fail;
    }
  else
    {
      /* Quit peacefully */
      return Success;
    }
}

/*
 * Function: GTC_version
 *
 * Locally defined function which prints out the GTALK version string
 *
 * Parameters:  Ctxt - Context of the program
 *              cmd  - the rest command line
 * History:
 * eml	Jul 11, 1994	Created
 */
static int GTC_version(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  /* print out the current version string
   */
  DISP_message(Ctxt, GTALK_);

  return Success;
}
/*
 * Echo back the command line.
 */
static int GTC_echo(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  if(cmd)
    DISP_message(Ctxt, cmd);
  else
    DISP_message(Ctxt, "No parameters to echo");
  return Success;
}

#ifdef TALKDTEST
/*
 * Function: GTC_delete, GTC_leave, GTC_lookup, GTC_announce
 * 
 * Each of these function test one section of the talk protocol by
 * sending one message to the destined talk daemon, and printing the
 * response message to the minibuffer.
 * 
 * Parameters: Ctxt - conext
 *             cmd - tail of the command line.
 * History:
 * eml 3/18/94
 */
static int GTC_delete(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  static char *deltype[] = { "INVITE", "ANNOUNCE" };
  char *id;			/* id type name          */
  char *host;			/* host name             */
  enum DMN_deletion_type t;	/* deletion type         */
  struct InputDevice *io;	/* io of thier daemon    */

  if(!Ctxt->myname)
    {
      DISP_message(Ctxt, "You must use the NAME command before deleting things.");
      return;
    }

  if(!cmd)
    {
      DISP_message(Ctxt, "usage: DELETE <ANNOUNCE | INVITE> [hostname]");
      return Fail;
    }

  id = strtok(cmd, " ");
  host = strtok(NULL, " \t\n\r");

  if(!id) {
    id = cmd;
  }

  switch(GTC_findmatch(deltype, 2, id))
    {
    case 0:
      t = DMN_invite;
      break;
    case 1:
      t = DMN_announce;
      break;
    default:
      DISP_message(Ctxt, "usage: DELETE <ANNOUNCE | INVITE> [hostname]");
      return Fail;
    }

  if(Ctxt->runstate == Socket)
    printf("Deleting id %s from %s\n", id, host);

  /*
   * If no host, then make sure that we use local host 127.0.0
   */
  if(host) {
    io = UDP_host(host);
    if(io == NULL) return Fail;
  }
  else
    io = Ctxt->local_daemon;

  if(Ctxt->runstate == Socket)
    {
      printf("IO device addr: ");
      print_sockaddr((struct sockaddr *)&io->raddr);
      printf("\n");
    }

  if(! DMN_Delete(Ctxt, io, t))
    {
      printf("Error, continuation action not taken.\n");
      return Fail;
    }

  /* Info is printed already if verbosity is on.
   */
  if(! verbose && (Ctxt->runstate == Socket))
    ETM_control_print(io->host->type);

  io->readme = (void *)DMN_get_and_display;

  return Success;
}
static int GTC_leave(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *user = NULL;		/* user name             */
  char *host;			/* host name             */
  char *tty;			/* tty of user           */
  struct InputDevice *io;	/* io of thier daemon    */

  if(!Ctxt->myname)
    {
      DISP_message(Ctxt, "You must use the NAME command before leaving an invite.");
      return Fail;
    }

  if(!cmd)
    {
      user = strtok(cmd, "!@");
      host = strtok(NULL, " \t\n");
      if(!host) {
	user  = strtok(cmd, " ");
      }
      tty = strtok(NULL, " \t\n");

      if(!tty) tty = "";
    }

  if(!user) {
    DISP_message(Ctxt, "usage: LEAVE user[[@]hostname][ TTY]");
    return Fail;
  }

  if(Ctxt->runstate == Socket)
    printf("leaving an invite for %s on %s\n", user, host);

  /*
   * If no host, then make sure that we use local host 127.0.0
   */
  if(host) {
    io = UDP_host(host);
    if(io == NULL) return Fail;
  }
  else
    io = Ctxt->local_daemon;

  if(Ctxt->runstate == Socket)
    {
      printf("IO device addr: ");
      print_sockaddr((struct sockaddr *)&io->raddr);
      printf("\n");
    }

  if(! DMN_LeaveInvite(Ctxt, user, tty))
    {
      DISP_message(Ctxt, "Error, continuation action not taken.");
      return Fail;
    }

  if(! verbose && (Ctxt->runstate == Socket))
    ETM_control_print(io->host->type);

  io->readme = (void *)DMN_get_and_display;

  return Success;
}
static int GTC_lookup(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *user = NULL;		/* user name             */
  char *host;			/* host name             */
  char *tty;			/* tty of user           */
  struct InputDevice *io;	/* io of thier daemon    */

  if(!Ctxt->myname)
    {
      DISP_message(Ctxt, "You must use the NAME command before requesting lookup.");
      return Fail;
    }

  if(cmd)
    {
      user = strtok(cmd, "!@");
      host = strtok(NULL, " \t\n\r");
      if(!host) {
	user = strtok(cmd, " ");
      }
      tty = strtok(NULL, " \t\n\r");

      if(!tty) tty = "";
    }

  if(!user) {
    DISP_message(Ctxt, "usage: LOOKUP user[[@]hostname][ TTY]");
    return Fail;
  }

  if(Ctxt->runstate == Socket)
    printf("Looking for invite from %s on %s\n", user, host);

  /*
   * If no host, then make sure that we use local host 127.0.0
   */
  if(host) {
    io = UDP_host(host);
    if(io == NULL) return Fail;
  }
  else
    io = Ctxt->local_daemon;

  if(Ctxt->runstate == Socket)
    {
      printf("IO device addr: ");
      print_sockaddr((struct sockaddr *)&io->raddr);
      printf("\n");
    }

  if(! DMN_Lookup(Ctxt, io, user, tty))
    {
      printf("Error, continuation action not taken.\n");
      return Fail;
    }

  if(! verbose && (Ctxt->runstate == Socket))
    ETM_control_print(io->host->type);

  io->readme = (void *)DMN_get_and_display;

  return Success;
}
static int GTC_announce(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char *user = NULL;		/* user name             */
  char *host;			/* host name             */
  char *tty;			/* tty of user           */
  struct InputDevice *io;	/* io of thier daemon    */

  if(!Ctxt->myname)
    {
      DISP_message(Ctxt, "You must use the NAME command before making announcing.");
      return Fail;
    }

  if(cmd)
    {
      user = strtok(cmd, "!@");
      host = strtok(NULL, " \t\n\r");
      if(!host) {
	user = strtok(cmd, " ");
      }
      tty = strtok(NULL, " \t\n\r");

      if(!tty) tty = "";
    }

  if(!user) {
    DISP_message(Ctxt, "usage: ANNOUNCE user[[@]hostname][ TTY]");
    return Fail;
  }

  if(Ctxt->runstate == Socket)
    printf("Announcing to %s on %s\n", user, host);

  /*
   * If no host, then make sure that we use local host 127.0.0
   */
  if(host) {
    io = UDP_host(host);
    if(io == NULL) return Fail;
  }
  else
    io = Ctxt->local_daemon;

  if(Ctxt->runstate == Socket)
    {
      printf("IO device addr: ");
      print_sockaddr((struct sockaddr *)&io->raddr);
      printf("\n");
    }

  if(! DMN_Announce(Ctxt, io, user, tty))
    {
      printf("Error, continuation action not taken.");
      return Fail;
    }

  if(! verbose && (Ctxt->runstate == Socket))
    ETM_control_print(io->host->type);

  io->readme = (void *)DMN_get_and_display;

  return Success;
}
static int GTC_query(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  struct InputDevice *io;	/* io of thier daemon    */

  if(!Ctxt->myname)
    {
      DISP_message(Ctxt, "You must use the NAME command before making announcing.");
      return Fail;
    }

  io = Ctxt->local_daemon;

  if(! DMN_Reply_Query(Ctxt, io))
    {
      DISP_message(Ctxt, "Error, continuation action not taken.");
      return Fail;
    }

  if(! verbose)
    ETM_control_print(io->host->type);

  io->readme = (void *)DMN_get_and_display;

  return Success;
}
#endif /* TALKDTEST */


/*
 * Following all the bottom level user commands are the variables
 * and structures using the commands.
 *
 * The help commands must be forward referenced for the
 * command definitions
 */
static int GTC_help();
static int GTC_sethelp();
static int GTC_showhelp();

static struct command_associate SetCommands[] =
{
  { "ACCEPTTCP", "set Accepttcp < NEVER | QUERY | ALWAYS >",      GTC_setaccept, 0 },
  { "APPLICATION", "set Application <this application name>",     GTC_app, 0 },
  { "ASSOCIATION", "set Association <suffix> <program>",          GTC_setassoc, 0 },
  { "CLEANUP",  "set Cleanup < Manual | Auto >",                  GTC_setclean, 0 },
  { "DOWNPATH", "set Downpath <path> to set directory for downloads.", GTC_setdownload, 0 },
  { "EDITCHAR", "set Editchar 123 to set the edit characters",    GTC_editc, 0 },
  { "EIGHTBIT", "set 8-bit characters enabled (disallows META)",  GTC_eightbit, 0 },
  { "FILTERDEFAULT", "set filterdefault <filtertype> for default filter method.", GTC_defaultfilter },
  { "HELP",     "These help messages",                            GTC_sethelp, 0 },
  { "NAME",	"set Name MYANNOUNCENAME to set your used name",  GTC_name, 0 },
  { "RINGER",   "set Ringer < ON | OFF > to set ringer status",   GTC_ringer, 0 },
  { "SHARED",   "set shared alias < proxy > < dispreq > command **", GTC_setshapp, 0 },
  { "VERBOSE",  "set Verbose [value] Toggle or set verbosity.",   GTC_verbose, 0 },
};


/*
 * Function: GTC_set
 *
 *   Locally defined function which is called when the SET command is
 * given.  This is then called to SET the desired value.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer to Character of cmd
 * History:
 * zappo   1/28/95    Created
 */
static int GTC_set(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char buffer[100];
  char *s = cmd;
  int   i, found = -1;

  if(cmd && (strlen(cmd) < 1))
    {
      DISP_message(Ctxt,"Type SET HELP for help.");
      return Fail;;
    }

  /* Nuke white space before set command */
  while(*cmd && (*cmd == ' ')) cmd++;

  /* find the end of the key word. */
  s = cmd;
  while(*s && (*s != ' ')) s++;

  /*
   * Now that the set command has been parsed, lets find the string
   * we have matched (case insensitive) and execute its function.
   */
  if(s - cmd)
    {
      for(i = 0; i < (sizeof(SetCommands) / sizeof(struct command_associate)); i++)
	{
	  /*
	   * case insensitive comparison of beginning of string.  Allow
	   * shortest version of word, but not shortest unique.  Close enough
	   * for something to be interfaced into some other program.
	   */
	  if(GTL_scmp(SetCommands[i].command, cmd, s - cmd) == Success)
	    {
	      if(found == -1)
		found = i;
	      else
		{
		  found = -2;
		  break;
		}
	    }
	}
      if(found == -1)
	{
	  sprintf(buffer, "Invalid SET command %s", cmd);
	  DISP_message(Ctxt, buffer);
	}
      else if(found == -2)
	{
	  sprintf(buffer, "Command SET %s is not unique.", cmd);
	  DISP_message(Ctxt, buffer);
	}
      else
	if(!SetCommands[found].parse(Ctxt, *s?++s:NULL) && verbose)
	  DISP_message(Ctxt, "That command did not succeed.");
    }
  return Success;;
}

static struct command_associate ShowCommands[] =
{
  { "ASSOCIATIONS", "Display list of suffix/app associations",GTC_showassoc, 0 },
  { "COPYRIGHT","Display the copyright for gtalk.",           GTC_copyright, 0 },
  { "DEVICES",  "Display a list of all active IO devices",    GTC_device, 0 },
  { "HELP",     "These help messages",                        GTC_showhelp, 0 },
  { "HOSTS",    "Display a list of all hosts we have accessed", GTC_host, 0 },
  { "JOBS",     "Display list of active subprocesses and data", GTC_showjobs, 0 },
  { "ROUTE",    "Display list of route filters",              GTC_showroute, 1 },
  { "SHARED",   "Display list of shared application definitions", GTC_showshapp, 0 },
  { "STATUS",   "Display compile time and runtime statistics.", GTC_status, 0 },
  { "USERS",    "Display list of all users gtalk is manageing.", GTC_users, 0 },
  { "VERSION",  "Display the version of gtalk running.",      GTC_version, 0 },
};

static int GTC_show(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char  buffer[100];
  char *s = cmd;
  int   i, found = -1;

  if(cmd && (strlen(cmd) < 1))
    {
      DISP_message(Ctxt, "Type SHOW HELP for help.");
      return Fail;
    }

  /* Nuke white space before set command */
  while(*cmd && (*cmd == ' ')) cmd++;

  /* find the end of the key word. */
  s = cmd;
  while(*s && (*s != ' ')) s++;

  /*
   * Now that the show command has been parsed, lets find the string
   * we have matched (case insensitive) and execute its function.
   */
  if(s - cmd)
    {
      for(i = 0; i < (sizeof(ShowCommands) / sizeof(struct command_associate)); i++)
	{
	  /*
	   * case insensitive comparison of beginning of string.  Allow
	   * shortest version of word, but not shortest unique.  Close enough
	   * for something to be interfaced into some other program.
	   */
	  if(GTL_scmp(ShowCommands[i].command, cmd, s - cmd) == Success)
	    {
	      if(found == -1)
		found = i;
	      else
		{
		  found = -2;
		  break;
		}
	    }
	}
      if(found == -1)
	{
	  sprintf(buffer, "Invalid SHOW command %s", cmd);
	  DISP_message(Ctxt, buffer);
	}
      else if(found == -2)
	{
	  sprintf(buffer, "Command SHOW %s is not unique.", cmd);
	  DISP_message(Ctxt, buffer);
	}
      else
	if(!ShowCommands[found].parse(Ctxt, *s?++s:NULL) && verbose)
	  DISP_message(Ctxt, "That command did not succeed.");
    }
  return Success;;
}

static struct command_associate Commands[] =
{
  { "ABORT",    "Abort any currently active call",            GTC_abort, 0 },
#ifdef TALKDTEST
  { "ANNOUNCE", "Announce USER[@MACHINE] to send a test announce message", GTC_announce, 1 },
#endif
  { "AUXILIARY","Auxiliary UID RUID REQ to create aux port for emacs", GTC_aux, 1 },
  { "CALL",     "Call UID USER[<@ | !>MACHINE][ TTY] to make connection", GTC_call, 1 },
  { "CLEAN",    "Clean all users and devs marked DEAD",       GTC_clean, 0 },
  { "CONNECT",  "Connect UID SOCKET USER[@MACHINE][ TTY] to talk someone", GTC_connect, 1 },
#ifdef TALKDTEST
  { "DELETE",   "Delete <ANNOUNCE | INVITE> [host] to delete id from host", GTC_delete, 1},
#endif
  { "ECHO",     "Echo command line back (Debug)",              GTC_echo, 0 },
  { "FILTER",   "FILTER HELP | UID TYPE to filter an existing connection.", GTC_filter, 0 },
  { "HANGUP",   "Hangup UID to hangup on this user",          GTC_hangup, 0 },
  { "HELP",     "These help messages",                        GTC_help, 0 },
#ifdef TALKDTEST
  { "LEAVE",    "leave USER[@MACHINE] to leave an invitation",GTC_leave, 1 },
  { "LOOKUP",   "Lookup USER[@MACHINE] to lookup an invite",  GTC_lookup, 1 },
#endif
  { "LOAD",     "Load [FILE] to read FILE, or default files for hosts.", GTC_load, 0 },
#ifdef TALKDTEST
  { "QUERY",    "Query local daemon for last person to announce", GTC_query, 1 },
#endif
  { "QUIT",     "Quit gtalk binary",                          GTC_quit, 0 },
  { "READ",     "Read [SCRIPT] to read script as commands.", GTC_read, 0 },
  { "REPLY",    "Reply UID to get machine id of last caller", GTC_reply, 0 },
  { "ROUTE",    "ROUTE DESTINATION ADDRESS to change local host id per network", GTC_route, 0 },
  { "SENDFILE", "Send UID FILE to transmit file to user",     GTC_sendfile, 0 },
  { "SET",      "SET < option | HELP > to set a state.",      GTC_set, 0 },
  { "SHOW",     "SHOW < option | HELP > to show information.", GTC_show, 0 },
  { "START",    "START UID ALIAS to start shared app with UID.", GTC_start, 0 },
  { "TEST",	"Test [VERBOSE] Run diagnostics within gtalk and exit.", GTC_test, 1 },
  { "WAIT",     "Wait UID USER[@MACHINE][ TTY] to wait for user to connect", GTC_wait, 1 },
};


/*
 * Function: GTC_interpret_string
 *
 *   Interprets the string via the parser.  Will receive input from
 * either the device reader (above) or the curses command prompt.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              buff - Pointer toCharacter of buffer
 * History:
 * zappo   8/24/95    Created
 */
void GTC_interpret_string(Ctxt, buff)
     struct TalkContext *Ctxt;
     char               *buff;
{
  char buffer[80];
  char *s;
  int   i, found = -1;

  /* Check for # comments */
  if(buff[0] == '#') return;

  /* Nuke white space before command */
  s = buff;
  while(*s && (*s != ' ')) s++;

  /*
   * Now that the first command has been parsed, lets find the string
   * we have matched (case insensitive) and execute its function.
   */
  if(s - buff)
    {
      for(i = 0; i < (sizeof(Commands) / sizeof(struct command_associate)); i++)
	{
	  /*
	   * case insensitive comparison of beginning of string.  Allow
	   * shortest version of word, but not shortest unique.  Close enough
	   * for something to be interfaced into some other program.
	   */
	  if(GTL_scmp(Commands[i].command, buff, s - buff) == Success)
	    {
	      if(found == -1)
		found = i;
	      else
		{
		  found = -2;
		  break;
		}
	    }
	}
      if(found == -1)
	{
	  sprintf(buffer, "Invalid command %s", buff);
	  DISP_message(Ctxt, buffer);
	}
      else if(found == -2)
	{
	  sprintf(buffer, "Command %s is not unique.", buff);
	  DISP_message(Ctxt, buffer);
	}
      else
	{
	  /* If we encountered a NULL, then pass no external commands */
	  if(!Commands[found].parse(Ctxt, *s?++s:NULL) && verbose)
	    DISP_message(Ctxt, "That command did not succeed.");
	}
      return;
    }
}

/*
 * Function: GTC_parse_command
 *
 * Reads in a command line, and parses it, then does some appropriate action.
 * 
 * Parameters: Ctxt - context of the program
 *             dev - device to read from (should be stdio)
 * History:
 * eml 3/1/94
 */
void GTC_parse_command(Ctxt, dev)
     struct TalkContext *Ctxt;
     struct InputDevice *dev;
{
  char  buff[BUFSIZE];

  /*
   * There will always be a ^M or ^L so make sure it's nuked.
   */
  memset(buff, 0, sizeof(buff));
  GT_recv(dev, buff, BUFSIZE);

  if(strlen(buff) < 1)
    {
      DISP_message(Ctxt, "Type HELP for help.");
      return;
    }

  GTC_interpret_string(Ctxt, buff);
}


/*
 * Function: GTC_help
 *
 *   Locally defined function which provides help for the command line
 * 
 * Parameters: None
 *
 * History:
 * eml 4/19/94
 */
static int GTC_help(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char buffer[90];
  int i;

#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 
		    75,
		    (sizeof(Commands) / sizeof(struct command_associate))+5);
#endif
  DISP_message(Ctxt, "Available commands:");
  for(i = 0; i < (sizeof(Commands) / sizeof(struct command_associate)); i++)
    {
      /*if((Commands[i].socket_only == 0) || (Ctxt->runstate == Socket))*/ 
      {
	sprintf(buffer," %-10s: %s", Commands[i].command, Commands[i].description);
	DISP_message(Ctxt, buffer);
      }
    }

  DISP_message(Ctxt, "For commands which refer to a connected user, such as HANGUP, START, and");
  DISP_message(Ctxt, "SEND, the [R]UID parameter can be an integer or the user's LOGIN NAME");

  return Success;
}

/*
 * Function: GTC_sethelp
 *
 *   Locally defined function which provides help for the set command
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - Pointer toCharacter of cmd
 * History:
 * zappo   1/28/95    Created
 */
static int GTC_sethelp(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char buffer[90];
  int i;

#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 
		    75,
		    (sizeof(SetCommands) / sizeof(struct command_associate))+3);
#endif
  DISP_message(Ctxt, "Available SET commands:");
  for(i = 0; i < (sizeof(SetCommands) / sizeof(struct command_associate)); i++)
    {
      if((SetCommands[i].socket_only == 0) || (Ctxt->runstate == Socket)) {
	sprintf(buffer," %-10s: %s", SetCommands[i].command, SetCommands[i].description);
	DISP_message(Ctxt, buffer);
      }
    }

  return Success;
}


/*
 * Function: GTC_showhelp
 *
 *   Locally defined function which provides help for the show command
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt - Context
 *              cmd  - rest of string
 * History:
 * zappo   1/28/95    Created
 */
static int GTC_showhelp(Ctxt, cmd)
     struct TalkContext *Ctxt;
     char               *cmd;
{
  char buffer[90];
  int i;

#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 
	     75,
	     (sizeof(ShowCommands) / sizeof(struct command_associate))+3);
#endif
  DISP_message(Ctxt, "Available SHOW commands:");
  for(i = 0; i < (sizeof(ShowCommands) / sizeof(struct command_associate)); i++)
    {
      if((ShowCommands[i].socket_only == 0) || (Ctxt->runstate == Socket)) {
	sprintf(buffer, " %-10s: %s", ShowCommands[i].command, ShowCommands[i].description);
	DISP_message(Ctxt, buffer);
      }
    }

  return Success;
}
