/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 */

#include <math.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include "../modutils.h"

module modinfo;

typedef struct {
    module_id id;
    shell *shl;
    GtkRange *treshold_control;
    GtkSpinButton *duration_control;
    GtkToggleButton *invert_control;
    GtkToggleButton *ganging_control;
    GtkToggleButton *delete_control;
    GtkWidget *dialog;
    AFframecount duration;
    float treshold;
    int invert;
    int ganging;
    int delete;
} amptreshold_params;

#ifdef HAVE_GNOME2
#define AMPTRESHOLD_GLADE_FILE "amptreshold-2.glade"
#else
#define AMPTRESHOLD_GLADE_FILE "amptreshold.glade"
#endif

struct extraction {
    AFframecount begin;
    AFframecount end;
};

void
extraction_list_dump(GList *l) {
    GList *l2;
    for(l2 = l; l2; l2 = l2->next) {
        DEBUG("start: %ld, end: %ld\n", 
              ((struct extraction *)l->data)->begin, 
              ((struct extraction *)l->data)->end); 
    }
}

AFframecount
extraction_list_apply(shell *shl,
                      int track,
                      GList *extraction,
                      int delete) {
    GList *l;
    struct extraction *e;
    AFframecount deleted = 0;
    rwlock_wlock(&shl->sr->tracks[track]->rwl);
    for(l = extraction; l; l = l->next) {
        e = (struct extraction *)l->data;
        /*
        DEBUG("erasing: %ld, end: %ld\n", 
              ((struct extraction *)l->data)->begin, 
              ((struct extraction *)l->data)->end); 
        */
        DEBUG("deleting %ld frames from %ld on track %d\n",
              e->end - e->begin, e->begin - deleted, track);
        blocklist_blocks_destroy(track_delete(shl->sr->tracks[track],
                                              e->begin - deleted,
                                              e->end - e->begin));
        if(!delete)
            track_null_insert(shl->sr->tracks[track],
                              e->begin,
                              e->end - e->begin);
        else
            deleted += (e->end - e->begin);
    }
    rwlock_wunlock(&shl->sr->tracks[track]->rwl);
    return deleted;
    
}

GList *
extraction_list_intersect(GList *el1, GList *el2) {
    GList *l1 = el1, *l2 = el2, *li = NULL;
    struct extraction *eli, *e1, *e2;

    while(l1 && l2) {
        e1 = (struct extraction *)l1->data;
        e2 = (struct extraction *)l2->data;
        
        if(e1->begin > e2->end) {
            l2 = l2->next;
            continue;
        } else if(e2->begin > e1->end) {
            l1 = l1->next;
            continue;
        }
        
        eli = mem_alloc(sizeof(struct extraction));
        if(!eli) {
            FAIL("could not allocate memory for extraction element!\n");
            return li;
        }
        eli->begin = MAX(e1->begin, e2->begin);
        eli->end = MIN(e1->end, e2->end);
        li = g_list_append(li, eli);

        if(e1->end <= e2->end)
            l1 = l1->next;
        else if(e2->end < e1->end) 
            l2 = l2->next;
    }
    return li;
}

void
extraction_list_destroy(GList *extraction) {
    GList *l;
    for(l = extraction; l; l = l->next) {
        mem_free(l->data);
        l->data = NULL;
    }
    g_list_free(l);
}

GList *
extraction_list_invert(GList *l,
                       AFframecount start_offset,
                       AFframecount end_offset) {
    GList *l2, *li = NULL;
    struct extraction store, *el, *eli;
    store.begin = start_offset;
    store.end = end_offset;
    for(l2 = l ; l2; l2 = l2->next) {
        el = (struct extraction *)l2->data;
        if(store.begin == start_offset &&
           el->begin == start_offset) {
            store.begin = el->begin;
            store.end = el->end;
            continue;
        }
        
        if(store.begin == start_offset &&
           el->begin != start_offset) 
            store.end = start_offset;
        
        eli = mem_alloc(sizeof(struct extraction));
        if(!eli) {
            FAIL("could not allocate memory for extraction element!\n");
            break;
        }
        eli->begin = store.end;
        eli->end = el->begin;
        li = g_list_append(li, eli);
        store.begin = el->begin;
        store.end = el->end;
    }
    if(store.end != end_offset || 
       (store.begin == start_offset && store.end == end_offset)) {
        if(store.end != end_offset) 
            store.begin = store.end;            
        eli = mem_alloc(sizeof(struct extraction));
        if(!eli) {
            FAIL("could not allocate memory for extraction element!\n");
        } else {
            eli->begin = store.begin;
            eli->end = end_offset;
            li = g_list_append(li, eli);
        }
    }            
    return li;
}

GList *
extraction_list_new(shell *shl,
                    int track,
                    AFframecount start_offset,
                    AFframecount end_offset,
                    float treshold,
                    AFframecount duration) {
    GList *l = NULL;
    int i, begin_found = 0;
    struct extraction store, *el;
    float begin_val = 0;
    AFframecount abs_offset;
    ITERATORF_INIT(start_offset, end_offset - start_offset);
    ITERATORF(shl, shl->sr->tracks[track],
              abs_offset = iter_frame_offset;
              for(i = 0; i < iter_read; i++, abs_offset++) {
                  if(!begin_found && (fabs(float_frame_bits[i]) <= treshold)) {
                      begin_val = fabs(float_frame_bits[i]);
                      begin_found = 1;
                      store.begin = abs_offset;
                  } else if(begin_found && (fabs(float_frame_bits[i]) > treshold)) {
                      begin_found = 0;
                      store.end = abs_offset;
                      if(store.end - store.begin > duration) {
                          DEBUG("range %ld-%ld\n",
                                store.begin, store.end);
                          el = mem_alloc(sizeof(struct extraction));
                          if(!el) {
                              FAIL("could not allocate memory for extraction element!\n");
                              ITERATOR_ESCAPE();
                              break;
                          }
                          *el = store;
                          l = g_list_append(l, el);
                      }
                  }
              });
    if(begin_found) {
        store.end = end_offset;
        if(store.end - store.begin > duration) {
            el = mem_alloc(sizeof(struct extraction));
            if(!el) 
                FAIL("could not allocate memory for extraction element!\n");
            else {
                *el = store;
                l = g_list_append(l, el);
            }
        }
    }
    ITERATORF_EXIT();
    return l;
    
}
                

module *
module_new() {
    MODULE_INIT(&modinfo, 
		"Amplitude treshold",
		"Pascal Haakmat",
		"Copyright (C) 2002,2003");
    return &modinfo;
}

void
on_apply_clicked(GtkWidget *w,
                 gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    amptreshold_params *p = (amptreshold_params *)module_state->data;
    p->duration = gtk_spin_button_get_value(p->duration_control) *
        p->shl->grid.gap;
    p->treshold = gtk_range_get_adjustment(p->treshold_control)->value;
    p->ganging = gtk_toggle_button_get_active(p->ganging_control) ? 1 : 0;
    p->invert = gtk_toggle_button_get_active(p->invert_control) ? 1 : 0;
    p->delete = gtk_toggle_button_get_active(p->delete_control) ? 1 : 0;
    action_do(ACTION_MODULE_EXECUTE_NEW(WITH_UNDO, p->shl, p->id));
}

void
on_ok_clicked(GtkWidget *w,
              gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    amptreshold_params *p = (amptreshold_params *)module_state->data;
    on_apply_clicked(w, NULL);
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_close_clicked(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    amptreshold_params *p = (amptreshold_params *)module_state->data;
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_dialog_destroy(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    module_state->is_open = 0;
    free(module_state->data);
}

void
module_open(module_id id,
            shell *shl, 
            int undo) {
    GladeXML *xml;
    GtkWidget *w;
    char path[4096];
    amptreshold_params *p = mem_calloc(1, sizeof(amptreshold_params));
    
    if(!p) {
        gui_alert("Not enough memory.");
        return;
    }

    snprintf(path, 4096, "%s/%s", dirname(modules[id].fname), AMPTRESHOLD_GLADE_FILE);
    DEBUG("loading interface %s\n", path);
    xml = glade_xml_new(path, "dialog", NULL);
    
    if(!xml) {
        gui_alert("Amplitude Treshold: could not load interface %s.", path);
        free(p);
        return;
    }

    shl->module_state[id].is_open = 1;
    shl->module_state[id].data = p;
    p->id = id;
    p->shl = shl;
    p->treshold_control = GTK_RANGE(glade_xml_get_widget(xml, "treshold"));
    p->duration_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "duration"));
    p->invert_control = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "invert"));
    p->ganging_control = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "ganging"));
    p->delete_control = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "delete"));
    p->dialog = GTK_WIDGET(glade_xml_get_widget(xml, "dialog"));
    g_signal_connect(GTK_OBJECT(p->dialog), "destroy", 
                     G_CALLBACK(on_dialog_destroy), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "ok"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_ok_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "close"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_close_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "apply"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_apply_clicked), shl);
    g_object_set_data(G_OBJECT(p->dialog), "user_data",
                      GINT_TO_POINTER(&shl->module_state[id]));
    g_object_unref(G_OBJECT(xml));
}

void
module_close(mod_state *module_state) {
    gtk_object_destroy(GTK_OBJECT(((amptreshold_params *)module_state->data)->dialog));
}

action_group *
module_execute(shell *shl, 
               int undo) {
    AFframecount start = shl->select_start,
        end = shl->select_end;
    int map = shl->select_channel_map;
    int t, id = shl->active_module_id, seltrks = 0, i = 0, invert, ganging, delete;
    amptreshold_params *p;
    AFframecount track_newlen[32], duration;
    action_group *undo_ag = NULL;
    snd *copy = NULL;
    GList *l = NULL, *extractions[32], *intersection = NULL;
    float treshold;

    if(!shl->module_state[id].is_open) { 
        gui_alert("Amplitude Treshold does not have defaults.");
        return NULL;
    }

    p = shl->module_state[id].data;
    invert = p->invert;
    delete = p->delete;
    treshold = p->treshold;
    duration = p->duration;
    ganging = p->ganging;

    if(shl->select_end - shl->select_start < duration) {
        gui_alert("The selection length is less than\n the minimum required duration.");
        return NULL;
    }

    rwlock_rlock(&shl->sr->rwl);

    if(undo) {
        copy = snd_copy(shl->sr, map, start, end - start);
        if(RESULT_IS_ERROR(shl->sr)) {
            gui_alert("Could not create undo: %s!", snd_error_get(shl->sr));
            rwlock_runlock(&shl->sr->rwl);
            return NULL;
        }
    }

    /* Calculate extraction lists for every selected track. */

    for(t = 0; t < snd_track_count(shl->sr); t++) {
        if((1 << t) & map) {
            extractions[seltrks++] = extraction_list_new(shl,
                                                         t,
                                                         start,
                                                         end,
                                                         treshold,
                                                         duration);
        }
    }

    /* If ganging, find the smallest intersection of all extraction
       lists. */

    if(ganging) {
        intersection = extractions[0];
        for(i = 0; i < seltrks - 1; i++) {
            intersection = extraction_list_intersect(extractions[i], extractions[i+1]);
            extraction_list_destroy(extractions[i]);
            extraction_list_destroy(extractions[i+1]);
            extractions[i] = intersection;
            extractions[i+1] = intersection;
        }
        for(i = 0; i < seltrks - 2; i++) {
            extraction_list_destroy(extractions[i]);
            extractions[i] = intersection;
        }
    }

    /* If inverting, invert all extraction lists. */

    if(invert) {
        for(i = 0; i < seltrks; i++) {
            if(!ganging || (ganging && i == 0)) {
                l = extraction_list_invert(extractions[i], start, end);
                extraction_list_destroy(extractions[i]);
            }
            extractions[i] = l;
        }
    }

    /* Finally apply the extraction lists. */

    for(t = 0, i = 0; t < snd_track_count(shl->sr); t++) {
        if((1 << t) & map) {
            track_newlen[t] = (end - start) -
                extraction_list_apply(shl, t, extractions[i], delete);
            if(!ganging)
                extraction_list_destroy(extractions[i]);
            i++;
        }
    }

    if(ganging)
        extraction_list_destroy(intersection);

    if(undo) {
        undo_ag = action_group_new_empty(2 + seltrks);

        if(!undo_ag) {
            snd_destroy(copy);
            rwlock_runlock(&shl->sr->rwl);
            return NULL;
        }

        undo_ag->a[seltrks] =
            ACTION_INSERT_NEW(DONT_UNDO,
                              shl->sr,
                              copy,
                              map,
                              start);
        undo_ag->a[seltrks + 1] =
            ACTION_SELECT_NEW(DONT_UNDO,
                              shl,
                              shl->sr,
                              map,
                              start,
                              snd_frame_count(copy));

        for(t = 0; t < snd_track_count(shl->sr); t++) {
            if((1 << t) & map) {
                undo_ag->a[--seltrks] =
                    ACTION_DELETE_ON_TRACK_NEW(DONT_UNDO,
                                               shl,
                                               shl->sr,
                                               t,
                                               start,
                                               track_newlen[t]);
            }
        }

    }

    rwlock_runlock(&shl->sr->rwl);
    return undo_ag;
}
