#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# FILE:
# Measurement.py
#
# DESCRIPTION:
# Class that contains a "measurement"
#
# NOTES:
#

__all__ = ['TextMeasurement']

from types import *
import string
from Measurement import Measurement

############################################################
#
# Represents a text measurement
#
# Initialization code can be pass an absolute measurement
# (anything that can be passed to Measurement()) or
# relative measurements of the form:
#   o "xx%"  -- Percentage of parent's measurement
#   o "12em" -- Size relative to font's height (1em = regular font height)
#   o "12ex" -- Size relative to the height of the letter 'x'
#   o "single" -- Same as "auto" / single spacing
#   o "double" -- Same as "auto * 2" / double spacing

class TextMeasurement(Measurement):
  def __init__(self, value, allowAuto=0):
    self.__value = value = string.lower(value)
    if value == 'single':
      self.__mode = 'percent'
      self.__multiplier = 1
    elif value == 'double':
      self.__mode = 'percent'
      self.__multiplier = 2
    elif value[-1:] == '%':
      try:
        self.__multiplier = float(value[:-1])/100
        self.__mode = 'percent'
      except ValueError:
        raise "Invalid percentage format: %s" % value
    elif value[-2:] == 'em':
      try:
        self.__multiplier = float(value[:-2])
        self.__mode = 'percent'
      except ValueError:
        raise "Invalid measurement format: %s" % value
    elif value[-2:] == 'ex':
      try:
        self.__multiplier = float(value[:-2])
        self.__mode = 'ex'
      except ValueError:
        raise "Invalid measurement format: %s" % value
    else:
      self.__mode = 'abs'
      Measurement.__init__(self, value)

  def points(self, font):
    inches = self.inches(fonts)
    if inches is None:
      return None
    else:
      return inches * 72

  def inches(self, font):
    if self.__mode == 'abs':
      return Measurement.inches(self)
    elif self.__mode == 'ex':
      return font.height_x * self.__multiplier
    else:
      return font.height * self.__multiplier

