/* gnobog_edit.c
 *
 * Copyright (C) 2000 Frdric LESPEZ & Renaud CHAILLAT
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <glade/glade.h>

#include "gnobog_bookmarks.h"
#include "gnobog_edit.h"

#define GBP_OK_BUTTON           0
#define GBP_APPLY_BUTTON        1
#define GBP_CANCEL_BUTTON       2

typedef struct _GnobogBookmarksPropertiesContext  GnobogBookmarksPropertiesContext;
struct _GnobogBookmarksPropertiesContext {
  GnobogBookmarks*     bookmarks;
  GnobogBookmarksNode  bookmarks_node;
};

GladeXML        *xml;
static gboolean gnobog_edit_dialog_unblock_flag;

/* Callbacks for "Bookmark Properties" dialog box*/
void gnobog_bookmark_properties_dialog_changed_cb (GtkWidget* widget, gpointer data);
void gnobog_bookmark_properties_dialog_ok_cb (GtkWidget* widget, gpointer user_data);
void gnobog_bookmark_properties_dialog_apply_cb (GtkWidget* widget, gpointer user_data);
void gnobog_bookmark_properties_dialog_cancel_cb (GtkWidget* widget, gpointer user_data);

/* This callback is called as soon as the user makes a modification
   in bookmark properties. Its purpose is to make "Ok" and "Apply"
   sensistive */
void
gnobog_bookmark_properties_dialog_changed_cb (GtkWidget* widget, gpointer data)
{
  GtkWidget* glade_widget;

  /* A change occured in bookmark properties,
     "Ok" and "Apply" button can be sensitive */
  glade_widget = glade_xml_get_widget (xml, "button_ok");
  gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), TRUE);
  glade_widget = glade_xml_get_widget (xml, "button_apply");
  gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), TRUE);

  return;
}

/* This callback is called when "Apply" button is pressed.
   Changes made are applied on bookmarks node and dialog box
   is updated when appropriated (modification time) */
void
gnobog_bookmark_properties_dialog_apply_cb (GtkWidget* widget, gpointer data)
{
  GnobogBookmarksPropertiesContext* context;
  GnobogBookmarks*     bookmarks;
  GnobogBookmarksNode  bookmarks_node;
  GtkWidget*           glade_widget;
  gchar*               name;
  gchar*               location;
  gchar*               description;
  gchar*               str_time;
  GTime                val_time;
        
  /* Get context */
  context = gtk_object_get_user_data (GTK_OBJECT (widget));               
        
  /* Get bookmark node we are are editing */
  bookmarks_node = context->bookmarks_node;               
                        
  /* Get bookmarks owning this node */
  bookmarks = context->bookmarks;         
                        
  /* Get new name */
  glade_widget = glade_xml_get_widget (xml, "entry_name");
  name = gtk_editable_get_chars (GTK_EDITABLE (glade_widget), 0, -1);

  /* Get new location */
  glade_widget = glade_xml_get_widget (xml, "entry_location");
  location = gtk_editable_get_chars (GTK_EDITABLE (glade_widget), 0, -1);
        
  /* Get new description */
  glade_widget = glade_xml_get_widget (xml, "text_box_description");
  description = gtk_editable_get_chars (GTK_EDITABLE (glade_widget), 0, -1);
        
  /* Apply modification on node */
  gnobog_bookmarks_node_modify (bookmarks, bookmarks_node,
                                name, location, description);

  /* Update modification time in dialog box */
  glade_widget = glade_xml_get_widget (xml, "label_modified_value");
  val_time = gnobog_bookmarks_node_get_modification_time (bookmarks_node);
  str_time = g_strdup (ctime ((const time_t*) &val_time));
  str_time[strlen (str_time)-1] = '\0';
  gtk_label_set_text (GTK_LABEL (glade_widget), str_time);
  g_free (str_time);

  /* Information in dialog box and bookmark node are now synched,
     "Ok" and "Apply" button can be unsensitive */
  glade_widget = glade_xml_get_widget (xml, "button_ok");
  gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);
  glade_widget = glade_xml_get_widget (xml, "button_apply");
  gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);

  /* FIXME : These string must be freed. But doing it cause a crash */
     g_free (name);
     g_free (location);
     g_free (description);
                
  return;
}

/* This callback is called when "Ok" button is pressed.
   This callback just calls "Apply" button callback */
void
gnobog_bookmark_properties_dialog_ok_cb (GtkWidget* widget, gpointer data)
{
  /* Basically OK implies APPLY ;-) */
  gnobog_bookmark_properties_dialog_apply_cb (widget, data);
        
  /* Destroy "Bookmark Properties" dialog Box */
  gtk_widget_destroy (GTK_WIDGET (widget));

  /* Set flag so that main function
     gnobog_ui_bookmark_properties_run_dialog no longer blocks */
  gnobog_edit_dialog_unblock_flag = TRUE;
  return;
}

/* This callback is called when "Cancel" Button is pressed.
   This one does nothing. */
void
gnobog_bookmark_properties_dialog_cancel_cb (GtkWidget* widget, gpointer data)
{
  /* Destroy "Bookmark Properties" dialog Box */
  gtk_widget_destroy (GTK_WIDGET (widget));

  /* Set flag so that main function
     gnobog_ui_bookmark_properties_run_dialog no longer blocks */
  gnobog_edit_dialog_unblock_flag = TRUE;
  return;
}

/* This function initializes values in "Bookmark Properties" dialog box.
   Then it displays and run the dialog box.
   It return the button number pressed in the dialog :
   GBP_OK_BUTTON, GBP_APPLY_BUTTON or GBP_CANCEL_BUTTON */
static gint
gnobog_ui_bookmark_properties_run_dialog (GnobogBookmarks* bookmarks, GnobogBookmarksNode bookmarks_node)
{
  GtkWidget*                         glade_widget;
  GnobogBookmarksPropertiesContext*  context;
  GnomeDialog*                       gnobog_bookmark_properties_dialog;
  GnomeDialog*                       dialog;
  gchar*                             description;
  gchar*                             str_time;
  gchar*                             str_nbalias;
  GTime                              val_time;
  gint                               position;       
  gint                               nbalias;
  gint                               reply;  
        
  /* Load "Bookmark Properties" dialog box from Glade file and process it */
  xml = glade_xml_new (GNOBOG_GLADEDIR "/gnobog.glade", "gnobog_bookmark_properties_dialog");
  /* Used for tests */
  //      xml = glade_xml_new ("glade/gnobog.glade", "gnobog_bookmark_properties_dialog");
  if (!xml) {
    g_warning ("Cannot load interface from Glade file");
    dialog = GNOME_DIALOG (gnome_error_dialog (_("Cannot load interface from Glade file")));
    gnome_dialog_run_and_close (dialog);
    /* Return GBP_CANCEL_BUTTON to act as if "CANCEL" Button has been pressed
       It is better than segfaulting */
    return GBP_CANCEL_BUTTON;
  }
  gnobog_edit_dialog_unblock_flag = FALSE;

  /* Create context */
  context = g_new(GnobogBookmarksPropertiesContext, 1);
  context->bookmarks = bookmarks;
  context->bookmarks_node = bookmarks_node;
        
  /* Attach 'context' to the dialog box */
  gnobog_bookmark_properties_dialog = GNOME_DIALOG (glade_xml_get_widget (xml, "gnobog_bookmark_properties_dialog"));
  gtk_object_set_user_data (GTK_OBJECT (gnobog_bookmark_properties_dialog), context);             
                
  /* Initialize "Name" field in "Bookmark Properties" dialog box */
  glade_widget = glade_xml_get_widget (xml, "entry_name");
  gtk_entry_set_text (GTK_ENTRY (glade_widget),
                      gnobog_bookmarks_node_get_name (bookmarks_node));

  /* Initialize "Location" field in "Bookmark Properties" dialog box */
  glade_widget = glade_xml_get_widget (xml, "entry_location");
  gtk_entry_set_text (GTK_ENTRY (glade_widget),
                      gnobog_bookmarks_node_get_location (bookmarks_node));
        
  /* Initialize "Description" field in "Bookmark Properties" dialog box */
  glade_widget = glade_xml_get_widget (xml, "text_box_description");
  description = gnobog_bookmarks_node_get_description (bookmarks_node);
  position = 0;
  gtk_editable_insert_text (GTK_EDITABLE (glade_widget),
                            description,
                            strlen (description),
                            &position);
        
  /* Initialize 'label_created_value' value with bookmarks_entry */
  glade_widget = glade_xml_get_widget (xml, "label_created_value");
  val_time = gnobog_bookmarks_node_get_creation_time (bookmarks_node);
  str_time = g_strdup (ctime ((const time_t*) &val_time));
  str_time[strlen (str_time)-1] = '\0';
  gtk_label_set_text (GTK_LABEL (glade_widget), str_time);
  g_free (str_time);
        
  /* Initialize 'label_visited_value' value with bookmarks_entry */
  glade_widget = glade_xml_get_widget (xml, "label_visited_value");
  val_time = gnobog_bookmarks_node_get_visit_time (bookmarks_node);
  str_time = g_strdup (ctime ((const time_t*) &val_time));
  str_time[strlen (str_time)-1] = '\0';
  gtk_label_set_text (GTK_LABEL (glade_widget), str_time);
  g_free (str_time);
        
  /* Initialize 'label_modified_value' value with bookmarks_entry */
  glade_widget = glade_xml_get_widget (xml, "label_modified_value");
  val_time = gnobog_bookmarks_node_get_modification_time (bookmarks_node);
  str_time = g_strdup (ctime ((const time_t*) &val_time));
  str_time[strlen (str_time)-1] = '\0';
  gtk_label_set_text (GTK_LABEL (glade_widget), str_time);
  g_free (str_time);
        
  /* Initialize 'label_nbalias_value' value with bookmarks_entry */
  glade_widget = glade_xml_get_widget (xml, "label_nbalias_value");
  nbalias = g_list_length (gnobog_bookmarks_node_get_alias_list (bookmarks_node))-1;
  str_nbalias = g_strdup_printf("%d", nbalias);
  gtk_label_set_text (GTK_LABEL (glade_widget), str_nbalias);
  g_free (str_nbalias);
        
  /* Mark as unsensitive unuseful part of the "Bookmark Properties" dialog Box */
  if ((gnobog_bookmarks_node_get_type (bookmarks_node) == FOLDER)
      || (gnobog_bookmarks_node_get_type (bookmarks_node) == TITLE)) {
    glade_widget = glade_xml_get_widget (xml, "entry_location");
    gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);
    glade_widget = glade_xml_get_widget (xml, "label_location");
    gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);
    glade_widget = glade_xml_get_widget (xml, "label_visited");
    gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);
    glade_widget = glade_xml_get_widget (xml, "label_visited_value");
    gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);
  }

  /* Mark as unsensitive "Ok" and "Apply" buttons */
  glade_widget = glade_xml_get_widget (xml, "button_ok");
  gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);
  glade_widget = glade_xml_get_widget (xml, "button_apply");
  gtk_widget_set_sensitive (GTK_WIDGET (glade_widget), FALSE);
                
  /* Connect all signal with functions name given in glade file */
  glade_xml_signal_autoconnect (xml);

  /* While user plays with "Apply" button, keep on waiting... */
  do {
    reply = gnome_dialog_run (gnobog_bookmark_properties_dialog);
  } while (gnobog_edit_dialog_unblock_flag == FALSE);

  /* Free context */
  g_free (context);

  return reply;
}

/* API entry for editing a bookmark */
void
gnobog_ui_edit_bookmark_properties (GnobogBookmarks* bookmarks, GnobogBookmarksNode bookmarks_node)
{
  /* Popup "Bookmark Properties" dialog box and *block* until it is closed */
  gnobog_ui_bookmark_properties_run_dialog (bookmarks, bookmarks_node);
}

/* Common part of  gnobog_ui_insert_new_bookmark / _folder functions
   Take care to delete the dummy node inserted if "CANCEL" button is pressed */
static void
gnobog_ui_insert_common (GnobogBookmarks* bookmarks, GnobogBookmarksNode bookmarks_node)
{
  GList* list;    
  gint reply;

  /* Popup "Bookmark Properties" dialog box and *block* until it is closed */
  reply = gnobog_ui_bookmark_properties_run_dialog (bookmarks, bookmarks_node);

  g_message ("Reply : %d", reply);

  if ((reply == GBP_CANCEL_BUTTON) || (reply == -1)) {
    /* If user press "CANCEL" or close the dialog box,
       removed the dummy node inserted */
    if (gnobog_bookmarks_node_get_modification_time (bookmarks_node)
        <= gnobog_bookmarks_node_get_creation_time (bookmarks_node)) {
      /* Remove node only if user have not modified if */
      list = g_list_prepend (NULL, bookmarks_node);
      gnobog_bookmarks_list_delete (bookmarks, list);
    }
  }
}

/* API entry for inserting a new bookmark */
void
gnobog_ui_insert_new_bookmark (GnobogBookmarks* bookmarks, GnobogBookmarksNode anchor_node)
{
  GnobogBookmarksNode bookmarks_node;

  /* Insert a dummy node */
  bookmarks_node = gnobog_bookmarks_node_insert_new (bookmarks, anchor_node,
                                                     INSERT_DEFAULT_MODE,
                                                     BOOKMARK,
                                                     _("New Bookmark"), "", "");
        
  /* Popup "Bookmark Properties" dialog box and edit the dummy node */
  gnobog_ui_insert_common (bookmarks, bookmarks_node);
}

/* API entry for inserting a new folder */
void
gnobog_ui_insert_new_folder (GnobogBookmarks* bookmarks, GnobogBookmarksNode anchor_node)
{
  GnobogBookmarksNode bookmarks_node;

  /* Insert a dummy node */
  bookmarks_node = gnobog_bookmarks_node_insert_new (bookmarks, anchor_node,
                                                     INSERT_DEFAULT_MODE,
                                                     FOLDER,
                                                     _("New Folder"), "", "");

  /* Popup "Bookmark Properties" dialog box and edit the dummy node */
  gnobog_ui_insert_common (bookmarks, bookmarks_node);
}
