#include "includes.h"

extern GlurpState *glurp;

GlurpConfig *config_init() {
  GlurpConfig *c = NULL;

  c = malloc(sizeof(GlurpConfig));

  c->server_host = g_strdup("");
  c->server_port = MPD_PORT;
  c->server_pass = g_strdup("");
  c->autoconnect = FALSE;
  c->playlist_vis_on_start = FALSE;
  c->time_display_left = FALSE;
  c->refresh_rate = 500;

  c->vis_filename = FALSE;
  c->vis_artist = FALSE;
  c->vis_title = FALSE;
  c->vis_album = FALSE;
  c->vis_name = FALSE;
  c->vis_pos = FALSE;
  c->vis_id = FALSE;

  return c;
}

gboolean config_load() {
  gchar *conf_str, *conf_path, **conf_items, **conf_item;
  GError *error = NULL;
  gint i = 0;

  glurp->config = config_init();

  conf_path = g_strdup_printf("%s/%s", g_strdup(g_get_home_dir()), GLURP_CONFIG_FILE);

  if( g_file_test(conf_path, G_FILE_TEST_EXISTS) ) {
    if( !g_file_test(conf_path, G_FILE_TEST_IS_REGULAR) ) {
      debug("Cannot open config file, it is not a regular file!");
      return FALSE;
    }
  }

  g_file_get_contents(conf_path, &conf_str, NULL, &error);

  if( conf_str ) {
    conf_items = g_strsplit(conf_str, "\n", NUM_SETTINGS);
  } else {
    debug("Config file empty, continuing.");
    config_set_defaults();
    return TRUE;
  }

  while(conf_items[i] && strlen(conf_items[i])) {
    conf_item = g_strsplit(conf_items[i], " = ", 2);
    if(conf_item[0] && conf_item[1] &&
       strlen(conf_item[0]) ) {
      config_load_item(g_strstrip(conf_item[0]), g_strstrip(conf_item[1]));
    } else debug("Invalid config item, ignoring.");

    i++;
  }

  return TRUE;
}

void config_load_item(const gchar *key, const gchar *value) {
  debug("%s = %s", key, value);

  if(!strcmp(key, "server")) {
    g_free(glurp->config->server_host);
    glurp->config->server_host = g_strdup(value);
    return;
  }
  if(!strcmp(key, "port")) {
    glurp->config->server_port = atoi(value);
    if( glurp->config->server_port < 1 || glurp->config->server_port > 65534 ) glurp->config->server_port = MPD_PORT;
    return;
  }
  if(!strcmp(key, "password")) {
    g_free(glurp->config->server_pass);
    glurp->config->server_pass = g_strdup(value);
    return;
  }
  if(!strcmp(key, "autoconnect")) {
    glurp->config->autoconnect = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "playlist-visible-on-start")) {
    glurp->config->playlist_vis_on_start = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "time-display-left")) {
    glurp->config->time_display_left = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "refresh-rate")) {
    glurp->config->refresh_rate = atol(value);
    if( glurp->config->refresh_rate < MIN_REFRESH_RATE ) glurp->config->refresh_rate = MIN_REFRESH_RATE;
    if( glurp->config->refresh_rate > MAX_REFRESH_RATE ) glurp->config->refresh_rate = MAX_REFRESH_RATE;
    return;
  }

  if(!strcmp(key, "stream-history")) {
    glurp->stream_history = get_stream_history((gchar *)value);
    return;
  }


  if(!strcmp(key, "vis-filename")) {
    glurp->config->vis_filename = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "vis-artist")) {
    glurp->config->vis_artist = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "vis-title")) {
    glurp->config->vis_title = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "vis-album")) {
    glurp->config->vis_album = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "vis-name")) {
    glurp->config->vis_name = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "vis-pos")) {
    glurp->config->vis_pos = yesno(atoi(value));
    return;
  }

  if(!strcmp(key, "vis-id")) {
    glurp->config->vis_id = yesno(atoi(value));
    return;
  }

  debug("Invalid config item, ignoring.");
  return;
}

void config_save() {
  gchar *conf_path = NULL, *stream_history = NULL;
  FILE *f;

  conf_path = g_strdup_printf("%s/%s", g_strdup(g_get_home_dir()), GLURP_CONFIG_FILE);

  if( g_file_test(conf_path, G_FILE_TEST_EXISTS) ) {
    if( !g_file_test(conf_path, G_FILE_TEST_IS_REGULAR) ) {
      debug("Cannot open config file!");
      return;
    }
  }

  if( !(f = fopen(conf_path, "w")) ) {
    debug("Could not write into config: %s", conf_path);
    statusbar_print("Could not write into config: %s", conf_path);
    return;
  }

  debug("Writing config file: %s", conf_path);

  if( glurp->config->server_host && strlen(glurp->config->server_host) ) {
    g_fprintf(f, "server = %s\n", glurp->config->server_host);
    debug("server = %s", glurp->config->server_host);
  }

  g_fprintf(f, "port = %d\n", glurp->config->server_port);
  debug("port = %d", glurp->config->server_port);

  if( glurp->config->server_pass && strlen(glurp->config->server_pass) ) {
    g_fprintf(f, "password = %s\n", glurp->config->server_pass);
    debug("password = %s", glurp->config->server_pass);
  }

  if( glurp->config->autoconnect ) {
    g_fprintf(f, "autoconnect = %d\n", yesno(glurp->config->autoconnect));
    debug("autoconnect = %d", yesno(glurp->config->autoconnect));
  }

  if( glurp->config->playlist_vis_on_start ) {
    g_fprintf(f, "playlist-visible-on-start = %d\n", yesno(glurp->config->playlist_vis_on_start));
    debug("playlist-visible-on-start = %d", yesno(glurp->config->playlist_vis_on_start));
  }

  if( glurp->config->time_display_left ) {
    g_fprintf(f, "time-display-left = %d\n", yesno(glurp->config->time_display_left));
    debug("time-display-left = %d", yesno(glurp->config->time_display_left));
  }

  g_fprintf(f, "refresh-rate = %ld\n", glurp->config->refresh_rate);
  debug("refresh-rate = %ld", glurp->config->refresh_rate);

  if( glurp->stream_history ) {
    stream_history = dump_stream_history();
    g_fprintf(f, "stream-history = %s\n", stream_history);
    debug("stream-history = %s", stream_history);
    g_free(stream_history);
  }

  if( glurp->config->vis_filename ) {
    g_fprintf(f, "vis-filename = %d\n", yesno(glurp->config->vis_filename));
    debug("vis-filename = %d", yesno(glurp->config->vis_filename));
  }

  if( glurp->config->vis_artist ) {
    g_fprintf(f, "vis-artist = %d\n", yesno(glurp->config->vis_artist));
    debug("vis-artist = %d", yesno(glurp->config->vis_artist));
  }

  if( glurp->config->vis_title ) {
    g_fprintf(f, "vis-title = %d\n", yesno(glurp->config->vis_title));
    debug("vis-title = %d", yesno(glurp->config->vis_title));
  }

  if( glurp->config->vis_album ) {
    g_fprintf(f, "vis-album = %d\n", yesno(glurp->config->vis_album));
    debug("vis-album = %d", yesno(glurp->config->vis_album));
  }

  if( glurp->config->vis_name ) {
    g_fprintf(f, "vis-name = %d\n", yesno(glurp->config->vis_name));
    debug("vis-name = %d", yesno(glurp->config->vis_name));
  }

  if( glurp->config->vis_pos ) {
    g_fprintf(f, "vis-pos = %d\n", yesno(glurp->config->vis_pos));
    debug("vis-pos = %d", yesno(glurp->config->vis_pos));
  }

  if( glurp->config->vis_id ) {
    g_fprintf(f, "vis-id = %d\n", yesno(glurp->config->vis_id));
    debug("vis-id = %d", yesno(glurp->config->vis_id));
  }

  debug("Config file saved");

  fclose(f);
}

void config_set_defaults() {
  glurp->config->vis_filename = FALSE;
  glurp->config->vis_artist = TRUE;
  glurp->config->vis_title = TRUE;
  glurp->config->vis_album = FALSE;
  glurp->config->vis_name = FALSE;
  glurp->config->vis_pos = TRUE;
  glurp->config->vis_id = FALSE;
}
