// MM2MDL.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm2mdl.h"

#include <algorithm>
#include <strstream>
using namespace std;

/*################################################################################################*/

mm2_mdl::mm2_mdl(ostream * p1, class_factory & p2, mm2_eng_param * p3) :
	model_extended(p1, p2), model_simple(p1, p2), trajectory_interface()
{
	prm.vdwrad = 1.325;				// 1.325	// average main-chain radius 0.218 nm -> 0.275 nm???
	prm.lenjon = 1.225;				// 1.225
	prm.wescc = 0.350;				// 0.350 r^-1
	prm.wescd = 0.450;				// 0.450 r^-2
	prm.dipole1 = 1.0;				// 1.0
	prm.dipole2 = 25.0;				// 16.0 / prm.dipole1 * prm.dipole1
	prm.screen1 = -20.929;				// -20.929
	prm.screen2 = 78.4 - prm.screen1;		// 78.4 - screen1
	prm.screen3 = 3.4781;				// 3.4781
	prm.screen4 = -0.01787;				// -0.01787
	prm.screen5 = 0.001;				// 0.001	// about 4 ...
	prm.sasa1[0] = -1.80;				// -1.80
	prm.sasa1[1] = +3.00;				// +3.00 ???
	prm.sasa2[0] = 0.60;				// 0.60		// about ???% of previous
	prm.sasa2[1] = 0.60 * 0.60;			// 0.60 * 0.60	// about ???% of previous
	prm.solvrad = 0.15;				// 0.15		// H2O -> 0.15
	prm.wtor1 = 1.00;				// 1.00
	prm.wtor2 = 3.50;				// 3.50
	prm.wrep = 0.00175;				// 0.00175	// angle rep fc
	
	prm.sasa9[0] = +0.00;							// a +0.0
	prm.sasa9[1] = +0.00; prm.sasa9[2] = -1.00; prm.sasa9[3] = -1.00;	// r +0.0 -1.0 +2.0
	prm.sasa9[4] = +0.00; prm.sasa9[5] = +1.00;				// n +0.0 +1.0
	prm.sasa9[6] = +0.00; prm.sasa9[7] = -1.00;				// d +0.0 +2.0
	prm.sasa9[8] = +0.00; prm.sasa9[9] = +0.00;				// c +0.0 +1.0
	prm.sasa9[10] = +0.00; prm.sasa9[11] = +1.00;				// q +0.0 +1.0
	prm.sasa9[12] = +0.00; prm.sasa9[13] = -1.00;				// e +0.0 +2.0
	prm.sasa9[14] = +0.00;							// g +0.0
	prm.sasa9[15] = +0.00; prm.sasa9[16] = +0.00;				// h +0.0 +1.0
	prm.sasa9[17] = +0.00; prm.sasa9[18] = -1.00;				// i +0.0 -1.0
	prm.sasa9[19] = +0.00; prm.sasa9[20] = -1.00;				// l +0.0 -1.0
	prm.sasa9[21] = +0.00; prm.sasa9[22] = -1.00; prm.sasa9[23] = -1.00;	// k +0.0 -1.0 +2.0
	prm.sasa9[24] = +0.00; prm.sasa9[25] = -1.00;				// m +0.0 -1.0
	prm.sasa9[26] = +0.00; prm.sasa9[27] = -1.00;				// f +0.0 -1.0
	prm.sasa9[28] = -1.00;							// p -1.0
	prm.sasa9[29] = +1.00;							// s +1.0
	prm.sasa9[30] = +1.00;							// t +1.0
	prm.sasa9[31] = +0.00; prm.sasa9[32] = +1.00; prm.sasa9[33] = -1.00;	// w +0.0 +1.0 -1.0
	prm.sasa9[34] = +0.00; prm.sasa9[35] = +0.00;				// y +0.0 +1.0
	prm.sasa9[36] = -1.00;							// v -1.0
	
	prm.charge_wes = 0.257;			// why so small???
	prm.charge_sasa2 = 1.090;
	prm.charge_sasa3 = 0.410;
	
	prm.charge_pKa[0] = 7.402;	// n-terminal		// 7.8 down
	prm.charge_pKa[1] = 4.753;	// c-terminal		// 3.6 up
	prm.charge_pKa[2] = 11.2;	// R2 (no data)		// 12.0 down? 0.8
	prm.charge_pKa[3] = 4.280;	// D1			// 4.0 up
	prm.charge_pKa[4] = 9.9;	// C1 (no data)		// 9.1 up? 0.8
	prm.charge_pKa[5] = 5.813;	// E1			// 4.5 up
	prm.charge_pKa[6] = 6.068;	// H1			// 6.4 down
	prm.charge_pKa[7] = 9.393;	// K2			// 10.5 down
	prm.charge_pKa[8] = 10.614;	// Y1			// 9.7 up
	
	prm.charge_sasa1[0] = 0.362;	// 1.5
	prm.charge_sasa1[1] = 0.464;	// 1.5
	prm.charge_sasa1[2] = 1.1;	// 4.4 no data? 25%
	prm.charge_sasa1[3] = 0.866;	// 3.2
	prm.charge_sasa1[4] = 0.2;	// 0.8 no data? 25%
	prm.charge_sasa1[5] = 0.747;	// 3.9
	prm.charge_sasa1[6] = 0.696;	// 3.1
	prm.charge_sasa1[7] = 1.326;	// 5.0
	prm.charge_sasa1[8] = 0.542;	// 2.0
	
	prm.charge_acid[0] = false;
	prm.charge_acid[1] = true;
	prm.charge_acid[2] = false;
	prm.charge_acid[3] = true;
	prm.charge_acid[4] = true;
	prm.charge_acid[5] = true;
	prm.charge_acid[6] = false;
	prm.charge_acid[7] = false;
	prm.charge_acid[8] = true;
	
//////////////////////////////////////////////////
//////////////////////////////////////////////////
//////////////////////////////////////////////////

	if (p3 != NULL)
	{
		prm.vdwrad = p3->vdwrad;
		prm.lenjon = p3->lenjon;
		prm.wescc = p3->wescc;
		prm.wescd = p3->wescd;
		prm.dipole1 = p3->dipole1;
		prm.dipole2 = p3->dipole2;
		prm.screen1 = p3->screen1;
		prm.screen2 = p3->screen2;
		prm.screen3 = p3->screen3;
		prm.screen4 = p3->screen4;
		prm.screen5 = p3->screen5;
		prm.sasa1[0] = p3->sasa1[0];
		prm.sasa1[1] = p3->sasa1[1];
		prm.sasa2[0] = p3->sasa2[0];
		prm.sasa2[1] = p3->sasa2[1];
		prm.solvrad = p3->solvrad;
		prm.wtor1 = p3->wtor1;
		prm.wtor2 = p3->wtor2;
		prm.wrep = p3->wrep;

		for (i32s n1 = 0;n1 < 37;n1++)
		{
			prm.sasa9[n1] = p3->sasa9[n1];
		}
		
		prm.charge_wes = p3->charge_wes;
		prm.charge_sasa2 = p3->charge_sasa2;
		prm.charge_sasa3 = p3->charge_sasa3;
		
		for (i32s n1 = 0;n1 < 9;n1++)
		{
			prm.charge_pKa[n1] = p3->charge_pKa[n1];
			prm.charge_sasa1[n1] = p3->charge_sasa1[n1];
			prm.charge_acid[n1] = p3->charge_acid[n1];
		}
	}
	
	pH = 7.0;
	
	ms = this;	// trajectory_interface
}

mm2_mdl::~mm2_mdl(void)
{
}

const char * mm2_mdl::GetProjectFileNameExtension(void)
{
	static const char ext[] = "mm2gp";
	return ext;
}

/*##############################################*/
/*##############################################*/

void mm2_mdl::PushCRDSets(i32u p1)
{
	for (i32u n1 = 0;n1 < p1;n1++) cs_vector.push_back(new crd_set());
	
	fGL_a3 newcrd = { 0.0, 0.0, 0.0 };
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				for (i32u n4 = 0;n4 < p1;n4++)
				{
					chn_vector[n1].res_vector[n2].crd_vector[n3].push_back(newcrd);
				}
			}
			
			for (i32u n3 = 0;n3 < p1;n3++)
			{
				chn_vector[n1].res_vector[n2].peptide_vector.push_back(0.0);
			}
		}
	}
}

void mm2_mdl::PopCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm2_mdl::CopyCRDSet(i32u p1, i32u p2)
{
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
	chn_vector[n1].res_vector[n2].crd_vector[n3][p2][0] = chn_vector[n1].res_vector[n2].crd_vector[n3][p1][0];
	chn_vector[n1].res_vector[n2].crd_vector[n3][p2][1] = chn_vector[n1].res_vector[n2].crd_vector[n3][p1][1];
	chn_vector[n1].res_vector[n2].crd_vector[n3][p2][2] = chn_vector[n1].res_vector[n2].crd_vector[n3][p1][2];
			}
			
	chn_vector[n1].res_vector[n2].peptide_vector[p2] = chn_vector[n1].res_vector[n2].peptide_vector[p1];
		}
	}
}

void mm2_mdl::SwapCRDSets(i32u, i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm2_mdl::CenterCRDSet(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm2_mdl::ReserveCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

/*##############################################*/
/*##############################################*/

i32s mm2_mdl::GetVirtAtomCount(void)
{
	i32s virt_atom_count = 0;
	
	for (i32u t1 = 0;t1 < chn_vector.size();t1++)
	{
		for (i32u t2 = 0;t2 < chn_vector[t1].res_vector.size();t2++)
		{
			virt_atom_count += chn_vector[t1].res_vector[t2].natm;
		}
	}
	
	return virt_atom_count;
}

void mm2_mdl::Reduce(mm1_mdl * mdl, mm1_readpdb_mdata * mdata)
{
	mdl->GatherGroups();
	mm1_mdl::amino_builder.Identify(mdl);
	DefineSecondaryStructure(mdl);
	
	// check that the number of crd-sets matches...
	
	i32s csets = cs_vector.size(); i32s new_csets = (mdl->cs_vector.size() - csets);
	cout << "there were " << csets << " old crd-sets, creating " << new_csets << " new..." << endl;
	
	PushCRDSets(new_csets);
	
	for (i32u n1 = 0;n1 < cs_vector.size();n1++) SetCRDSetVisible(n1, true);
	
	// ok, start working...
	
	vector<mm1_chn_info> & ci_vector = (* mdl->ref_civ);
	
////////////////////////////////////////////////// debug!!!
////////////////////////////////////////////////// debug!!!
//	for (iter_mm1al it = mdl->atom_list.begin();it != mdl->atom_list.end();it++)
//	{
//		const fGL * color = (* it).el.GetColor();
//		mm2_debug newdb;
//		for (i32s n1 = 0;n1 < 3;n1++)
//		{
//			newdb.crd[n1] = (* it).crd_vector[0][n1];
//			newdb.color[n1] = color[n1] * 0.75;
//		}
//		debug_vector.push_back(newdb);
//	}
////////////////////////////////////////////////// debug!!!
////////////////////////////////////////////////// debug!!!

	// here we should check that we don't have any nucleic acid chains...
	// here we should check that we don't have any nucleic acid chains...
	// here we should check that we don't have any nucleic acid chains...
	
	// now, we must relate mdata and ci_vector!!! for each ci, find exactly matching mdata...
	
	i32s * ci_to_mdata_table = new i32s[ci_vector.size()];
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		vector<mm1_atom *> calpha_vector;
		
		iter_mm1al range1[2]; mdl->GetRange(1, n1, range1);
		iter_mm1al range2[2]; i32s res = 0;
		
		while (true)
		{
			mdl->GetRange(2, range1, res, range2);
			if (range2[0] == range2[1]) break;
			
			for (iter_mm1al it1 = range2[0];it1 != range2[1];it1++)
			{
				if (((* it1).res_id & 0xFF) != 0x01) continue;
				else calpha_vector.push_back(& (* it1));
			}
			
			res++;		// move to next residue...
		}
		
		ci_to_mdata_table[n1] = NOT_DEFINED;
		
		if (mdata != NULL)
		{
			for (i32u n2 = 0;n2 < mdata->chn_vector.size();n2++)
			{
				if (calpha_vector.size() != mdata->chn_vector[n2]->alpha_carbons.size()) continue;
				
				bool flag = true;
				for (i32u n3 = 0;n3 < calpha_vector.size();n3++)
				{
					mm1_atom * ref1 = calpha_vector[n3];
					mm1_atom * ref2 = mdata->chn_vector[n2]->alpha_carbons[n3];
					if (ref1 != ref2) flag = false;
				}
				
				if (flag)
				{
					ci_to_mdata_table[n1] = (i32s) n2;
					break;
				}
			}
			
			if (ci_to_mdata_table[n1] == NOT_DEFINED)
			{
				cout << "failed to map ci " << n1 << " to any mdata record!!!" << endl;
				exit(EXIT_FAILURE);
			}
		}
	}
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		iter_mm1al range1[2];
		mdl->GetRange(1, n1, range1);
		
		mm2_chn newchn;
		chn_vector.push_back(newchn);
		
		if (ci_to_mdata_table[n1] != NOT_DEFINED)	// copy mdata!!!
		{
			i32s index = ci_to_mdata_table[n1];
			
			chn_vector.back().pdb_chn_id = mdata->chn_vector[index]->chn_id;
			
			if (mdata->chn_vector[index]->seqres != NULL)
			{
				chn_vector.back().pdb_seqres = new char[strlen(mdata->chn_vector[index]->seqres) + 1];
				strcpy(chn_vector.back().pdb_seqres, mdata->chn_vector[index]->seqres);
			}
			else
			{
				chn_vector.back().pdb_seqres = NULL;
			}
			
			chn_vector.back().pdb_missing_residues = mdata->chn_vector[index]->missing_residues;
		}
		
		iter_mm1al range2[2]; i32s res = 0;
		
		while (true)
		{
			mdl->GetRange(2, range1, res, range2);
			if (range2[0] == range2[1]) break;
			
			i32s tmp1[2]; tmp1[0] = (* range2[0]).res_id >> 8;
			for (tmp1[1] = 0;tmp1[1] < (i32s) mm1_mdl::amino_builder.residue_vector.size();tmp1[1]++)
			{
				if (mm1_mdl::amino_builder.residue_vector[tmp1[1]].id == tmp1[0]) break;
			} if (tmp1[1] == (i32s) mm1_mdl::amino_builder.residue_vector.size()) continue;
			
			mm2_res newres = mm2_res(mm1_mdl::amino_builder.residue_vector[tmp1[1]].symbol, cs_vector.size(), & prm);
			
			vector<i32s> idv; idv.push_back(0x01);
			for (i32u n2 = 0;n2 < cs_vector.size();n2++)
			{
				Reduce_sub1(range2[0], idv, newres.crd_vector[0][n2].data, n2);
			}
			
			if (newres.natm > 1)
			{
				idv.resize(0);
				switch (newres.symbol)
				{
					case 'R':
					idv.push_back(0x21); idv.push_back(0x22);
					break;
					
					case 'N':
					idv.push_back(0x21);
					break;
					
					case 'D':
					idv.push_back(0x21);
					break;
					
					case 'C':
					idv.push_back(0x21);
					break;
					
					case 'Q':
					idv.push_back(0x21); idv.push_back(0x22);
					break;
					
					case 'E':
					idv.push_back(0x21); idv.push_back(0x22);
					break;
					
					case 'H':
					idv.push_back(0x21); idv.push_back(0x22);
					idv.push_back(0x23); idv.push_back(0x24); idv.push_back(0x25);
					break;
					
					case 'I':
					idv.push_back(0x22);
					break;
					
					case 'L':
					idv.push_back(0x21);
					break;
					
					case 'K':
					idv.push_back(0x21);
					break;
					
					case 'M':
					idv.push_back(0x21); idv.push_back(0x22); idv.push_back(0x23);
					break;
					
					case 'F':
					idv.push_back(0x21); idv.push_back(0x24);
					break;
					
					case 'W':
					idv.push_back(0x21); idv.push_back(0x23); idv.push_back(0x24);
					break;
					
					case 'Y':
					idv.push_back(0x21); idv.push_back(0x24);
					break;
				}
				
				for (i32u n2 = 0;n2 < cs_vector.size();n2++)
				{
					Reduce_sub1(range2[0], idv, newres.crd_vector[1][n2].data, n2);
				}
			}
			
			if (newres.natm > 2)
			{
				idv.resize(0);
				switch (newres.symbol)
				{
					case 'R':
					idv.push_back(0x24);
					break;
					
					case 'K':
					idv.push_back(0x23);
					break;
					
					case 'W':
					idv.push_back(0x25); idv.push_back(0x28);
					break;
				}
				
				for (i32u n2 = 0;n2 < cs_vector.size();n2++)
				{
					Reduce_sub1(range2[0], idv, newres.crd_vector[2][n2].data, n2);
				}
			}
			
			chn_vector.back().res_vector.push_back(newres);
			res++;		// move to next residue...
		}
		
		// peptide units...
		
		for (i32s n2 = 1;n2 < ((i32s) chn_vector.back().res_vector.size()) - 2;n2++)
		{
			mdl->GetRange(2, range1, n2, range2);
			
			iter_mm1al itc = range2[0];
			while (((* itc).res_id & 0xFF) != 0x01) itc++;
			
			iter_mm1al ito = range2[0];
			while (((* ito).res_id & 0xFF) != 0x10) ito++;
			
			for (i32u n3 = 0;n3 < cs_vector.size();n3++)
			{
				fGL * curr = chn_vector.back().res_vector[n2].crd_vector[0][n3].data;
				fGL * prev = chn_vector.back().res_vector[n2 - 1].crd_vector[0][n3].data;
				fGL * next = chn_vector.back().res_vector[n2 + 1].crd_vector[0][n3].data;
				
				v3d<fGL> v1(curr, prev); v3d<fGL> v2(curr, next);
				v3d<fGL> v3((* ito).crd_vector[n3].data, (* itc).crd_vector[n3].data);
				
				chn_vector.back().res_vector[n2].peptide_vector[n3] = v1.tor(v2, v3);
			}
		}
	}
	
	delete[] ci_to_mdata_table;
	
	// do some output at this stage (the output should be in FASTA format)...
	
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		cout << "imported chain " << n1 << ":" << endl;
		
		if (chn_vector[n1].pdb_seqres != NULL)
		{
			i32s pdb_chn_length = strlen(chn_vector[n1].pdb_seqres);
			
			cout << "> ORIGINAL sequence, length " << pdb_chn_length;
			cout << ", id '" << chn_vector[n1].pdb_chn_id << "':" << endl;
			
			for (i32s n2 = 0;n2 < pdb_chn_length;n2++)
			{
				cout << chn_vector[n1].pdb_seqres[n2];
				
				bool is_break = !((n2 + 1) % 50);
				bool is_end = ((n2 + 1) == pdb_chn_length);
				
				if (is_break || is_end) cout << endl;
			}
		}
		
		cout << "> IMPORTED sequence, length ";
		cout << chn_vector[n1].res_vector.size() << ":" << endl;
		
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			cout << chn_vector[n1].res_vector[n2].symbol;
			
			bool is_break = !((n2 + 1) % 50);
			bool is_end = ((n2 + 1) == chn_vector[n1].res_vector.size());
			
			if (is_break || is_end) cout << endl;
		}
	}
	
	// disulphide bridges...
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		iter_mm1al range1a[2];
		mdl->GetRange(1, n1, range1a);
		
		// intra-chain ones...
		
		vector<i32s> cys_data1;
		vector<mm1_atom *> cys_ref1;
		
		for (i32s n2 = 0;n2 < ci_vector[n1].length;n2++)
		{
			if (ci_vector[n1].sequence[n2] == 'C')
			{
				bool flag = true;
				
				iter_mm1al range1b[2];
				mdl->GetRange(2, range1a, n2, range1b);
				if (range1b[0] == range1b[1]) flag = false;
				
				iter_mm1al it1 = range1b[0];
				while (it1 != range1b[1] && ((* it1).res_id & 0xFF) != 0x21) it1++;
				if (it1 == range1b[1]) flag = false;
				
				if (flag)
				{
					cys_data1.push_back(n2);
					cys_ref1.push_back(& (* it1));
				}
			}
		}
		
		for (i32s n2 = 0;n2 < ((i32s) cys_ref1.size()) - 1;n2++)
		{
			for (i32s n3 = n2 + 1;n3 < (i32s) cys_ref1.size();n3++)
			{
				mm1_bond tb = mm1_bond(cys_ref1[n2], cys_ref1[n3], bondtype('S'));
				iter_mm1bl it1 = find(mdl->bond_list.begin(), mdl->bond_list.end(), tb);
				if (it1 != mdl->bond_list.end())
				{
					mm2_dsb newdsb;
					newdsb.chn[0] = n1; newdsb.res[0] = cys_data1[n2];
					newdsb.chn[1] = n1; newdsb.res[1] = cys_data1[n3];
					dsb_vector.push_back(newdsb);
				}
			}
		}
		
		// interchain ones...
		
		for (i32u n2 = n1 + 1;n2 < ci_vector.size();n2++)
		{
			iter_mm1al range2a[2];
			mdl->GetRange(1, n2, range2a);
			
			vector<i32s> cys_data2;
			vector<mm1_atom *> cys_ref2;
			
			for (i32s n3 = 0;n3 < ci_vector[n2].length;n3++)
			{
				if (ci_vector[n2].sequence[n3] == 'C')
				{
					bool flag = true;
					
					iter_mm1al range2b[2];
					mdl->GetRange(2, range2a, n3, range2b);
					if (range2b[0] == range2b[1]) flag = false;
					
					iter_mm1al it1 = range2b[0];
					while (it1 != range2b[1] && ((* it1).res_id & 0xFF) != 0x21) it1++;
					if (it1 == range2b[1]) flag = false;
					
					if (flag)
					{
						cys_data2.push_back(n3);
						cys_ref2.push_back(& (* it1));
					}
				}
			}
			
			for (i32u n3 = 0;n3 < cys_ref1.size();n3++)
			{
				for (i32u n4 = 0;n4 < cys_ref2.size();n4++)
				{
					mm1_bond tb = mm1_bond(cys_ref1[n3], cys_ref2[n4], bondtype('S'));
					iter_mm1bl it1 = find(mdl->bond_list.begin(), mdl->bond_list.end(), tb);
					if (it1 != mdl->bond_list.end())
					{
						mm2_dsb newdsb;
						newdsb.chn[0] = n1; newdsb.res[0] = cys_data1[n3];
						newdsb.chn[1] = n2; newdsb.res[1] = cys_data2[n4];
						dsb_vector.push_back(newdsb);
					}
				}
			}
		}
	}
	
	// secondary structure...
	
	// the default state is always loop. strands have the same places as in K&S.
	// helices are shifted: the smallest helical peptide is "4...." -> "LHHHL"!!!!!
	
	for (i32u n1 = 0;n1 < ci_vector.size();n1++)
	{
		for (i32s n2 = 0;n2 < ci_vector[n1].length;n2++)
		{
			switch (ci_vector[n1].state[n2])
			{
				case 'S':
				chn_vector[n1].res_vector[n2].state = STATE_STRAND;
				break;
				
				default:
				chn_vector[n1].res_vector[n2].state = STATE_LOOP;
				break;
			}
		}
				
		for (i32s n2 = 2;n2 < ci_vector[n1].length - 2;n2++)
		{
			switch (ci_vector[n1].state[n2 - 2])
			{
				case '4':
				chn_vector[n1].res_vector[n2 - 1].state = STATE_HELIX;
				chn_vector[n1].res_vector[n2 + 0].state = STATE_HELIX;
				chn_vector[n1].res_vector[n2 + 1].state = STATE_HELIX;
				break;
			}
		}
	}
	
	// still it would be best to check that results of the above are ok:
	// * secondary structural elements should be separated by at least one loop-residue!!!
	// * what is the smallest meaningful element ??? FOR HELIX 3, FOR STRAND 1 !!!!!!!!!!!!
	
	UpdateConstraints();
}

void mm2_mdl::Reduce_sub1(iter_mm1al it1, vector<i32s> & idv, fGL * crd, i32u cset)
{
	vector<mm1_atom *> refv;
	for (i32u n1 = 0;n1 < idv.size();n1++)
	{
		iter_mm1al it2 = it1;
		while (((* it2).res_id & 0xFF) != idv[n1]) it2++;
		refv.push_back(& (* it2));
	}
	
	for (i32u n1 = 0;n1 < 3;n1++)
	{
		crd[n1] = 0.0;
		for (i32u n2 = 0;n2 < refv.size();n2++)
		{
			crd[n1] += refv[n2]->crd_vector[cset][n1];
		}
		
		crd[n1] /= (f64) refv.size();
	}
}

// the name of this function is quite misleading...
// the name of this function is quite misleading...
// the name of this function is quite misleading...

// here we only set charges for terminal residues!!!!!!!

// HOW TO PROPERLY DETERMINE THE STATES???? DEPENDENT ON pH AND/OR GEOMETRY?!?!?!?
// HOW TO PROPERLY DETERMINE THE STATES???? DEPENDENT ON pH AND/OR GEOMETRY?!?!?!?
// HOW TO PROPERLY DETERMINE THE STATES???? DEPENDENT ON pH AND/OR GEOMETRY?!?!?!?

void mm2_mdl::UpdateConstraints(void)
{
	// first, we calculate energy just to get the sasa values...
	
	mm2_eng * eng = new mm2_eng(* this);
	CopyCRD(this, eng, 0); eng->Compute(0);
	
	i32u counter = 0;
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				f64 sasa = eng->stored_sasa_values[counter++];
				chn_vector[n1].res_vector[n2].tmp_sasa[n3] = sasa;
			}
		}
	}
	
	delete eng;
	
	// initialize the charges...
	
	const i32s smoothing = 10;
	
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < 3;n3++) chn_vector[n1].res_vector[n2].tmp_smooth[n3] = NULL;
			
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				chn_vector[n1].res_vector[n2].charge[n3] = 0.0;
				
				bool is_acid = false; bool is_base = false;
				f64 pKa = NOT_DEFINED; f64 sasa1 = NOT_DEFINED;
				GetDefaultDConst(n1, n2, n3, is_acid, is_base, pKa, sasa1);
				if (!is_acid && !is_base) continue;
				
				// pKa = pH + log([HX]/[X-])
				// log([X-]/[HX]) = pH - pKa = log(x/(1-x))
				// x = 10^(pH - pKa) / (1 + 10^(pH - pKa))
				
				f64 tmp1 = pow(10.0, pH - pKa);
				f64 tmp2 = tmp1 / (1.0 + tmp1);
				
				if (is_acid) chn_vector[n1].res_vector[n2].charge[n3] = -tmp2;
				if (is_base) chn_vector[n1].res_vector[n2].charge[n3] = 1.0 - tmp2;
				
				f64 charge = chn_vector[n1].res_vector[n2].charge[n3];
				chn_vector[n1].res_vector[n2].tmp_smooth[n3] = new f64[smoothing];
				for (i32s n4 = 0;n4 < smoothing;n4++)
				{
					chn_vector[n1].res_vector[n2].tmp_smooth[n3][n4] = charge;
				}
			}
		}
	}
	
	// the loop...
	
//ofstream logf("/tmp/pKa.log", ios::out);
	for (i32s loop = 0;loop < 250;loop++)
	{
		for (i32u n1 = 0;n1 < chn_vector.size();n1++)
		{
			for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
			{
				for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
				{
					bool is_acid = false; bool is_base = false;
					f64 pKa = NOT_DEFINED; f64 sasa1 = NOT_DEFINED;
					GetDefaultDConst(n1, n2, n3, is_acid, is_base, pKa, sasa1);
					if (!is_acid && !is_base) continue;
					
					f64 k1 = pow(10.0, -pKa);
					f64 dg = -8.31451 * 300.0 * log(k1);
					
					f64 delta = 0.0;
					f64 charge1 = chn_vector[n1].res_vector[n2].charge[n3];
					fGL * crd1 = chn_vector[n1].res_vector[n2].crd_vector[n3][0].data;
					for (i32u n4 = 0;n4 < chn_vector.size();n4++)
					{
						for (i32u n5 = 0;n5 < chn_vector[n4].res_vector.size();n5++)
						{
							for (i32u n6 = 0;n6 < chn_vector[n4].res_vector[n5].natm;n6++)
							{
								if (n1 == n4 && n2 == n5 && n3 == n6) continue;
								if (chn_vector[n4].res_vector[n5].charge[n6] == 0.0) continue;
								
								fGL * crd2 = chn_vector[n4].res_vector[n5].crd_vector[n6][0].data;
								
								f64 r2 = 0.0;
								for (i32s n9 = 0;n9 < 3;n9++)
								{
									fGL tmp1 = crd1[n9] - crd2[n9];
									r2 += tmp1 * tmp1;
								}
								
								if (r2 < 0.01) continue;	// always skip if r < 1  !!!
								f64 r1 = sqrt(r2);
								
					if (r1 > 1.5) continue;		// also skip large distances; charge neutralization???
								
								// e = A + B/(1+k*exp(L*B*r))
								// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2
								
							f64 t7a = exp(prm.screen2 * prm.screen4 * r1); 	f64 t7b = 1.0 + prm.screen3 * t7a;
							f64 t7d = prm.screen1 + prm.screen2 / t7b;
								
								// f = Q/(e*r)
								// df/dr = -Q * ((1/e*r^2) + (de/dr)/(e^2*r))
								
								f64 qq = 138.9354518 * charge1 * chn_vector[n4].res_vector[n5].charge[n6];
								delta += qq / (t7d * r1);
							}
						}
					}
					
					// here "cc" is a coupling constant used to stabilize the fit...
					// here "cc" is a coupling constant used to stabilize the fit...
					// here "cc" is a coupling constant used to stabilize the fit...
					
					f64 cc = loop / 100.0;		// this is for 250 steps!!!
					const f64 limit = 1.00; if (cc > limit) cc = limit;
					
					if (is_acid) dg += cc * prm.charge_wes * delta * 1000.0;	// plus!!!
					if (is_base) dg -= cc * prm.charge_wes * delta * 1000.0;	// minus!!!
					
					f64 tmpA = chn_vector[n1].res_vector[n2].tmp_sasa[n3] - sasa1;
		//	if (tmpA < 0.0) tmpA = 0.0; else tmpA = tmpA * tmpA;		// square?
					if (tmpA < 0.0) tmpA /= 2.0;	// enhance the "burial" effect???
					f64 tmpB = tmpA * (prm.charge_sasa2 + prm.charge_sasa3 * charge1 * charge1);
					if (is_acid) dg -= cc * tmpB * 1000.0;		// minus?
					if (is_base) dg += cc * tmpB * 1000.0;		// plus?
					
					f64 k2 = exp(-dg / (8.31451 * 300.0));
					pKa = -log10(k2);
					
	chn_vector[n1].res_vector[n2].tmp_z1[n3] = delta;
	chn_vector[n1].res_vector[n2].tmp_z2[n3] = chn_vector[n1].res_vector[n2].tmp_sasa[n3];
					
					f64 tmp1 = pow(10.0, pH - pKa);
					f64 tmp2 = tmp1 / (1.0 + tmp1);
					
					if (is_acid) chn_vector[n1].res_vector[n2].tmp_charge[n3] = -tmp2;
					if (is_base) chn_vector[n1].res_vector[n2].tmp_charge[n3] = 1.0 - tmp2;
					
					chn_vector[n1].res_vector[n2].tmp_newpKa[n3] = pKa;	// do these oscillate as well???
				}
			}
		}
		
		f64 delta = 0.0;
		
		for (i32u n1 = 0;n1 < chn_vector.size();n1++)
		{
			for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
			{
				for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
				{
					bool is_acid = false; bool is_base = false;
					f64 pKa = NOT_DEFINED; f64 sasa1 = NOT_DEFINED;
					GetDefaultDConst(n1, n2, n3, is_acid, is_base, pKa, sasa1);
					if (!is_acid && !is_base) continue;	// obs!!!
					
					f64 ch1 = chn_vector[n1].res_vector[n2].charge[n3];
					f64 ch2 = chn_vector[n1].res_vector[n2].tmp_charge[n3];
					
					for (i32u n4 = 0;n4 < smoothing - 1;n4++)
					{
						f64 tmp1 = chn_vector[n1].res_vector[n2].tmp_smooth[n3][n4 + 1];
						chn_vector[n1].res_vector[n2].tmp_smooth[n3][n4 + 0] = tmp1;
					}
					
					chn_vector[n1].res_vector[n2].tmp_smooth[n3][smoothing - 1] = ch2;
					
					ch2 = 0.0;	// now do the smoothing by averaging the array...
					for (i32s n4 = 0;n4 < smoothing;n4++) ch2 += chn_vector[n1].res_vector[n2].tmp_smooth[n3][n4];
					ch2 /= (f64) smoothing;
					
					f64 dch = ch2 - ch1;
					delta += dch * dch;
					
					chn_vector[n1].res_vector[n2].charge[n3] = ch2;
//logf << ch2 << " ";
				}
			}
		}
//logf << endl;
		
		cout << "cycle = " << loop << " delta = " << delta << endl;
//		if ((loop % 50) == 49) { int qqq; cin >> qqq; }
	}
//logf.close();
	
	// report the results briefly...
	
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				chn_vector[n1].res_vector[n2].label[n3] = 0.0;		// LABELS!!!
				
				bool is_acid = false; bool is_base = false;
				f64 pKa = NOT_DEFINED; f64 sasa1 = NOT_DEFINED;
				GetDefaultDConst(n1, n2, n3, is_acid, is_base, pKa, sasa1);
				if (!is_acid && !is_base) continue;
				
				f64 dpKa = chn_vector[n1].res_vector[n2].tmp_newpKa[n3] - pKa;
				f64 charge = chn_vector[n1].res_vector[n2].charge[n3];

				chn_vector[n1].res_vector[n2].label[n3] = charge;	// LABELS!!!
				
				cout << "chn " << n1 << " res " << n2 << " " << chn_vector[n1].res_vector[n2].symbol << " ";
				cout << "pKa = " << chn_vector[n1].res_vector[n2].tmp_newpKa[n3] << " (shift = " << dpKa << ") ";
				cout << "charge = " << charge << endl;
				
			}
			
			for (i32u n3 = 0;n3 < 3;n3++)
			{
				if (!chn_vector[n1].res_vector[n2].tmp_smooth[n3]) continue;
				delete[] chn_vector[n1].res_vector[n2].tmp_smooth[n3];
			}
			
			// finally, update the sasa values as well since they are charge-related!!!
			// finally, update the sasa values as well since they are charge-related!!!
			// finally, update the sasa values as well since they are charge-related!!!
			
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				const char symbols[20 + 1] = "ARNDCQEGHILKMFPSTWYV";
				
				const i32s ntypes = 37;
				const i32s types[ntypes] =
				{
					0x0100,
					0x0200, 0x0201, 0x0202,
					0x0300, 0x0301,
					0x0400, 0x0401,
					0x0500, 0x0501,
					0x0600, 0x0601,
					0x0700, 0x0701,
					0x0800,
					0x0900, 0x0901,
					0x0A00, 0x0A01,
					0x0B00, 0x0B01,
					0x0C00, 0x0C01, 0x0C02,
					0x0D00, 0x0D01,
					0x0E00, 0x0E01,
					0x0F00,
					0x1000,
					0x1100,
					0x1200, 0x1201, 0x1202,
					0x1300, 0x1301,
					0x1400
				};
				
				char tmp1a = chn_vector[n1].res_vector[n2].symbol;
				i32s tmp1b = 0; while (tmp1b < 20 && symbols[tmp1b] != tmp1a) tmp1b++;
				if (tmp1b == 20) { cout << "symbol error #1"; exit(EXIT_FAILURE); }
				
				tmp1b++; tmp1b = tmp1b << 8; tmp1b += n3;
				i32s tmp1c = 0; while (tmp1c < ntypes && types[tmp1c] != tmp1b) tmp1c++;
				if (tmp1c == ntypes) { cout << "type error #1"; exit(EXIT_FAILURE); }
				
				f64 charge = chn_vector[n1].res_vector[n2].charge[n3];
				f64 sasa = prm.sasa1[0] * (prm.sasa9[tmp1c] + charge * charge * prm.sasa1[1]);
				
				chn_vector[n1].res_vector[n2].sasa[n3] = sasa;
			}
		}
	}
}

void mm2_mdl::GetDefaultDConst(i32u n1, i32u n2, i32u n3, bool & is_acid, bool & is_base, f64 & pKa, f64 & sasa1)
{
	i32s index = NOT_DEFINED;
	
	if (!n2 && n3 == 0) index = 0;							// n-terminal
	if ((n2 + 1) == chn_vector[n1].res_vector.size() && n3 == 0) index = 1;		// c-terminal
	
	if (chn_vector[n1].res_vector[n2].symbol == 'R' && n3 == 2) index = 2;
	if (chn_vector[n1].res_vector[n2].symbol == 'D' && n3 == 1) index = 3;
	if (chn_vector[n1].res_vector[n2].symbol == 'C' && n3 == 1) index = 4;		// cysteine!!!
	if (chn_vector[n1].res_vector[n2].symbol == 'E' && n3 == 1) index = 5;
	if (chn_vector[n1].res_vector[n2].symbol == 'H' && n3 == 1) index = 6;
	if (chn_vector[n1].res_vector[n2].symbol == 'K' && n3 == 2) index = 7;
	if (chn_vector[n1].res_vector[n2].symbol == 'Y' && n3 == 1) index = 8;
	
	// check the cases where cysteine is disulphide-bridged.
	// check the cases where cysteine is disulphide-bridged.
	// check the cases where cysteine is disulphide-bridged.
	
	if (index == 4)		// test for cysteine, see above...
	{
		for (i32u n8 = 0;n8 < dsb_vector.size();n8++)
		{
			for (i32u n9 = 0;n9 < 2;n9++)
			{
				bool test1 = (dsb_vector[n8].chn[n9] == (i32s) n1);
				bool test2 = (dsb_vector[n8].res[n9] == (i32s) n2);
				if (test1 && test2) return;
			}
		}
	}
	
	if (index != NOT_DEFINED)
	{
		is_acid = prm.charge_acid[index];
		is_base = !prm.charge_acid[index];
		
		pKa = prm.charge_pKa[index];
		sasa1 = prm.charge_sasa1[index];
	}
}

/*##############################################*/
/*##############################################*/

// THIS IS NOT YET READY FOR MULTIPLE CRD-SETS!!!!!!!!!
// THIS IS NOT YET READY FOR MULTIPLE CRD-SETS!!!!!!!!!
// THIS IS NOT YET READY FOR MULTIPLE CRD-SETS!!!!!!!!!

bool mm2_mdl::ReadStream(istream & istr, bool selected, bool)
{
	i32s ncs; istr >> buffer >> ncs;
	PushCRDSets(ncs - cs_vector.size());
	
	i32s n1; i32s n2; i32s n3; i32s n4;
	char tmp1; char tmp2; fGL tmp3;
	
	while (istr.peek() != '#')
	{
		char peek1 = istr.peek();
		
		if (peek1 == 'c')		// chn
		{
			mm2_chn newchn;
			chn_vector.push_back(newchn);
		}
		
		if (peek1 == 'p')		// pdb_???
		{
			for (n1 = 0;n1 < 4;n1++) istr.get();
			char peek2 = istr.peek();
			
			if (peek2 == 'c')		// pdb_chain
			{
				istr.getline(buffer, sizeof(buffer), '\'');
				chn_vector.back().pdb_chn_id = istr.get();
			}
			
			if (peek2 == 's')		// pdb_seqres
			{
				char seqres[65536];
				istr >> buffer >> seqres;
				
				chn_vector.back().pdb_seqres = new char[strlen(seqres) + 1];
				strcpy(chn_vector.back().pdb_seqres, seqres);
			}
		
			if (peek2 == 'm')		// pdb_missing
			{
				istr >> buffer >> n1;
				istr.getline(buffer, sizeof(buffer), ':');
				
				for (n2 = 0;n2 < n1;n2++)
				{
					istr >> n3;
					chn_vector.back().pdb_missing_residues.push_back(n3);
				}
			}
		}
		
		if (peek1 == 'r')		// res
		{
			istr >> buffer >> n1 >> tmp1 >> tmp2 >> tmp3;
			istr.getline(buffer, sizeof(buffer));
			
			mm2_res newres = mm2_res(tmp1, cs_vector.size(), & prm);
			
			switch (tmp2)
			{
				case 'H':	newres.state = STATE_HELIX; break;
				case 'S':	newres.state = STATE_STRAND; break;
				default:	newres.state = STATE_LOOP;
			}
			
			for (n1 = 0;n1 < ncs;n1++) newres.peptide_vector[n1] = tmp3;
			
			for (n1 = 0;n1 < (i32s) newres.natm;n1++)
			{
				for (n2 = 0;n2 < ncs;n2++)
				{
					istr >> tmp3; newres.crd_vector[n1][n2][0] = tmp3;
					istr >> tmp3; newres.crd_vector[n1][n2][1] = tmp3;
					istr >> tmp3; newres.crd_vector[n1][n2][2] = tmp3;
				}
				
				if (n1 + 1 == (i32s) newres.natm) continue;
				else istr.getline(buffer, sizeof(buffer));
			}
			
			chn_vector.back().res_vector.push_back(newres);
		}
		
		if (peek1 == 'B')		// bridge
		{
			mm2_dsb newdsb; istr.get();
			istr >> n1 >> n2 >> n3 >> n4;
			
			newdsb.chn[0] = n1; newdsb.res[0] = n2;
			newdsb.chn[1] = n3; newdsb.res[1] = n4;
			
			dsb_vector.push_back(newdsb);
		}
		
		istr.getline(buffer, sizeof(buffer));
	}
	
	UpdateConstraints();
	
	return true;
}

void mm2_mdl::WriteStream(ostream & ostr)
{
	ostr << "ncs " << cs_vector.size() << endl;
	
	for (i32u n1 = 0;n1 < chn_vector.size();n1++)
	{
		ostr << "chn " << n1 << endl;
		
		if (chn_vector[n1].pdb_seqres != NULL)
		{
			ostr << "pdb_chain '" << chn_vector[n1].pdb_chn_id << "' " << endl;
			ostr << "pdb_seqres " << chn_vector[n1].pdb_seqres << " " << endl;
			
			ostr << "pdb_missing " << chn_vector[n1].pdb_missing_residues.size() << " : ";
			for (i32u n2 = 0;n2 < chn_vector[n1].pdb_missing_residues.size();n2++)
			{
				ostr << chn_vector[n1].pdb_missing_residues[n2] << " ";
			}	ostr << endl;
		}
		
		for (i32u n2 = 0;n2 < chn_vector[n1].res_vector.size();n2++)
		{
			ostr << "res " << n2 << " ";
			ostr << chn_vector[n1].res_vector[n2].symbol << " ";
			
			switch (chn_vector[n1].res_vector[n2].state)
			{
				case STATE_HELIX:	ostr << "H"; break;
				case STATE_STRAND:	ostr << "S"; break;
				default:		ostr << "L";
			} ostr << " ";
			
			for (i32u n3 = 0;n3 < cs_vector.size();n3++)
			{
				ostr << chn_vector[n1].res_vector[n2].peptide_vector[n3] << endl;
			}
			
			for (i32u n3 = 0;n3 < chn_vector[n1].res_vector[n2].natm;n3++)
			{
				for (i32u n4 = 0;n4 < cs_vector.size();n4++)
				{
					ostr << chn_vector[n1].res_vector[n2].crd_vector[n3][n4][0] << " ";
					ostr << chn_vector[n1].res_vector[n2].crd_vector[n3][n4][1] << " ";
					ostr << chn_vector[n1].res_vector[n2].crd_vector[n3][n4][2] << " ";
				} ostr << endl;
			}
		}
		
		ostr << "endchn" << endl;
	}
	
	for (i32u n1 = 0;n1 < dsb_vector.size();n1++)
	{
		ostr << "B ";
		ostr << dsb_vector[n1].chn[0] << " ";
		ostr << dsb_vector[n1].res[0] << " ";
		ostr << dsb_vector[n1].chn[1] << " ";
		ostr << dsb_vector[n1].res[1] << endl;
	}
	
	ostr << "#end" << endl;
}

void mm2_mdl::ImportPDB(const char * fn, i32s chain_index)
{
	mm1_mdl * mdl1 = new mm1_mdl(NULL, * console_class_factory::GetInstance());
	
	mm1_readpdb_mdata * mdata = mdl1->readpdb_ReadMData(fn);
	mdl1->readpdb_ReadData(fn, mdata, chain_index);
	Reduce(mdl1, mdata);
	
	delete mdl1; delete mdata;
}

/*##############################################*/
/*##############################################*/

void mm2_mdl::DoEnergy(void)
{
	mm2_eng * eng = new mm2_eng(* this);
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Calculating Energy ";
	str1 << "(simplified forcefield";
	str1 << ", engine = " << "the default engine";
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, eng, 0); eng->Compute(0);
	
	char buffer[1024];
	ostrstream str(buffer, sizeof(buffer)); str.setf(ios::fixed); str.precision(8);
	
// this will print also the components...
// this will print also the components...
// this will print also the components...
cout << "ENERGY COMPONENTS:" << endl;
cout << eng->energy_bt1 << " ";
cout << eng->energy_bt2 << " ";
cout << eng->energy_bt3a << " " << eng->energy_bt3b << " ";
cout << eng->energy_bt4a << " " << eng->energy_bt4b << " ";
cout << eng->energy_nbt1a << " " << eng->energy_nbt1b << " " << eng->energy_nbt1c << " ";
cout << eng->energy_nbt2a << " " << eng->energy_nbt2b << " " << eng->energy_nbt2c << " ";
cout << eng->energy_nbt3a << " " << eng->energy_nbt3b << " ";
cout << eng->energy_pnlt << endl;

	str << "Energy = " << eng->energy << " kJ/mol + " << eng->constraints << " kJ/mol = ";
	str << (eng->energy + eng->constraints) << " kJ/mol" << endl << ends;
	
	PrintToLog(buffer);
	delete eng;
}

void mm2_mdl::DoGeomOpt(mm2_geomopt_param & param)
{
	GeomOptGetParam(param);
	if (!param.confirm) return;
	
	mm2_eng * eng = new mm2_eng(* this);
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Starting Geometry Optimization ";
	str1 << "(simplified forcefield";
	str1 << ", engine = " << "the default engine";
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, eng, 0);
	mm2_geomopt * opt = new mm2_geomopt(eng, 50, 0.025);	// optimal settings?!?!?
	
//eng->SetPenaltyData();	// save the template...
//eng->penalty_weight = ???;	// use the penalty function...

	char buffer[1024];
	f64  last_energy = 0.0;		// this is for output and delta_e test...

	PrintToLog("Cycle    Energy    Gradient       Step        Delta E\n");

	i32s n1 = 0;	// n1 counts the number of steps...
	while (true)
	{
		opt->TakeCGStep(conjugate_gradient::Newton2An);
		
// problem: the gradient information is in fact not precise in this stage. the current gradient
// is the one that was last calculated in the search, and it is not necessarily the best one.
// to update the gradient, we need to Compute(1) here...
	eng->Compute(1);	// this is not vital, but will update the gradient vector length...
	
		if (n1 != 0)
		  {
		    sprintf(buffer, "%4d %12.5f %10.4e %10.4e %10.4e \n", n1,
			    opt->optval, eng->GetGradientVectorLength(),
			    opt->optstp, last_energy - opt->optval);
		  }
		else
		  {
		    sprintf(buffer, "%4d %12.5f %10.4e %10.4e ********** \n", n1,
			    opt->optval, eng->GetGradientVectorLength(),
			    opt->optstp);
		  }
		PrintToLog(buffer);
		
		bool terminate = false;
		
		if (param.enable_nsteps)	// the nsteps test...
		{
			if (n1 >= param.treshold_nsteps)
			{
				terminate = true;
				PrintToLog("the nsteps termination test was passed.\n");
			}
		}
		
		// no other tests implemented yet...
		// no other tests implemented yet...
		// no other tests implemented yet...
		
		if (!(n1 % 10) || terminate)
		{
			CopyCRD(eng, this, 0); CenterCRDSet(0);
			UpdateAllGraphicsViews(true);
		}
		
		if (terminate) break;		// exit the loop here!!!
		
		n1++;	// update the number of steps...
	}
	
	delete opt;
	delete eng;
}

void mm2_mdl::GeomOptGetParam(mm2_geomopt_param & param)
{
	param.confirm = true;
}

#define START 15000
#define S_TC_NSTEPS 100
#define S_TC_COUPLING 1.0

#define EQUIL 5000
#define E_TC_NSTEPS 250
#define E_TC_COUPLING 0.25

void mm2_mdl::DoMolDyn(mm2_moldyn_param & param)
{
	MolDynGetParam(param);
	if (!param.confirm) return;
	
	mm2_eng * eng = new mm2_eng(* this);
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Starting Molecular Dynamics ";
	str1 << "(simplified forcefield";
	str1 << ", engine = " << "the default engine";
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, eng, 0);
	
	mm2_moldyn * dyn = new mm2_moldyn(eng, param.temperature, param.timestep);
	
//eng->SetPenaltyData();		// save the template...
//eng->penalty_weight = 0.0;	// use the penalty function...

	// WHAT IS THE LARGEST STABLE INTEGRATION TIME STEP !?!?!??!
	// WHAT IS THE LARGEST STABLE INTEGRATION TIME STEP !?!?!??!
	// WHAT IS THE LARGEST STABLE INTEGRATION TIME STEP !?!?!??!
	
	ofstream ofile;
	ofile.open(param.filename, ios::out | ios::binary);
	
	const char file_id[10] = "traj_v10";
	const int number_of_atoms = GetVirtAtomCount();
	const int total_frames = param.nsteps / 100;
	
	ofile.write((char *) file_id, 8);					// file id, 8 chars.
	ofile.write((char *) & number_of_atoms, sizeof(number_of_atoms));	// number of atoms, int.
	ofile.write((char *) & total_frames, sizeof(total_frames));		// total number of frames, int.
	
	i32s ccount = 0; f64 control = 0.0;
	
	for (i32s n1 = 0;n1 < START + EQUIL + param.nsteps;n1++)
	{
		dyn->TakeMDStep();
		if (!(n1 % 50))		// 250
		{
			ostrstream str2(mbuff1, sizeof(mbuff1));
			str2 << "step " << n1 << " T = " << dyn->ConvEKinTemp(dyn->kin);
			str2 << " Epot = " << dyn->pot << " Etot = " << (dyn->kin + dyn->pot) << endl << ends;
			PrintToLog(mbuff1);
		}
		
	//	if (n1 < START + EQUIL)		// remove this to enable TC in simulation...
		{
			ccount++; control += dyn->kin;
			
			bool flag = false;
			if (n1 < START && ccount == S_TC_NSTEPS) flag = true;
			else if (ccount == E_TC_NSTEPS) flag = true;
			
			if (flag)
			{
				f64 ekin = control / (f64) ccount;
				
				ostrstream str2(mbuff1, sizeof(mbuff1));
				str2 << "average Ekin = " << ekin << " kJ/mol = " << dyn->ConvEKinTemp(ekin) << " K." << endl << ends;
				PrintToLog(mbuff1);
				
				f64 coupling = E_TC_COUPLING; f64 rtemp = 1.0;
				if (n1 < START)
				{
					coupling = S_TC_COUPLING;
					rtemp = (f64) (n1 + 1) / (f64) START;
				}
				
				dyn->SetEKin(dyn->kin + coupling * (dyn->ConvTempEKin(dyn->temp * rtemp) - ekin));
				
				ccount = 0; control = 0.0;
			}
		}
/*
if (!(n1 % 10))		// penalty function adjustment...
{
	f64 rms = sqrt(eng->penalty_sum2 / number_of_atoms);
	cout << "RMS = " << rms << "   ";
	
	f64 e1 = eng->penalty_weight * eng->penalty_sum2;
	
	eng->penalty_weight += 2.5 * (rms - 0.2);
	if (eng->penalty_weight < 0.0) eng->penalty_weight = 0.0;
	cout << "penalty_weight = " << eng->penalty_weight << endl;
	
	f64 e2 = eng->penalty_weight * eng->penalty_sum2;
	
	f64 new_ekin = dyn->kin - (e2 - e1);
	dyn->SetEKin(new_ekin < 0.0 ? 0.0 : new_ekin);
}
*/
		if (!(n1 < START + EQUIL) && !(n1 % 100))
		{
			CopyCRD(eng, this, 0);
			
			const float ekin = dyn->kin;
			const float epot = dyn->pot;
			
			ofile.write((char *) & ekin, sizeof(ekin));	// kinetic energy, float.
			ofile.write((char *) & epot, sizeof(epot));	// potential energy, float.
			
			for (i32u t1 = 0;t1 < chn_vector.size();t1++)
			{
				for (i32u t2 = 0;t2 < chn_vector[t1].res_vector.size();t2++)
				{
					for (i32u t3 = 0;t3 < chn_vector[t1].res_vector[t2].natm;t3++)
					{
						for (i32u t4 = 0;t4 < 3;t4++)	// all coordinates, float.
						{
							float t1a = chn_vector[t1].res_vector[t2].crd_vector[t3][0][t4];
							ofile.write((char *) & t1a, sizeof(t1a));
						}
						
						float t1a = chn_vector[t1].res_vector[t2].label[t3];	// PENALTYDEBUG!!!
						ofile.write((char *) & t1a, sizeof(t1a));
					}
				}
			}
		}
		
		if (!(n1 % 50))
		{
			CopyCRD(eng, this, 0);
			UpdateAllGraphicsViews(true);
		}
	}
	
	ofile.close();
	
	delete dyn; delete eng;
	UpdateAccumValues();
}

void mm2_mdl::MolDynGetParam(mm2_moldyn_param & param)
{
	param.confirm = true;
}

/*##############################################*/
/*##############################################*/

void mm2_mdl::OpenTrajectory(const char * fn)
{
	if (!trajfile)
	{
		trajfile = new ifstream(fn, ios::in | ios::binary);
		
		trajfile->seekg(8, ios::beg);	// skip the file id...
		
		int natoms;
		trajfile->read((char *) & natoms, sizeof(natoms));
		
		if (natoms != GetVirtAtomCount())
		{
			PrintToLog("incompatible file : different number of atoms!\n");
			CloseTrajectory(); return;
		}
		
		trajfile->read((char *) & total_traj_frames, sizeof(total_traj_frames));

		char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
		str1 << "the trajectory file contains " << total_traj_frames << " frames." << endl << ends;
		PrintToLog(mbuff1);
		
		current_traj_frame = 0;
	}
	else PrintToLog("trajectory already exists!\n");
}

void mm2_mdl::CloseTrajectory(void)
{
	if (trajfile != NULL)
	{
		trajfile->close();
		delete trajfile;
		
		trajfile = NULL;
	}
}

void mm2_mdl::ReadFrame(void)
{
	i32s atoms = GetVirtAtomCount();
	
	i32s place = 8 + 2 * sizeof(int);					// skip the header...
//	place += (2 + 3 * atoms) * sizeof(float) * current_traj_frame;		// get the correct frame... ORIGINAL!!!
	place += (2 + 4 * atoms) * sizeof(float) * current_traj_frame;		// get the correct frame... PENALTYDEBUG!!!
	place += 2 * sizeof(float);						// skip epot and ekin...
	
	trajfile->seekg(place, ios::beg);
	
	for (i32u t1 = 0;t1 < chn_vector.size();t1++)
	{
		for (i32u t2 = 0;t2 < chn_vector[t1].res_vector.size();t2++)
		{
			for (i32u t3 = 0;t3 < chn_vector[t1].res_vector[t2].natm;t3++)
			{
				for (i32u t4 = 0;t4 < 3;t4++)
				{
					float t1a; trajfile->read((char *) & t1a, sizeof(t1a));
					chn_vector[t1].res_vector[t2].crd_vector[t3][0][t4] = t1a;
				}
				
				float t1a; trajfile->read((char *) & t1a, sizeof(t1a));		// PENALTYDEBUG!!!
				chn_vector[t1].res_vector[t2].label[t3] = t1a;
			}
		}
	}
}

/*################################################################################################*/
/*################################################################################################*/

// f = a/r
// df/dr = -a/r^2

// f = a/r^2
// df/dr = -2a/r^-3

fGL mm2_GetESPValue(fGL * crd, mm2_mdl * ref, fGL * grad)
{
	fGL vdwsv = 0.0;
	if (grad != NULL) grad[0] = grad[1] = grad[2] = 0.0;
	
	// contribution from charges. PRIMARY!!! this will give the large-scale shape...
	// contribution from charges. PRIMARY!!! this will give the large-scale shape...
	// contribution from charges. PRIMARY!!! this will give the large-scale shape...
	
	for (i32u chn = 0;chn < ref->chn_vector.size();chn++)
	{
		for (i32u res = 0;res < ref->chn_vector[chn].res_vector.size();res++)
		{
			for (i32u atm = 0;atm < ref->chn_vector[chn].res_vector[res].natm;atm++)
			{
				fGL charge = ref->chn_vector[chn].res_vector[res].charge[atm];
				if (charge == 0.0) continue;
				
				fGL tmp1[3]; fGL r2 = 0.0;
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					tmp1[n1] = crd[n1] - ref->chn_vector[chn].res_vector[res].crd_vector[atm][0][n1];
					r2 += tmp1[n1] * tmp1[n1];
				}
				
				if (r2 == 0.0) return +1.0e+35;		// numeric_limits<fGL>::max()?!?!?!
				fGL r1 = sqrt(r2);
				
				// do we have a correct constant here??? I think so, if we define
				// electrostatic potential as potential energy of a unit positive charge.
				
	// e = A + B/(1+k*exp(L*B*r))
	// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2
	
	f64 t7a = exp(ref->prm.screen2 * ref->prm.screen4 * r1); f64 t7b = 1.0 + ref->prm.screen3 * t7a;
	f64 t7d = ref->prm.screen1 + ref->prm.screen2 / t7b;
	
	f64 t7c = t7b * t7b; f64 t7e = t7d * t7d;
	
	f64 t7f = t7a * ref->prm.screen2 * ref->prm.screen2 * ref->prm.screen3 * ref->prm.screen4;
	
				// f = Q/(e*r)
				// df/dr = -Q * ((1/e*r^2) + (de/dr)/(e^2*r))
				
				fGL qq = 138.9354518 * charge;
				vdwsv += qq / (t7d * r1);
				
				if (grad != NULL)	// sign??? constants ???
				{
					fGL df = -qq * (1.0 / (t7d * r2) - t7f / (t7c * t7e * r1));
					
					for (i32s n1 = 0;n1 < 3;n1++)
					{
						fGL dr = tmp1[n1] / r1;
						grad[n1] += df * dr;
					}
				}
			}
		}
	}
	
	// contribution from dipoles. SECONDARY!!! mostly just adds complexity (and runtime)...
	// contribution from dipoles. SECONDARY!!! mostly just adds complexity (and runtime)...
	// contribution from dipoles. SECONDARY!!! mostly just adds complexity (and runtime)...
	
/*	// DISABLED, TO MAKE THIS RUN FASTER...

	for (i32u chn = 0;chn < ref->chn_vector.size();chn++)
	{
		for (i32s res = 1;res < ((i32s) ref->chn_vector[chn].res_vector.size()) - 2;res++)
		{
			if (ref->chn_vector[chn].res_vector[res + 1].symbol == 'P') continue;
			
			fGL * pp = ref->chn_vector[chn].res_vector[res - 1].crd_vector[0][0].data;
			fGL * p1 = ref->chn_vector[chn].res_vector[res + 0].crd_vector[0][0].data;
			fGL * p2 = ref->chn_vector[chn].res_vector[res + 1].crd_vector[0][0].data;
			
			fGL comp[3];
			comp[0] = crd[0] - (p1[0] + p2[0]) / 2.0;
			comp[1] = crd[1] - (p1[1] + p2[1]) / 2.0;
			comp[2] = crd[2] - (p1[2] + p2[2]) / 2.0;
			
			fGL r2 = comp[0] * comp[0] + comp[1] * comp[1] + comp[2] * comp[2];
			fGL r1 = sqrt(r2);
			
			comp[0] /= r1; comp[1] /= r1; comp[2] /= r1;
			
			v3d<fGL> va = v3d<fGL>(p1, pp); va = va / va.len();
			v3d<fGL> vb = v3d<fGL>(p1, p2); vb = vb / vb.len();
			v3d<fGL> vc = va.vpr(vb); vc = vc / vc.len();
			v3d<fGL> vn = vb.vpr(vc); vn = vn / vn.len();
			
			fGL angle = ref->chn_vector[chn].res_vector[res].peptide_vector[0];
			v3d<fGL> vp = vn * cos(angle) - vc * sin(angle);
			
			fGL cosine = vp[0] * comp[0] + vp[1] * comp[1] + vp[2] * comp[2];
			fGL qd = 138.9354518 * 0.077711 * ref->prm.dipole1;
			
	f64 t9x = r2 * r2;	// this is r^4 !!!!!!!
	f64 t9y = t9x * t9x;	// this is r^8 !!!!!!!
	
	// e = A + B/(1+k*exp(L*B*r)) + Z/r^9
	// de/dr = -(exp(L*B*r)*B^2*k*L)/(1+k*exp(L*B*r))^2 - 9Z/r^10
	
	f64 t7a = exp(ref->prm.screen2 * ref->prm.screen4 * r1); f64 t7b = 1.0 + ref->prm.screen3 * t7a;
	f64 t7d = ref->prm.screen1 + ref->prm.screen2 / t7b + ref->prm.screen5 / (t9y * r1);
	
	f64 t7c = t7b * t7b; f64 t7e = t7d * t7d;
	
	f64 t7f = t7a * ref->prm.screen2 * ref->prm.screen2 * ref->prm.screen3 * ref->prm.screen4;
	f64 t7g = 9.0 * ref->prm.screen5 / (t9y * r2);
	
			// f = Q/(e*r^2)
			// df/dr = -Q * ((2/e*r^3) + (de/dr)/(e^2*r^2))
			
			vdwsv += qd * cosine / (t7d * r2);
			
			if (grad != NULL)	// sign??? constants???
			{
				fGL t3a = -qd * cosine * (2.0 / (t7d * r2 * r1) - (t7f / t7c + t7g) / (t7e * r2));
				fGL t3b = qd / (t7d * r2);
				
				fGL deri[3];
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					deri[n1] = 0.0;
					for (i32s n2 = 0;n2 < 3;n2++)
					{
						fGL dunitr;
						if (n1 != n2) dunitr = -comp[n1] * comp[n2] / r1;
						else dunitr = (1.0 - comp[n1] * comp[n2]) / r1;
						
						deri[n1] += dunitr * vp[n2];
					}
				}
				
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					grad[n1] += t3a * comp[n1] + t3b * deri[n1];
				}
			}
		}
	}
	
*/	// DISABLED, TO MAKE THIS RUN FASTER...

	return vdwsv;
}

/*################################################################################################*/

// f = sum[(r/a)^-3] = sum[a^3 * r^-3]		// is it now r^-12 ?!?!?
// df/dr = -3 * sum[a^3 * r^-4]

fGL mm2_GetSASValue(fGL * crd, mm2_mdl * ref, fGL * grad)
{
	fGL sasv = 0.0;
	if (grad != NULL) grad[0] = grad[1] = grad[2] = 0.0;
	
	for (i32u chn = 0;chn < ref->chn_vector.size();chn++)
	{
		for (i32u res = 0;res < ref->chn_vector[chn].res_vector.size();res++)
		{
			for (i32u atm = 0;atm < ref->chn_vector[chn].res_vector[res].natm;atm++)
			{
				fGL tmp1[3]; fGL r2 = 0.0;
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					tmp1[n1] = crd[n1] - ref->chn_vector[chn].res_vector[res].crd_vector[atm][0][n1];
					r2 += tmp1[n1] * tmp1[n1];
				}
				
				if (r2 == 0.0) return +1.0e+35;		// numeric_limits<fGL>::max()?!?!?!
				fGL r1 = sqrt(r2);
				
				fGL tmp2 = r1 / (ref->chn_vector[chn].res_vector[res].vdwr[atm] + 0.15);
				fGL qqq = tmp2 * tmp2 * tmp2 * tmp2;
				
				fGL tmp3 = 1.0 / (qqq * qqq * qqq);
				sasv += tmp3;
				
				if (grad != NULL)	// sign ??? constant ???
				{
					for (i32s n1 = 0;n1 < 3;n1++)
					{
						fGL tmp4 = tmp1[n1] / r1;
						fGL tmp5 = tmp4 * tmp3 / tmp2;
						grad[n1] += tmp5;
					}
				}
			}
		}
	}
	
	return sasv;
}

/*################################################################################################*/
/*################################################################################################*/

mm2_superimpose::mm2_superimpose(mm2_mdl * p1, i32s p2, i32s p3) : superimpose(p2, p3)
{
	mdl = p1;
}

mm2_superimpose::~mm2_superimpose(void)
{
}

f64 mm2_superimpose::GetValue(void)
{
	value = 0.0; counter = 0;
	
	for (i32u n1 = 0;n1 < mdl->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < mdl->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < mdl->chn_vector[n1].res_vector[n2].natm;n3++)
/////////////////////////////////////////////
//i32u n3 = 0;	// MAIN-CHAIN ONLY !!!!!
/////////////////////////////////////////////
			{
				f64 d1[3];
				d1[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[0];
				d1[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[1];
				d1[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[2];
				
				f64 d2[3];
				d2[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0];
				d2[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1];
				d2[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2];
				
				Compare(d1, d2, false);
			}
		}
	}
	
	return value;
}

f64 mm2_superimpose::GetGradient(void)
{
	value = 0.0; counter = 0;
	dloc[0] = dloc[1] = dloc[2] = 0.0;
	drot[0] = drot[1] = drot[2] = 0.0;
	
	for (i32u n1 = 0;n1 < mdl->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < mdl->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < mdl->chn_vector[n1].res_vector[n2].natm;n3++)
/////////////////////////////////////////////
//i32u n3 = 0;	// MAIN-CHAIN ONLY !!!!!
/////////////////////////////////////////////
			{
				f64 d1[3];
				d1[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[0];
				d1[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[1];
				d1[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[2];
				
				f64 d2[3];
				d2[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0];
				d2[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1];
				d2[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2];
				
				Compare(d1, d2, true);
			}
		}
	}
	
//////////////////////////////
//////////////////////////////
//
//cout << "analytical:" << endl;
//cout << dloc[0] << endl;
//cout << dloc[1] << endl;
//cout << dloc[2] << endl;
//cout << drot[0] << endl;
//cout << drot[1] << endl;
//cout << drot[2] << endl;
//
//cout << "numerical:" << endl;
//f64 vara1 = value;
//
//for (i32s qq = 0;qq < 3;qq++)
//{
//	f64 vara2 = loc[qq];
//	
//	loc[qq] = vara2 + 0.00001;
//	cout << ((GetValue() - vara1) / 0.00001) << endl;
//	
//	loc[qq] = vara2;
//}
//
//for (i32s qq = 0;qq < 3;qq++)
//{
//	f64 vara2 = rot[qq];
//	
//	rot[qq] = vara2 + 0.00001;
//	cout << ((GetValue() - vara1) / 0.00001) << endl;
//	
//	rot[qq] = vara2;
//}
//
//value = vara1;
//int zzz; cin >> zzz;
//
//////////////////////////////
//////////////////////////////

	return value;
}

void mm2_superimpose::Transform(void)
{
	value = 0.0; counter = 0;
	
	for (i32u n1 = 0;n1 < mdl->chn_vector.size();n1++)
	{
		for (i32u n2 = 0;n2 < mdl->chn_vector[n1].res_vector.size();n2++)
		{
			for (i32u n3 = 0;n3 < mdl->chn_vector[n1].res_vector[n2].natm;n3++)
/////////////////////////////////////////////
//i32u n3 = 0;	// MAIN-CHAIN ONLY !!!!!
/////////////////////////////////////////////
			{
				f64 d1[3];
				d1[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[0];
				d1[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[1];
				d1[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[0]].data[2];
				
				f64 d2[3];
				d2[0] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0];
				d2[1] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1];
				d2[2] = mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2];
				
				f64 d3[3];
				Compare(d1, d2, false, d3);
				
				mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[0] = d3[0];
				mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[1] = d3[1];
				mdl->chn_vector[n1].res_vector[n2].crd_vector[n3][index[1]].data[2] = d3[2];
			}
		}
	}
}

/*################################################################################################*/

// eof
