// MM1ALG.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm1alg.h"

#include <strstream>
using namespace std;

/*################################################################################################*/

mm1_geomopt::mm1_geomopt(mm1_eng * p1, i32s p2, f64 p3) : conjugate_gradient(p2, p3)
{
	eng = p1;
	
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			AddVar(& eng->crd[n1][n2], & eng->d1[n1][n2]);
		}
	}
}

mm1_geomopt::~mm1_geomopt(void)
{
}

f64 mm1_geomopt::GetValue(void)
{
	eng->Compute(0);	// request energy
	return eng->energy;
}

f64 mm1_geomopt::GetGradient(void)
{
	eng->Compute(1);	// request energy and gradient
	return eng->energy;
}

/*################################################################################################*/

mm1_moldyn::mm1_moldyn(mm1_eng * p1, f64 p2, f64 p3)
{
	eng = p1;
	
	vel = new f64[eng->GetModel()->atom_list.size() * 3];		// [1.0E+3 m/s]
	acc = new f64[eng->GetModel()->atom_list.size() * 3];		// [1.0E+12 m/s^2]
	
	mass = new f64[eng->GetModel()->atom_list.size()];
	
	iter_mm1al it1 = eng->GetModel()->atom_list.begin(); i32s n1 = 0;
	while (it1 != eng->GetModel()->atom_list.end())
	{
		mass[n1] = (* it1).el.GetAtomicMass();
		mass[n1] *= 1.6605402e-27 * 6.0221367e+23;
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			vel[n1 * 3 + n2] = 0.0;
			acc[n1 * 3 + n2] = 0.0;
		}
		
		it1++; n1++;
	}
	
	temp = p2; step1 = p3;
	step2 = step1 * step1;		// [1.0E-15 s]
}

mm1_moldyn::~mm1_moldyn(void)
{
	delete[] vel;
	delete[] acc;
	
	delete[] mass;
}

void mm1_moldyn::TakeMDStep(void)
{
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			f64 tmp1 = step1 * vel[n1 * 3 + n2] * 1.0e-3;
			f64 tmp2 = step2 * acc[n1 * 3 + n2] * 0.5e-9;
			eng->crd[n1][n2] += tmp1 + tmp2;
			
			vel[n1 * 3 + n2] += step1 * acc[n1 * 3 + n2] * 0.5e-6;
		}
	}
	
	eng->Compute(1);
	pot = eng->energy;
	
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			acc[n1 * 3 + n2] = -eng->d1[n1][n2] / mass[n1];
			vel[n1 * 3 + n2] += step1 * acc[n1 * 3 + n2] * 0.5e-6;
		}
	}
	
	kin = KineticEnergy();
}

f64 mm1_moldyn::KineticEnergy(void)
{
	f64 energy = 0.0;
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		f64 sum = 0.0;
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 tmp = vel[n1 * 3 + n2];
			sum += tmp * tmp;
		}
		
		energy += 500.0 * mass[n1] * sum;
	}
	
	return energy;
}

f64 mm1_moldyn::ConvTempEKin(f64 p1)
{
	return (3.0 / 2.0) * eng->GetModel()->atom_list.size() * 8.314510 * p1 / 1000.0;
}

f64 mm1_moldyn::ConvEKinTemp(f64 p1)
{
	return (2.0 / 3.0) * p1 * 1000.0 / (eng->GetModel()->atom_list.size() * 8.314510);
}

void mm1_moldyn::SetEKin(f64 p1)
{
	f64 tmp1 = p1 / KineticEnergy();
	f64 tmp2 = (tmp1 < 0.0 ? 0.0 : sqrt(tmp1));
	
	i32u tmp3 = eng->GetModel()->atom_list.size() * 3;
	for (i32u n1 = 0;n1 < tmp3;n1++) vel[n1] *= tmp2;
}

/*################################################################################################*/

mm1_random_search::mm1_random_search(mm1_mdl * p1, i32s p2, i32s p3, i32s p4, i32s p5, i32s p6)
{
	mdl = p1;
	molnum = p2;
	in_crdset = p3;
	out_crdset = p4;
	
	cycles = p5;
	optsteps = p6;
	
	mdl->GatherGroups();	// mm1_intcrd requires this!!!
	ic = new mm1_intcrd((* mdl), molnum, in_crdset);
	
	eng = mdl->CreateDefaultEngine(); go = NULL;
	
	counter1 = 0;
	counter2 = NOT_DEFINED;
	
	if (!ic->GetVariableCount())
	{
		cout << "ERROR: no rotatable bonds!!!" << endl;
		counter1 = cycles;	// skip the search...
	}
	
	CopyCRD(mdl, eng, in_crdset); CopyCRD(eng, mdl, out_crdset);
	eng->Compute(0); min_energy = eng->energy;
}

mm1_random_search::~mm1_random_search(void)
{
	if (go != NULL) delete go;
	delete eng;
	
	delete ic;
}

i32s mm1_random_search::TakeStep(void)
{
	if (counter1 < cycles)
	{
		if (counter2 == NOT_DEFINED)	// start a new cycle...
		{
			counter1++; counter2 = 0;
			
			const fGL rotprob = 1.0 / sqrt((fGL) ic->GetVariableCount());
			for (i32s n1 = 0;n1 < ic->GetVariableCount();n1++)
			{
				fGL random;
				
				random = (fGL) rand() / (fGL) RAND_MAX;
				if (random > rotprob) continue;
				
				random = (fGL) rand() / (fGL) RAND_MAX;
				ic->SetVariable(n1, 2.0 * M_PI * random);
			}
			
			ic->UpdateCartesian();
			mdl->CenterCRDSet(in_crdset);
			CopyCRD(mdl, eng, in_crdset);
			
			if (go != NULL) delete go;
			go = new mm1_geomopt(eng, 50, 0.001);
		}
		
		for (i32s n1 = 0;n1 < UPDATEFRQ && counter2 < optsteps;n1++)	// optimize...
		{
			counter2++;
			go->TakeCGStep(conjugate_gradient::Newton2An);
		}
		
		CopyCRD(eng, mdl, in_crdset);
		
		i32s retval = counter2;
		if (counter2 >= optsteps)
		{
			eng->Compute(0);
			if (eng->energy < min_energy)
			{
				CopyCRD(eng, mdl, out_crdset);
				min_energy = eng->energy;
			}
			
char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
str1 << "step " << (counter1 + 1) << "/" << cycles << "   energy = " << eng->energy << " kJ/mol" << endl << ends;
mdl->PrintToLog(mbuff1);
			
			counter2 = NOT_DEFINED;
		}
		
		return retval;
	}
	else return -1;
}

/*################################################################################################*/

mm1_systematic_search::mm1_systematic_search(mm1_mdl * p1, i32s p2, i32s p3, i32s p4, i32s p5, i32s p6)
{
	mdl = p1;
	molnum = p2;
	in_crdset = p3;
	out_crdset = p4;
	
	divisions = p5;
	optsteps = p6;
	
	mdl->GatherGroups();	// mm1_intcrd requires this!!!
	ic = new mm1_intcrd((* mdl), molnum, in_crdset);
	
	eng = mdl->CreateDefaultEngine(); go = NULL;
	
	nvar = ic->GetVariableCount();
	if (nvar != 0)
	{
		counter1 = new i32s[nvar];
		for (i32s n1 = 0;n1 < nvar;n1++)
		{
			counter1[n1] = 0;
		}
	}
	else
	{
		cout << "ERROR: no rotatable bonds!!!" << endl;
		counter1 = NULL;	// skip the search...
	}
	
	counter2 = NOT_DEFINED;
	
	CopyCRD(mdl, eng, in_crdset); CopyCRD(eng, mdl, out_crdset);
	eng->Compute(0); min_energy = eng->energy;
}

mm1_systematic_search::~mm1_systematic_search(void)
{
	if (counter1 != NULL) delete[] counter1;	// this should never happen...
	
	if (go != NULL) delete go;
	delete eng;
	
	delete ic;
}

i32s mm1_systematic_search::TakeStep(void)
{
	if (counter1 != NULL)
	{
		if (counter2 == NOT_DEFINED)	// start a new cycle...
		{
			counter2 = 0;
			
			i32s n1; bool overflow = false;
			for (n1 = 0;n1 < nvar;n1++)
			{
				overflow = false;
				
				counter1[n1]++;
				if (counter1[n1] >= divisions)
				{
					counter1[n1] = 0;
					overflow = true;
				}
				
				fGL value = (fGL) counter1[n1] / (fGL) divisions;
				ic->SetVariable(n1, 2.0 * M_PI * value);
				
				if (!overflow) break;
			}
			
			if (overflow && n1 == nvar)	// if overflow happened in the last counter, the search is ready!!!
			{
				delete[] counter1;
				counter1 = NULL;
			}
			
			ic->UpdateCartesian();
			mdl->CenterCRDSet(in_crdset);
			CopyCRD(mdl, eng, in_crdset);
			
			if (go != NULL) delete go;
			go = new mm1_geomopt(eng, 50, 0.001);
		}
		
		for (i32s n1 = 0;n1 < UPDATEFRQ && counter2 < optsteps;n1++)	// optimize...
		{
			counter2++;
			go->TakeCGStep(conjugate_gradient::Newton2An);
		}
		
		CopyCRD(eng, mdl, in_crdset);
		
		i32s retval = counter2;
		if (counter2 >= optsteps)
		{
			eng->Compute(0);
			if (eng->energy < min_energy)
			{
				CopyCRD(eng, mdl, out_crdset);
				min_energy = eng->energy;
			}
			
if (counter1 != NULL)
{
	char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
	str1 << "step "; for (int v = 0;v < nvar;v++) { str1 << (char) ('A' + counter1[(nvar - 1) - v]); }
	str1 << "   energy = " << eng->energy << " kJ/mol" << endl << ends;
	mdl->PrintToLog(mbuff1);
}
			
			counter2 = NOT_DEFINED;
		}
		
		return retval;
	}
	else return -1;
}

/*################################################################################################*/

mm1_monte_carlo_search::mm1_monte_carlo_search(mm1_mdl * p1, i32s p2, i32s p3, i32s p4, i32s p5, i32s p6, i32s p7)
{
	mdl = p1;
	molnum = p2;
	in_crdset = p3;
	out_crdset = p4;
	
	n_init_steps = p5;
	n_simul_steps = p6;
	optsteps = p7;
	
// OPTIMIZE WITH NO NONBONDED TERMS TO GET AN IDEAL SYMMETRIC STARTING GEOMETRY?!?!?
// OPTIMIZE WITH NO NONBONDED TERMS TO GET AN IDEAL SYMMETRIC STARTING GEOMETRY?!?!?
// OPTIMIZE WITH NO NONBONDED TERMS TO GET AN IDEAL SYMMETRIC STARTING GEOMETRY?!?!?
	
	mdl->GatherGroups();	// mm1_intcrd requires this!!!
	ic = new mm1_intcrd((* mdl), molnum, in_crdset);
	
	eng = mdl->CreateDefaultEngine(); go = NULL;
	
	counter1 = -n_init_steps;
	counter2 = NOT_DEFINED;
	
	if (!ic->GetVariableCount())
	{
		cout << "ERROR: no rotatable bonds!!!" << endl;
		counter1 = n_simul_steps;	// skip the search...
	}
	
	nvar = ic->GetVariableCount();
	curr_ic1 = new f64[nvar]; curr_ic2 = new f64[nvar];
	for (i32s n1 = 0;n1 < nvar;n1++)
	{
		curr_ic1[n1] = ic->GetVariable(n1);
	}
	
	CopyCRD(mdl, eng, in_crdset); CopyCRD(eng, mdl, out_crdset);
	eng->Compute(0); curr_energy = min_energy = eng->energy;
	
//DEBUG
////mdl->PushCRDSets(n_simul_steps);	//DEBUG - this also SHOULD work, but DOES NOT!!!

	// intcrd class uses pointer to coordinates, that might be invalidated if more memory is allocated
	// when adding new coordinate sets! this is fixed in CVS HEAD by adding new member functions
	// atom::GetCRD() and atom::SetCRD() that strictly control the access to coordinate data. might
	// be fixed here as well, later, but is troublesome since we have many models, not just one.
	
//for (i32s n1 = 0;n1 < n_simul_steps;n1++) mdl->SetCRDSetVisible(n1 + 2, false);
}

mm1_monte_carlo_search::~mm1_monte_carlo_search(void)
{
	delete[] curr_ic1;
	delete[] curr_ic2;
	
	if (go != NULL) delete go;
	delete eng;
	
	delete ic;
	
//DEBUG
//mdl->SetCRDSetVisible(0, false);
//mdl->SetCRDSetVisible(1, false);
//for (i32s n1 = 0;n1 < n_simul_steps;n1++) mdl->SetCRDSetVisible(n1 + 2, true);
//mdl->UpdateAllGraphicsViews(true);
}

i32s mm1_monte_carlo_search::TakeStep(void)
{
	if (counter1 < n_simul_steps)
	{
		if (counter2 == NOT_DEFINED)	// start a new cycle...
		{
			counter1++; counter2 = 0;
			
			const fGL rotprob = 1.0 / sqrt((fGL) nvar);
			for (i32s n1 = 0;n1 < nvar;n1++)
			{
				fGL value = curr_ic1[n1];
				curr_ic2[n1] = value;
				
				fGL random;
				
				random = (fGL) rand() / (fGL) RAND_MAX;
				if (random > rotprob) continue;
				
				random = (fGL) rand() / (fGL) RAND_MAX;
				curr_ic2[n1] = 2.0 * M_PI * random;
			}
			
			for (i32s n1 = 0;n1 < nvar;n1++)
			{
				ic->SetVariable(n1, curr_ic2[n1]);
			}
			
			ic->UpdateCartesian();
			mdl->CenterCRDSet(in_crdset);
			CopyCRD(mdl, eng, in_crdset);
			
			if (go != NULL) delete go;
			go = new mm1_geomopt(eng, 50, 0.001);
		}
		
		for (i32s n1 = 0;n1 < UPDATEFRQ && counter2 < optsteps;n1++)	// optimize...
		{
			counter2++;
			go->TakeCGStep(conjugate_gradient::Newton2An);
		}
		
		CopyCRD(eng, mdl, in_crdset);
		
		i32s retval = counter2;
		if (counter2 >= optsteps)
		{
			bool accept = false;
			
			eng->Compute(0);
			if (eng->energy < curr_energy) accept = true;
			
			f64 tmp1 = eng->energy - curr_energy;
			f64 tmp2 = 1000.0 * tmp1 / (8.31451 * 300.0);
			
			if (!accept && ((fGL) rand() / (fGL) RAND_MAX < exp(-tmp2))) accept = true;
//cout << counter1 << "   " << eng->energy << "   " << curr_energy << "   TESTVALUE = " << exp(-tmp2) << endl;
			
			if (accept)
			{
				for (i32s n1 = 0;n1 < nvar;n1++)
				{
					curr_ic1[n1] = curr_ic2[n1];
				}
				
				curr_energy = eng->energy;
				
char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
str1 << "step " << (counter1 + 1) << "/" << n_simul_steps << "   energy = " << eng->energy << " kJ/mol" << endl << ends;
mdl->PrintToLog(mbuff1);
				
//DEBUG
//if (counter1 >= 0) {
//cout << "STORE STRUCTURE " << counter1 << endl;
//CopyCRD(eng, mdl, counter1 + 2); }
// set the weight???
// set the weight???
// set the weight???
			}
			else
			{
				counter1--;
				// increase the weighting factor???
			}
			
			counter2 = NOT_DEFINED;
			
			if (eng->energy < min_energy)
			{
				CopyCRD(eng, mdl, out_crdset);
				min_energy = eng->energy;
			}
		}
		
		return retval;
	}
	else return -1;
}

/*################################################################################################*/

// eof
