/*
    send.c - Provide the send dialog and related routines
    Copyright (c) 1999  Martin Schulze <joey@infodrom.north.de>

    This file is part of the Gerstensaft package.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111, USA.
*/

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <stdio.h>
#include "main.h"
#include "beer.h"
#include "send.h"
#include "saft.h"
#include "nls.h"
#include "config.h"

gint send_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
#ifdef DEBUG
  printf ("send_delete()\n");
#endif

  return (FALSE);
}

void send_destroy (GtkWidget *widget, gpointer data)
{
#ifdef DEBUG
  printf ("send_destroy()\n");
#endif

  if (((struct s_send_data *) data) != NULL &&  ((struct s_send_data *) data)->numfiles) {
    if  (((struct s_send_data *) data)->selected != NULL)
      free (((struct s_send_data *) data)->selected);
    ((struct s_send_data *) data)->selected = NULL;
    ((struct s_send_data *) data)->numfiles = 0;
    free (((struct s_send_data *) data));
  }
}

void send_addr_change (GtkWidget *widget, gpointer data)
{
  /* De/-activate the Send button only if needed */
  if (strlen(gtk_entry_get_text (GTK_ENTRY(widget))) > 0) {
    if (!((struct s_send_data *) data)->addrok) {
      if (((struct s_send_data *) data)->numselected > 0)
	gtk_widget_set_sensitive (((struct s_send_data *) data)->sendbutton, TRUE);
      ((struct s_send_data *) data)->addrok = 1;
    }
  } else {
    if (((struct s_send_data *) data)->addrok) {
      gtk_widget_set_sensitive (((struct s_send_data *) data)->sendbutton, FALSE);
      ((struct s_send_data *) data)->addrok = 0;
    }
  }
}

void send_row_select (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
  /* Only select and increase if it wasn't before, adjust button */
  if (row < ((struct s_send_data *) data)->numfiles)
    if (((struct s_send_data *) data)->selected[row] != 1) {
      ((struct s_send_data *) data)->selected[row] = 1;
      ((struct s_send_data *) data)->numselected++;
      if (((struct s_send_data *) data)->numselected == 1 && ((struct s_send_data *) data)->addrok == 1)
	gtk_widget_set_sensitive (((struct s_send_data *) data)->sendbutton, TRUE);
    }
}

void send_row_unselect (GtkCList *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
  /* Only unselect and decrease if it wasn't before, adjust button */
  if (row < ((struct s_send_data *) data)->numfiles)
    if (((struct s_send_data *) data)->selected[row] != 0) {
      ((struct s_send_data *) data)->selected[row] = 0;
      if (((struct s_send_data *) data)->numselected > 0)
	((struct s_send_data *) data)->numselected--;
      if (((struct s_send_data *) data)->numselected == 0)
	gtk_widget_set_sensitive (((struct s_send_data *) data)->sendbutton, FALSE);
    }
}

void send_button_select_all (GtkWidget *widget, gpointer data)
{
  int i;

  gtk_clist_freeze (GTK_CLIST(((struct s_send_data *) data)->listbox));
  for (i=0; i<((struct s_send_data *) data)->numfiles; i++)
    if (!((struct s_send_data *) data)->selected[i]) {
      ((struct s_send_data *) data)->selected[i] = 1;
      ((struct s_send_data *) data)->numselected++;
      gtk_clist_select_row (GTK_CLIST(((struct s_send_data *) data)->listbox), i, 0);
    }
  gtk_clist_thaw (GTK_CLIST(((struct s_send_data *) data)->listbox));
  if (((struct s_send_data *) data)->numselected > 0 && ((struct s_send_data *) data)->addrok == 1)
    gtk_widget_set_sensitive (((struct s_send_data *) data)->sendbutton, TRUE);
}

void send_button_unselect_all (GtkWidget *widget, gpointer data)
{
  int i;

  gtk_clist_freeze (GTK_CLIST(((struct s_send_data *) data)->listbox));
  for (i=0; i<((struct s_send_data *) data)->numfiles; i++)
    if (((struct s_send_data *) data)->selected[i]) {
      ((struct s_send_data *) data)->selected[i] = 0;
      ((struct s_send_data *) data)->numselected--;
      gtk_clist_unselect_row (GTK_CLIST(((struct s_send_data *) data)->listbox), i, 0);
    }
  gtk_clist_thaw (GTK_CLIST(((struct s_send_data *) data)->listbox));
  if (((struct s_send_data *) data)->numselected == 0)
    gtk_widget_set_sensitive (((struct s_send_data *) data)->sendbutton, FALSE);
}

void send_button_clear (GtkWidget *widget, gpointer data)
{
  gtk_clist_clear (GTK_CLIST(((struct s_send_data *) data)->listbox));

  if (((struct s_send_data *) data)->numfiles) {
    if  (((struct s_send_data *) data)->selected != NULL)
      free (((struct s_send_data *) data)->selected);
    ((struct s_send_data *) data)->selected = NULL;
    ((struct s_send_data *) data)->numfiles = 0;
    ((struct s_send_data *) data)->numselected = 0;
    gtk_widget_set_sensitive (((struct s_send_data *) data)->sendbutton, FALSE);
  }

  gtk_entry_set_text (GTK_ENTRY(((struct s_send_data *) data)->addrinput->entry), "");
  main_clear (((struct s_send_data *) data)->wdata);
}

void send_button_send (GtkWidget *widget, gpointer data)
{
  int i, n;
  gchar *fname;
  gchar *addr;
  char **files;
  char *xp;

  addr = gtk_entry_get_text (GTK_ENTRY(((struct s_send_data *) data)->addrinput->entry));

  /* Find out if the address is already listed in the history */
  for (i=0,n=0;i<g_list_length (config.addrs); i++) {
    if (!strcmp (addr, g_list_nth (config.addrs, i)->data)) {
      n++;
      break;
    }
  }
  /* Add address to history if it isn't in already */
  if (!n && (xp = (char *)malloc (strlen(addr)+1)) != NULL) {
    sprintf (xp, addr);
    config.addrs = g_list_insert_sorted (config.addrs, xp, (GCompareFunc)strcmp);
    gtk_combo_set_popdown_strings (((struct s_send_data *) data)->addrinput, config.addrs);
    gtk_entry_set_text (GTK_ENTRY(((struct s_send_data *) data)->addrinput->entry),addr);
  }

  for (i=0,n=0; i < ((struct s_send_data *) data)->numfiles; i++)
    if (((struct s_send_data *) data)->selected[i])
      n++;

  if (n>0 && (files = (char **)malloc ((n+1)*sizeof (char *))) != NULL) {
    memset (files, 0, sizeof (files));
    for (i=0,n=0; i < ((struct s_send_data *) data)->numfiles; i++)
      if (((struct s_send_data *) data)->selected[i]) {
	gtk_clist_get_text (GTK_CLIST(((struct s_send_data *) data)->listbox), i, 1, &fname);
	files[n++] = fname;
      }
    files[n] = NULL;
    saft_send (addr, files);
    free (files);
    send_button_unselect_all (widget, data);
  }
}

void send_dialog (int num, char **files, unsigned long int *size, gpointer data)
{
  GtkWidget *dialog;
  GtkWidget *mainbox, *hbox, *vbox;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *button;
#if GTK_MAJOR_VERSION >= 1 && GTK_MINOR_VERSION >= 1
  GtkWidget *scrollwin;
#endif
  GtkTooltips *tooltips;
  char *titles[3] = {"", _("Filename"), _("Size")};
  struct stat st;
  char name[1024], asize[10], *dummy="";
  char *clp[3] = {dummy, name, asize};
  GdkColor background;
  GdkColormap *colormap;
  int i;
  struct s_send_data *sdata;

  if ((sdata = (struct s_send_data *) malloc (sizeof(struct s_send_data))) == NULL)
    return;
  memset (sdata, 0, sizeof (struct s_send_data));

  sdata->wdata = data;

  /* Create a new dialog box */
  dialog = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_grab_add (dialog);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Send files"));
  gtk_widget_set_usize (GTK_WIDGET (dialog), 400, 240);

  /* Connect the appropriate signal handlers */
  gtk_signal_connect (GTK_OBJECT (dialog), "delete_event", GTK_SIGNAL_FUNC (send_delete), (gpointer) sdata);
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy", GTK_SIGNAL_FUNC (send_destroy), (gpointer) sdata);

  mainbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER (dialog), mainbox);

  frame = gtk_frame_new (NULL);
  gtk_container_border_width (GTK_CONTAINER(frame), 5);
  gtk_container_add (GTK_CONTAINER(mainbox), frame);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER(vbox), 5);
  gtk_container_add (GTK_CONTAINER(frame), vbox);

  /* Input field for directory, will be updated automatically */
  label = gtk_label_new (_("Recipient"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  sdata->addrinput = GTK_COMBO(gtk_combo_new());
  if (config.addrs) {
    gtk_combo_set_popdown_strings (sdata->addrinput, config.addrs);
    gtk_entry_set_text (GTK_ENTRY(sdata->addrinput->entry),"");
  }
  gtk_signal_connect (GTK_OBJECT(sdata->addrinput->entry), "changed", GTK_SIGNAL_FUNC(send_addr_change), (gpointer) sdata);
  /*  The following doesn't work since [enter] means, open the history for a lookup
  gtk_signal_connect (GTK_OBJECT(sdata->addrinput->entry), "activate", GTK_SIGNAL_FUNC(send_button_send), (gpointer) sdata);
  */
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, sdata->addrinput->entry, _("Enter recipient's address"), NULL);
  gtk_box_pack_start (GTK_BOX(vbox), GTK_WIDGET(sdata->addrinput), FALSE, FALSE, FALSE);
  gtk_widget_show (GTK_WIDGET(sdata->addrinput));

  background.red = background.green = background.blue =  ~0;
  colormap = gdk_colormap_get_system();
  gdk_color_alloc (colormap, &background);

  sdata->listbox = gtk_clist_new_with_titles (3, titles);
  gtk_clist_column_titles_passive (GTK_CLIST(sdata->listbox));
  gtk_clist_set_selection_mode (GTK_CLIST(sdata->listbox), GTK_SELECTION_MULTIPLE);
  gtk_clist_set_column_width (GTK_CLIST(sdata->listbox), 0, 16);
  gtk_clist_set_column_justification (GTK_CLIST(sdata->listbox), 0, GTK_JUSTIFY_CENTER);
  gtk_clist_set_column_width (GTK_CLIST(sdata->listbox), 1, 450);
  gtk_clist_set_column_justification (GTK_CLIST(sdata->listbox), 1, GTK_JUSTIFY_LEFT);
  gtk_clist_set_column_width (GTK_CLIST(sdata->listbox), 2, 40);
  gtk_clist_set_column_justification (GTK_CLIST(sdata->listbox), 2, GTK_JUSTIFY_RIGHT);

  gtk_signal_connect(GTK_OBJECT(sdata->listbox), "select_row", GTK_SIGNAL_FUNC(send_row_select), (gpointer) sdata);
  gtk_signal_connect(GTK_OBJECT(sdata->listbox), "unselect_row", GTK_SIGNAL_FUNC(send_row_unselect), (gpointer) sdata);
#if GTK_MAJOR_VERSION >= 1 && GTK_MINOR_VERSION >= 1
  scrollwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrollwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);
  gtk_container_add (GTK_CONTAINER(scrollwin), sdata->listbox);
  gtk_widget_show (scrollwin);
#else
  gtk_box_pack_start (GTK_BOX(vbox) , sdata->listbox, TRUE, TRUE, 0);
#endif

  for (i=0,sdata->numfiles=0; i<num; i++) {
    if (files[i] && strlen (files[i])) {
      snprintf (name, sizeof (name), "%s", files[i]);
      snprintf (asize, sizeof (asize), "%d", size[i]);
      gtk_clist_append (GTK_CLIST(sdata->listbox), clp);
      if (!stat (files[i], &st)) {
	if (S_ISDIR (st.st_mode))
	  gtk_clist_set_pixmap (GTK_CLIST(sdata->listbox), sdata->numfiles, 0, dir_pixmap, dir_mask);
	else
	  gtk_clist_set_pixmap (GTK_CLIST(sdata->listbox), sdata->numfiles, 0, doc_pixmap, doc_mask);
      }
      gtk_clist_set_background (GTK_CLIST(sdata->listbox), sdata->numfiles, &background);
      sdata->numfiles++;
    }
  }
  if ((sdata->selected = (int *)malloc (sdata->numfiles*sizeof(int))) == NULL)
    return;
  memset (sdata->selected, 0, sdata->numfiles*sizeof(int));

  gtk_widget_show (sdata->listbox);
  gtk_widget_show (vbox);

  /* horizontal box containing the buttons */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER(hbox), 5);
  gtk_box_pack_start (GTK_BOX(mainbox), hbox, FALSE, TRUE, 0);

  button = gtk_button_new_with_label (_(" Select all "));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (send_button_select_all), (gpointer) sdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Select all files"), NULL);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, FALSE, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_(" Unselect all "));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (send_button_unselect_all), (gpointer) sdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Unselected all files"), NULL);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, FALSE, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label (_(" Clear "));
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (send_button_clear), (gpointer) sdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Clear list of files"), NULL);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, FALSE, 0);
  gtk_widget_show (button);

  sdata->sendbutton = gtk_button_new_with_label (_(" Send "));
  gtk_widget_set_sensitive (sdata->sendbutton, FALSE);
  gtk_signal_connect (GTK_OBJECT (sdata->sendbutton), "clicked", GTK_SIGNAL_FUNC (send_button_send), (gpointer) sdata);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, sdata->sendbutton, _("Send the selected files to recipient"), NULL);
  gtk_box_pack_start (GTK_BOX(hbox), sdata->sendbutton, TRUE, FALSE, 0);
  gtk_widget_show (sdata->sendbutton);

  button = gtk_button_new_with_label (_(" Back "));
  gtk_signal_connect_object (GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),  GTK_OBJECT(dialog));
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip (tooltips, button, _("Get back to main program"), NULL);
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, FALSE, 0);
  gtk_widget_show (button);

  gtk_widget_show (hbox);
  gtk_widget_show (frame);
  gtk_widget_show (mainbox);
  gtk_widget_show (dialog);

}
