/*
** 2003-11-04 -	Incremental search. Better late than never, right?
*/

#include <ctype.h>

#include <gdk/gdkkeysyms.h>

#include "gentoo.h"
#include "cmdseq.h"
#include "dialog.h"
#include "dirpane.h"

#include "cmd_dpfocusisrch.h"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	GtkWidget	*hbox;			/* Really should start with hbox. */
	GtkWidget	*label, *entry;
} ISearchWidgetry;

/* Various state, kept as a static global. Ugly, but simplifies things. */
static struct {
	DirPane		*pane;				/* Non-NULL during ISearch, NULL when not in use. */
	guint		page;
	MainInfo	*min;
	char *	(*search)(const char *haystack, const char *needle);
	gboolean	select;
	ISearchWidgetry	*wid;
	gint		old_focus;

	guint	sig_activate, sig_changed, sig_focus_out, sig_key_press;
} the_isearch_info = { NULL };

/* ----------------------------------------------------------------------------------------- */

/* 2003-11-06 -	A case-insensitive version of the standard strstr() function. */
static gchar * nocase_strstr(const gchar *haystack, const gchar *needle)
{
	gint	nlen = -1;

	for(; *haystack; haystack++)
	{
		if(tolower(*haystack) == tolower(*needle))
		{
			if(nlen < 0)
				nlen = strlen(needle);
			if(g_strncasecmp(haystack, needle, nlen) == 0)	/* I find "nocase" clearer than "case", myself. */
				return (gchar *) haystack;
		}
	}
	return NULL;
}

/* 2003-11-06 -	Perform an incremental search, from row <start> and looking for <string>. */
static gboolean do_isearch(guint start, const gchar *string)
{
	guint	i;

	for(i = start; i < the_isearch_info.pane->dir.num_rows; i++)
	{
		if(the_isearch_info.search(the_isearch_info.pane->dir.row[i].dr_name, string) != NULL)
		{
			dp_focus(the_isearch_info.pane, i);
			if(the_isearch_info.select)
				dp_select(the_isearch_info.pane, i);
			return TRUE;
		}
	}
	return FALSE;
}

/* 2003-11-08 -	Close down incremental search. Hides widgetry. */
static void isearch_close(void)
{
	MainInfo	*min = the_isearch_info.min;
	GtkWidget	*entry = the_isearch_info.wid->entry;

	dp_pathwidgetry_show(the_isearch_info.pane, 0U);
	the_isearch_info.pane = NULL;

	gtk_signal_disconnect(GTK_OBJECT(entry), the_isearch_info.sig_activate);
	gtk_signal_disconnect(GTK_OBJECT(entry), the_isearch_info.sig_changed);
	gtk_signal_disconnect(GTK_OBJECT(entry), the_isearch_info.sig_key_press);
	gtk_signal_disconnect(GTK_OBJECT(entry), the_isearch_info.sig_focus_out);

	kbd_context_attach(min->gui->kbd_ctx, GTK_WINDOW(min->gui->window));
}

/* 2003-11-04 -	Handle keypresses. Traps special keys for handy functionality. */
static gint evt_isearch_keypress(GtkWidget *wid, GdkEventKey *evt, gpointer user)
{
	if(evt->keyval == GDK_Escape || (evt->keyval == GDK_g && evt->state & GDK_CONTROL_MASK))
	{
		dp_focus(the_isearch_info.pane, the_isearch_info.old_focus);
		isearch_close();
		return TRUE;
	}
	else if(evt->keyval == GDK_Up || evt->keyval == GDK_Down ||
	   evt->keyval == GDK_Page_Up || evt->keyval == GDK_Page_Down ||
	   evt->keyval == GDK_Tab)
	{
		if(evt->keyval == GDK_Up || evt->keyval == GDK_Down)
			csq_execute_format(the_isearch_info.min, "DpFocus %s", evt->keyval == GDK_Up ? "prev" : "next");
		else if(evt->keyval == GDK_Page_Up || evt->keyval == GDK_Page_Down)
			csq_execute_format(the_isearch_info.min, "DpFocus %s", evt->keyval == GDK_Page_Up ? "pageprev" : "pagenext");
		else if(evt->keyval == GDK_Tab && the_isearch_info.pane->focus_row >= 0)
		{
			const gchar	*string = gtk_entry_get_text(GTK_ENTRY(the_isearch_info.wid->entry));

			/* Use TAB as (hardcoded) repeat-search key, since we don't know isearch binding (if any). */
			if(string && *string)
			{
				if(!do_isearch(the_isearch_info.pane->focus_row + 1, string))
					do_isearch(0, string);
			}
		}
		gtk_signal_emit_stop_by_name(GTK_OBJECT(wid), "key_press_event");
	}
	return TRUE;
}

static gint evt_isearch_changed(GtkWidget *wid, gpointer user)
{
	const gchar	*string = gtk_entry_get_text(GTK_ENTRY(wid));

	dp_unfocus(the_isearch_info.pane);
	if(string == NULL || *string == '\0')
		return TRUE;
	do_isearch(0, string);

	return TRUE;
}

static gint evt_isearch_activate(GtkWidget *wid, gpointer user)
{
	isearch_close();
	return TRUE;
}

static gint evt_isearch_focus_out(GtkWidget *wid, GdkEventFocus *evt, gpointer user)
{
	isearch_close();
	return TRUE;
}

/* 2003-11-04 -	Incremental search. Initial implementation, with a rather non-satisfying UI. */
gint cmd_dpfocusisrch(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GtkWidget	*entry;
	const gchar	*text;

	if(the_isearch_info.pane != NULL)
		return FALSE;
	kbd_context_detach(min->gui->kbd_ctx, GTK_WINDOW(min->gui->window));

	the_isearch_info.pane = src;
	the_isearch_info.min  = min;
	the_isearch_info.wid  = (ISearchWidgetry *) dp_pathwidgetry_show(src, the_isearch_info.page);
	entry = the_isearch_info.wid->entry;
	gtk_widget_grab_focus(entry);

	the_isearch_info.sig_activate  = gtk_signal_connect(GTK_OBJECT(entry), "activate", GTK_SIGNAL_FUNC(evt_isearch_activate), NULL);
	the_isearch_info.sig_changed   = gtk_signal_connect(GTK_OBJECT(entry), "changed", GTK_SIGNAL_FUNC(evt_isearch_changed), NULL);
	the_isearch_info.sig_key_press = gtk_signal_connect(GTK_OBJECT(entry), "key_press_event", GTK_SIGNAL_FUNC(evt_isearch_keypress), NULL);
	the_isearch_info.sig_focus_out = gtk_signal_connect(GTK_OBJECT(entry), "focus_out_event", GTK_SIGNAL_FUNC(evt_isearch_focus_out), NULL);

	the_isearch_info.old_focus = src->focus_row;

	if(car_keyword_get_boolean(ca, "nocase", FALSE))
		the_isearch_info.search = nocase_strstr;
	else
		the_isearch_info.search = strstr;
	the_isearch_info.select = car_keyword_get_boolean(ca, "select", FALSE);

	if((text = car_keyword_get_value(ca, "text", NULL)) != NULL)
		gtk_entry_set_text(GTK_ENTRY(entry), text);
	else if(src->focus_row >= 0)
		gtk_entry_set_text(GTK_ENTRY(entry), src->dir.row[src->focus_row].dr_name);

	return TRUE;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-11-08 -	Build dirpane path widgetry for incremental search. */
static GtkWidget ** widgetry_builder(MainInfo *min)
{
	ISearchWidgetry	*wid;

	wid = g_malloc(sizeof *wid);
	wid->hbox = gtk_hbox_new(FALSE, 0);
	wid->label = gtk_label_new(_("ISearch"));
	gtk_box_pack_start(GTK_BOX(wid->hbox), wid->label, FALSE, FALSE, 5);
	wid->entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(wid->hbox), wid->entry, TRUE, TRUE, 0);

	return (GtkWidget **) wid;
}

/* 2003-11-08 -	Configure DpFocusISrch. Does not add user-settable options, just registers
**		a new pathwidgetry builder with the dirpane subsystem. Italian food.
*/
void cfg_dpfocusisrch(MainInfo *min)
{
	the_isearch_info.page = dp_pathwidgetry_add(widgetry_builder);
}
