#!/usr/bin/env python
# -*- coding: UTF-8 -*-
"""
    Modul:          sah
    Description:    Beschreibung
    Version:        V0.1
    Copyright:      2003 by Fritz Cizmarov fritz@sol.at
    Last modified:  27. Sep. 2003
    License:        
    Exports:        Classes and Functions to export
"""

import os
from time import time

# Constants for status
OK = 1
CONNECTING = -1
NO_SETI_PATH = -2
NOT_RUNNING = 0

class Sah_Data:
    SAH_NAME = ""
    def __init__(self, setipath=""):
        self.path = setipath
        self.init_data()
        self.notifys = {}
        if self.path:
            self.update()
    
    def __getitem__(self, name):
        return self.data[name]
    
    def init_data(self):
        self.last_update = 0.0
        self.data = {}
    
    def type_of(self, name, value=False):
        if value and value != "no":
            return type(self.data[name])(value)
        else:
            return type(self.data[name])()

    def clear_data(self):
        self.last_update = 0.0
        for key, value in self.data.iteritems():
            newvalue = self.type_of(key)
            self.data[key] = newvalue
            self.have_changed(key, newvalue)
                
    def new_path(self, path):
        self.path = path
        self.update(chk_time = False)
        
    def set_notify(self, name, func):
        self.notifys[name] = func
        
    def notify(self, name, value):
        if self.notifys.has_key(name):
            self.notifys[name](name, value)
    
    def have_changed(self, name, value):
        self.notify(name, value)

    def add_data(self, key, value):
        value = self.type_of(key, value)
        if self.data[key] != value:
            self.data[key] = value
            self.have_changed(key, value)
    
    def do_line(self, line):
        if "=" in line:
            key, value = line.split("=")
            key = key.strip()
            value = value.strip()
            self.add_data(key, value)
        return True
    
    def read(self):
        path = os.path.join(self.path, self.SAH_NAME)
        try:
            sah_file = open(path, "r")
        except IOError:
            return False
        
        for line in sah_file:
            if not self.do_line(line): break
        sah_file.close()
        return True
        
    def update(self, chk_time=True):
        if not self.SAH_NAME: return False
        path = os.path.join(self.path, self.SAH_NAME)
        if not os.path.exists(path):
            self.clear_data()
            return True
        if chk_time:
            try:
                mtime = os.path.getmtime(path)
            except:
                self.clear_data()
                return False
            if self.last_update > mtime: return True
        self.last_update = time()
        return self.read()
    
    def need_update(self):
        if not self.SAH_NAME: return False
        path = os.path.join(self.path, self.SAH_NAME)
        if not os.path.exists(path): return True
        mtime = os.path.getmtime(path)
        return self.last_update < os.path.getmtime(path)
        
class Version(Sah_Data):
    SAH_NAME = "version.sah"
    
    def __init__(self, setipath=""):
        Sah_Data.__init__(self, setipath)

    def init_data(self):
        Sah_Data.init_data(self)
        self.data["major_version"] = 0
        self.data["minor_version"] = 0
        self.version = (0,0)

    def have_changed(self, name, value):
        Sah_Data.have_changed(self, name, value)
        self.version = (self["major_version"], self["minor_version"])
        
class User_Info(Sah_Data):
    SAH_NAME = "user_info.sah"
    def __init__(self, setipath=""):
        Sah_Data.__init__(self, setipath)
    
    def init_data(self):
        Sah_Data.init_data(self)
        self.data["type"] = ""
        self.data["id"] = 0
        self.data["key"] = 0
        self.data["email_addr"] = ""
        self.data["name"] = ""
        self.data["url"] = ""
        self.data["country"] = ""
        self.data["postal_code"] = ""
        self.data["show_name"] = False
        self.data["show_email"] = False
        self.data["venue"] = 0
        self.data["register_time"] = 0.0
        self.data["last_wu_time"] = 0.0
        self.data["last_result_time"] = 0.0
        self.data["nwus"] = 0
        self.data["nresults"] = 0
        self.data["total_cpu"] = 0.0
        self.data["params_index"] = 0

    def type_of(self, name, value=False):
        if name[-4:] == "time" and value:
            return value.split()[0]
        else:
            return Sah_Data.type_of(self, name, value)


class State(Sah_Data):
    SAH_NAME = "state.sah"
    def __init__(self, setipath=""):
        Sah_Data.__init__(self, setipath)

    def init_data(self):
        Sah_Data.init_data(self)
        self.data["ncfft"] = 0
        self.data["cr"] = 0.0
        self.data["fl"] = 0
        self.data["cpu"] = 0.0
        self.data["prog"] = 0.0
        self.data["potfreq"] = 0
        self.data["potactivity"] = 0
        self.data["outfilepos"] = 0
        self.data["bs_power"] = 0.0
        self.data["bs_score"] = 0.0
        self.data["bs_bin"] = 0
        self.data["bs_fft_ind"] = 0
        self.data["bs_chirp_rate"] = 0.0
        self.data["bs_fft_len"] = 0
        self.data["bg_score"] = 0.0
        self.data["bg_sigma"] = 0.0
        self.data["bg_power"] = 0.0
        self.data["bg_true_mean"] = 0.0
        self.data["bg_chisq"] = 0.0
        self.data["bg_bin"] = 0
        self.data["bg_fft_ind"] = 0
        self.data["bg_chirp_rate"] = 0.0
        self.data["bg_fft_len"] = 0
        self.data["bg_pot"] = [0.0]*64
        self.data["bp_score"] = 0.0
        self.data["bp_power"] = 0.0
        self.data["bp_mean"] = 0.0
        self.data["bp_period"] = 0.0
        self.data["bp_freq_bin"] = 0
        self.data["bp_time_bin"] = 0
        self.data["bp_chirp_rate"] = 0.0
        self.data["bp_fft_len"] = 0
        self.data["bp_pot"] = ""
        self.data["bt_score"] = 0.0
        self.data["bt_power"] = 0.0
        self.data["bt_mean"] = 0.0
        self.data["bt_period"] = 0.0
        self.data["bt_bperiod"] = 0.0
        self.data["bt_tpotind0_0"] = 0
        self.data["bt_tpotind0_1"] = 0
        self.data["bt_tpotind1_0"] = 0
        self.data["bt_tpotind1_1"] = 0
        self.data["bt_tpotind2_0"] = 0
        self.data["bt_tpotind2_1"] = 0
        self.data["bt_freq_bin"] = 0
        self.data["bt_time_bin"] = 0.0
        self.data["bt_chirp_rate"] = 0.0
        self.data["bt_scale"] = 0.0
        self.data["bt_fft_len"] = 0
        self.data["bt_pot"] = ""
        self.bg_pot_changed = False
    
    def __getitem__(self, name):
        if name[:7] == "bg_pot ":
            return self.data[name[:6]][int(name[7:])]
        else:
            return Sah_Data.__getitem__(self, name)
    
    def add_data(self, key, value):
        if key[:7] == "bg_pot ":
            value = float(value)
            i = int(key[7:])
            if i >= len(self.data["bg_pot"]):
                self.data["bg_pot"].append(value)
                self.bg_pot_changed = True
            elif self.data["bg_pot"][i] != value:
                self.data["bg_pot"][i] = value
                self.bg_pot_changed = True
        else:
            if self.bg_pot_changed:
                self.have_changed("bg_pot", self.data["bg_pot"])
                self.bg_pot_changed = False
            Sah_Data.add_data(self, key, value)

class OutFile(Sah_Data):
    SAH_NAME = "outfile.sah"
    def __init__(self, setipyth=""):
        from warnings import warn
        warn("OutFile not implementet yet!")
        Sah_Data.__init__(self)
        
class Work_Unit(Sah_Data):
    SAH_NAME = "work_unit.sah"
    def __init__(self, setipath=""):
        Sah_Data.__init__(self, setipath)

    def init_data(self):
        Sah_Data.init_data(self)
        self.data["type"] = ""
        self.data["task"] = ""
        self.data["version"] = 0
        self.data["name"] = ""
        self.data["data_type"] = ""
        self.data["data_class"] = 0
        self.data["splitter_version"] = ""
        self.data["start_ra"] = 0.0
        self.data["start_dec"] = 0.0
        self.data["end_ra"] = 0.0
        self.data["end_dec"] = 0.0
        self.data["angle_range"] = 0.0
        self.data["true_angle_range"] = 0.0
        self.data["beam_width"] = 0.0
        self.data["time_recorded"] = ""
        self.data["subband_center"] = 0.0
        self.data["subband_base"] = 0.0
        self.data["subband_sample_rate"] = 0.0
        self.data["fft_len"] = 0
        self.data["ifft_len"] = 0
        self.data["subband_number"] = 0
        self.data["receiver"] = ""
        self.data["nsamples"] = 0
        self.data["tape_version"] = 0.0
        self.data["num_positions"] = 0
        self.data["coord"] = [(0.0,0.0,0.0)]*self.data["num_positions"]
        self.coord_changed = False
    
    def type_of(self, name, value=False):
        if name == "coord":
            if value:
                return value
            else:
                return [(0.0,0.0,0.0)]*self.data["num_positions"]
        
        return Sah_Data.type_of(self, name, value)
                
    def add_data(self, key, value):
        if key[:5] == "coord":
            value = tuple(map(float,value.split()))
            i = int(key[5:])
            if self.data["coord"][i] != value:
                self.data["coord"][i] = value
                self.coord_changed = True
        else:
            if self.coord_changed:
                self.have_changed("coord", self.data["coord"])
                self.coord_changed = False            
            if key == "num_positions" and (self.data[key] != int(value)):
                self.data["coord"] = [(0.0, 0.0, 0.0)] * int(value)
            Sah_Data.add_data(self, key, value)

    def do_line(self, line):
        if line.strip() == "end_header": return True
        if line.strip() == "end_seti_header": return False
        return Sah_Data.do_line(self, line)


class Seti_Data:
    maybe = [
        "/usr/setiathome",
        "/usr/seti",
        "/usr/local/setiathome",
        "/usr/local/seti",
        os.path.expanduser("~/setiathome"),
        os.path.expanduser("~/.setiathome"),
        os.path.expanduser("~/seti"),
        os.path.expanduser("~/.seti")]

    def __init__(self, path=""):
        self.path = path
        if not path:
            for mb in self.maybe:
                if os.path.exists(mb):
                    self.path = mb
        
        self.version = Version()
        self.user_info = User_Info()
        self.state = State()
        self.work_unit = Work_Unit()
        #self.outfile = OutFile()
        self.new_path(self.path)
    
    def new_path(self, path):
        self.status = OK
        path = os.path.expanduser(path)
        if path == "" or not os.path.exists(self.path):
            self.clear_all()
            self.status = NO_SETI_PATH
            return False

        self.path = path
        self.version.new_path(self.path)
        self.user_info.new_path(self.path)
        self.state.new_path(self.path)
        self.work_unit.new_path(self.path)
        #self.outfile.new_path(self.path)
        return self.update_all()
    
    def update_all(self):
        if self.status_ok():
            self.version.update()
            self.user_info.update()
            self.state.update()
            self.work_unit.update()
            return True
        else:
            return False
    
    def clear_all(self):
        self.version.clear_data()
        self.user_info.clear_data()
        self.state.clear_data()
        self.work_unit.clear_data()
        
    def set_notify(self, sah_name, entry, func):
        if sah_name == self.version.SAH_NAME:
            self.version.set_notify(entry, func)
        elif sah_name == self.user_info.SAH_NAME:
            self.user_info.set_notify(entry, func)
        elif sah_name == self.state.SAH_NAME:
            self.state.set_notify(entry, func)
        elif sah_name == self.work_unit.SAH_NAME:
            self.work_unit.set_notify(entry, func)
        else:
            raise ValueError("%s unknown!" % sah_name)
    
    def status_ok(self):
        self.status = OK
        if self.path == "" or not os.path.exists(self.path):
            self.clear_all()
            self.status = NO_SETI_PATH
            return False
        if not os.path.exists(os.path.join(self.path, "lock.sah")):
            self.clear_all()
            self.status = NOT_RUNNING
            return False
        if not os.path.exists(os.path.join(self.path, "state.sah")):
            self.clear_all()
            self.status = CONNECTING
            return False
        return True
        
    def real_cpu_time(self):
        if self.status_ok() and self.state.last_update:
            akt = int(round(self.state["cpu"] + (
                            time() - self.state.last_update) - 4))
            return akt
        elif self.status == NOT_RUNNING:
            akt = int(self.state.last_update)
        else:
            return False
            
    def seti_version(self):
        return self.version.version

if __name__ == "__main__":
    seti = Seti_Data()
    print seti.path
    print seti.seti_version()
    print seti.user_info.data
    print seti.state.data
    print seti.work_unit.data
