/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-part-viewer-app.c,v 1.18 2005/02/01 02:32:00 hoa Exp $
 */

#include "etpan-part-viewer-app.h"

#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <ncurses.h>
#include <string.h>
#include "etpan-viewer-common.h"
#include "etpan-imf-helper.h"
#include "etpan-mime-tools.h"
#include "etpan-subapp-thread.h"
#include "etpan-help-viewer.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_part_viewer_app_driver = {
  .name = "part-viewer",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .get_idle_udelay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  /* msg */
  struct mailmime * mime;
  mailmessage * msg;
  struct mailfolder * folder;
  int part_type;
  
  struct etpan_viewer_common_app_state common_state;
};

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case 'q':
  case KEY_LEFT:
    etpan_app_quit_subapp(app);
    break;
  }
}

static void set_title(struct etpan_subapp * app);

static int display_init(struct etpan_subapp * app)
{
  set_title(app);
  return etpan_app_subapp_display_init(app);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_set_color(app, &state->common_state);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_viewer_common_display(app, &state->common_state, w, "q: exit");
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->msg = NULL;
  state->folder = NULL;
  state->mime = NULL;
  state->part_type = ETPAN_PART_VIEWER_NONE;
  
  r = etpan_viewer_common_init(subapp, &state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  /* do nothing */
  etpan_viewer_common_done(subapp, &state->common_state);
  free(state);
}

struct etpan_subapp * etpan_part_viewer_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_part_viewer_app_driver);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;

  etpan_part_viewer_app_flush(app);
}

#define TITLE_NO_MSG_ID "etPan! - message part viewer"
#define TITLE_MSG_ID "etPan! - message part viewer - "

static void set_title(struct etpan_subapp * app)
{
  char * content_type_str;
  int title_set;
  struct app_state * state;
  struct mailmime * mime;
  
  state = app->data;
  mime = state->mime;
  
  content_type_str = NULL;
  if (mime != NULL)
    if (mime->mm_content_type != NULL)
      content_type_str = etpan_get_content_type_str(mime->mm_content_type);
  
  title_set = 0;
  
  if (content_type_str != NULL) {
    char * title;
    size_t len;
    
    len = sizeof(TITLE_MSG_ID) + strlen(content_type_str);
    
    title = malloc(len);
    if (title != NULL) {
      snprintf(title, len, "%s%s", TITLE_MSG_ID, content_type_str);
      etpan_subapp_set_title(app, title);
      free(title);
      title_set = 1;
    }
    free(content_type_str);
  }
  
  if (!title_set) {
    etpan_subapp_set_title(app, TITLE_NO_MSG_ID);
  }
}

int etpan_part_viewer_app_set_part(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime, int part_type,
    char * filename, FILE * f, carray * attr)
{
  struct app_state * state;
  int r;
  int res;
  
  state = app->data;
  
  etpan_part_viewer_app_flush(app);
  
  r = etpan_viewer_common_set_file(&state->common_state,
      filename, f, attr,
      ETPAN_VIEWER_FREE_FLAGS_CLOSE | ETPAN_VIEWER_FREE_FLAGS_REMOVE);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  r = etpan_queue_ref_msg_mime(app, msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "view message - not enough memory"));
    res = r;
    goto flush;
  }
  
  state->folder = folder;
  state->msg = msg;
  state->mime = mime;
  state->part_type = part_type;

  set_title(app);
  
  return NO_ERROR;

 flush:
  etpan_viewer_common_flush(&state->common_state);
 err:
  return res;
}

void etpan_part_viewer_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->msg != NULL) {
    etpan_queue_unref_msg_mime(app, state->msg);
  }
  
  state->part_type = ETPAN_PART_VIEWER_NONE;
  state->folder = NULL;
  state->mime = NULL;
  state->msg = NULL;
  etpan_viewer_common_flush(&state->common_state);
}


struct mailmime * etpan_part_viewer_app_get_mime(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->mime;
}

int etpan_part_viewer_app_get_part_type(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->part_type;
}

mailmessage * etpan_part_viewer_app_get_msg(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->msg;
}

struct mailfolder *
etpan_part_viewer_app_get_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->folder;
}


#define HELP_TEXT \
"\
Help for MIME part viewer\n\
-------------------------\n\
\n\
This application will let you view the content of a MIME part.\n\
\n\
- Ctrl-W,\n\
  Ctrl-X     : switch between applications\n\
\n\
- arrow keys : scroll\n\
\n\
- m          : compose new message\n\
- a          : reply to the author of this MIME part\n\
- r          : reply to the message in this MIME part\n\
- g          : reply to all recipient of the message in this MIME part\n\
- f          : forward this MIME part\n\
- F          : forward this MIME part as attachment\n\
- b          : bounce this MIME part\n\
- B          : redirect (Subject line is modified)\n\
\n\
- q          : exit\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
