/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-mime-params.c,v 1.9 2003/12/15 16:08:53 hoa Exp $
 */

#include "etpan-mime-params.h"

#include <stdlib.h>

#include <libetpan/libetpan.h>

#include "etpan-errors.h"
#if 0
#include "etpan-security-types.h"
#endif

struct etpan_mime_param {
  int opened;
  int visibility;
  char * privacy_driver;
  char * privacy_encryption;
};

static inline struct etpan_mime_param * param_new(void)
{
  struct etpan_mime_param * param;

  param = malloc(sizeof(struct etpan_mime_param));
  if (param == NULL)
    return param;
  
  param->opened = 1;
  param->visibility = 1;
  param->privacy_driver = NULL;
  param->privacy_encryption = NULL;
  
  return param;
}

static inline void param_free(struct etpan_mime_param * param)
{
  free(param);
}

/*
  create the mime states
*/

int etpan_mime_params_add(struct etpan_mime_params * params,
    struct mailmime * mime)
{
  int r;
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);

  r = chash_get(params->hash, &key, &data);
  if (r == 0)
    return NO_ERROR;

  param = param_new();
  if (param == NULL)
    goto err;

  r = carray_add(params->tab, mime, NULL);
  if (r < 0) {
    param_free(param);
    goto err;
  }
  
  data.data = param;
  data.len = 0;

  r = chash_set(params->hash, &key, &data, NULL);
  if (r < 0)
    goto err;

  return NO_ERROR;

 err:
  return ERROR_MEMORY;
}

int etpan_mime_params_add_recursive(struct etpan_mime_params * params,
    struct mailmime * root)
{
  int r;
  clistiter * cur;
  struct mailmime * child;

  r = etpan_mime_params_add(params, root);
  if (r != NO_ERROR)
    goto err;

  switch (root->mm_type) {
  case MAILMIME_SINGLE:
    break;

  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(root->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      child = cur->data;
      
      r = etpan_mime_params_add_recursive(params, child);
      if (r != NO_ERROR)
        goto err;
    }
    break;
  case MAILMIME_MESSAGE:
    if (root->mm_data.mm_message.mm_msg_mime != NULL) {
      child = root->mm_data.mm_message.mm_msg_mime;

      r = etpan_mime_params_add_recursive(params, child);
      if (r != NO_ERROR)
        goto err;
    }
    break;
  }

  return NO_ERROR;

 err:
  return ERROR_MEMORY;
}


/* initializes structures */

struct etpan_mime_params * etpan_mime_params_new(void)
{
  struct etpan_mime_params * params;

  params = malloc(sizeof(struct etpan_mime_params));
  if (params == NULL)
    goto err;

  params->hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (params->hash == NULL)
    goto free;
  params->tab = carray_new(128);
  if (params->tab == NULL)
    goto free_hash;

  return params;

 free_hash:
  chash_free(params->hash);
 free:
  free(params);
 err:
  return NULL;
}

void etpan_mime_params_free(struct etpan_mime_params * params)
{
  etpan_mime_params_clear(params);
  carray_free(params->tab);
  chash_free(params->hash);
  free(params);
}


/*
  destroys the mime states
*/

void etpan_mime_params_clear(struct etpan_mime_params * params)
{
  unsigned int i;
  int r;
  chashdatum key;
  chashdatum data;

  for(i = 0 ; i < carray_count(params->tab) ; i ++) {
    struct etpan_mime_param * param;
    struct mailmime * mime;

    mime = carray_get(params->tab, i);

    key.data = &mime;
    key.len = sizeof(mime);
    r = chash_delete(params->hash, &key, &data);
    if (r == 0) {
      param = data.data;
      param_free(param);
    }
  }
  carray_set_size(params->tab, 0);
}


/* stat mime */

void etpan_mime_set_visibility(struct etpan_mime_params * mime_params,
    struct mailmime * mime, int visible)
{
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);
  chash_get(mime_params->hash, &key, &data);

  param = data.data;

  param->visibility = visible;
}

int etpan_mime_is_visible(struct etpan_mime_params * mime_params,
    struct mailmime * mime)
{
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);
  chash_get(mime_params->hash, &key, &data);

  param = data.data;

  return param->visibility;
}

static void update_visibility(struct etpan_mime_params * params,
    struct mailmime * mime, int visible);

static inline void update_visibility_sub(struct etpan_mime_params * params,
    struct mailmime * child, int visible)
{
  chashdatum key;
  chashdatum data;
  struct etpan_mime_param * child_param;
  
  key.data = &child;
  key.len = sizeof(child);
  chash_get(params->hash, &key, &data);
  
  child_param = data.data;
  
  if (child_param->visibility != visible) {
    child_param->visibility = visible;
    
    if (visible)
      update_visibility(params, child, child_param->opened);
    else
      update_visibility(params, child, 0);
  }
}

static void update_visibility(struct etpan_mime_params * params,
    struct mailmime * mime, int visible)
{
  clistiter * cur;

  switch (mime->mm_type) {
  case MAILMIME_SINGLE:
    break;

  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      struct mailmime * child;
      
      child = clist_content(cur);
      
      update_visibility_sub(params, child, visible);
    }
    break;

  case MAILMIME_MESSAGE:
    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      struct mailmime * child;

      child = mime->mm_data.mm_message.mm_msg_mime;

      update_visibility_sub(params, child, visible);
    }
    break;
  }
}

int etpan_mime_is_opened(struct etpan_mime_params * mime_params,
    struct mailmime * mime)
{
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);
  chash_get(mime_params->hash, &key, &data);

  param = data.data;

  return param->opened;
}

void etpan_mime_set_opened(struct etpan_mime_params * mime_params,
    struct mailmime * mime, int opened)
{
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);
  chash_get(mime_params->hash, &key, &data);

  param = data.data;

  param->opened = opened;

  update_visibility(mime_params, mime, param->opened);
}

void etpan_mime_change_opened(struct etpan_mime_params * mime_params,
    struct mailmime * mime)
{
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);
  chash_get(mime_params->hash, &key, &data);

  param = data.data;

  param->opened = !param->opened;

  update_visibility(mime_params, mime, param->opened);
}

void etpan_mime_set_privacy(struct etpan_mime_params * mime_params,
    struct mailmime * mime,
    char * privacy_driver, char * privacy_encryption)
{
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);
  chash_get(mime_params->hash, &key, &data);

  param = data.data;
  
  param->privacy_driver = privacy_driver;
  param->privacy_encryption = privacy_encryption;
}

void etpan_mime_get_privacy(struct etpan_mime_params * mime_params,
    struct mailmime * mime,
    char ** privacy_driver, char ** privacy_encryption)
{
  struct etpan_mime_param * param;
  chashdatum key;
  chashdatum data;

  key.data = &mime;
  key.len = sizeof(mime);
  chash_get(mime_params->hash, &key, &data);

  param = data.data;

  * privacy_driver = param->privacy_driver;
  * privacy_encryption = param->privacy_encryption;
}
