/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-imf-helper.c,v 1.14 2004/12/12 12:38:57 hoa Exp $
 */

#include "etpan-imf-helper.h"

#include <string.h>
#include <stdlib.h>

#include <libetpan/libetpan.h>

#include "etpan-errors.h"
#include "etpan-cfg-mime.h"
#include "etpan-msg-new.h"
#include "etpan-tools.h"

struct mailimf_mailbox_list *
etpan_address_to_mailbox_list(struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  struct mailimf_mailbox_list * mb_list;
  int r;
  
  mb_list = mailimf_mailbox_list_new_empty();
  if (mb_list == NULL)
    goto err;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox_list * new_mb_list;
    
    addr = clist_content(cur);
    switch (addr->ad_type) {
    case MAILIMF_ADDRESS_MAILBOX:
      mb = etpan_dup_mailbox(addr->ad_data.ad_mailbox);
      if (mb == NULL)
        goto free_list;
      r = mailimf_mailbox_list_add(mb_list, mb);
      if (r < 0) {
        mailimf_mailbox_free(mb);
        goto free_list;
      }
      break;
      
    case MAILIMF_ADDRESS_GROUP:
      if (addr->ad_data.ad_group->grp_mb_list != NULL) {
        new_mb_list =
          etpan_dup_mailbox_list(addr->ad_data.ad_group->grp_mb_list);
        if (new_mb_list)
          goto free_list;
        
        r = etpan_append_mailbox_list(mb_list, new_mb_list);
        mailimf_mailbox_list_free(new_mb_list);
        if (r < 0) {
          goto free_list;
        }
      }
      break;
    }
  }
  
  return mb_list;

 free_list:
  mailimf_mailbox_list_free(mb_list);
 err:
  return NULL;
}

struct mailimf_mailbox * etpan_dup_mailbox(struct mailimf_mailbox * mb)
{
  struct mailimf_mailbox * new_mb;
  char * display_name;
  char * addr_spec;
  
  display_name = NULL;
  if (mb->mb_display_name != NULL) {
    display_name = strdup(mb->mb_display_name);
    if (display_name == NULL)
      goto err;
  }

  addr_spec = NULL;
  if (mb->mb_addr_spec != NULL) {
    addr_spec = strdup(mb->mb_addr_spec);
    if (addr_spec == NULL)
      goto free_dsp_name;
  }
  
  new_mb = mailimf_mailbox_new(display_name, addr_spec);
  if (new_mb == NULL)
    goto free_addr_spec;
  
  return new_mb;

 free_addr_spec:
  if (addr_spec != NULL)
    free(addr_spec);
 free_dsp_name:
  if (display_name != NULL)
    free(display_name);
 err:
  return NULL;
}

struct mailimf_mailbox_list *
etpan_dup_mailbox_list(struct mailimf_mailbox_list * mb_list)
{
  struct mailimf_mailbox_list * new_mb_list;
  clist * list;
  clistiter * cur;
  int r;
  
  list = clist_new();
  if (list == NULL)
    goto err;

  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox * new_mb;
    
    mb = clist_content(cur);
    new_mb = etpan_dup_mailbox(mb);
    if (new_mb == NULL)
      goto free_list;

    r = clist_append(list, new_mb);
    if (r < 0) {
      mailimf_mailbox_free(new_mb);
      goto free_list;
    }
  }
  
  new_mb_list = mailimf_mailbox_list_new(list);
  if (new_mb_list == NULL)
    goto free_list;

  return new_mb_list;

 free_list:
  clist_foreach(list, (clist_func) mailimf_mailbox_free, NULL);
  clist_free(list);
 err:
  return NULL;
}

struct mailimf_address_list *
etpan_mailbox_to_address_list(struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  clist * list;
  struct mailimf_address_list * addr_list;
  int r;
  
  list = clist_new();
  if (list == NULL)
    goto err;

  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox * new_mb;
  
    mb = clist_content(cur);
    new_mb = etpan_dup_mailbox(mb);
    if (new_mb == NULL)
      goto free_list;

    addr = mailimf_address_new(MAILIMF_ADDRESS_MAILBOX, new_mb, NULL);
    if (addr == NULL) {
      mailimf_mailbox_free(new_mb);
      goto free_list;
    }

    r = clist_append(list, addr);
    if (r < 0) {
      mailimf_address_free(addr);
      goto free_list;
    }
  }

  addr_list = mailimf_address_list_new(list);
  if (addr_list == NULL)
    goto free_list;

  return addr_list;

 free_list:
  clist_foreach(list, (clist_func) mailimf_address_free, NULL);
  clist_free(list);
 err:
  return NULL;
}

struct mailimf_group * etpan_dup_group(struct mailimf_group * group)
{
  struct mailimf_group * new_group;
  struct mailimf_mailbox_list * mb_list;
  char * dsp_name;
  
  mb_list = NULL;
  if (group->grp_mb_list != NULL) {
    mb_list = etpan_dup_mailbox_list(group->grp_mb_list);
    if (mb_list == NULL)
      goto err;
  }
  
  dsp_name = strdup(group->grp_display_name);
  if (dsp_name == NULL)
    goto free_mb_list;
  
  new_group = mailimf_group_new(strdup(group->grp_display_name), mb_list);
  if (new_group == NULL)
    goto free_dsp_name;
  
  return new_group;

 free_dsp_name:
  free(dsp_name);
 free_mb_list:
  if (mb_list != NULL)
    mailimf_mailbox_list_free(mb_list);
 err:
  return NULL;
}

struct mailimf_address * etpan_dup_address(struct mailimf_address * addr)
{
  struct mailimf_address * new_addr;
  struct mailimf_mailbox * new_mb;
  struct mailimf_group * group;
	
  new_mb = NULL;
  group = NULL;
  
  switch (addr->ad_type) {
  case MAILIMF_ADDRESS_MAILBOX:
    new_mb = etpan_dup_mailbox(addr->ad_data.ad_mailbox);
    if (new_mb == NULL)
      goto err;
    break;
    
  case MAILIMF_ADDRESS_GROUP:
    group = etpan_dup_group(addr->ad_data.ad_group);
    if (group == NULL)
      goto err;
    break;
  }
  
  new_addr = mailimf_address_new(addr->ad_type, new_mb, group);
  if (new_addr == NULL)
    goto free;
  
  return new_addr;
  
 free:
  if (group != NULL)
    mailimf_group_free(group);
  if (new_mb != NULL)
    mailimf_mailbox_free(new_mb);
 err:
  return NULL;
}

int etpan_append_address_list(struct mailimf_address_list * new_addr_list,
    struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  int r;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_address * addr;
    struct mailimf_address * new_addr;
    
    addr = clist_content(cur);
    
    new_addr = etpan_dup_address(addr);
    if (new_addr == NULL)
      goto err;

    r = clist_append(new_addr_list->ad_list, new_addr);
    if (r < 0) {
      mailimf_address_free(new_addr);
      goto err;
    }
  }

  return MAILIMF_NO_ERROR;

 err:
  return MAILIMF_ERROR_MEMORY;
}

int etpan_append_mailbox_list(struct mailimf_mailbox_list * new_mb_list,
    struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  int r;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;
    struct mailimf_mailbox * new_mb;
    
    mb = clist_content(cur);
    
    new_mb = etpan_dup_mailbox(mb);
    if (new_mb == NULL)
      goto err;

    r = clist_append(new_mb_list->mb_list, new_mb);
    if (r < 0) {
      mailimf_mailbox_free(new_mb);
      goto err;
    }
  }

  return MAILIMF_NO_ERROR;

 err:
  return MAILIMF_ERROR_MEMORY;
}

struct mailimf_address_list *
etpan_dup_address_list(struct mailimf_address_list * addr_list)
{
  clist * list;
  struct mailimf_address_list * new_addr_list;
  int r;
  
  list = clist_new();
  if (list == NULL)
    goto err;

  new_addr_list = mailimf_address_list_new(list);
  if (new_addr_list == NULL) {
    goto free_list;
  }

  if (addr_list != NULL) {
    r = etpan_append_address_list(new_addr_list, addr_list);
    if (r != MAILIMF_NO_ERROR)
      goto free_list;
  }

  return new_addr_list;

 free_list:
  clist_foreach(list, (clist_func) mailimf_address_free, NULL);
  clist_free(list);
 err:
  return NULL;
}



static int set_agent_name(struct mailimf_fields * new_fields,
    char * field_name)
{
  struct mailimf_optional_field * opt_field;
  struct mailimf_field * field;
  int r;
  char * name;
  char * value;
  int res;

  name = strdup(field_name);
  if (name == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  value = strdup(MAILER_NAME);
  if (value == NULL) {
    res = ERROR_MEMORY;
    goto free_name;
  }

  opt_field = mailimf_optional_field_new(name, value);
  if (opt_field == NULL) {
    res = ERROR_MEMORY;
    goto free_value;
  }

  field = mailimf_field_new(MAILIMF_FIELD_OPTIONAL_FIELD,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, opt_field);
  if (field == NULL) {
    res = ERROR_MEMORY;
    goto free_opt_field;
  }

  r = clist_append(new_fields->fld_list, field);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto free_field;
  }

  return NO_ERROR;

 free_field:
  mailimf_field_free(field);
  goto err;
 free_opt_field:
  mailimf_optional_field_free(opt_field);
  goto err;
 free_value:
  free(value);
 free_name:
  free(name);
 err:
  return res;
}

int etpan_set_xmailer(struct mailimf_fields * new_fields,
    int nntp_protocol)
{
  char * field_name;
  
  if (nntp_protocol)
    field_name = "X-Newsreader";
  else
    field_name = "X-Mailer";

  return set_agent_name(new_fields, field_name);
}

int etpan_set_resentxmailer(struct mailimf_fields * new_fields,
    int nntp_protocol)
{
  char * field_name;
 
  if (nntp_protocol)
    field_name = "Resent-X-Newsreader";
  else
    field_name = "Resent-X-Mailer";
  
  return set_agent_name(new_fields, field_name);
}


int etpan_is_nntp_protocol(struct mailfolder * folder)
{
  if (folder == NULL)
    return 0;
  
  if (folder->fld_storage == NULL)
    return 0;

  if (strcasecmp(folder->fld_storage->sto_driver->sto_name, "nntp") == 0)
    return 1;
  
  return 0;
}



int etpan_set_current_newsgroups_header(struct mailimf_fields * new_fields,
    struct mailfolder * post_folder)
{
  struct mailimf_optional_field * opt_field;
  struct mailimf_field * field;
  int r;
  int res;
  char * name;
  char * value;

  name = strdup("Newsgroups");
  if (name == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  value = strdup(post_folder->fld_pathname);
  if (value == NULL) {
    res = ERROR_MEMORY;
    goto free_name;
  }
  
  opt_field = mailimf_optional_field_new(name, value);
  if (opt_field == NULL) {
    res = ERROR_MEMORY;
    goto free_value;
  }

  field = mailimf_field_new(MAILIMF_FIELD_OPTIONAL_FIELD,
      NULL, NULL, NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL, opt_field);
  if (field == NULL) {
    res = ERROR_MEMORY;
    goto free_opt_field;
  }

  r = clist_append(new_fields->fld_list, field);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto free_field;
  }

  return NO_ERROR;

 free_field:
  mailimf_field_free(field);
  goto err;
 free_opt_field:
  mailimf_optional_field_free(opt_field);
  goto err;
 free_value:
  free(value);
 free_name:
  free(name);
 err:
  return res;
}





#if 0
struct mailmime_content * mailmime_content_new_with_str(char * str)
{
  int r;
  size_t cur_token;
  struct mailmime_content * content;

  cur_token = 0;
  r =  mailmime_content_parse(str, strlen(str), &cur_token, &content);
  if (r != MAILIMF_NO_ERROR)
    return NULL;
  
  return content;
}

struct mailmime_fields * mailmime_fields_new_encoding(int type)
{
  struct mailmime_mechanism * encoding;
  struct mailmime_fields * mime_fields;

  encoding = mailmime_mechanism_new(type, NULL);
  if (encoding == NULL)
    goto err;

  mime_fields = mailmime_fields_new_with_data(encoding,
      NULL, NULL, NULL, NULL);
  if (mime_fields == NULL)
    goto free;

  return mime_fields;

 free:
  mailmime_mechanism_free(encoding);
 err:
  return NULL;
}


struct mailmime * mailmime_multiple_new(char * type)
{
  struct mailmime_fields * mime_fields;
  struct mailmime_content * content;
  struct mailmime * mp;

  mime_fields = mailmime_fields_new_encoding(MAILMIME_MECHANISM_8BIT);
  if (mime_fields == NULL)
    goto err;
  
  content = mailmime_content_new_with_str(type);
  if (content == NULL)
    goto free_fields;
  
  mp = mailmime_new_empty(content, mime_fields);
  if (mp == NULL)
    goto free_content;
  
  return mp;

 free_content:
  mailmime_content_free(content);
 free_fields:
  mailmime_fields_free(mime_fields);
 err:
  return NULL;
}


int etpan_add_part(struct mailmime * mime,
    struct mailmime * mime_sub)
{
  struct mailmime * saved_sub;
  struct mailmime * mp;
  int res;
  int r;

  switch (mime->type) {
  case MAILMIME_SINGLE:
    res = ERROR_INVAL;
    goto err;

  case MAILMIME_MULTIPLE:
    r = mailmime_add_part(mime, mime_sub);
    if (r != MAILIMF_NO_ERROR) {
      res = ERROR_MEMORY;
      goto err;
    }

    return NO_ERROR;
  }

  /* MAILMIME_MESSAGE */

  if (mime->msg_mime == NULL) {
    r = mailmime_add_part(mime, mime_sub);
    if (r != MAILIMF_NO_ERROR) {
      res = ERROR_MEMORY;
      goto err;
    }

    return NO_ERROR;
  }
  
  if (mime->msg_mime->type == MAILMIME_MULTIPLE)
    return etpan_add_part(mime->msg_mime, mime_sub);
  
  saved_sub = mime->msg_mime;
  
  mp = mailmime_multiple_new("multipart/mixed");
  if (mp == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  mailmime_remove_part(saved_sub);
  
  r = mailmime_add_part(mime, mp);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_mp;
  }
  
  r = mailmime_add_part(mp, saved_sub);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_saved_sub;
  }

  r = mailmime_add_part(mp, mime_sub);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_saved_sub;
  }

  return NO_ERROR;

 free_mp:
  mailmime_free(mp);
 free_saved_sub:
  mailmime_free(saved_sub);
 err:
  return res;
}
#endif

#if 0
int etpan_delete_possible(struct mailmime * mime)
{
  struct mailmime * parent;

  parent = mime->parent;
  if (parent == NULL)
    return 0;

  switch (mime->type) {
  case MAILMIME_MESSAGE:
    if (mime->msg_mime != NULL)
      return 0;
      
    return 1;

  case MAILMIME_MULTIPLE:
    if (!clist_isempty(mime->list))
      return 0;
      
    return 1;

  case MAILMIME_SINGLE:
    return 1;
    
  default:
    return 0;
  }
}
#endif

#if 0
int etpan_delete_part(struct mailmime * mime)
{
  struct mailmime * parent;
  int res;

  parent = mime->parent;
  if (parent == NULL) {
    res = ERROR_INVAL;
    goto err;
  }

  switch (mime->type) {
  case MAILMIME_MESSAGE:
    if (mime->msg_mime != NULL) {
      res = ERROR_INVAL;
      goto err;
    }

    mailmime_remove_part(mime);
    
    mailmime_free(mime);

    return NO_ERROR;

  case MAILMIME_MULTIPLE:
    if (!clist_isempty(mime->list)) {
      res = ERROR_INVAL;
      goto err;
    }
      
    mailmime_remove_part(mime);
    
    mailmime_free(mime);
    
    return NO_ERROR;

  case MAILMIME_SINGLE:
    mailmime_remove_part(mime);

    mailmime_free(mime);

    return NO_ERROR;
    
  default:
    return ERROR_INVAL;
  }

 err:
  return res;
}
#endif







#if 0
#define MAX_MAIL_COL 72
#endif

#if 0
static int
etpan_mailbox_write(FILE * f, int * col,
    struct etpan_global_config * global_config,
    struct mailimf_mailbox * mb)
{
  int r;

  if (* col > 1) {
    
    if (* col + strlen(mb->addr_spec) >= MAX_MAIL_COL) {
      r = mailimf_string_write(f, col, "\r\n ", 3);
      if (r != MAILIMF_NO_ERROR)
	return ERROR_FILE;
      * col = 1;
    }
  }
  
  if (mb->display_name) {
    char * decoded_from;
    size_t cur_token;

    cur_token = 0;
    r = mailmime_encoded_phrase_parse(global_config->message_charset,
        mb->display_name, strlen(mb->display_name),
        &cur_token, global_config->display_charset,
        &decoded_from);
    if (r != MAILIMF_NO_ERROR) {
      decoded_from = strdup(mb->display_name);
      if (decoded_from == NULL)
        return ERROR_MEMORY;
    }

    r = mailimf_quoted_string_write(f, col, decoded_from,
        strlen(decoded_from));
    if (r != MAILIMF_NO_ERROR) {
      free(decoded_from);
      return ERROR_FILE;
    }

    if (* col > 1) {
      
      if (* col + strlen(decoded_from) + 3 >= MAX_MAIL_COL) {
	r = mailimf_string_write(f, col, "\r\n ", 3);
	if (r != MAILIMF_NO_ERROR) {
          free(decoded_from);
	  return r;
        }
	* col = 1;
      }
    }

    free(decoded_from);
    
    r = mailimf_string_write(f, col, " <", 2);
    if (r != MAILIMF_NO_ERROR)
      return ERROR_FILE;

    r = mailimf_string_write(f, col, mb->addr_spec, strlen(mb->addr_spec));
    if (r != MAILIMF_NO_ERROR)
      return ERROR_FILE;

    r = mailimf_string_write(f, col, ">", 1);
    if (r != MAILIMF_NO_ERROR)
      return ERROR_FILE;
  }
  else {
    r = mailimf_string_write(f, col, mb->addr_spec, strlen(mb->addr_spec));
    if (r != MAILIMF_NO_ERROR)
      return ERROR_FILE;
  }


  return NO_ERROR;

}

int
etpan_mailbox_list_write(FILE * f, int * col,
    struct etpan_global_config * global_config,
    struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  int r;
  int first;

  first = TRUE;

  for(cur = clist_begin(mb_list->list) ; cur != NULL ; cur = cur->next) {
    struct mailimf_mailbox * mb;

    mb = cur->data;

    if (!first) {
      r = mailimf_string_write(f, col, ", ", 2);
      if (r != MAILIMF_NO_ERROR)
	return ERROR_FILE;
    }
    else {
      first = FALSE;
    }

    r = etpan_mailbox_write(f, col, global_config, mb);
    if (r != NO_ERROR)
      return r;
  }

  return NO_ERROR;
}

static int
etpan_group_write(FILE * f, int * col,
    struct etpan_global_config * global_config,
    struct mailimf_group * group)
{
  int r;

  r = mailimf_string_write(f, col, group->display_name,
			   strlen(group->display_name));
  if (r != MAILIMF_NO_ERROR)
    return ERROR_FILE;

  r = mailimf_string_write(f, col, ": ", 2);
  if (r != MAILIMF_NO_ERROR)
    return ERROR_FILE;
  
  if (group->mb_list != NULL) {
    r = etpan_mailbox_list_write(f, col, global_config, group->mb_list);
    if (r != NO_ERROR)
      return r;
  }

  r = mailimf_string_write(f, col, ";", 1);
  if (r != MAILIMF_NO_ERROR)
    return ERROR_FILE;

  return NO_ERROR;
}

int
etpan_address_write(FILE * f, int * col,
    struct etpan_global_config * global_config,
    struct mailimf_address * addr)
{
  int r;

  switch(addr->type) {
  case MAILIMF_ADDRESS_MAILBOX:
    r = etpan_mailbox_write(f, col, global_config, addr->mailbox);
    if (r != NO_ERROR)
      return r;

    break;

  case MAILIMF_ADDRESS_GROUP:
    r = etpan_group_write(f, col, global_config, addr->group);
    if (r != NO_ERROR)
      return r;
    
    break;
  }

  return MAILIMF_NO_ERROR;
}

int
etpan_address_list_write(FILE * f, int * col,
    struct etpan_global_config * global_config,
    struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  int r;
  int first;

  first = TRUE;

  for(cur = clist_begin(addr_list->list) ; cur != NULL ; cur = cur->next) {
    struct mailimf_address * addr;

    addr = cur->data;

    if (!first) {
      r = mailimf_string_write(f, col, ", ", 2);
      if (r != MAILIMF_NO_ERROR)
	return ERROR_FILE;
    }
    else {
      first = FALSE;
    }

    r = etpan_address_write(f, col, global_config, addr);
    if (r != NO_ERROR)
      return r;
  }

  return NO_ERROR;
}
#endif

int
etpan_get_to_cc_address(struct mailimf_single_fields * single_fields,
    struct mailimf_address_list ** result)
{
  struct mailimf_address_list * addr_list;
  int r;

  addr_list = NULL;

  if (single_fields->fld_to != NULL) {  
    if (single_fields->fld_to->to_addr_list != NULL) {
      if (addr_list == NULL) {
        addr_list = etpan_dup_address_list(NULL);
        if (addr_list == NULL)
          goto free_addr_list;
      }

      r = etpan_append_address_list(addr_list,
          single_fields->fld_to->to_addr_list);
      if (r != NO_ERROR)
        goto free_addr_list;
    }
  }

  if (single_fields->fld_cc != NULL) {
    if (single_fields->fld_cc->cc_addr_list != NULL) {
      if (addr_list == NULL) {
        addr_list = etpan_dup_address_list(NULL);
        if (addr_list == NULL)
          goto free_addr_list;
      }

      r = etpan_append_address_list(addr_list,
          single_fields->fld_cc->cc_addr_list);
      if (r != NO_ERROR)
        goto free_addr_list;
    }
  }

  * result = addr_list;

  return NO_ERROR;

 free_addr_list:
  if (addr_list != NULL)
    mailimf_address_list_free(addr_list);
  return ERROR_MEMORY;
}

#define MAX_DATE_STR 256

/* 0 = Sunday */
/* y > 1752 */

static int dayofweek(int year, int month, int day)
{
  static int offset[] = {0, 3, 2, 5, 0, 3, 5, 1, 4, 6, 2, 4};

  year -= month < 3;

  return (year + year/4 - year/100 + year/400 + offset[month-1] + day) % 7;
}

char * mailimf_date_time_to_string(struct mailimf_date_time * date_time)
{
  char date_str[MAX_DATE_STR];
  char * week_of_day_str[] = { "Sun", "Mon", "Tue", "Wed", "Thu",
			       "Fri", "Sat"};
  char * month_str[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
			 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
  int wday;
  char * date_str_dup;
  
  wday = dayofweek(date_time->dt_year, date_time->dt_month,
      date_time->dt_day);

  snprintf(date_str, MAX_DATE_STR, "%s, %i %s %i %02i:%02i:%02i %+05i",
      week_of_day_str[wday], date_time->dt_day,
      month_str[date_time->dt_month - 1],
      date_time->dt_year, date_time->dt_hour,
      date_time->dt_min, date_time->dt_sec,
      date_time->dt_zone);

  date_str_dup = strdup(date_str);
  if (date_str_dup == NULL)
    return NULL;

  return date_str_dup;
}

#if 0
char *
mailimf_mailbox_to_string(struct etpan_app * app,
    struct mailimf_mailbox * mb)
#endif

char *
mailimf_mailbox_to_string(struct mailimf_mailbox * mb)
{
  char * mb_str;
  size_t len;

  if (mb->mb_display_name != NULL) {
#if 0
    char * decoded_from;
    int r;
    size_t cur_token;

    cur_token = 0;
    r = mailmime_encoded_phrase_parse(app->global_config->message_charset,
        mb->display_name, strlen(mb->display_name),
        &cur_token, app->global_config->display_charset,
        &decoded_from);
    if (r != MAILIMF_NO_ERROR) {
      decoded_from = strdup(mb->display_name);
      if (decoded_from == NULL)
        goto err;
    }
    
    len = strlen(decoded_from) + strlen(mb->addr_spec) + 4;
    mb_str = malloc(len);
    if (mb_str == NULL) {
      free(decoded_from);
      goto err;
    }

    snprintf(mb_str, len, "%s <%s>", decoded_from, mb->addr_spec);
    
    free(decoded_from);
#endif
    len = strlen(mb->mb_display_name) + strlen(mb->mb_addr_spec) + 4;
    mb_str = malloc(len);
    if (mb_str == NULL)
      goto err;
    
    snprintf(mb_str, len, "%s <%s>", mb->mb_display_name, mb->mb_addr_spec);
    return mb_str;
  }
  else {
    len = strlen(mb->mb_addr_spec) + 3;
    mb_str = malloc(len);
    if (mb_str == NULL)
      goto err;

    snprintf(mb_str, len, "<%s>", mb->mb_addr_spec);

    return mb_str;
  }

 err:
  return NULL;
}

/*
char *
mailimf_mailbox_list_to_string(struct etpan_app * app,
    struct mailimf_mailbox_list * mb_list)
*/
char *
mailimf_mailbox_list_to_string(struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  int first;
  MMAPString * mmapstr;
  char * mb_list_str;

  mmapstr = mmap_string_new("");
  if (mmapstr == NULL)
    goto err;
  
  first = 1;
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    char * mb_str;
    size_t len_mb;
    
    mb_str = mailimf_mailbox_to_string(clist_content(cur));
    if (mb_str == NULL)
      goto free;
    
    len_mb = strlen(mb_str);

    if (!first) {
      if (mmap_string_append(mmapstr, ", ") == NULL) {
        free(mb_str);
        goto free;
      }
      if (mmap_string_append(mmapstr, mb_str) == NULL) {
        free(mb_str);
        goto free;
      }
    }
    else {
      first = 0;
      if (mmap_string_append(mmapstr, mb_str) == NULL) {
        free(mb_str);
        goto free;
      }
    }

    free(mb_str);
  }

  mb_list_str = strdup(mmapstr->str);
  if (mb_list_str == NULL)
    goto free;

  mmap_string_free(mmapstr);

  return mb_list_str;
  
 free:
  mmap_string_free(mmapstr);
 err:
  return NULL;
}

/*
char *
mailimf_group_to_string(struct etpan_app * app,
    struct mailimf_group * group)
*/
char *
mailimf_group_to_string(struct mailimf_group * group)
{
  char * group_str;

  if (group->grp_mb_list != NULL) {
    char * mb_list_str;
    size_t len;

    mb_list_str = mailimf_mailbox_list_to_string(group->grp_mb_list);
    if (mb_list_str == NULL)
      return NULL;

    len = strlen(group->grp_display_name) + strlen(mb_list_str) + 4;
    group_str = malloc(len);
    if (group_str == NULL) {
      free(mb_list_str);
      return NULL;
    }
    
    snprintf(group_str, len, "%s: %s;", group->grp_display_name, mb_list_str);
    free(mb_list_str);
    
    return group_str;
  }
  else {
    char * group_str;
    size_t len;
    
    len = strlen(group->grp_display_name) + 3;
    group_str = malloc(len);
    if (group_str == NULL)
      return NULL;
    
    snprintf(group_str, len, "%s:;", group->grp_display_name);
    
    return group_str;
  }
}

/*
char *
mailimf_address_to_string(struct etpan_app * app,
    struct mailimf_address * address)
*/
char *
mailimf_address_to_string(struct mailimf_address * address)
{
  if (address->ad_type == MAILIMF_ADDRESS_GROUP)
    return mailimf_group_to_string(address->ad_data.ad_group);
  else
    return mailimf_mailbox_to_string(address->ad_data.ad_mailbox);
}

/*
char *
mailimf_address_list_to_string(struct etpan_app * app,
struct mailimf_address_list * addr_list)
*/
char *
mailimf_address_list_to_string(struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  int first;
  MMAPString * mmapstr;
  char * addr_list_str;

  mmapstr = mmap_string_new("");
  if (mmapstr == NULL)
    goto err;

  first = 1;
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    char * addr_str;
    size_t len_addr;
    
    addr_str = mailimf_address_to_string(clist_content(cur));
    if (addr_str == NULL)
      goto free;
    
    len_addr = strlen(addr_str);

    if (!first) {
      if (mmap_string_append(mmapstr, ", ") == NULL) {
        free(addr_str);
        goto free;
      }
      if (mmap_string_append(mmapstr, addr_str) == NULL) {
        free(addr_str);
        goto free;
      }
    }
    else {
      first = 0;
      if (mmap_string_append(mmapstr, addr_str) == NULL) {
        free(addr_str);
        goto free;
      }
    }

    free(addr_str);
  }

  addr_list_str = strdup(mmapstr->str);
  if (addr_list_str == NULL)
    goto free;

  mmap_string_free(mmapstr);

  return addr_list_str;
  
 free:
  mmap_string_free(mmapstr);
 err:
  return NULL;
}



/*
  TODO : better get_from_value that will return
  an allocated rendered string
*/

void etpan_get_from_value(struct mailimf_single_fields * fields,
    char ** from, int * is_addr)
{
  struct mailimf_mailbox * mb;
  
  if (fields->fld_from == NULL) {
    * from = NULL;
    if (is_addr != NULL)
      * is_addr = FALSE;
    return;
  }
  
  if (clist_isempty(fields->fld_from->frm_mb_list->mb_list)) {
    * from = NULL;
    if (is_addr != NULL)
      * is_addr = FALSE;
    return;
  }
  
  mb = clist_begin(fields->fld_from->frm_mb_list->mb_list)->data;

  if (mb->mb_display_name != NULL) {
    * from = mb->mb_display_name;
    if (is_addr != NULL)
      * is_addr = FALSE;
  }
  else {
    * from = mb->mb_addr_spec;
    if (is_addr != NULL)
      * is_addr = TRUE;
  }
}


char * etpan_dup_imf_file(char * source_filename)
{
  char filename[PATH_MAX];
  FILE * dest_f;
  int r;
  struct stat stat_info;
  char * dest_filename;
  char * mapping;
  int fd;
  int col;
  
  dest_f = etpan_get_tmp_file(filename, sizeof(filename));
  if (dest_f == NULL)
    goto err;
  
  dest_filename = strdup(filename);
  if (dest_filename == NULL)
    goto close_dest;
  
  fd = open(source_filename, O_RDONLY);
  if (fd < 0)
    goto free_dest;
  
  r = fstat(fd, &stat_info);
  if (r < 0)
    goto close_src;
  
  mapping = mmap(NULL, stat_info.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (mapping == MAP_FAILED)
    goto close_src;
  
  col = 0;
  r = mailimf_string_write(dest_f, &col, mapping, stat_info.st_size);
  if (r != MAILIMF_NO_ERROR)
    goto unmap;
  
  munmap(mapping, stat_info.st_size);
  close(fd);
  fclose(dest_f);
  
  return dest_filename;
  
 unmap:
  munmap(mapping, stat_info.st_size);
 close_src:
  close(fd);
 free_dest:
  free(dest_filename);
 close_dest:
  fclose(dest_f);
 err:
  return NULL;
}
