#include "etpan-cfg-sender.h"

#include "etpan-errors.h"
#include "etpan-db-read.h"
#include <stdlib.h>
#include <string.h>

int etpan_sender_config_read(char * filename,
    struct etpan_sender_config ** pconfig)
{
  struct etpan_db * db;
  int r;
  int res;
  unsigned int i;
  struct etpan_sender_config * config;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  config = etpan_sender_config_new();
  if (config == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }

  config->default_sender[0] = '\0';
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    char * type_str;
    struct etpan_sender_item * item;
    char * command;
    char * host;
    char * login;
    char * password;
    char * name;
    int type;
    char * default_sender;
    unsigned int port;
    
    entry = carray_get(db->data, i);
    
    default_sender = etpan_db_entry_get_value(entry, "default-sender");
    if (default_sender != NULL) {
      strncpy(config->default_sender, default_sender,
          sizeof(config->default_sender));
      config->default_sender[sizeof(config->default_sender) - 1] = '\0';
      continue;
    }
    
    type_str = etpan_db_entry_get_value(entry, "type");
    if (type_str == NULL)
      continue;
    
    name = etpan_db_entry_get_value(entry, "name");
    if (name == NULL)
      continue;
    
    port = 0;
    command = NULL;
    host = NULL;
    login = NULL;
    password = NULL;
    
    type = SENDER_TYPE_COMMAND;
    if (strcmp(type_str, "command") == 0) {
      type = SENDER_TYPE_COMMAND;
      
      command = etpan_db_entry_get_value(entry, "command");
      if (command == NULL)
        continue;
    }
    else if (strcmp(type_str, "smtp") == 0) {
      char * port_str;
      
      type = SENDER_TYPE_SMTP;
      
      host = etpan_db_entry_get_value(entry, "host");
      if (host == NULL)
        continue;
      
      port_str = etpan_db_entry_get_value(entry, "port");
      if (port_str == NULL) {
        port = 25;
      }
      else {
        port = strtoul(port_str, NULL, 10);
      }
      
      login = etpan_db_entry_get_value(entry, "login");
      password = etpan_db_entry_get_value(entry, "password");
    }
    
    item = etpan_sender_item_new(name, type,
        command, host, port, login, password);
    if (item == NULL)
      continue;
    
    r = etpan_sender_config_add_item(config, item);
    if (r != NO_ERROR) {
      etpan_sender_item_free(item);
      continue;
    }
  }
  
  etpan_db_free(db);
  
  * pconfig = config;
  
  return NO_ERROR;
  
 free_config:
  etpan_sender_config_free(config);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}

void etpan_sender_config_free(struct etpan_sender_config * config)
{
  chashiter * iter;
  
  for(iter = chash_begin(config->sender_hash) ; iter != NULL ;
      iter = chash_next(config->sender_hash, iter)) {
    chashdatum value;
    struct etpan_sender_item * item;
    
    chash_value(iter, &value);
    item = value.data;
    
    etpan_sender_item_free(item);
  }
  
  chash_free(config->sender_hash);
  free(config);
}

struct etpan_sender_config * etpan_sender_config_new(void)
{
  struct etpan_sender_config * config;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    return NULL;
  
  config->sender_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (config->sender_hash == NULL) {
    free(config);
    return NULL;
  }
  
  config->default_sender[0] = '\0';
  
  return config;
}

void etpan_sender_item_free(struct etpan_sender_item * item)
{
  free(item);
}

struct etpan_sender_item * etpan_sender_item_new(char * name,
    int type, char * command, char * host, int port,
    char * login, char * password)
{
  struct etpan_sender_item * item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    return NULL;
    
  item->type = type;
    
  strncpy(item->name, name, sizeof(item->name));
  item->name[sizeof(item->name) - 1] = '\0';
    
  if (type == SENDER_TYPE_COMMAND) {
    item->command[0] = '\0';
    if (command != NULL) {
      strncpy(item->command, command, sizeof(item->command));
      item->command[sizeof(item->command) - 1] = '\0';
    }
  }
  else {
    item->host[0] = '\0';
    if (host != NULL) {
      strncpy(item->host, host, sizeof(item->host));
      item->host[sizeof(item->host) - 1] = '\0';
    }
      
    item->port = port;
    
    item->login[0] = '\0';
    if (login != NULL)
      strncpy(item->login, login, sizeof(item->login));
    
    item->password[0] = '\0';
    if (password != NULL)
      strncpy(item->password, password, sizeof(item->password));
  }
  
  return item;
}

int etpan_sender_config_add_item(struct etpan_sender_config * config,
    struct etpan_sender_item * item)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = item->name;
  key.len = strlen(item->name);
  value.data = item;
  value.len = 0;
  
  r = chash_set(config->sender_hash, &key, &value, NULL);
  if (r < 0)
    return ERROR_MEMORY;
  
  return NO_ERROR;
}

void etpan_sender_config_remove_item(struct etpan_sender_config * config,
    char * name)
{
  struct etpan_sender_item * item;
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name);
  r = chash_get(config->sender_hash, &key, &value);
  if (r < 0)
    return;
  
  item = value.data;
  etpan_sender_item_free(item);
  
  chash_delete(config->sender_hash, &key, NULL);
}

struct etpan_sender_item *
etpan_sender_config_get_item(struct etpan_sender_config * config)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = config->default_sender;
  key.len = strlen(config->default_sender);
  r = chash_get(config->sender_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

void etpan_sender_config_set_item(struct etpan_sender_config * config,
    char * name)
{
  strncpy(config->default_sender, name, sizeof(config->default_sender));
  config->default_sender[sizeof(config->default_sender) - 1] = '\0';
}

