/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: stones_simple.cc,v 1.53.2.5 2003/10/11 13:48:06 dheck Exp $
 */
#include "stones.hh"
#include "objects.hh"
#include "object_mixins.hh"
#include "game.hh"
#include "laser.hh"

#include "stones_internal.hh"

#include <cassert>
#include <vector>

using namespace std;
using namespace world;
using namespace enigma;
using namespace stones;

//----------------------------------------
// SimpleStoneTraits
//
// This class stores some atrributes for SimpleStones.
// Only one instance is created for each type of SimpleStone.
//----------------------------------------
namespace
{
    class SimpleStoneTraits {
        string sound;           // collision sound
        bool   hollow;          // whether stone is hollow
        bool   glass;        // whether its a glass stone

        // static list of all allocated SimpleStoneTraits (never freed yet)
        static vector<SimpleStoneTraits*> simple_stone_traits;

        SimpleStoneTraits(const string& sound_, bool hollow_, bool glass_)
            : sound(sound_) , hollow(hollow_) , glass(glass_)
        {}

    public:
        SimpleStoneTraits() {}

//         static void clear() {
//             vector<SimpleStoneTraits*>::iterator i = simple_stone_traits.begin();
//             vector<SimpleStoneTraits*>::iterator e = simple_stone_traits.end();
//             for (; i != e; ++e)
//                 delete simple_stone_traits[i];
//             simple_stone_traits.clear();
//         }

        static const SimpleStoneTraits* Register(const string& sound_, bool hollow_, bool glass_) {
            simple_stone_traits.push_back(new SimpleStoneTraits(sound_, hollow_, glass_));
            return simple_stone_traits.back();
        }

        const string& get_sound() const { return sound; }
        bool is_hollow() const { return hollow; }
        bool is_glass() const { return glass; }
    };

    vector<SimpleStoneTraits*> SimpleStoneTraits::simple_stone_traits;
}

//----------------------------------------
// SimpleStone
//
// This kind of stone can be defined from Lua programs with
// `DefineSimpleStone'.
//----------------------------------------
namespace
{
    class SimpleStone : public Stone {
    public:
        SimpleStone(const string &knd, const string & snd, bool hollow, bool is_glass)
            : Stone(knd.c_str())
            , traits(SimpleStoneTraits::Register(snd, hollow, is_glass))
            , sunglas(false)
        {}
    private:
        SimpleStone(const SimpleStone& other)
            : Stone(other.get_kind())
            , traits(other.traits)
        {}

        Object *clone() { return new SimpleStone(*this); }
        void dispose() { delete this; }

        const char *collision_sound() {
            return traits->get_sound().c_str();
        }
        StoneResponse collision_response(const StoneContact &/*sc*/) {
            return traits->is_hollow() ? STONE_PASS : STONE_REBOUND;
        }

        bool is_floating() const {
            return traits->is_hollow();
        }
        bool on_laserhit(Direction) {
            return traits->is_hollow() || traits->is_glass();
        }

        void message(const string& msg, const Value &val) {
            if (traits->is_hollow() && msg == "glasses") {
                if (to_int(val)) {
                    if (!sunglas) {
                        sunglas = true;
                        set_model( "invisible");
                    }
                }
                else {
                    if (sunglas) {
                        sunglas = false;
                        set_model( this->get_kind());
                    }
                }
            }
        }

        const SimpleStoneTraits *traits; // owned by simple_stone_traits
        bool sunglas; // seen through glasses
    };
}

//----------------------------------------
// SimpleStoneMovable
//
// This kind of stone can be defined from Lua programs with
// `DefineSimpleStone'.
//----------------------------------------
namespace
{
    class SimpleStoneMovable : public MovableStone {
    public:
        SimpleStoneMovable(const string &knd, const string & snd, bool is_glass)
            : MovableStone(knd.c_str())
            , traits(SimpleStoneTraits::Register(snd, false, is_glass))
        {}

    private:
        SimpleStoneMovable(const SimpleStoneMovable& other)
            : MovableStone(other.get_kind())
            , traits(other.traits)
        {}

        Object *clone() { return new SimpleStoneMovable(*this); }
        void dispose() { delete this; }

        const char *collision_sound() {
            return traits->get_sound().c_str();
        }
        bool on_laserhit(Direction) {
            return traits->is_glass();
        }

        const SimpleStoneTraits *traits; // owned by simple_stone_traits
    };
}

//----------------------------------------
// DummyStone
//
// used for debugging
// prints its own oxyd code when hit
//----------------------------------------
namespace
{
    class DummyStone : public Stone {
        CLONEOBJ(DummyStone);
    public:
        DummyStone() : Stone("st-dummy") {}
    private:

        StoneResponse collision_response(const StoneContact &/*sc*/) {
            static int lastCode = -1;
            int        code     = int_attrib("code");
            if (code != lastCode) {
                fprintf(stderr, "Collision with stone 0x%02x\n", code);
                lastCode = code;
            }
            return STONE_REBOUND;
        }
    };
}


//----------------------------------------
//  EasyModeStone
//----------------------------------------

/** \page st-easymode Easy-Mode Stone

In easy game mode this stone converts the floor at its
position to fl-normal.
In normal game mode the stone kills any item at its position.
Then in both modes it kills itself.

E.g. it can be used to hide water-barriers or to insert helper
items that make the level easier in easy game mode.

\subsection easye Example
\verbatim
set_stone("st-easymode", 10,10)
\endverbatim

\ref it-easymode
*/

namespace
{
    class EasyModeStone : public Stone {
        SINGLETONOBJ(EasyModeStone);
    public:
        EasyModeStone() : Stone("st-easymode") {}

        void on_creation() {
            if (options::Difficulty == DIFFICULTY_EASY) {
                SetFloor (get_pos(), MakeFloor ("fl-normal"));
            } else {
                KillItem (get_pos());
            }
            KillStone (get_pos());
        }

        StoneResponse collision_response(const StoneContact &/*sc*/) {
            return STONE_PASS;
        }
    };
}


//----------------------------------------
// Grate1/Grate2
//
// Actor may pass if on floor
//----------------------------------------
namespace
{
    class GrateBase : public Stone {
    public:
        GrateBase(const char *kind) : Stone(kind) {}
    private:
        bool is_floating() const { return true; }

        virtual StoneResponse collision_response(const StoneContact &sc) {
            // tested with per.oxyd
            return sc.actor->is_on_floor() ? STONE_PASS : STONE_REBOUND;
        }
        bool on_laserhit(Direction) { return true; }
    };

    class Grate1 : public GrateBase {
        CLONEOBJ(Grate1);
    public:
        Grate1() : GrateBase("st-grate1") {}
    };

    class Grate2 : public GrateBase {
        CLONEOBJ(Grate2);
    public:
        Grate2() : GrateBase("st-grate2") {}
    };
}


//----------------------------------------
// Grate3
//
// Horses and small marbles can move through this stone, but normal
// marbles can't.
//----------------------------------------
namespace
{
    class Grate3 : public GrateBase {
        CLONEOBJ(Grate3);
    public:
        Grate3() : GrateBase("st-grate3") {}

        StoneResponse collision_response(const StoneContact &sc) {
            string actorkind = sc.actor->get_kind();
            if (actorkind == "ac-horse" || actorkind == "ac-whiteball-small")
                return STONE_PASS;
            else
                return STONE_REBOUND;
        }
    };
}


/** \page st-chameleon Chameleon Stone

This stone takes on the look of the floor beneath it.  Actors can
move through it, so these stones are perfect for hiding stuff under
them...

*/
namespace
{
    class ChameleonStone : public Stone {
        CLONEOBJ(ChameleonStone);
    public:
        ChameleonStone() : Stone("st-chameleon"){}
    private:
        void init_model() {
            string modelname = "fl-gray";
            if (Floor *fl = GetFloor(get_pos()))
                modelname = fl->get_kind();
            set_model(modelname);
        }
        StoneResponse collision_response(const StoneContact &/*sc*/) {
            return STONE_PASS;
        }
    };
}


//----------------------------------------
// SwapStone
//----------------------------------------
namespace
{
    class SwapStone : public Stone, public TimeHandler {
    public:
        SwapStone();
    private:
        // Object interface
        SwapStone *clone();
        void       dispose();

        // GridObject interface
        void init_model();
        void on_removal();

        // Stone interface
        void on_impulse (const Impulse &impulse);
        bool is_removable() { return state == IDLE; }
        void actor_hit (const StoneContact &sc);
        void actor_inside(Actor *a) { SendMessage(a, "shatter"); }

        // TimeHandler interface
        void alarm();

        // Variables :
        enum State { IDLE, COME, GO } state;
        YieldedGridStone *in_exchange_with;
        Direction         move_dir;
    };
}

SwapStone::SwapStone()
: Stone("st-swap"),
  state(IDLE),
  in_exchange_with(0),
  move_dir(NODIR)
{}

SwapStone *
SwapStone::clone() 
{
    SwapStone *other        = new SwapStone(*this);
    other->in_exchange_with = 0;
    return other;
}

void SwapStone::dispose() 
{
    if (state == COME && in_exchange_with != 0) {
        in_exchange_with->dispose();
        delete in_exchange_with;
    }
    delete this;
}

void SwapStone::on_removal() 
{
    if (state == COME) {
        g_timer.remove_alarm(this);
    }
}

/* Animation finished; put the "swapped" stone to its new position. */
void SwapStone::alarm() 
{
    GridPos oldPos = move(get_pos(), reverse(move_dir));

    // Set the swapped stone (this also kills the old (inactive) swap stone)
    in_exchange_with->set_stone(oldPos);
    delete in_exchange_with;
    in_exchange_with = 0;

    state = IDLE;
    init_model();
    play_sound("st-move");
}

void 
SwapStone::on_impulse(const Impulse& impulse) 
{
    if (state == IDLE) {
        GridPos oldp = get_pos();
        GridPos newp = move(oldp, impulse.dir);

        if (!IsLevelBorder(newp)) {
            Stone *other = GetStone(newp);
            if (other && other->is_removable()) {
                SwapStone *newStone = new SwapStone;
                newStone->state            = COME;
                newStone->in_exchange_with = new YieldedGridStone(other); // yields 'other'
                newStone->move_dir         = impulse.dir;

                g_timer.set_alarm(newStone, 0.1, false);

                SetStone(newp, newStone);

                state    = GO;
                move_dir = impulse.dir;
                init_model();

                play_sound ("st-move");
                player::IncMoveCounter(1);
            }
        }
    }
}

void 
SwapStone::actor_hit(const StoneContact &sc) 
{
    Direction dir = get_push_direction (sc);
    if (dir != NODIR) {
        send_impulse(get_pos(), dir);
    }
}

void SwapStone::init_model() 
{
    static char *models_come[] = { "st-swap-w", "st-swap-s", "st-swap-e", "st-swap-n" };
    static char *models_go[] =   { "st-swap-e", "st-swap-n", "st-swap-w", "st-swap-s" };

    const char *model = 0;
    switch (state) {
    case IDLE: model = "st-swap"; break;
    case COME: model = models_come[move_dir]; break;
    case GO:   model = models_go[move_dir]; break;
    }

    set_model(model);
}


//----------------------------------------
// BlockStone
//----------------------------------------

/** \page st-block Block Stone

This stone is movable.

\subsection block Example
\verbatim
set_stone("st-block", 10,10)
\endverbatim

\image html st-block.png
*/
namespace
{
    class BlockStone : public MovableStone {
        CLONEOBJ(BlockStone);
    public:
        BlockStone() : MovableStone("st-block") {}
    private:
        V2 get_center() const {
            return get_pos().center();
        }

        void floor_change() {
            if (Floor *fl=GetFloor(get_pos())) {
                const string &k = fl->get_kind();
                if (k=="fl-water" || k=="fl-abyss" || k == "fl-swamp") {
                    display::AddEffect (get_center(), "ring-anim");
                    KillStone(get_pos());
                }
            }
        }
    };

};

//----------------------------------------
// Window
//----------------------------------------

/** \page st-window Breakable Stone

Hit this window heavily with your marble to blast it into smithereens.

\image html st-window.png
*/

namespace
{
    class Window : public Stone {
        CLONEOBJ(Window);
        const char *collision_sound() {return "ballcollision";}
    public:
        Window() : Stone("st-window"), state(IDLE) {}
    private:
        bool on_laserhit(Direction /*dir*/) { return true; }
        enum State { IDLE, BREAK } state;
        void actor_hit(const StoneContact &sc)
        {
            Actor *a = sc.actor;
            if (state == IDLE)
            {
	    	double impulse = -(a->get_vel() * sc.normal) * a->get_mass();
        	if (impulse > 30) {
        	    SendMessage(a, "shatter");
        	}
                
		if (impulse > 25) {
                    play_sound("shatter");
                    state = BREAK;
                    set_anim("st-window-anim");
                }
            }
        }
        void animcb() {
            KillStone(get_pos());
        }
    };
}

// -----------------------
//      BreakableStone
// -----------------------
// base class for Stone_break, Break_Bolder, Break_acwhite and Break_acblack
//
// breakable stones can be destroyed using
// hammer, laser, dynamite, bombs or bombstones

namespace {
    class BreakableStone : public Stone {
    public:
        BreakableStone(const char *kind) : Stone(kind), state(IDLE) {}
    protected:
        void break_me() {
            if (state == IDLE) {
                state = BREAK;
                play_sound("explosion1");
                set_anim(get_break_anim());
            }
        }
    private:
        const char *collision_sound() { return "st-stone"; }

        virtual void actor_hit(const StoneContact &sc) {
            if (may_be_broken_by(sc.actor))
                break_me();
        }
        bool on_laserhit(Direction) {
            break_me();
            return false;
        }
        void animcb() {
            KillStone(get_pos());
        }
        virtual void message(const string &msg, const Value &) {
            if (msg =="ignite" || msg == "expl" || msg == "bombstone")
                break_me();
        }

        virtual string get_break_anim() const  {
            return string(get_kind())+"-anim";
        }
        virtual bool may_be_broken_by(Actor *a) const = 0;

        // variables:

        enum State { IDLE, BREAK };
        State state;
    };
}

//----------------------------------------
// Stone_break
//----------------------------------------

/** \page st-stone_break Breakable Stone

This stone can be destroyed by an actor having a
hammer and by laser, dynamite, bombs and bombstones.

\subsection stone_breake Example
\verbatim
set_stone("st-stone_break", 10,10)
\endverbatim

\image html st-stone_break.png
*/
namespace
{
    class Stone_break : public BreakableStone {
        CLONEOBJ(Stone_break);
    public:
        Stone_break(const char *kind) : BreakableStone(kind) { }
    private:
        bool may_be_broken_by(Actor *a) const {
            return player::wielded_item_is(a, "it-hammer");
        }
    };
}

//----------------------------------------
// Break_bolder
//----------------------------------------

/** \page st-break_bolder Breakable Stone

This stone can be destroyed by an actor having a
hammer and by laser, dynamite, bombs and bombstones, bolder

\subsection break_bolder Example
\verbatim
set_stone("st-break_bolder", 10,10)
\endverbatim

\image html st-break_bolder.png
*/
namespace
{
    class Break_bolder : public BreakableStone {
        CLONEOBJ(Break_bolder);
    public:
        Break_bolder() : BreakableStone("st-break_bolder") {}
    private:
        bool may_be_broken_by(Actor *a) const {
            return player::wielded_item_is(a, "it-hammer");
        }
        virtual void message(const string &msg, const Value &) {
            if (msg == "trigger")
                break_me();
        }
    };
}

//----------------------------------------
// Stone_movebreak
//----------------------------------------

/** \page st-rock3_movebreak Breakable Movable Stone

This stone can be destroyed by an actor having a
hammer and by laser, dynamite, bombs and bombstones.

\subsection stone_breake Example
\verbatim
set_stone("st-rock3_movebreak", 10,10)
\endverbatim

\image html st-rock3.png
*/
namespace
{
    class Stone_movebreak : public BreakableStone {
        CLONEOBJ(Stone_movebreak);
    public:
        Stone_movebreak() : BreakableStone("st-rock3_movebreak") {}
    private:

        string get_break_anim() const  {
            return "st-rock3_break-anim";
        }
        bool may_be_broken_by(Actor *a) const {
            return player::wielded_item_is(a, "it-hammer");
        }

        bool is_movable() { return true; }
        void actor_inside (Actor *a) { SendMessage(a, "shatter"); }

        void actor_hit(const StoneContact &sc) {
            if (may_be_broken_by(sc.actor))
                break_me();
            else
                maybe_push_stone (sc);
        }
        void on_impulse(const Impulse& impulse) {
            move_stone(impulse.dir);
        }

    };
}

//----------------------------------------
// Break_acwhite
//----------------------------------------

/** \page st-break_acwhite Breakable Stone

This stone can be destroyed by actor (whiteball) having a
hammer and by laser, dynamite, bombs and bombstones.

\subsection break_acwhite Example
\verbatim
set_stone("st-break_acwhite", 10,10)
\endverbatim

\image html st-break_acwhite.png
*/
namespace
{
    class Break_acwhite : public BreakableStone {
        CLONEOBJ(Break_acwhite);
    public:
        Break_acwhite() : BreakableStone("st-break_acwhite") {}
    private:
        bool may_be_broken_by(Actor *a) const {
            return a->get_attrib("whiteball") &&
                player::wielded_item_is(a, "it-hammer");
        }
    };
}

//----------------------------------------
// Break_acblack
//----------------------------------------

/** \page st-break_acblack Breakable Stone

This stone can be destroyed by actor (blackball) having a
hammer.

\subsection break_acblack Example
\verbatim
set_stone("st-break_acblack", 10,10)
\endverbatim

\image html st-break_acblack.png
*/
namespace
{
    class Break_acblack : public BreakableStone {
        CLONEOBJ(Break_acblack);
    public:
        Break_acblack() : BreakableStone("st-break_acblack") {}
    private:
        bool may_be_broken_by(Actor *a) const {
            return a->get_attrib("blackball") &&
                player::wielded_item_is(a, "it-hammer");
        }
    };
}


//----------------------------------------
// Brick Magic
//----------------------------------------

/** \page st-brick_magic Magic Brick Stone

This stone does initially look like a "st-brick". If touched by the
actor, having a magic wand, it turns into a "st-glass" stone and
allows lasers to go through it.

\subsection brick_magicke Example
\verbatim
set_stone("st-brick_magic", 10,10)
\endverbatim

\image html st-brick.png
*/
namespace
{
    class BrickMagic : public Stone {
        CLONEOBJ(BrickMagic);
        const char *collision_sound() {return "st-stone";}
    public:
        BrickMagic() : Stone("st-brick_magic"), state(BRICK) {}
    private:
        enum State { BRICK, GLASS } state;
        void actor_hit(const StoneContact &sc)
        {
            if( state == BRICK)
            {
                if (player::wielded_item_is(sc.actor, "it-magicwand")) {
                    play_sound("st-magic");
                    state = GLASS;
                    set_model("st-glass");
                    laser::MaybeRecalcLight(get_pos());
                }
            }
        }
        bool on_laserhit(Direction /*dir*/) {
            return state == GLASS;
        }
    };
}

//----------------------------------------
// Stonebrush
//----------------------------------------

/** \page st-stonebrush Brush Stone

This stone is initially invisible. If touched by an actor
having a brush it turns into a "st-rock4".

\subsection stonebrushe Example
\verbatim
set_stone("st-stonebrush", 10,10)
\endverbatim

\image html st-rock4.png
*/
namespace
{
    class Stonebrush : public Stone {
        CLONEOBJ(Stonebrush);
        const char *collision_sound() {return "st-stone";}
    public:
        Stonebrush() : Stone("st-stonebrush"), state(INVISIBLE) {}
    private:
        enum State { INVISIBLE, BRUSH } state;

        void actor_hit(const StoneContact &sc) {
            if( state == INVISIBLE) {
                if (player::wielded_item_is(sc.actor, "it-brush")) {
                    play_sound("st-magic");
                    state = BRUSH;
                    if (enigma::GameCompatibility == GAMET_PEROXYD) {
                        set_model("st-likeoxydc-open");
                    }
                    else {
                        set_model("st-rock4");
                    }
                }
            }
        }
    };
}

//----------------------------------------
// Break_invisible
//----------------------------------------

/** \page st-break_invisible Brush Stone

This stone is initially invisible. If touched by an actor having a
brush it turns into a "st_stone_break".  This stone can be destroyed
by an actor having a hammer.

\subsection break_invisible Example
\verbatim
set_stone("st-break_invisible", 10,10)
\endverbatim

\image html st-stone_break.png
*/
namespace
{
    class Break_invisible : public Stone {
        CLONEOBJ(Break_invisible);
        const char *collision_sound() {return "st-stone";}
    public:
        Break_invisible() : Stone("st-break_invisible"), state(INVISIBLE) {}
    private:
        enum State { INVISIBLE, BRUSH };
        State state;
        void actor_hit(const StoneContact &sc)
        {
            if (state == INVISIBLE)
            {
                if (player::wielded_item_is(sc.actor, "it-brush")) {
                    play_sound("st-magic");
                    state = BRUSH;
                    set_model("st-stone_break");
                }
            }
            else if (state == BRUSH)
            {
                if (player::wielded_item_is(sc.actor, "it-hammer")) {
                    play_sound("explosion1");
                    set_anim("st-stone_break-anim");
                }
            }
        }
        void animcb() {
            if (state == BRUSH) {
                KillStone(get_pos());
            }
        }
    };
}

//----------------------------------------
// Invisible Magic
//----------------------------------------

/** \page st-invisible_magic Magic Invisible Stone

This stone is initially invisible, and laserlight can pass through.
If touched by an actor having a magic wand, it will mutate into a
"st-greenbrown" and laserlight is blocked.

\subsection invisible_magice Example
\verbatim
set_stone("st-invisible_magic", 10,10)
\endverbatim

\image html st-greenbrown.png
*/
namespace
{
    class InvisibleMagic : public Stone {
        CLONEOBJ(InvisibleMagic);
        const char *collision_sound() {return "st-thud";}
    public:
        InvisibleMagic() : Stone("st-invisible_magic"), state(INVISIBLE) {}
    private:
        enum State { INVISIBLE, STONE } state;
        void actor_hit(const StoneContact &sc)
        {
            if( state == INVISIBLE)
            {
                if (player::wielded_item_is(sc.actor, "it-magicwand")) {
                    play_sound("st-magic");
                    state = STONE;
                    set_model("st-greenbrown");
                    laser::MaybeRecalcLight(get_pos());
                }
            }
        }
        bool on_laserhit(Direction /*dir*/) {return state==INVISIBLE;}
    };
}

//----------------------------------------
// WoodenStone
//----------------------------------------

/** \page st-wood Wooden Stone

This stone is movable.  If moved into abyss, water or swamp it builds
a wooden plank.

\subsection woode Example
\verbatim
set_stone("st-wood", 10,10)
\endverbatim

Note: There are two flavors of st-wood which may be specified
by using st-wood1 or st-wood2.

\image html st-wood.png
*/
namespace
{
    class WoodenStone : public MovableStone {
	CLONEOBJ(WoodenStone);
    public:
        WoodenStone(const char *kind) : MovableStone(kind) {}

    private:
        void fall() {
            GridPos p = get_pos();
            if (!world::IsLevelBorder(p)) {
                if (Floor *fl=GetFloor(p)) {
                    const string &k = fl->get_kind();
                    if (k == "fl-abyss" || k=="fl-water" || k=="fl-swamp") {
                        SetFloor(p, MakeFloor(is_kind("st-wood1") ? "fl-stwood1" : "fl-stwood2"));
                        KillStone(p);
                    }
                }
            }
        }

        // in oxyd1 only fall when moving
        void on_move() {
            if (enigma::GameCompatibility == GAMET_OXYD1)
                fall();
        }
        // other oxyds versions: fall everytime the floor changes
        void floor_change() {
            if (enigma::GameCompatibility != GAMET_OXYD1)
                fall();
        }
    };

    class RandomWoodenStone : public Stone {
    public:
	RandomWoodenStone() : Stone("st-wood") {}
    private:
        // When st-wood is created it randomly becomes st-wood1 or st-wood2.
	Stone *clone() {
	    return new WoodenStone (IntegerRand(0, 1) ? "st-wood1" :  "st-wood2");
	}
	void dispose() {delete this;}
    };
}

//----------------------------------------
// Growing stones used by it-seed
//----------------------------------------
namespace
{
    class WoodenStone_Growing : public Stone {
        CLONEOBJ(WoodenStone_Growing);
    public:
        WoodenStone_Growing() : Stone("st-wood-growing") {}
    private:
        void init_model() { set_anim("st-wood-growing"); }
        void animcb() {
            Stone *st;
            if( DoubleRand(0, 1) > 0.0)
                st = world::MakeStone("st-wood1");
            else
                st = world::MakeStone("st-wood2");
            world::ReplaceStone(get_pos(), st);
            st->floor_change(); // instantly builds a bridge on fl-swamp etc
        }
        void actor_contact(Actor *a) {SendMessage(a, "shatter");}
        void actor_inside(Actor *a) {SendMessage(a, "shatter");}
        void actor_hit(const StoneContact &sc) {SendMessage(sc.actor, "shatter");}
    };

    class GreenbrownStone_Growing : public Stone {
        CLONEOBJ(GreenbrownStone_Growing);
    public:
        GreenbrownStone_Growing() : Stone("st-greenbrown-growing") {}
    private:
        void init_model() { set_anim("st-greenbrown-growing"); }
        void animcb() {
            Stone *st = world::MakeStone("st-greenbrown");
            world::ReplaceStone(get_pos(), st);
            st->floor_change(); // instantly builds a bridge on fl-swamp etc
        }
        void actor_contact(Actor *a) {SendMessage(a, "shatter");}
        void actor_inside(Actor *a) {SendMessage(a, "shatter");}
        void actor_hit(const StoneContact &sc) {SendMessage(sc.actor, "shatter");}
    };

    class VolcanoStone_Growing : public Stone {
        CLONEOBJ(VolcanoStone_Growing);
    public:
        VolcanoStone_Growing() : Stone("st-volcano-growing") {}
    private:
        void init_model() { set_anim("st-volcano-growing"); }
        void animcb() {
            Stone *st = world::MakeStone("st-volcano_active");
            world::ReplaceStone(get_pos(), st);
            st->floor_change(); // instantly builds a bridge on fl-swamp etc
        }
        void actor_contact(Actor *a) {SendMessage(a, "shatter");}
        void actor_inside(Actor *a) {SendMessage(a, "shatter");}
        void actor_hit(const StoneContact &sc) {SendMessage(sc.actor, "shatter");}
    };
}

//----------------------------------------
// ScissorsStone
//----------------------------------------

/** \page st-scissors Scissors stone

This stone cuts \c all rubber bands attached to an actor that touches
it.

\image html st-scissors
*/
namespace
{
    class ScissorsStone : public Stone {
        CLONEOBJ(ScissorsStone);
    public:
        ScissorsStone() : Stone("st-scissors") {}
    private:
        void actor_hit(const StoneContact &sc) {
            world::KillRubberBand (sc.actor, (Stone*)0);
            world::KillRubberBand (sc.actor, (Actor*)0);
            play_sound("snip");
            set_anim("st-scissors-snip");
        }
        void animcb() {
            set_model("st-scissors");
        }
    };
}

//----------------------------------------
// RubberBand stone
//----------------------------------------

/** \page st-rubberband Rubberband stone

If hit by a marble, this stone first removes existing connections with
other rubberband stones and then attaches a new elastic between the
marble and itself.  Nothing happens if the marble was already attached
to this particular stone.

This stone can be moved if hit with a magic wand.

\subsection rubberbanda Attributes

- \c length  The natural length of the rubberband (default: 1)
- \c strength The strength of the rubberband (default: 10)

\image html st-rubberband.png
*/
namespace
{
    class RubberBandStone : public MovableStone {
        CLONEOBJ(RubberBandStone);
    public:
        RubberBandStone () : MovableStone ("st-rubberband") {
            set_attrib("length", 1.0);
            set_attrib("strength", 10.0);
        }
    private:


        void actor_hit(const StoneContact &sc) {
            double strength = 10.0;
            double length = 1.0;
            double_attrib ("length", &length);
            double_attrib ("strength", &strength);

            if (!world::HasRubberBand (sc.actor, this)) {
                play_sound ("boing");
                world::KillRubberBand (sc.actor, (Stone*)0);
                world::AddRubberBand (sc.actor, this, strength, length);
            }
            if (player::wielded_item_is (sc.actor, "it-magicwand"))
                MovableStone::actor_hit(sc);
        }
    };
}

//----------------------------------------
// TimerStone
//
// Attributes:
//
// :interval        seconds between two "ticks"
// :loop
// :action,target   as usual
// :invisible
//----------------------------------------

/** \page st-timer Timer Stone

This stone can be used to trigger periodic events or to trigger one
single event after a certain amount of time.

\subsection timera Attributes

- \b on: 1 if the timer is running
- \b interval:  number of seconds before \b action is performed
- \b loop:      if 1, restart the timer after performing \b action
- \b action, \b target: as usual
- \b invisible : if 1, stone is invisible

\subsection timerm Messages

- \b on, \b off, \b onoff: as usual

\subsection timere Example

\verbatim
-- activate a laser after 5 seconds
set_stone("st-laser", 10,11, {name="laser"})
set_stone("st-timer", 10,10,
          {loop=0, action="onoff", target="laser", interval=5})
\endverbatim
*/
namespace
{
    class TimerStone : public OnOffStone, public TimeHandler
    {
        CLONEOBJ(TimerStone);
    public:
        TimerStone() : OnOffStone("st-timer") {
            set_attrib("interval", 1.0);
            set_attrib("loop", 1.0);
            set_attrib("on", 1.0);
            set_attrib("invisible", 0.0);

            // set_on(true);   DOESN'T WORK! calls init_model()
        }
    private:
        double get_interval() const {
            double interval = 100;
            double_attrib("interval", &interval);
            return interval;
        }
        void init_model() {
            if (int_attrib("invisible")) {
                set_model("invisible");
            }
            else {
                set_model(is_on() ? "st-timer" : "st-timeroff");
            }
        }

        void on_creation() {
            set_alarm();
            Stone::on_creation();
        }

        void set_alarm() {
            if (is_on())
                g_timer.set_alarm(this, get_interval(), true);
        }

        void alarm() {
            if (is_on())
            {
//                 sound::PlaySound("st-timer");
                PerformAction(this, is_on());
            }
        }
    };
}

//----------------------------------------
// Switch
//
// Attributes:
//
// :on              1 or 0
// :target,action   as usual
//----------------------------------------

/** \page st-switch Switch Stone

On actor hit this stone can trigger actions

\image html st-switch1.png
*/
namespace
{
    class SwitchStone : public OnOffStone {
        CLONEOBJ(SwitchStone);
    public:
        SwitchStone() : OnOffStone("st-switch") {}
    private:
        void init_model() {
            set_model(is_on() ? "st-switch1" : "st-switch2");
        }

        void actor_hit(const StoneContact &/*sc*/) {
            set_on(!is_on());
            PerformAction(this, is_on());
            play_sound("st-switch");
        }
        const char *collision_sound() { return "st-metal"; }
    };
}

//----------------------------------------
// Switch_black
//
// Attributes:
//
// :on              1 or 0
// :target,action   as usual
//----------------------------------------

//On actor (black only) hit this stone can trigger actions

namespace
{
    class Switch_black : public OnOffStone {
        CLONEOBJ(Switch_black);
    public:
        Switch_black() : OnOffStone("st-switch_black") {}
    private:
        void init_model() {
            set_model(is_on() ? "st-switch_black1" : "st-switch_black2");
        }

        void actor_hit(const StoneContact &sc)
        {
            if (sc.actor->get_attrib("blackball"))
                set_on(!is_on());
            PerformAction(this, is_on());
            play_sound("st-switch");
        }

        const char *collision_sound() { return "st-metal"; }
    };
}

//----------------------------------------
// Switch_white
//
// Attributes:
//
// :on              1 or 0
// :target,action   as usual
//----------------------------------------

//On actor (white only) hit this stone can trigger actions

namespace
{
    class Switch_white : public OnOffStone {
        CLONEOBJ(Switch_white);
    public:
        Switch_white() : OnOffStone("st-switch_white") {}
    private:
        void init_model() {
            set_model(is_on() ? "st-switch_white1" : "st-switch_white2");
        }

        void actor_hit(const StoneContact &sc)
        {
            if (sc.actor->get_attrib("whiteball"))
                set_on(!is_on());
            PerformAction(this, is_on());
            play_sound("st-switch");
        }

        const char *collision_sound() { return "st-metal"; }
    };
}

//----------------------------------------
// Timeswitch
//
// Attributes:
//
// :on              1 or 0
// :target,action   as usual
//----------------------------------------

/** \page st-switch Switch Stone

On actor hit this stone can trigger actions

\image html st-timeswitch.png
*/
namespace
{
    class TimeSwitch : public OnOffStone {
        CLONEOBJ(TimeSwitch);
    public:
        TimeSwitch() : OnOffStone("st-timeswitch"), state(IDLE) {}
    private:
        enum State {IDLE, ON, OFF };
        State state;
        void change_state (State newstate)
       {
            if (newstate == IDLE) {
                state = IDLE;
            }
        }

        void actor_hit(const StoneContact &sc)
        {

        if ( state == IDLE) {
            if (sc.actor)
                state = ON;
                set_on(!is_on());
                PerformAction(this, is_on());
                play_sound("st-switch");
                set_anim("st-time1switch");
            }
        }

        void animcb() {
            if (state == ON) {
            set_on(!is_on());
            PerformAction(this, is_on());
            play_sound("st-switch");
            change_state (IDLE);
        }
     }
        const char *collision_sound() { return "st-metal"; }
   };
}
//----------------------------------------
// FourSwitch
//
// Attributes:
//
// :on              1 or 0
// :target,action   as usual
//----------------------------------------

/** \page st-fourswitch Switch Stone

On actor hit this stone can trigger actions

\image html st-fourswitch.png
*/

namespace
{
    class FourSwitch : public OnOffStone {
        CLONEOBJ(FourSwitch);
    public:
        FourSwitch() : OnOffStone("st-fourswitch"), m_direction(NORTH) {}
    private:
        Direction m_direction;

        void init_model() {
            switch (m_direction) {
            case NORTH: set_model("st-fourswitch");   break;
            case EAST:  set_model("st-fourswitch_e"); break;
            case SOUTH: set_model("st-fourswitch_s"); break;
            case WEST:  set_model("st-fourswitch_w"); break;
            case NODIR: assert(0);
            }
        }

        void actor_hit(const StoneContact &/*sc*/)
        {
            set_on(!is_on());
            PerformAction(this, is_on());
            play_sound("st-switch");

            m_direction = rotate(m_direction, true);
            init_model();
        }
        const char *collision_sound() { return "st-metal"; }
    };
}

//----------------------------------------
// Laser Switch
//----------------------------------------

/** \page st-laserswitch Laserswitch Stone

Activated by laser light this switch can trigger actions

*/
namespace
{
    class LaserSwitch : public laser::PhotoStone {
        CLONEOBJ(LaserSwitch);
    public:
        LaserSwitch();

    private:
        // Stone interface
        void on_creation();
        void on_removal();
        const char *collision_sound() { return "st-metal"; }

        // PhotoStone interface.
        void notify_laseron() { set_model("st-laserswitch1"); PerformAction(this, true);}
        void notify_laseroff() { set_model("st-laserswitch0"); PerformAction(this, false);}
    };
}

LaserSwitch::LaserSwitch()
: PhotoStone("st-laserswitch")
{
}

void
LaserSwitch::on_creation()
{
    set_model("st-laserswitch0");
    photo_activate();
}

void
LaserSwitch::on_removal()
{
    photo_deactivate();
}

//----------------------------------------
// Floppy Switch
//
// Attributes:
//
// :on              1 or 0
// :target,action   as usual
//----------------------------------------
namespace
{
    class FloppyStone : public OnOffStone
    {
        CLONEOBJ(FloppyStone);
    public:
        FloppyStone() : OnOffStone("st-floppy") {}
    private:
        // Stone interface
        void init_model();
        void actor_hit(const StoneContact &sc);
        const char *collision_sound() { return "st-metal"; }
    };
}

void FloppyStone::init_model() 
{
    set_model(is_on() ? "st-floppy1" : "st-floppy0");
}


void FloppyStone::actor_hit (const StoneContact &sc)
{
    if (player::Inventory *inv = player::GetInventory(sc.actor))
    {
        if (is_on())
        {
            if (!inv->is_full()) {
                inv->add_item(MakeItem("it-floppy"));
                set_on(false);
                PerformAction(this, is_on());
            }
        }
        else if (player::wielded_item_is(sc.actor, "it-floppy"))
        {
            DisposeObject(inv->yield_first());
            set_on(true);
            PerformAction(this, is_on());
        }
    }
}


//----------------------------------------
// FartStone
//----------------------------------------

/** \page st-fart Fart Stone

The fart stone has the unpleasant habit of "blowing off" when
triggered (by actor contact or signal) and will close all oxyd stones.

\subsection fartm Messages

- \b trigger: as usual

*/

namespace
{
    class FartStone : public Stone {
        CLONEOBJ(FartStone);
    public:
        FartStone() : Stone("st-fart"), state(IDLE) {}
    private:
        enum State { IDLE, FARTING, BREAKING };
        State state;

        void change_state(State newstate) {
            if (state == newstate)
                return;

            switch (newstate) {
            case IDLE:
                state = IDLE;
                init_model();
                break;
            case FARTING:
            case BREAKING:
                if (state == IDLE) {
                    Object *ox = world::GetObjectTemplate("st-oxyd");
                    SendMessage(ox, "closeall");
                    play_sound("fart");
                    set_anim(newstate == FARTING ? "st-farting" : "st-fartbreak-anim");
                    state = newstate;
                }
                break;
            }
        }

        void animcb() {
            if (state == FARTING)
                change_state(IDLE);
            else if (state == BREAKING)
                KillStone(get_pos());
        }

        void actor_hit(const StoneContact &sc) {
            if (player::wielded_item_is(sc.actor, "it-hammer"))
                change_state(BREAKING);
            else
                change_state(FARTING);
        }
        void message(const string &m, const Value &) {
            if (m=="trigger")
                change_state(FARTING);
            else if (m == "ignite" || m == "expl")
                change_state(BREAKING);
        }

        bool on_laserhit(Direction) {
            change_state(BREAKING);
            return false;
        }
    };
}

//----------------------------------------
// Thief Stone
//
// Steals one item from the player's inventory when hit.
//----------------------------------------
namespace
{
    class ThiefStone : public Stone {
	CLONEOBJ(ThiefStone);

        enum State { IDLE, EMERGING, RETREATING } state;
        Actor *m_affected_actor;
    public:
	ThiefStone() : Stone("st-thief"), state(IDLE) { 
	    m_affected_actor = 0;
	}
    private:

        void actor_hit(const StoneContact &sc) {
            if (state==IDLE) {
                set_anim("st-thief-emerge");
		state = EMERGING;
		m_affected_actor = sc.actor;
            }
	}

        void animcb() {
	    switch (state) {
	    case EMERGING:
		steal_from_player();
		state = RETREATING;
		set_anim("st-thief-retreat");
		break;
	    case RETREATING:
		state = IDLE;
		init_model();
		break;
	    default:
		assert(0);
	    }
        }

	void steal_from_player() {
	    if (m_affected_actor && !m_affected_actor->has_shield()) {
                int iplayer;
		m_affected_actor->int_attrib("player", &iplayer);
		player::Inventory *inv = player::GetInventory(iplayer);
		if (inv && inv->size() > 0)
		{
		    int i = IntegerRand (0, inv->size()-1);
		    delete inv->yield_item(i);
			
		    play_sound("thief");
		}
	    }
	}

        const char *collision_sound() {
            return "st-thud";
        }
    };
}

// -------------------------
//      ActorImpulseBase
// -------------------------

namespace
{
    class ActorImpulseBase : public Stone {
    public:
        ActorImpulseBase(const char *kind) : Stone(kind), state(IDLE) {
//             set_attrib("force", Value());
        }

    protected:
        virtual void actor_hit (const StoneContact &sc) {
            if (state == IDLE) {
                // actor_hit is called before reflect, but the force added below
                // is applied to actor after the reflection.

                double forcefac = enigma::BumperForce;
                double_attrib("force", &forcefac);

                V2 vec = normalize(sc.actor->get_pos() - get_pos().center());
                sc.actor->add_force (vec * forcefac);

                play_sound("impulse");
                set_anim("st-actorimpulse-anim");
                state = PULSING;
            }
        }

    private:
        void animcb() {
            if (state == PULSING) {
                state = IDLE;
                init_model();
            }
        }

        // Variables
        enum State { IDLE, PULSING, BROKEN };
        State state;
    };


    class ActorImpulseStone : public ActorImpulseBase {
        CLONEOBJ(ActorImpulseStone);
    public:
        ActorImpulseStone() : ActorImpulseBase("st-actorimpulse") {}
    };


    class ActorImpulseStoneInvisible : public ActorImpulseBase {
        CLONEOBJ(ActorImpulseStoneInvisible);
    public:
        ActorImpulseStoneInvisible() : ActorImpulseBase("st-actorimpulse_invisible") {}

        void actor_hit(const StoneContact& sc) {
            if (player::wielded_item_is(sc.actor, "it-brush")) {
                Stone *st = MakeStone("st-actorimpulse");
                SetStone(get_pos(), st);
                st->actor_hit(sc);
            }
            else
                ActorImpulseBase::actor_hit(sc);
        }
    };
}

//----------------------------------------
// FakeOxydStone
//----------------------------------------

/** \page st-fakeoxyd Fake Oxyd Stone

These stones look like real Oxyd stones, but they only blink a little
when touched and do not open or have other special abilities.

\image html st-fakeoxyd-blink_0001.png
*/
namespace
{
    class FakeOxydStone : public Stone {
        CLONEOBJ(FakeOxydStone);
    public:
        FakeOxydStone() : Stone("st-fakeoxyd"), state(IDLE) {}
    private:
        enum State { IDLE, BLINKING } state;
        void actor_hit(const StoneContact &/*sc*/) {
            if (state == IDLE) {
                set_anim("st-fakeoxyd-blink");
                state = BLINKING;
            }
        }
        const char *collision_sound() {
            return "st-fakeoxyd";
        }
        void animcb() {
            set_model("st-fakeoxyd");
            state = IDLE;
        }
    };
}


//----------------------------------------
// Black stones
//----------------------------------------
namespace
{
    class BlackStone : public Stone {
        CLONEOBJ(BlackStone);
    public:
        BlackStone() : Stone("st-black1") {}
        BlackStone(const char *name) : Stone( name) {}

        StoneResponse collision_response(const StoneContact &sc) {
            if (sc.actor->get_attrib("blackball"))
                return STONE_PASS;
            else
                return STONE_REBOUND;
        }
    };

    class BlackStone2 : public BlackStone {
        CLONEOBJ(BlackStone2);
    public:
        BlackStone2() : BlackStone("st-black2") {}
    };

    class BlackStone3 : public BlackStone {
        CLONEOBJ(BlackStone3);
    public:
        BlackStone3() : BlackStone("st-black3") {}
    };

    class BlackStone4 : public BlackStone {
        CLONEOBJ(BlackStone4);
    public:
        BlackStone4() : BlackStone("st-black4") {}
    };

}

//----------------------------------------
// White stones
//----------------------------------------
namespace
{
    class WhiteStone : public Stone {
        CLONEOBJ(WhiteStone);
    public:
        WhiteStone() : Stone("st-white1") {}
        WhiteStone(const char *name) : Stone( name) {}

        StoneResponse collision_response(const StoneContact &sc) {
	    if (sc.actor->get_attrib("whiteball"))
                return STONE_PASS;
	    else
                return STONE_REBOUND;
        }
    };

    class WhiteStone2 : public WhiteStone {
        CLONEOBJ(WhiteStone2);
    public:
        WhiteStone2() : WhiteStone("st-white2") {}
    };

    class WhiteStone3 : public WhiteStone {
        CLONEOBJ(WhiteStone3);
    public:
        WhiteStone3() : WhiteStone("st-white3") {}
    };

    class WhiteStone4 : public WhiteStone {
        CLONEOBJ(WhiteStone4);
    public:
        WhiteStone4() : WhiteStone("st-white4") {}
    };

}

// ---------------------------------
//      YinYangStone (baseclass)
// ---------------------------------
namespace
{
    class YinYangStone : public Stone {
    public:
        YinYangStone(const char *kind) : Stone(kind), state(NOCOLOR) {}

    protected:

        void turn_white() {
            assert(state == NOCOLOR);
            state = WHITE;
            init_model();
            play_sound("st-magic");
        }
        void turn_black() {
            assert(state == NOCOLOR);
            state = BLACK;
            init_model();
            play_sound("st-magic");
        }

        bool is_uncolored() const { return state == NOCOLOR; }

    private:
        enum State { NOCOLOR, WHITE, BLACK } state;

        void init_model() {
            switch (state) {
                case NOCOLOR: set_model(get_kind()); break;
                case WHITE: set_model("st-white1"); break;
                case BLACK: set_model("st-black1"); break;
            }
        }

        StoneResponse collision_response(const StoneContact &sc)
        {
            if ((state==BLACK && sc.actor->get_attrib("blackball")) ||
                (state==WHITE && sc.actor->get_attrib("whiteball")))
                return STONE_PASS;
            return STONE_REBOUND;
        }
    };
}

//----------------------------------------
// YinYang stone 1
//----------------------------------------
namespace
{
    class YinYangStone1 : public YinYangStone {
        CLONEOBJ(YinYangStone1);
    public:
        YinYangStone1() : YinYangStone("st-yinyang1") {}

    private:
        void actor_hit(const StoneContact &sc) {
            if (is_uncolored()) {
                if      (sc.actor->get_attrib("blackball")) turn_white();
                else if (sc.actor->get_attrib("whiteball")) turn_black();
            }
        }
    };
}

//----------------------------------------
// YinYang stone 2
//----------------------------------------
namespace
{
    class YinYangStone2 : public YinYangStone {
        CLONEOBJ(YinYangStone2);
    public:
        YinYangStone2() : YinYangStone("st-yinyang2") {}
    private:
        void actor_hit(const StoneContact &sc) {
            if (is_uncolored()) {
                if      (sc.actor->get_attrib("blackball")) turn_black();
                else if (sc.actor->get_attrib("whiteball")) turn_white();
            }
        }
    };
}


//----------------------------------------
// BombStone
//
// These stones add a bomb to the player's inventory when touched.
//----------------------------------------
namespace
{
    class BombStone : public Stone {
        CLONEOBJ(BombStone);
        const char *collision_sound() {return "st-stone";}
    public:
        BombStone() : Stone("st-bombs"),state(IDLE) {}
    private:
        enum State { IDLE, BREAK };
        State state;
        void actor_hit(const StoneContact &sc);
        void change_state (State newstate) {
            if (state == IDLE && newstate==BREAK) {
                state = newstate;
                play_sound("explosion1");
                set_anim("st-bombs-anim");
            }
        }
        void animcb() {
            assert(state == BREAK);
            GridPos p = get_pos();
            SendExplosionEffect(p, BOMBSTONE);
            KillStone(p);
            SetItem(p, MakeItem("it-explosion1"));
        }
        void BombStone::message(const string &msg, const Value &) {
            if (msg =="expl" || msg =="bombstone")
                change_state(BREAK);
        }
    };
}
void
BombStone::actor_hit(const StoneContact &sc)
{
    if (player::Inventory *inv = player::GetInventory(sc.actor))
    {
        if (!inv->is_full())
        {
            Item *it = MakeItem("it-blackbomb");
            inv->add_item(it);
        }
    }
}

//----------------------------------------
// MagicStone
//----------------------------------------
namespace
{
    class MagicStone : public Stone {
        CLONEOBJ(MagicStone);
    public:
        MagicStone() : Stone("st-magic") {}
    private:
        void actor_hit(const StoneContact &/*sc*/) {
            KillStone(get_pos());
            display::GetStatusBar()->show_text("We don't sell books..", display::TEXT_2SECONDS);
        }
    };
}


//----------------------------------------
// DeathStone
//----------------------------------------

/** \page st-death Death's Head Stone

Simply kills all actors that touch it (except for actors that are
immune to these stones).

\image html st-death.png
*/
namespace
{
    class DeathStone : public Stone {
        CLONEOBJ(DeathStone);
    public:
        DeathStone() : Stone("st-death"), active(false) {}
    private:
        bool active;
        void actor_hit(const StoneContact &sc)
        {
            SendMessage(sc.actor, "shatter");
            if (!active) {
		active=true;
		set_anim("st-death-anim");
	    }
        }
	void animcb() { set_model("st-death"); active=false; }
    };
}


//----------------------------------------
// DeathStone Invisible
//----------------------------------------

/** \page st-death_invisible Death's Head Stone invivible

Simply kills all actors that touch it (except for actors that are
immune to these stones). This variant is invisible.

\image html st-death.png
*/
namespace
{
    class DeathStoneInvisible : public Stone {
        CLONEOBJ(DeathStoneInvisible);
    public:
        DeathStoneInvisible() : Stone("st-death_invisible"), active(false), visible(false) {}
    private:
        bool active;
        bool visible; // seen through glasses

        void actor_hit(const StoneContact &sc)
        {
            SendMessage(sc.actor, "shatter");
            if (!active) {
                active=true;
                set_anim("st-death-anim");
            }
        }

        void set_visible_model() {
            set_model(visible ? "st-death" : "st-death_invisible");
        }

        void animcb() {
            set_visible_model();
            active=false;
        }

        void message(const string& msg, const Value &val) {
            if (msg == "glasses") {
                if (to_int(val)) {
                    if (!visible) {
                        visible = true;
                        set_visible_model();
                    }
                }
                else {
                    if (visible) {
                        visible = false;
                        set_visible_model();
                    }
                }
            }
        }
    };
}

// -------------------
//      BrakeStone
// -------------------

/** \page st-brake Brake

Blocks bolder stones and other movable stones.
May be carried.

\image html st-brake.png
*/

namespace
{
    class BrakeStone : public Stone {
        CLONEOBJ(BrakeStone);
    public:
        BrakeStone() : Stone("st-brake") {}

        void on_creation() {
            Stone::on_creation();

            GridPos  p  = get_pos();
            Item    *it = GetItem(p);
            if (it && it->is_kind("it-blocker")) {
                KillItem(p);
                play_sound("explosion1");
            }
        }

        StoneResponse collision_response(const StoneContact &/*sc*/) {
            return STONE_PASS;
        }

        void actor_inside(Actor *a) {
            const double BRAKE_RADIUS = 0.3;
            GridPos      p            = get_pos();
            double       dist         = length(a->get_pos() - p.center());

            if (dist < BRAKE_RADIUS) {
                player::PickupStoneAsItem(a, p);
            }
        }

        void explode() {
            GridPos p = get_pos();
            KillStone(p);
            SetItem(p, MakeItem("it-explosion1"));
        }

        bool on_laserhit(Direction) {
            explode();
            return false;       // block laser
        }

        void message(const string &msg, const Value &) {
            if (msg == "expl") {
                explode();
            }
        }
    };
}

//----------------------------------------
// Disco stones
//----------------------------------------
namespace
{
    class DiscoStone : public Stone {
        CLONEOBJ(DiscoStone);
    public:
        DiscoStone(const char *kind) : Stone(kind) {}

        StoneResponse collision_response(const StoneContact &/*sc*/) {
            return STONE_PASS;
        }

        virtual void lighten() {}
        virtual void darken() {}
    protected:
        static void visit_lighten(GridPos p) {
            if (DiscoStone *st = dynamic_cast<DiscoStone*>(GetStone(p)))
                st->lighten();
        }
        static void visit_darken(GridPos p) {
            if (DiscoStone *st = dynamic_cast<DiscoStone*>(GetStone(p)))
                st->darken();
        }

    private:
        bool is_floating() const { return true; }

        void message(const string &msg, const Value &val) {
            if (msg == "signal") {
                int ival = to_int (val);
                if (ival > 0)
                    lighten();
                else
                    darken();
            } 
            else if (msg == "lighten") 
                lighten();
            else if (msg == "darken")
                darken();
        }
    };
    class DiscoLight : public DiscoStone {
        CLONEOBJ(DiscoLight);
        virtual void darken() {
            GridPos p = get_pos();
            SetStone (p, MakeStone("st-disco-dark"));
            visit_darken (move(p, NORTH));
            visit_darken (move(p, EAST));
            visit_darken (move(p, SOUTH));
            visit_darken (move(p, WEST));
        }
    public:
        DiscoLight() : DiscoStone("st-disco-light") {}
    };
    class DiscoMedium : public DiscoStone {
        CLONEOBJ(DiscoMedium);
    public:
        DiscoMedium() : DiscoStone("st-disco-medium") {}
    };

    class DiscoDark : public DiscoStone {
        CLONEOBJ(DiscoDark);
        virtual void lighten() {
            GridPos p = get_pos();
            SetStone (p, MakeStone("st-disco-light"));
            visit_lighten (move(p, NORTH));
            visit_lighten (move(p, EAST));
            visit_lighten (move(p, SOUTH));
            visit_lighten (move(p, WEST));
        }
        virtual void darken() {}
    public:
        DiscoDark() : DiscoStone("st-disco-dark") {}
    };
}

//----------------------------------------
// Knight Stone
//----------------------------------------
namespace
{
    class Knight : public Stone {
        CLONEOBJ(Knight);
        int subtype;
        enum {MIN_SUBTYPE=1, MAX_SUBTYPE=5};
    public:
        Knight() : Stone("st-knight"), subtype (MIN_SUBTYPE) {}
    private:
        StoneResponse collision_response(const StoneContact &/*sc*/) {
            return (subtype == MAX_SUBTYPE) ? STONE_PASS : STONE_REBOUND;
        }
        void actor_hit(const StoneContact &sc)
        {
            if (subtype != MAX_SUBTYPE) {
                if (player::wielded_item_is (sc.actor, "it-sword")) {
                    subtype += 1;
                    if (subtype == MAX_SUBTYPE) {
                        display::StatusBar *sb = display::GetStatusBar();
                        sb->show_text ("All right, we'll call it a draw",
                                       display::TEXT_5SECONDS,
                                       true);
                            }
                    init_model();
                } else {
                    SendMessage(sc.actor, "shatter");
                }
            }
        }
        void init_model() {
            char mname[20];
            sprintf(mname, "st-knight%d", subtype);
            set_model(mname);
        }
    };
}

// --------------------------------------------------------------------------------


void
world::DefineSimpleStone(const std::string &kind, const std::string &sound,
                         int hollow, int glass)
{
    Register(new SimpleStone(kind, sound, hollow, glass));
}

void
world::DefineSimpleStoneMovable(const std::string &kind, const std::string &sound, int glass)
{
    Register(new SimpleStoneMovable(kind, sound, glass));
}

// --------------------------------------------------------------------------------

void stones::Init_simple()
{
    Register(new ActorImpulseStone);
    Register(new ActorImpulseStoneInvisible);

    Register(new BlackStone);
    Register(new BlackStone2);
    Register(new BlackStone3);
    Register(new BlackStone4);

    Register(new BlockStone);
    Register(new BombStone);
    Register(new BrakeStone);

    Register(new Break_acblack);
    Register(new Break_acwhite);
    Register(new Break_bolder);
    Register(new Break_invisible);

    Register(new BrickMagic);

    Register(new ChameleonStone);

    Register(new DeathStone);
    Register(new DeathStoneInvisible);
    Register(new DiscoLight);
    Register(new DiscoMedium);
    Register(new DiscoDark);
    Register(new DummyStone);
    Register(new EasyModeStone);
    Register(new FakeOxydStone);
    Register(new FartStone);
    Register(new FloppyStone);
    Register(new FourSwitch);
    Register(new Grate1);
    Register(new Grate2);
    Register(new Grate3);
    Register(new InvisibleMagic);
    Register(new Knight);
    Register(new LaserSwitch);
    Register(new MagicStone);
    Register(new RubberBandStone);
    Register(new ScissorsStone);
    Register(new Stone_break("st-stone_break"));
    Register(new Stone_break("st-rock3_break"));
    Register(new Stone_movebreak);
    Register(new Stonebrush);
    Register(new SwapStone);

    Register(new SwitchStone);
    Register(new Switch_black);
    Register(new Switch_white);

    Register(new ThiefStone);
    Register(new TimeSwitch);
    Register(new TimerStone);

    Register(new WhiteStone);
    Register(new WhiteStone2);
    Register(new WhiteStone3);
    Register(new WhiteStone4);

    Register(new Window);

    Register(new RandomWoodenStone); // random flavor
    Register(new WoodenStone("st-wood1")); // horizontal planks
    Register(new WoodenStone("st-wood2")); // vertical planks
    Register(new WoodenStone_Growing);
    Register(new GreenbrownStone_Growing);
    Register(new VolcanoStone_Growing);

    Register(new YinYangStone1);
    Register(new YinYangStone2);

}
