/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: stones.cc,v 1.19.2.1 2003/09/22 02:41:37 dheck Exp $
 */
#include "items.hh"
#include "stones.hh"
#include "objects.hh"
#include "object_mixins.hh"
#include "stones_internal.hh"
#include <cassert>

using namespace std;
using namespace world;
using namespace enigma;
using namespace stones;

// *******************************************************************************
//  Stones under development :

//----------------------------------------
// Explosion stone
//----------------------------------------

namespace
{
    class ExplosionStone : public Stone {
        CLONEOBJ(ExplosionStone);

        StoneResponse collision_response(const StoneContact &) {
            return STONE_PASS;
        }

        void actor_contact (Actor *a) {
            SendMessage(a, "shatter");
        }
        void init_model() {
            set_anim("st-explosion");
        }

        void animcb() {
            KillStone(get_pos());
        }
    public:
        ExplosionStone(): Stone("st-explosion")
        {}

    };
        
}


//----------------------------------------
// MailStone
//----------------------------------------
namespace
{
    class MailStone : public Stone {
        CLONEOBJ(MailStone);
        Direction m_dir;
    public:
        MailStone (const char *kind, Direction dir) : Stone(kind), m_dir(dir)
        {}

        void actor_hit (const StoneContact &sc) {
            if (player::Inventory *inv = player::GetInventory(sc.actor)) {
                if (Item *it = inv->get_item(0)) {
                    GridPos p = find_pipe_endpoint();

                    if (it->is_kind("it-brake")) {
                        if (!GetStone(p)) {
                            it = inv->yield_first();
                            world::SetItemAsStone(p, it);
                        }
                    }
                    else {
                        if (!GetItem (p)) {
                            it = inv->yield_first();
                            world::SetItem(p, it);
                            it->on_drop(sc.actor);
                        }
                    }
                }
            }
        }
    private:
        GridPos find_pipe_endpoint() {
            using namespace items;

            GridPos p = get_pos();
            Direction move_dir = m_dir;

            while (move_dir != NODIR) {
                p.move (move_dir);
                if (Item *it = world::GetItem(p)) {
                    switch (it->get_id()) {
                    case IT_PIPE_H:
                        if (!(move_dir == EAST || move_dir == WEST))
                            move_dir = NODIR;
                        break;
                    case IT_PIPE_V:
                        if (!(move_dir == SOUTH || move_dir == NORTH))
                            move_dir = NODIR;
                        break;
                    case IT_PIPE_NE:
                        if (move_dir == SOUTH)     move_dir = EAST;
                        else if (move_dir == WEST) move_dir = NORTH;
                        else                       move_dir = NODIR;
                        break;
                    case IT_PIPE_ES:
                        if (move_dir == NORTH)     move_dir = EAST;
                        else if (move_dir == WEST) move_dir = SOUTH;
                        else                       move_dir = NODIR;
                        break;
                    case IT_PIPE_SW:
                        if (move_dir == NORTH)     move_dir = WEST;
                        else if (move_dir == EAST) move_dir = SOUTH;
                        else                       move_dir = NODIR;
                        break;
                    case IT_PIPE_WN:
                        if (move_dir == SOUTH)     move_dir = WEST;
                        else if (move_dir == EAST) move_dir = NORTH;
                        else                       move_dir = NODIR;
                        break;
                    default:
                        move_dir = NODIR;; // enf of pipe reached
                    }
                } else
                    move_dir = NODIR;
            }
            return p;
        }

    };
}


//----------------------------------------
// ChargeStone
//
// Attributes:
//
// :charge  + - 0
//----------------------------------------
namespace
{
    class ChargeStone : public Stone {
        CLONEOBJ(ChargeStone);
    public:
        ChargeStone(const char *kind, const double charge)
        : Stone(kind)
        {
            set_attrib("charge", charge);
        }
    private:
        double get_charge() {
            double q = 0;
            double_attrib("charge", &q);
            return q;
        }
        void animcb() { init_model(); }
        void actor_hit (const StoneContact &sc) {
            ActorInfo *ai = sc.actor->get_actorinfo();
            ai->charge = max(-1.0, min(+1.0, get_charge()));
            set_anim(string(get_kind())+"-anim");
        }
    };
}

// --------------------------------------------------------------------------------




YieldedGridStone::YieldedGridStone(Stone *st)
: stone(st)
{
    GridPos pos = stone->get_pos();
    model       = display::YieldModel(GridLoc(GRID_STONES, pos));
    YieldStone(pos);
}

YieldedGridStone::~YieldedGridStone() 
{
    assert(!stone);
    assert(!model);
}

void YieldedGridStone::set_stone(GridPos pos) 
{
    SetStone(pos, stone);
    display::SetModel(GridLoc(GRID_STONES, stone->get_pos()), model);
    stone->on_move();
    stone = 0;
    model = 0;
}

void YieldedGridStone::dispose() 
{
    stone->dispose();
    delete model;
    stone = 0;
    model = 0;
}

// --------------------------------------------------------------------------------

void stones::Init()
{
    // Register(new ...);

    Register (new ExplosionStone);
    Register (new MailStone ("st-mail-n", NORTH));
    Register (new MailStone ("st-mail-e", EAST));
    Register (new MailStone ("st-mail-s", SOUTH));
    Register (new MailStone ("st-mail-w", WEST));
    Register (new ChargeStone ("st-chargeplus", +1.0));
    Register (new ChargeStone ("st-chargeminus", -1.0));
    Register (new ChargeStone ("st-chargezero", 0.0));

    // Init stones from stones_simple.cc and stones_complex.cc:
    Init_simple();
    Init_complex();
}
