/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: lua.cc,v 1.15 2003/07/13 19:30:29 reallysoft Exp $
 */
#include "lua.hh"
#include "world.hh"
#include "objects.hh"
#include "enigma.hh"
#include "config.h"
#include "actors.hh"
#include "video.hh"

#include "display-lua.hh"
#include "enigma-lua.hh"
#include "px-lua.hh"

extern "C" {
#include "lualib.h"
#include "tolua.h"
}
#include "px/px.hh"
#include "SDL.h"
#include <string>
#include <cassert>

using namespace lua;
using namespace std;

using world::GridPos;
using world::ForceField;

namespace lua
{
    lua_State *state = 0;       // global Lua state
    int object_tag;             // Lua tag for `Object's
    ForceField *cff = 0;	// constant force field for levels
}

//----------------------------------------------------------------------
// Helper routines
//----------------------------------------------------------------------

using enigma::Value;

static void
push_value(lua_State *L, const Value &val)
{
    switch (val.get_type()) {
    case Value::NIL: lua_pushnil(L); break;
    case Value::DOUBLE: lua_pushnumber(L, to_double(val)); break;
    case Value::STRING: lua_pushstring(L, to_string(val)); break;
    }
}

static Value
to_value(lua_State *L, int idx)
{
    switch (lua_type(L, idx)) {
    case LUA_TNIL: return Value();
    case LUA_TNUMBER: return Value(lua_tonumber(L,idx));
    case LUA_TSTRING: return Value(lua_tostring(L,idx));
    default: lua_error(L,"Cannot convert type to Value.");
    }
    return Value();
}

static bool
is_object(lua_State *L, int idx)
{
    return lua_isuserdata(L,idx) && lua_tag(L,idx)==object_tag;
}

static world::Object *
to_object(lua_State *L, int idx)
{
    if (lua_isnil(L,idx))
        return 0;

    if (!is_object(L,idx)) {
        lua_error(L, "Cannot convert type to an Object");
        return 0;
    }
    return static_cast<world::Object*>(lua_touserdata(L,idx));
}

static void
pushobject(lua_State *L, world::Object *obj)
{
    /*
       Arghh, now that's a surprise.  Lua does not allow us to store
       NULL pointers in a userdata variable!  Don't ask me why, it's
       not in the documentation (afaik).  This cost me almost an hour
       to track down and fix...
    */
    if (obj == 0)
        lua_pushnil(L);
    else
        lua_pushusertag(L, obj, object_tag);
}

//----------------------------------------------------------------------
// Enigma interface routines
//----------------------------------------------------------------------
static int
en_make_object (lua_State *L)
{
    world::Object *obj = world::MakeObject(lua_tostring(L, 1));
    pushobject(L, obj);
    return 1;
}

static int
get_object_template(lua_State *L)
{
    world::Object *obj = world::GetObjectTemplate(lua_tostring(L, 1));
    pushobject(L, obj);
    return 1;
}

static int
en_set_attrib(lua_State *L)
{
    world::Object *obj = to_object(L,1);
    const char *key = lua_tostring(L,2);
    if (obj && key)
        obj->set_attrib(key, to_value(L, 3));
    else
        lua_error(L, "SetAttrib: invalid object or attribute name");
    return 0;
}

static int
en_get_attrib(lua_State *L)
{
    world::Object *obj = to_object(L,1);
    const char *key = lua_tostring(L,2);

    if (!obj) {
        lua_error(L, "GetAttrib: invalid object");
        return 0;
    }

    const Value *v =  obj->get_attrib(key);
    if (!v) {
        lua_error(L, "GetAttrib: unknown attribute");
        lua_pushnil(L);
    }
    else
        push_value(L, *v);
    return 1;
}

static int
en_set_floor(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::Floor *fl=0;

    if (lua_isnil(L, 3))
        fl = 0;
    else if (is_object(L,3)) {
        fl = static_cast<world::Floor*>(lua_touserdata(L,3));
    	if( ! fl)
	    lua_error(L, "object is no valid floor");
    } else
        lua_error(L, "argument 3 must be an Object or nil");
    world::SetFloor(GridPos(x,y), fl);
    return 0;
}

static int
en_set_item(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::Item *it = dynamic_cast<world::Item*>(to_object(L, 3));
    if( ! it)
        lua_error(L, "object is no valid item");
    world::SetItem(GridPos(x,y), it);
    return 0;
}
static int
en_set_stone(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::Stone *st = dynamic_cast<world::Stone*>(to_object(L, 3));
    if( ! st)
        lua_error(L, "object is no valid stone");
    world::SetStone(GridPos(x,y), st);
    return 0;
}

static int en_kill_stone(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::KillStone(GridPos(x,y));
    return 0;
}

static int en_kill_item(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::KillItem(GridPos(x,y));
    return 0;
}

static int
en_set_actor(lua_State *L)
{
    double x = lua_tonumber(L,1);
    double y = lua_tonumber(L,2);
    world::Actor *ac = dynamic_cast<world::Actor*>(to_object(L, 3));
    if( ! ac)
        lua_error(L, "object is no valid actor");
    world::AddActor(x, y, ac);
    return 0;
}


static int
en_send_message(lua_State *L)
{
    world::Object *obj = to_object(L, 1);
    const char *msg = lua_tostring(L, 2);
    if (!msg)
        lua_error(L,"Illegal message");
    else if (obj)
        obj->message(msg, to_value(L, 3));
    return 0;
}

static int
en_play_sound(lua_State *L)
{
    world::Object *obj       = to_object(L, 1);
    const char    *soundname = lua_tostring(L, 2);

    if (!soundname)
        lua_error(L,"Illegal sound");
    else if (obj) {
        world::GridObject *gobj = dynamic_cast<world::GridObject*>(obj);
        if (gobj)
            gobj->play_sound(soundname);
    }

    return 0;
}

static int
en_name_object(lua_State *L)
{
    world::Object *obj = to_object(L, 1);
    const char *name = lua_tostring(L,2);

    world::NameObject(obj, name);

    return 0;
}

static int
en_get_named_object(lua_State *L)
{
    world::Object *o = world::GetNamedObject(lua_tostring(L,1));
    pushobject(L, o);
    return 1;
}

static int
en_get_floor(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::Object *o = world::GetFloor(GridPos(x, y));
    pushobject(L, o);
    return 1;
}
static int
en_get_item(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::Object *o = world::GetItem(GridPos(x, y));
    pushobject(L, o);
    return 1;
}
static int
en_get_stone(lua_State *L)
{
    int x = int(lua_tonumber(L, 1));
    int y = int(lua_tonumber(L, 2));
    world::Object *o = world::GetStone(GridPos(x, y));
    pushobject(L, o);
    return 1;
}

static int
en_get_pos(lua_State *L)
{
    world::Object *obj = to_object(L, 1);
    GridPos        p;

    if (world::GridObject *gobj = dynamic_cast<world::GridObject*>(obj))
        p = gobj->get_pos();
    else
        p = GridPos(-1, -1);

    lua_pushnumber(L, double(p.x));
    lua_pushnumber(L, double(p.y));
    return 2;
}

 // LUA functions

int lua::FindDataFile(lua_State *L)
{
    const char *filename = lua_tostring(L, 1);
    string absfile = enigma::FindDataFile(filename);
    lua_pushstring(L, absfile.c_str());
    return 1;
}

static int
add_constant_force(lua_State *L)
{
    if (cff) {
	world::RemoveForceField(cff);
	delete cff;
	cff = 0;
    }

    px::V2 v;
    v[0] = lua_tonumber(L, 1);
    v[1] = lua_tonumber(L, 2);

    cff = new world::ConstantForce(v);
    world::AddForceField(cff);

    return 0;
}

static int
add_rubber_band (lua_State *L)
{
    using namespace world;

    Actor  *a1       = dynamic_cast<Actor*> (to_object(L, 1));
    Object *o2       = to_object(L, 2);
    Actor  *a2       = dynamic_cast<Actor*>(o2);
    Stone  *st       = dynamic_cast<Stone*>(o2);
    double  strength = lua_tonumber(L, 3);
    double  length   = lua_tonumber(L, 4);

    if (!a1)
        lua_error(L, "AddRubberBand: First argument must be an actor\n");
    else if (!a2 && !st)
        lua_error(L, "AddRubberBand: Second argument must be actor or stone\n");
    else {
        if (a2)
            world::AddRubberBand (a1, a2, strength,  length);
        else
            world::AddRubberBand (a1, st, strength,  length);
    }
    return 0;
}

static int
get_ticks(lua_State *L)
{
    lua_pushnumber(L, SDL_GetTicks());
    return 1;
}

static int
en_add_scramble(lua_State *L)
{
    int         x       = int(lua_tonumber(L, 1));
    int         y       = int(lua_tonumber(L, 2));
    const char *dir     = lua_tostring(L,3);
    const char *allowed = "wsen";
    char       *found   = strchr(allowed, dir[0]);

    if (found && found[0]) {
        world::AddScramble(GridPos(x,y), enigma::Direction(found-allowed));
    }
    else {
        lua_error(L, "AddScramble: Third argument must be one character of \"wsen\"");
    }

    return 0;
}

static int
en_set_scramble_intensity(lua_State *L)
{
    world::SetScrambleIntensity(int(lua_tonumber(L, 1)));
    return 0;
}

static CFunction luafuncs[] = {
    {en_set_attrib,         "SetAttrib"},
    {en_get_attrib,         "GetAttrib"},
    {en_make_object,        "MakeObject"},
    {get_object_template,   "GetObjectTemplate"},
    {en_set_floor,          "SetFloor"},
    {en_set_item,           "SetItem"},
    {en_set_stone,          "SetStone"},
    {en_kill_stone,         "KillStone"},
    {en_kill_item,          "KillItem"},
    {en_set_actor,          "SetActor"},
    {en_send_message,       "SendMessage"},
    {en_play_sound,         "PlaySound"},
    {en_name_object,        "NameObject"},
    {en_get_named_object,   "GetNamedObject"},
    {en_get_floor,          "GetFloor"},
    {en_get_item,           "GetItem"},
    {en_get_stone,          "GetStone"},
    {en_get_pos,            "GetPos"},
    {add_constant_force,    "AddConstantForce"},
    {add_rubber_band,       "AddRubberBand"},
    {lua::FindDataFile,     "FindDataFile"},
    {get_ticks,             "GetTicks"},
    {en_add_scramble,       "AddScramble"},
    {en_set_scramble_intensity, "SetScrambleIntensity"},
    {0,0}
};

//----------------------------------------------------------------------
// lua:: functions
//----------------------------------------------------------------------

void lua::RegisterFuncs(CFunction *funcs)
{
    lua_getglobal(state, "enigma");
    for (unsigned i=0; funcs[i].func; ++i) {
        lua_pushstring(state, funcs[i].name);
        lua_pushcfunction(state, funcs[i].func);
        lua_settable(state, -3);
    }
    lua_pop(state, 1);
}

int
lua::CallFunc(const char *funcname, const Value& arg)
{
    lua_getglobal(state, funcname);
    push_value(state, arg);
    return lua_call(state, 1, 0);
}

int lua::Dofile(const string &filename)
{
    string fname = enigma::FindDataFile(filename);

    int oldtop = lua_gettop(state);
    int retval = lua_dofile(state, fname.c_str());
    lua_settop(state, oldtop);
    return retval;
}

int lua::Dobuffer (const vector<char> &luacode)
{
    return lua_dobuffer (state, &luacode[0], luacode.size(), "buffer");
}


int
lua::DoSubfolderfile(const string &basefolder, const string &filename)
{
    std::list <string> fnames = enigma::FindDataFiles(basefolder, filename);
    int retval = 0;
    while ( fnames.size() > 0)
    {
        int oldtop = lua_gettop(state);
	string fname = fnames.front();
        retval = lua_dofile(state, fname.c_str());
	fnames.pop_front();
        lua_settop(state, oldtop);
    }
    return retval;
}

void
lua::Init()
{
    state = lua_open(0);
    lua_baselibopen(state);
    lua_strlibopen(state);
    lua_mathlibopen(state);
    lua_iolibopen(state);

    tolua_open(state);
    tolua_enigma_open(state);
    tolua_px_open(state);
    tolua_display_open(state);

    // Create a new tag for world::Object objects
    object_tag = lua_newtag(state);

    // Register all functions
    RegisterFuncs(luafuncs);
    cff = 0;
}

void lua::Shutdown()
{
    if (state) {
        lua_close(state);
        state = 0;
	if (cff) {
	    delete cff;
	    cff = 0;
	}
    }
}
