%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs unrecord}
\begin{code}
module Unrecord ( unrecord, unpull ) where
import Monad ( when )

import SignalHandler ( withSignalsBlocked )
import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag( Verbose ),
                        working_repo_dir, nocompress, verbose,
                        match_one,
                        ignoretimes,
                      )
import Repository ( PatchSet, read_repo, read_pending,
                    get_unrecorded,
                    write_inventory, write_pending, write_patch,
                    slurp_recorded, slurp_recorded_and_unrecorded,
                    am_in_repo, sync_repo,
                  )
import Pristine ( identifyPristine, write_dirty_Pristine )
import Patch ( Patch, invert, apply_to_slurpy, patch2patchinfo,
               join_patches, commute, flatten, null_patch,
             )
import SlurpDirectory ( slurp_write_dirty, )
import Depends ( deep_optimize_patchset )
import Lock ( withLock )
import Unrevert ( remove_from_unrevert_context )
import SelectChanges ( with_selected_patch_from_repo )
#include "impossible.h"
\end{code}
\begin{code}
unrecord_description :: String
unrecord_description =
 "Unrecord a named patch."
\end{code}

\options{unrecord}

\haskell{unrecord_help}

Unrecord can be thought of as undo-record.
If a record is followed by an unrecord, everything looks like before
the record; all the previously unrecorded changes are back, and can be
recorded again in a new patch. The unrecorded patch however is actually
removed from your repository, so there is no way to record it again to get
it back.\footnote{The patch file itself is not actually deleted, but its
context is lost, so it cannot be reliably read---your only choice would be
to go in by hand and read its contents.}.

If you want to remove
the change from the working copy too (where they otherwise will show
up as unrecorded changes again), you'll also need to \verb!darcs revert!.
To do unrecord and revert in one go, you can use \verb!darcs unpull!.

If you don't revert after unrecording, then the changes made by the
unrecorded patches are left in your working tree.  If these patches are
actually from another repository, interaction (either pushes or pulls) with
that repository may be massively slowed down, as darcs tries to cope with
the fact that you appear to have made a large number of changes that
conflict with those present on the other repository.  So if you really want
to undo the result of a \emph{pull} operation, use unpull! Unrecord is
primarily intended for when you record a patch, realize it needs just one
more change, but would rather not have a separate patch for just that one
change.

\begin{code}
unrecord_help :: String
unrecord_help =
 "Unrecord is used to undo a single recorded patch.  It will prompt you\n"++
 "for which patch to unrecord, and then will undo that patch.\n"++
 "Unrecord doesn't affect your working copy of the tree at all.\n"
\end{code}
\begin{code}
unrecord :: DarcsCommand
unrecord = DarcsCommand {command_name = "unrecord",
                         command_help = unrecord_help,
                         command_description = unrecord_description,
                         command_extra_args = 0,
                         command_extra_arg_help = [],
                         command_command = unrecord_cmd,
                         command_prereq = am_in_repo,
                         command_get_arg_possibilities = return [],
                         command_argdefaults = nodefaults,
                         command_darcsoptions = [match_one,
                                                 verbose,nocompress,
                                                 working_repo_dir]}
\end{code}
\begin{code}
unrecord_cmd :: [DarcsFlag] -> [String] -> IO ()
unrecord_cmd opts _ = withLock "./_darcs/lock" $ do
  recorded <- slurp_recorded "."
  pend <- do aack <- read_pending
             return $ case aack of Nothing -> []
                                   Just p -> flatten p
  with_selected_patch_from_repo "unrecord" opts True $
    \ (p, skipped) ->
    case apply_to_slurpy (invert p) recorded of
    Nothing -> putStrLn "Unable to apply inverse patch!"
    Just recorded' -> do
       remove_from_unrevert_context p
       withSignalsBlocked $ do
          when (Verbose `elem` opts) $
               putStrLn "About to write out (potentially) modified patches..."
          sequence_ $ map (write_patch opts) skipped
          patches' <- read_repo "."
          when (Verbose `elem` opts) $
               putStrLn "About to write inventory..."
          write_inventory "." $ rempatch p patches'
          when (Verbose `elem` opts) $ putStrLn "Updating pristine tree..."
          identifyPristine >>= write_dirty_Pristine recorded'
          when (Verbose `elem` opts) $ putStrLn "Updating pending..."
          write_pending $ join_patches (p : pend)
          sync_repo
          putStrLn "Finished unrecording."

rempatch :: Patch -> PatchSet -> PatchSet
rempatch p (pps:ppss) =
    case patch2patchinfo p of
    Nothing -> bug "Weird problem in unrecord or unpull - in rempatch."
    Just pinfo -> if pinfo `elem` simple_infos
                  then (filter ((/= pinfo).fst) pps) : ppss
                  else deep_optimize_patchset $
                       map (filter ((/= pinfo).fst)) (pps:ppss)
    where simple_infos = init $ map fst pps
rempatch _ [] = impossible
\end{code}

\subsection{darcs unpull}

\begin{code}
unpull_description :: String
unpull_description =
 "Unpull a named patch."
\end{code}

\begin{code}
unpull_help :: String
unpull_help =
 "Unpull is used to undo a single patch that has been pulled from another\n"++
 "repository.  It will prompt you for which patch to unpull, and then will\n"++
 "undo that patch.  Beware that unpull undoes the patch both from the repo\n"++
 "records AND from the current working directory, and does NOT check that\n"++
 "the patch originated with a pull.  In other words, you could lose precious\n"++
 "code by unpulling!\n"
\end{code}

\options{unpull}

\haskell{unpull_help}

Like unrecord, unpull does not just apply an inverse patch to the
repository, it actually deletes the patch from the repository.  This makes
unpull a particularly dangerous command, as it not only deletes the patch
from the repo, but also removes the changes from the working directory.  It
is equivalent to an unrecord followed by a revert, except that revert can
be unreverted.

Contrary to what its name suggests, there is nothing in unpull that
requires that the ``unpulled'' patch originate from a different repository.
The name was chosen simply to suggest a situation in which it is ``safe''
to used unpull.  If the patch was originally from another repo, then
unpulling is safe, because you can always pull the patch again if you
decide you want it after all.  If you unpull a locally recorded patch, all
record of that change is lost, which is what makes this a ``dangerous''
command, and thus deserving of a obscure name which is more suggestive of
when it is safe to use than precisely what it does.

\begin{options}
--match, --patch, --tag
\end{options}

Each of these options is used to select a single patch to unpull. In the case of
\verb!--tag!, what you are unpulling is the tag itself, not any other patches. 

\begin{code}
unpull :: DarcsCommand
unpull = DarcsCommand {command_name = "unpull",
                       command_help = unpull_help,
                       command_description = unpull_description,
                       command_extra_args = 0,
                       command_extra_arg_help = [],
                       command_command = unpull_cmd,
                       command_prereq = am_in_repo,
                       command_get_arg_possibilities = return [],
                       command_argdefaults = nodefaults,
                       command_darcsoptions = [match_one,
                                               verbose,nocompress,
                                               ignoretimes,
                                               working_repo_dir]}
\end{code}
\begin{code}
unpull_cmd :: [DarcsFlag] -> [String] -> IO ()
unpull_cmd opts _ = withLock "./_darcs/lock" $ do
  (recorded, work) <- slurp_recorded_and_unrecorded "."
  mpend <- get_unrecorded opts
  let pend = case mpend of
             Nothing -> null_patch
             Just p -> p
  with_selected_patch_from_repo "unpull" opts False $
    \ (p, skipped) ->
    case commute (pend, p) of
    Nothing -> fail $ "Couldn't commute patch past pending."
    Just (p_after_pending, pend') ->
      case apply_to_slurpy (invert p) recorded of
      Nothing -> fail "Unable to apply inverse patch!"
      Just rec' ->
          case apply_to_slurpy (invert p_after_pending) work of
          Nothing -> fail "Couldn't undo patch in working dir."
          Just work' -> do
             remove_from_unrevert_context p
             withSignalsBlocked $ do
                sequence_ $ map (write_patch opts) skipped
                patches' <- read_repo "."
                write_inventory "." $ rempatch p patches'
                identifyPristine >>= write_dirty_Pristine rec'
                slurp_write_dirty [] work'
                write_pending pend'
                sync_repo
                putStrLn "Finished unpulling."
\end{code}

