/*
 * (CAPI*)
 *
 * An implementation of Common ISDN API 2.0 for Asterisk
 *
 * Copyright (C) 2002, 2003, 2004, Junghanns.NET GmbH
 *
 * Klaus-Peter Junghanns <kapejod@ns1.jnetdns.de>
 *
 * This program is free software and may be modified and 
 * distributed under the terms of the GNU Public License.
 */

#include <asterisk/lock.h>
#include <asterisk/frame.h> 
#include <asterisk/channel.h>
#include <asterisk/channel_pvt.h>
#include <asterisk/logger.h>
#include <asterisk/module.h>
#include <asterisk/pbx.h>
#include <asterisk/config.h>
#include <asterisk/options.h>
#include <asterisk/features.h>
#include <asterisk/utils.h>
#include <asterisk/cli.h>
#include <sys/time.h>
#include <sys/signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <linux/capi.h>
#include <capi20.h>
#include <asterisk/dsp.h>
#include "xlaw.h"
#include "chan_capi_pvt.h"

unsigned ast_capi_ApplID;
_cword ast_capi_MessageNumber=1;
static char *desc = "Common ISDN API for Asterisk";
#ifdef CAPI_ULAW
#ifdef UNSTABLE_CVS
static char *tdesc = "Common ISDN API Driver (0.3.5) muLaw CVS HEAD";
#else
static char *tdesc = "Common ISDN API Driver (0.3.5) muLaw";
#endif
#else
#ifdef UNSTABLE_CVS
static char *tdesc = "Common ISDN API Driver (0.3.5) aLaw CVS HEAD";
#else
static char *tdesc = "Common ISDN API Driver (0.3.5) aLaw";
#endif
#endif
static char *type = "CAPI";


static int usecnt;
#ifdef UNSTABLE_CVS
AST_MUTEX_DEFINE_STATIC(usecnt_lock);
AST_MUTEX_DEFINE_STATIC(iflock);
AST_MUTEX_DEFINE_STATIC(pipelock);
AST_MUTEX_DEFINE_STATIC(monlock);
AST_MUTEX_DEFINE_STATIC(contrlock);
AST_MUTEX_DEFINE_STATIC(capi_send_buffer_lock);
AST_MUTEX_DEFINE_STATIC(capi_put_lock);
#else
static ast_mutex_t usecnt_lock = AST_MUTEX_INITIALIZER;
static ast_mutex_t iflock = AST_MUTEX_INITIALIZER;
static ast_mutex_t pipelock = AST_MUTEX_INITIALIZER;
static ast_mutex_t monlock = AST_MUTEX_INITIALIZER;
static ast_mutex_t contrlock = AST_MUTEX_INITIALIZER;
static ast_mutex_t capi_send_buffer_lock = AST_MUTEX_INITIALIZER;
static ast_mutex_t capi_put_lock = AST_MUTEX_INITIALIZER;
#endif

#ifdef CAPI_ULAW
static int capi_capability = AST_FORMAT_ULAW;
#else
static int capi_capability = AST_FORMAT_ALAW;
#endif
static struct ast_capi_profile profile;

static pthread_t monitor_thread = -1;

static struct ast_capi_pvt *iflist = NULL;
static struct capi_pipe *pipelist = NULL;
static int capi_last_plci = 0;
static struct ast_capi_controller *capi_controllers[AST_CAPI_MAX_CONTROLLERS];
static int capi_num_controllers = 0;
static int capi_counter = 0;
static unsigned long capi_used_controllers=0;

static char capi_send_buffer[AST_CAPI_MAX_B3_BLOCKS * AST_CAPI_MAX_B3_BLOCK_SIZE];
static int capi_send_buffer_handle = 0;

char capi_national_prefix[AST_MAX_EXTENSION];
char capi_international_prefix[AST_MAX_EXTENSION];

int capidebug = 0;

MESSAGE_EXCHANGE_ERROR _capi_put_cmsg(_cmsg *CMSG) {
    MESSAGE_EXCHANGE_ERROR error;
    if (ast_mutex_lock(&capi_put_lock)) {
        ast_log(LOG_WARNING,"Unable to lock capi put!\n");
        return -1;
    } 
    error = capi20_put_cmsg(CMSG);    
    if (ast_mutex_unlock(&capi_put_lock)) {
	ast_log(LOG_WARNING,"Unable to unlock capi put!\n");
        return -1;
    }
    return error;
}


MESSAGE_EXCHANGE_ERROR check_wait_get_cmsg(_cmsg *CMSG) {
    MESSAGE_EXCHANGE_ERROR Info;
    struct timeval tv;
    tv.tv_sec = 0;
    tv.tv_usec = 10000;
    Info = capi20_waitformessage(ast_capi_ApplID,&tv);
    if ((Info != 0x0000) && (Info != 0x1104)) {
	printf("Error waiting for cmsg... INFO = %#x\n", Info);
	return Info;
    }
    
    if (Info == 0x0000) {
        Info = capi_get_cmsg(CMSG,ast_capi_ApplID);
    }
    return Info;
}


unsigned ListenOnController(unsigned long CIPmask,unsigned controller) {
    MESSAGE_EXCHANGE_ERROR  error;
    _cmsg		    CMSG,CMSG2;

    LISTEN_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, controller);
#ifdef NEVER_EVER_EARLY_B3_CONNECTS
    LISTEN_REQ_INFOMASK(&CMSG) = 0x00ff; // lots of info ;)
#else
    LISTEN_REQ_INFOMASK(&CMSG) = 0x03ff; // lots of info ;) + early B3 connect
#endif
    LISTEN_REQ_CIPMASK(&CMSG) = CIPmask;
    if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	return error;
    }
    while (!IS_LISTEN_CONF(&CMSG2)) {
	error = check_wait_get_cmsg(&CMSG2);
    }
    return 0;
}

unsigned Listen(unsigned long CIPmask) {
    return ListenOnController(CIPmask,0);
}

// Echo cancellation is for cards w/ integrated echo cancellation only
// (i.e. Eicon active cards support it)

#define EC_FUNCTION_ENABLE		1
#define EC_FUNCTION_DISABLE		2
#define EC_FUNCTION_FREEZE		3
#define EC_FUNCTION_RESUME		4
#define EC_FUNCTION_RESET		5
#define EC_OPTION_DISABLE_NEVER		0
#define EC_OPTION_DISABLE_G165	 	(1<<1)
#define EC_OPTION_DISABLE_G164_OR_G165	(1<<1 | 1<<2)
#define EC_DEFAULT_TAIL			64

static int capi_echo_canceller(struct ast_channel *c, int function) {
    struct ast_capi_pvt *i = c->pvt->pvt;
    MESSAGE_EXCHANGE_ERROR  error;
    _cmsg		    CMSG;
    char   buf[7];

	/* If echo cancellation is not requested or supported, don't attempt to enable it */
	ast_mutex_lock(&contrlock);
	if (!capi_controllers[i->controller]->echocancel || !i->doEC) {
		ast_mutex_unlock(&contrlock);
		return 0;
	}
	ast_mutex_unlock(&contrlock);

	if (option_verbose > 2) 
		ast_verbose(VERBOSE_PREFIX_3 "Setting up echo canceller (PLCI=%#x, function=%d, options=%d, tail=%d)\n",i->PLCI,function,i->ecOption,i->ecTail);

	FACILITY_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
	FACILITY_REQ_NCCI(&CMSG) = i->NCCI;
	FACILITY_REQ_FACILITYSELECTOR(&CMSG) = 6; /* Echo canceller */

        buf[0]=6; /* msg size */
        buf[1]=function;
	if (function == EC_FUNCTION_ENABLE) {
	        buf[3]=i->ecOption; /* bit field - ignore echo canceller disable tone */
		buf[5]=i->ecTail;   /* Tail length, ms */
	}
	else {
	        buf[3]=0;
	        buf[5]=0;
	}

	// Always null:
        buf[2]=0;
        buf[4]=0;
        buf[6]=0;

	FACILITY_REQ_FACILITYREQUESTPARAMETER(&CMSG) = buf;
        
	if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	    ast_log(LOG_ERROR,"error sending FACILITY_REQ (error=%#x)\n",error);
	    return error;
	}

	if (option_verbose > 5) 
	   ast_verbose(VERBOSE_PREFIX_4 "sent FACILITY_REQ (PLCI=%#x)\n",i->PLCI);

    return 0;
}

int capi_detect_dtmf(struct ast_channel *c, int flag) {
    struct ast_capi_pvt *i = c->pvt->pvt;
    MESSAGE_EXCHANGE_ERROR  error;
    _cmsg		    CMSG;
    char	buf[9];
#ifndef FORCE_SOFTWARE_DTMF
    // does the controller support dtmf? and do we want to use it?
    ast_mutex_lock(&contrlock);
    if ((capi_controllers[i->controller]->dtmf == 1) && (i->doDTMF == 0)) {
	ast_mutex_unlock(&contrlock);
	FACILITY_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
	FACILITY_REQ_PLCI(&CMSG) = i->PLCI;
	FACILITY_REQ_FACILITYSELECTOR(&CMSG) = 1;
	buf[0] = 8;
	if (flag == 1) {
	    buf[1] = 1;
	} else {
	    buf[1] = 2;
	}
	buf[2] = 0;
	buf[3] = AST_CAPI_DTMF_DURATION;
	buf[4] = 0;
	buf[5] = AST_CAPI_DTMF_DURATION;
	buf[6] = 0;
	buf[7] = 0;
	buf[8] = 0;
	FACILITY_REQ_FACILITYREQUESTPARAMETER(&CMSG) = buf;
        
	if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	    ast_log(LOG_ERROR,"error sending FACILITY_REQ (error=%#x)\n",error);
	    return error;
	} else {
	    if (option_verbose > 5) {
		ast_verbose(VERBOSE_PREFIX_4 "sent FACILITY_REQ (PLCI=%#x)\n",i->PLCI);
	    }
	}
    } else {
	ast_mutex_unlock(&contrlock);

#endif
	// do software dtmf detection
	i->doDTMF = 1; // just being paranoid again...
#ifndef FORCE_SOFTWARE_DTMF
    }
#endif
    return 0;
}
static int capi_send_digit(struct ast_channel *c,char digit) {
    struct ast_capi_pvt *i = c->pvt->pvt;
    MESSAGE_EXCHANGE_ERROR  error;
    _cmsg		    CMSG;
    char	buf[10];
    
    if (i->state != CAPI_STATE_BCONNECTED) {
	return 0;
    }
    
    
#ifndef NEVER_EVER_EARLY_B3_CONNECTS
    if(i->earlyB3 == 1)
    /* we should really test for the network saying the number is incomplete
    since i'm only doing a test and this is true at the right time
    i'm going with this */
    {

	INFO_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
	INFO_REQ_PLCI(&CMSG) = i->PLCI;
	buf[0] = 2;
	buf[1] = 0x80;
	buf[2] = digit;
	INFO_REQ_CALLEDPARTYNUMBER(&CMSG) = buf;


	if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	    ast_log(LOG_ERROR,"error sending CALLEDPARTYNUMBER INFO (error=%#x)\n",error);
	    return error;
	} else {
	    if (option_verbose > 5) {
		ast_verbose(VERBOSE_PREFIX_4 "sent CALLEDPARTYNUMBER INFO digit = %c (PLCI=%#x)\n", digit, i->PLCI);
	    }
	}

    } else {
#endif
#ifndef FORCE_SOFTWARE_DTMF
	ast_mutex_lock(&contrlock);
	if ((capi_controllers[i->controller]->dtmf == 0) || (i->doDTMF == 1)) {
#endif
	    // let * fake it
#ifndef FORCE_SOFTWARE_DTMF
	    ast_mutex_unlock(&contrlock);
#endif
	    return -1;
#ifndef FORCE_SOFTWARE_DTMF
	}
	ast_mutex_unlock(&contrlock);
	
	FACILITY_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
	FACILITY_REQ_PLCI(&CMSG) = i->NCCI;
        FACILITY_REQ_FACILITYSELECTOR(&CMSG) = 1;
        buf[0] = 8;
    
        buf[1] = 3;
        buf[2] = 0;
    
        buf[3] = AST_CAPI_DTMF_DURATION;
        buf[4] = 0;
    
        buf[5] = AST_CAPI_DTMF_DURATION;
        buf[6] = 0;
    
        buf[7] = 1;
	buf[8] = digit;
        buf[9] = 0;
	FACILITY_REQ_FACILITYREQUESTPARAMETER(&CMSG) = buf;
        
        if ((error = _capi_put_cmsg(&CMSG)) != 0) {
    	    ast_log(LOG_ERROR,"error sending FACILITY_REQ (error=%#x)\n",error);
	    return error;
        } else {
	    if (option_verbose > 4) {
		ast_verbose(VERBOSE_PREFIX_3 "sent dtmf '%c'\n",digit);
	    }
	}
#endif
#ifndef NEVER_EVER_EARLY_B3_CONNECTS
    }
#endif
    return 0;
}

static int capi_alert(struct ast_channel *c) {
    struct ast_capi_pvt *i = c->pvt->pvt;
    MESSAGE_EXCHANGE_ERROR  error;
    _cmsg	CMSG;
    
    ALERT_REQ_HEADER(&CMSG, ast_capi_ApplID, i->MessageNumber, 0);
    ALERT_REQ_PLCI(&CMSG) = i->PLCI;

    if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	ast_log(LOG_ERROR,"error sending ALERT_REQ PLCI = %#x\n",i->PLCI);
	return -1;
    } else {
	if (option_verbose > 5) {
	    ast_verbose(VERBOSE_PREFIX_4 "sent ALERT_REQ PLCI = %#x\n",i->PLCI);
	}
    }

    i->state = CAPI_STATE_ALERTING;
    return 0;
}

#ifdef DEFLECT_ON_CIRCUITBUSY
static int capi_deflect(struct ast_channel *chan, void *data)
{
    struct ast_capi_pvt *i = chan->pvt->pvt;
    MESSAGE_EXCHANGE_ERROR Info;
    _cmsg	CMSG;
    char	bchaninfo[1];
    char	fac[60];
    int res=0;
    int ms=3000;

    if (!data) {
    	ast_log(LOG_WARNING, "cd requires an argument (destination phone number)\n");
    	return -1;
    }

    if ((i->state == CAPI_STATE_CONNECTED) || (i->state == CAPI_STATE_BCONNECTED)) {
	ast_log(LOG_ERROR, "call deflection does not work with calls that are already connected!\n");
	return -1;
    }
    // wait until the channel is alerting, so we dont drop the call and interfer with msgs
    while ((ms > 0) && (i->state != CAPI_STATE_ALERTING)) {
	sleep(100);
	ms -= 100;
    }

    // make sure we hang up correctly
    i->state = CAPI_STATE_CONNECTPENDING;

    fac[0]=0; // len 
    fac[1]=0; //len 
    fac[2]=0x01; // Use D-Chan
    fac[3]=0; // Keypad len
    fac[4]=31;	// user user data? len = 31 = 29 + 2
    fac[5]=0x1c;	// magic?
    fac[6]=0x1d;	// strlen destination + 18 = 29
    fac[7]=0x91;	// ..
    fac[8]=0xA1;
    fac[9]=0x1A;	// strlen destination + 15 = 26
    fac[10]=0x02;
    fac[11]=0x01;
    fac[12]=0x70;
    fac[13]=0x02;
    fac[14]=0x01;
    fac[15]=0x0d;
    fac[16]=0x30;
    fac[17]=0x12;	// strlen destination + 7 = 18
    fac[18]=0x30;	// ...hm 0x30
    fac[19]=0x0d;	// strlen destination + 2	
    fac[20]=0x80;	// CLIP
    fac[21]=0x0b;	//  strlen destination 
    fac[22]=0x01;	//  destination start
    fac[23]=0x01;	//  
    fac[24]=0x01;	//  
    fac[25]=0x01;	//  
    fac[26]=0x01;	//  
    fac[27]=0x01;	//  
    fac[28]=0x01;	//  
    fac[29]=0x01;	//  
    fac[30]=0x01;	//  
    fac[31]=0x01;	//  
    fac[32]=0x01;	//  
    fac[33]=0x01;	// 0x1 = sending complete
    fac[34]=0x01;
    fac[35]=0x01;
				   
    memcpy((unsigned char *)fac+22,data,strlen(data));
    fac[22+strlen(data)]=0x01;	// fill with 0x01 if number is only 6 numbers (local call)
    fac[23+strlen(data)]=0x01;
    fac[24+strlen(data)]=0x01;
    fac[25+strlen(data)]=0x01;
    fac[26+strlen(data)]=0x01;
     
    fac[6]=18+strlen(data);
    fac[9]=15+strlen(data);
    fac[17]=7+strlen(data);
    fac[19]=2+strlen(data);
    fac[21]=strlen(data);

    bchaninfo[0] = 0x1;
    INFO_REQ_HEADER(&CMSG,ast_capi_ApplID,ast_capi_MessageNumber++,0);
    INFO_REQ_CONTROLLER(&CMSG) = i->controller;
    INFO_REQ_PLCI(&CMSG) = i->PLCI;
    INFO_REQ_BCHANNELINFORMATION(&CMSG) = (unsigned char*)bchaninfo; // use D-Channel
    INFO_REQ_KEYPADFACILITY(&CMSG) = 0;
    INFO_REQ_USERUSERDATA(&CMSG) = 0;
    INFO_REQ_FACILITYDATAARRAY(&CMSG) = (unsigned char*) fac + 4;

    if ((Info = _capi_put_cmsg(&CMSG)) != 0) {
	ast_log(LOG_ERROR,"Error sending INFO_REQ\n");
	return Info;
    } else {
	if (capidebug) {
	    // ast_log(LOG_NOTICE,"%s\n",capi_cmsg2str(&CMSG));
	    ast_log(LOG_NOTICE,"sent INFO_REQ PLCI = %#x\n",i->PLCI);
	}
    }

    return res;
}
#endif

void remove_pipe(int PLCI) {
    struct capi_pipe *p,*ptmp;

    ast_mutex_lock(&pipelock);
    p = pipelist;
    ptmp = NULL;
    while (p) {
	if (p->PLCI == PLCI) {
	    if (ptmp == NULL) {
		// mypipe == head of pipelist
		pipelist = p->next;
	        if(p->fd > -1) close(p->fd);
    		if(p->i != NULL && p->i->fd > -1) close(p->i->fd);
		free(p);
		if (option_verbose > 4) {
		    ast_verbose(VERBOSE_PREFIX_3 "removed pipe for PLCI = %#x\n",PLCI);
		}
		break;
	    } else {
		// somehwere inbetween or at the end
		ptmp->next = p->next;
		if (p->next == NULL) {
		    capi_last_plci = p->PLCI;
		}
    		if(p->fd > -1) close(p->fd);
    		if(p->i != NULL && p->i->fd > -1) close(p->i->fd);
		free(p);
		if (option_verbose > 4) {
		    ast_verbose(VERBOSE_PREFIX_3 "removed pipe for PLCI = %#x\n",PLCI);
		}
		break;
	    }
	}
	ptmp = p;
	p = p->next;
    }
    ast_mutex_unlock(&pipelock);
}

static int capi_activehangup(struct ast_channel *c) {
    struct ast_capi_pvt *i;
    MESSAGE_EXCHANGE_ERROR  error;
    _cmsg	CMSG;
    i = c->pvt->pvt;

    if (option_verbose > 2) {
	if (capidebug)
            ast_verbose(VERBOSE_PREFIX_4 "activehangingup\n");
    }

    if (i == NULL) {
	return 0;
    }

    if (c->_state == AST_STATE_RING) {
	CONNECT_RESP_HEADER(&CMSG, ast_capi_ApplID, i->MessageNumber, 0);
	CONNECT_RESP_PLCI(&CMSG) = i->PLCI;
	CONNECT_RESP_REJECT(&CMSG) = 2;
	if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	    ast_log(LOG_ERROR,"error sending CONNECT_RESP for PLCI = %#x\n",i->PLCI);
	} else {
	    if (option_verbose > 5) {
    	        ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_RESP for PLCI = %#x\n",i->PLCI);
	    }
	}
	return 0;
    }

    // active disconnect
    if (i->state == CAPI_STATE_BCONNECTED) {
	DISCONNECT_B3_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
	DISCONNECT_B3_REQ_NCCI(&CMSG) = i->NCCI;

	if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	    ast_log(LOG_ERROR, "error sending DISCONNECT_B3_REQ NCCI=%#x\n",i->NCCI);
	} else {
	    if (option_verbose > 5) {
    		ast_verbose(VERBOSE_PREFIX_4 "sent DISCONNECT_B3_REQ NCCI=%#x\n",i->NCCI);
	    }
	}
	// wait for the B3 layer to go down
	while (i->state != CAPI_STATE_CONNECTED) {
	    usleep(10000);
	}
    }
    if ((i->state == CAPI_STATE_CONNECTED) || (i->state == CAPI_STATE_CONNECTPENDING)){
	DISCONNECT_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
	DISCONNECT_REQ_PLCI(&CMSG) = i->PLCI;

	if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	    ast_log(LOG_ERROR, "error sending DISCONNECT_REQ PLCI=%#x\n",i->PLCI);
	} else {
	    if (option_verbose > 5) {
    		ast_verbose(VERBOSE_PREFIX_4 "sent DISCONNECT_REQ PLCI=%#x\n",i->PLCI);
	    }
	}
	// wait for the B1 layer to go down
	while (i->state != CAPI_STATE_DISCONNECTED) {
	    usleep(10000);
	}
    }
    return 0;
}

static int capi_hangup(struct ast_channel *c) {
    struct ast_capi_pvt *i;
    i = c->pvt->pvt;

    // hmm....ok...this is called to free the capi interface (passive disconnect)
    // or to bring down the channel (active disconnect)
    
    if (option_verbose > 3)
    	    ast_verbose(VERBOSE_PREFIX_3 "CAPI Hangingup\n");

    if (i == NULL) {
	ast_log(LOG_ERROR,"channel has no interface!\n");
	return -1;
    }
    
    // are we down, yet?
    if (i->state != CAPI_STATE_DISCONNECTED) {
	// no
	capi_activehangup(c);
    }

    remove_pipe(i->PLCI);
    i->PLCI = 0;
    i->NCCI = 0;
    if ((i->doDTMF == 1) && (i->vad != NULL)) {
	ast_dsp_free(i->vad);
    }
    ast_smoother_free(i->smoother); // discard any frames left hanging
    i->smoother=ast_smoother_new(AST_CAPI_MAX_B3_BLOCK_SIZE * 2);
    memset(i->cid,0,sizeof(i->cid));
    i->owner=NULL;
    ast_mutex_lock(&usecnt_lock);
	usecnt--;
    ast_mutex_unlock(&usecnt_lock);
    ast_update_use_count();
    i->mypipe = NULL;
    i = NULL;
    c->pvt->pvt = NULL;
    ast_setstate(c,AST_STATE_DOWN);
    return 0;
}

static char *capi_number(char *data,int strip) {
    unsigned len = *data;
    // XXX fix me
    // convert a capi struct to a \0 terminated string
    if (!len || len < (unsigned int) strip) return NULL;
    len = len - strip;
    data = (char *)(data + 1 + strip);
    return strndup((char *)data,len);
}

int capi_call(struct ast_channel *c, char *idest, int timeout)
{
	struct ast_capi_pvt *i;
	struct capi_pipe *p = NULL;
	int fds[2];
	char *dest,*msn;
	char buffer[AST_MAX_EXTENSION];
	char called[AST_MAX_EXTENSION],calling[AST_MAX_EXTENSION];
	char bchaninfo[3];
	
	_cmsg CMSG;
	MESSAGE_EXCHANGE_ERROR  error;
	
	strncpy(buffer,(char *)idest,sizeof(buffer)-1);
	msn = strtok(buffer, ":");
	dest = strtok(NULL, ":");


	if (!dest) {
		ast_log(LOG_WARNING, "Destination %s requres a real destination\n", idest);
		return -1;
	}
	i = c->pvt->pvt;
	i->doB3 = AST_CAPI_B3_DONT; // <homer>DOH</homer>

	// always B3
	if (((char *)dest)[0] == 'b') {
	    i->doB3 = AST_CAPI_B3_ALWAYS;
	}
	// only do B3 on successfull calls
	if (((char *)dest)[0] == 'B') {
	    i->doB3 = AST_CAPI_B3_ON_SUCCESS;
	} 
	
	if (i->doB3 != AST_CAPI_B3_DONT) {
	    dest++;
	}

	if (option_verbose > 1) {
	    if (capidebug)
		ast_verbose(VERBOSE_PREFIX_2 "CAPI Call %s %s", c->name, i->doB3?"with B3":"");
	}
    
	if (((char *)msn)[0] == '@') {
	    i->CLIR = 1;
	    msn++;
	} else {
	    i->CLIR = 0;
	}

	if (pipe(fds) == 0) {
	    ast_mutex_lock(&pipelock);
	    i->fd = fds[0];
	    p = malloc(sizeof(struct capi_pipe));
	    memset(p, 0, sizeof(struct capi_pipe));
	    p->fd = fds[1];
	    c->fds[0] = fds[1];
	    p->PLCI = -1;
	    p->i = i;
	    p->c = c;
	    i->mypipe = p;
	    p->next = pipelist;
	    pipelist = p;
	    if (option_verbose > 4) {
		ast_verbose(VERBOSE_PREFIX_3 "creating pipe for PLCI=-1\n");
	    }
	    ast_mutex_unlock(&pipelock);
	}
	i->outgoing = 1;
	
	i->MessageNumber = ast_capi_MessageNumber++;
	CONNECT_REQ_HEADER(&CMSG, ast_capi_ApplID, i->MessageNumber, i->controller);
	CONNECT_REQ_CONTROLLER(&CMSG) = i->controller;
	CONNECT_REQ_CIPVALUE(&CMSG) = 0x10;	// Telephony, could also use 0x04 (3.1Khz audio)
	called[0] = strlen(dest)+1;
	called[1] = 0x80;
	strncpy(&called[2],dest,sizeof(called)-2);
	CONNECT_REQ_CALLEDPARTYNUMBER(&CMSG) = called;
	CONNECT_REQ_CALLEDPARTYSUBADDRESS(&CMSG) = NULL;

	if (i->isdnmode && (strlen(msn)>strlen(i->msn)) && !strncmp(msn, i->msn, strlen(i->msn)))
	    msn += strlen(i->msn);

	calling[0] = strlen(msn)+2;
	calling[1] = 0x0;
	if (i->CLIR == 1) {
	    calling[2] = 0xA0; // CLIR
	} else {
	    calling[2] = 0x80; // CLIP
	}
	strncpy(&calling[3],msn,sizeof(calling)-3);
	CONNECT_REQ_CALLINGPARTYNUMBER(&CMSG) = calling;
	CONNECT_REQ_CALLINGPARTYSUBADDRESS(&CMSG) = NULL;

	CONNECT_REQ_B1PROTOCOL(&CMSG) = 1;
	CONNECT_REQ_B2PROTOCOL(&CMSG) = 1; // 1
	CONNECT_REQ_B3PROTOCOL(&CMSG) = 0;

	bchaninfo[0] = 2;
	bchaninfo[1] = 0x0;
	bchaninfo[2] = 0x0;
	CONNECT_REQ_BCHANNELINFORMATION(&CMSG) = bchaninfo; // 0

        if ((error = _capi_put_cmsg(&CMSG))) {
    	    ast_log(LOG_ERROR,"error sending CONNECT_REQ (error=%#x)\n",error);
	    return error;
	} else {
	    if (option_verbose > 5) {
		ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_REQ MN =%#x\n",CMSG.Messagenumber);
	    }
	}

	i->state = CAPI_STATE_CONNECTPENDING;

	ast_setstate(c, AST_STATE_DIALING);

	// XXX fixme, not nice:
/*	if (i->controller > 0) {
	    capi_controllers[i->controller]->nfreebchannels--;
	} */

	// now we shall return .... the rest has to be done by handle_msg
    return 0;
}


static int capi_answer(struct ast_channel *c) {
    struct ast_capi_pvt *i = c->pvt->pvt;
    MESSAGE_EXCHANGE_ERROR  error;
    _cmsg		    CMSG;
    char buf[AST_MAX_EXTENSION];
    char *dnid;
    
    if (i->isdnmode && (strlen(i->incomingmsn)<strlen(i->dnid)))
	dnid = i->dnid + strlen(i->incomingmsn);
    else
	dnid = i->dnid;

    CONNECT_RESP_HEADER(&CMSG, ast_capi_ApplID, i->MessageNumber, 0);
    CONNECT_RESP_PLCI(&CMSG) = i->PLCI;
    CONNECT_RESP_REJECT(&CMSG) = 0;
    buf[0] = strlen(dnid)+2;
    buf[1] = 0x0;
    buf[2] = 0x80;
    strncpy(&buf[3],dnid,sizeof(buf)-4);
    CONNECT_RESP_CONNECTEDNUMBER(&CMSG) = buf;
    CONNECT_RESP_CONNECTEDSUBADDRESS(&CMSG) = NULL;
    CONNECT_RESP_LLC(&CMSG) = NULL;
    CONNECT_RESP_B1PROTOCOL(&CMSG) = 1;
    CONNECT_RESP_B2PROTOCOL(&CMSG) = 1;
    CONNECT_RESP_B3PROTOCOL(&CMSG) = 0;

    if (option_verbose > 3)
	ast_verbose(VERBOSE_PREFIX_3 "CAPI Answering for MSN %s\n", dnid);
    if ((error = _capi_put_cmsg(&CMSG)) != 0) {
	return -1;	
    } else {
	if (option_verbose > 5) {
	    if (capidebug) 
		ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_RESP PLCI = %#x DNID = %s\n",i->PLCI,i->dnid);
	}
    }
    
    i->state = CAPI_STATE_ANSWERING;
    i->doB3 = AST_CAPI_B3_DONT;
    i->outgoing = 0;
    i->earlyB3 = -1;

    return 0;
}

struct ast_frame *capi_read(struct ast_channel *c) 
{
	struct ast_capi_pvt *i = c->pvt->pvt;
	int readsize = 0;

	if ((i->state == CAPI_STATE_REMOTE_HANGUP)) {
    	    ast_log(LOG_ERROR,"this channel is not connected\n");
	    return NULL;
	}
	if (i->state == CAPI_STATE_ONHOLD) {
    	    i->fr.frametype = AST_FRAME_NULL;
	    return &i->fr;
	}
	
	if (i == NULL) {
    	    ast_log(LOG_ERROR,"channel has no interface\n");
	    return NULL;
	}
	i->fr.frametype = AST_FRAME_NULL;
	i->fr.subclass = 0;
#ifdef UNSTABLE_CVS
	i->fr.delivery.tv_sec = 0;
	i->fr.delivery.tv_usec = 0;
#endif	
	readsize = read(i->fd,&i->fr,sizeof(struct ast_frame));
	if (readsize != sizeof(struct ast_frame)) {
	    ast_log(LOG_ERROR,"did not read a whole frame\n");
	}
	if (i->fr.frametype == AST_FRAME_VOICE) {
	    readsize = read(i->fd,i->fr.data,i->fr.datalen);
	    if (readsize != i->fr.datalen) {
		ast_log(LOG_ERROR,"did not read whole frame data\n");
	    }
	}
	i->fr.mallocd = 0;	
	if (i->fr.frametype == AST_FRAME_NULL) {
	    return NULL;
	}
	return &i->fr;
}

int capi_write(struct ast_channel *c, struct ast_frame *f)
{
	struct ast_capi_pvt *i = c->pvt->pvt;
	_cmsg CMSG;
	MESSAGE_EXCHANGE_ERROR  error;
	int j=0;
	char buf[1000];
	struct ast_frame *fsmooth;
#ifdef CAPI_ES
	int txavg=0;
#endif

#ifndef NEVER_EVER_EARLY_B3_CONNECTS
	// dont send audio to the local exchange!
	if (i->earlyB3 == 1 || !i->NCCI) {
	    return 0;
	}
#endif

	if (!i) {
	    ast_log(LOG_ERROR,"channel has no interface\n");
	    return -1;
	} 

	if (f->frametype == AST_FRAME_NULL) {
	    return 0;
	}
	if (f->frametype == AST_FRAME_DTMF) {
	    ast_log(LOG_ERROR,"dtmf frame should be written\n");
	    return 0;
	}
	if (f->frametype != AST_FRAME_VOICE) {
	    ast_log(LOG_ERROR,"not a voice frame\n");
	    return -1;
	}
	if (f->subclass != capi_capability) {
	    ast_log(LOG_ERROR,"dont know how to write subclass %d\n",f->subclass);
	    return -1;
	}
//	ast_log(LOG_NOTICE,"writing frame %d %d\n",f->frametype,f->subclass);

    if (ast_smoother_feed(i->smoother, f)!=0) {
        ast_log(LOG_ERROR,"failed to fill smoother\n");
        return -1;
    }

    fsmooth=ast_smoother_read(i->smoother);
    while(fsmooth != NULL) {
        DATA_B3_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
        DATA_B3_REQ_NCCI(&CMSG) = i->NCCI;
        DATA_B3_REQ_DATALENGTH(&CMSG) = fsmooth->datalen;
        DATA_B3_REQ_FLAGS(&CMSG) = 0; 

        if (ast_mutex_lock(&capi_send_buffer_lock)) {
            ast_log(LOG_WARNING,"Unable to lock B3 send buffer!\n");
            return -1;
        }
#ifndef CAPI_ES
#ifdef CAPI_GAIN
        for (j=0;j<fsmooth->datalen;j++) {
	    buf[j] = i->g.txgains[reversebits[((unsigned char *)fsmooth->data)[j]]]; 
        }
#else
        for (j=0;j<fsmooth->datalen;j++) {
	    buf[j] = reversebits[ ((unsigned char *)fsmooth->data)[j] ]; 
        }
#endif
#else
	if ((i->doES == 1)) {
    	    for (j=0;j<fsmooth->datalen;j++) {
		buf[j] = reversebits[ ((unsigned char *)fsmooth->data)[j] ]; 
		txavg += abs( capiXLAW2INT(reversebits[ ((unsigned char*)fsmooth->data)[j]]) );
    	    }
	    txavg = txavg/j;
	    for(j=0;j<ECHO_TX_COUNT-1;j++) {
		i->txavg[j] = i->txavg[j+1];
	    }
	    i->txavg[ECHO_TX_COUNT-1] = txavg;
	    
//	    ast_log(LOG_NOTICE,"txavg = %d\n",txavg);
	} else {
#ifdef CAPI_GAIN
    	    for (j=0;j<fsmooth->datalen;j++) {
		buf[j] = i->g.txgains[reversebits[((unsigned char *)fsmooth->data)[j]]]; 
    	    }
#else
    	    for (j=0;j<fsmooth->datalen;j++) {
		buf[j] = reversebits[ ((unsigned char *)fsmooth->data)[j] ]; 
	    }
#endif
	}
#endif
    
        DATA_B3_REQ_DATAHANDLE(&CMSG) = capi_send_buffer_handle;
    	memcpy((char *)&capi_send_buffer[(capi_send_buffer_handle % AST_CAPI_MAX_B3_BLOCKS) * AST_CAPI_MAX_B3_BLOCK_SIZE],&buf,fsmooth->datalen);
        DATA_B3_REQ_DATA(&CMSG) = (unsigned char *)&capi_send_buffer[(capi_send_buffer_handle % AST_CAPI_MAX_B3_BLOCKS) * AST_CAPI_MAX_B3_BLOCK_SIZE];
    	capi_send_buffer_handle++;
    
        if (ast_mutex_unlock(&capi_send_buffer_lock)) {
            ast_log(LOG_WARNING,"Unable to unlock B3 send buffer!\n");
            return -1;
        }


#ifdef CAPI_SYNC    
    ast_mutex_lock(&i->lockB3in);
    if ((i->B3in >= 1) && (i->B3in <= AST_CAPI_MAX_B3_BLOCKS)) {
	ast_mutex_unlock(&i->lockB3in);
        if ((error = _capi_put_cmsg(&CMSG)) != 0) {
            ast_log(LOG_ERROR,"error sending DATA_B3_REQ (error=%#x, datalen=%d) B3in=%d\n",error,fsmooth->datalen,i->B3in);
//    	ast_log(LOG_NOTICE,"f: timelen %d b = %d MN = %d \n",fsmooth->timelen,b,CMSG.Messagenumber);
    	} else {
      	    if (option_verbose > 5) {
 		if (capidebug) 
             	    ast_verbose(VERBOSE_PREFIX_4 "sent DATA_B3_REQ (NCCI=%#x) (%d bytes)\n",i->NCCI,fsmooth->datalen);
      	    }
    	}
	i->B3in--;
    } else {
	if (i->B3in > 0) i->B3in--;
	ast_mutex_unlock(&i->lockB3in);
    }
#else
        if ((error = _capi_put_cmsg(&CMSG)) != 0) {
            ast_log(LOG_ERROR,"error sending DATA_B3_REQ (error=%#x, datalen=%d)\n",error,fsmooth->datalen);
//    	ast_log(LOG_NOTICE,"f: timelen %d b = %d MN = %d \n",fsmooth->timelen,b,CMSG.Messagenumber);
    	} else {
      	    if (option_verbose > 5) {
 		if (capidebug) 
             	    ast_verbose(VERBOSE_PREFIX_4 "sent DATA_B3_REQ (NCCI=%#x) (%d bytes)\n",i->NCCI,fsmooth->datalen);
      	    }
    	}
#endif
    
//	ast_frfree(fsmooth);

        fsmooth=ast_smoother_read(i->smoother);
    }
	return 0;
}

static int capi_fixup(struct ast_channel *oldchan, struct ast_channel *newchan)
{
	struct ast_capi_pvt *p = newchan->pvt->pvt;
	p->owner = newchan;
	return 0;
}

int capi_indicate(struct ast_channel *c,int condition) {
    return -1;
}

int capi_bridge(struct ast_channel *c0, struct ast_channel *c1, int flags, struct ast_frame **fo, struct ast_channel **rc) {
    return -1;
}


struct ast_channel *capi_new(struct ast_capi_pvt *i,int state) {
    struct ast_channel *tmp;
    int fmt;

    tmp = ast_channel_alloc(1);
    if (tmp != NULL) {
//	ast_log(LOG_NOTICE,"allocated channel for PLCI=%#x!\n",i->PLCI);
	snprintf(tmp->name,sizeof(tmp->name),"CAPI[contr%d/%s]/%d",i->controller,i->dnid,capi_counter++);
	tmp->type = type;
	tmp->nativeformats = capi_capability;
	ast_setstate(tmp,state);
	tmp->fds[0] = i->fd;
	i->smoother = ast_smoother_new(AST_CAPI_MAX_B3_BLOCK_SIZE);
	if (i->smoother == NULL) {
	    ast_log(LOG_ERROR, "smoother NULL!\n");
	}
	i->fr.frametype = 0;
	i->fr.subclass = 0;
#ifdef UNSTABLE_CVS
	i->fr.delivery.tv_sec = 0;
	i->fr.delivery.tv_usec = 0;
#endif
	i->state = CAPI_STATE_DISCONNECTED;
	i->CLIR = 0;
	i->calledPartyIsISDN = 0; // let's be pessimistic
	i->earlyB3 = -1;
	i->doB3 = AST_CAPI_B3_DONT;
	i->outgoing = 0;
	i->onholdPLCI = 0;
#ifdef CAPI_SYNC
	i->B3in = 0;
	ast_mutex_init(&i->lockB3in);
#endif		
#ifdef CAPI_ES
	memset(i->txavg,0,ECHO_TX_COUNT);
#endif

#ifndef FORCE_SOFTWARE_DTMF
	    if (i->doDTMF == 1) {
#endif
		i->vad = ast_dsp_new();
		ast_dsp_set_features(i->vad,DSP_FEATURE_DTMF_DETECT);
#ifndef FORCE_SOFTWARE_DTMF
	    }
#endif

	if (tmp->pvt == NULL) {
	    free(tmp);
	    return NULL;
	}
	tmp->pvt->pvt = i;

	tmp->callgroup = i->callgroup;
	tmp->pvt->call = capi_call;
	tmp->pvt->fixup = capi_fixup;
	tmp->pvt->indicate = capi_indicate;
	tmp->pvt->bridge = capi_bridge;
	tmp->pvt->answer = capi_answer;
	tmp->pvt->hangup = capi_hangup;
	tmp->pvt->read = capi_read;
	tmp->pvt->write = capi_write;
	tmp->pvt->send_digit = capi_send_digit;
	tmp->nativeformats = capi_capability;
	fmt = ast_best_codec(tmp->nativeformats);
//	fmt = capi_capability;
	tmp->readformat = fmt;
	tmp->writeformat = fmt;
	tmp->pvt->rawreadformat = fmt;
	tmp->pvt->rawwriteformat = fmt;
	strncpy(tmp->context,i->context,sizeof(tmp->context)-1);
	tmp->callerid = strdup(i->cid);
	tmp->dnid = strdup(i->dnid);
	strncpy(tmp->exten,i->dnid,sizeof(tmp->exten)-1);
	strncpy(tmp->accountcode,i->accountcode,sizeof(tmp->accountcode)-1);
	i->owner = tmp;
	ast_mutex_lock(&usecnt_lock);
	usecnt++;
	ast_mutex_unlock(&usecnt_lock);
	ast_update_use_count();
	if (state != AST_STATE_DOWN) {
	    // we are alerting (phones ringing)
	    if (state == AST_STATE_RING)
		capi_alert(tmp);
	    if (ast_pbx_start(tmp)) {
		ast_log(LOG_ERROR,"Unable to start pbx on channel!\n");
		ast_hangup(tmp);
		tmp = NULL;
	    } else {
		if (option_verbose > 2) {
    		    ast_verbose(VERBOSE_PREFIX_3 "started pbx on channel (callgroup=%d)!\n",tmp->callgroup);
		}
	    }
	}
    } else {
	ast_log(LOG_ERROR,"Unable to allocate channel!\n");
    }
    return tmp;
}


struct ast_channel *capi_request(char *type, int format, void *data)
{
	struct ast_capi_pvt *i;
	struct ast_channel *tmp = NULL;
	char *dest,*omsn,*msn;
	char buffer[AST_MAX_EXTENSION];
	int c=0;
	char *stringp,tmpstr[AST_MAX_EXTENSION];

	if (option_verbose > 1) {
	    if (capidebug)
    		ast_verbose(VERBOSE_PREFIX_3 "data = %s\n",(char *)data);
	}
	strncpy(buffer,(char *)data,sizeof(buffer)-1);

	omsn = strtok(buffer, ":");
	dest = strtok(NULL, ":");

	if (option_verbose > 1) {
	    if (capidebug)
		ast_verbose(VERBOSE_PREFIX_3 "capi request omsn = %s\n",omsn);
	}
	
	if (((char *)omsn)[0] == '@') {
	    omsn++;
	}

	ast_mutex_lock(&iflock);
	i = iflist;
	while (i) {
	    strncpy(tmpstr,i->msn,sizeof(tmpstr)-1);
	    stringp=tmpstr;
	    msn = strsep(&stringp,",");
	    while (msn != NULL) {
		if (!strcmp(omsn,msn) ||
	    	    (i->isdnmode && (strlen(msn)<=strlen(omsn)) && !strncmp(omsn, msn,strlen(msn)))) {
		    // unused channel
		    if (!i->owner) {
			if (option_verbose > 1) {
			    if (capidebug)
				ast_verbose(VERBOSE_PREFIX_2 "found capi with omsn = %s\n",omsn);
			}

			ast_mutex_lock(&contrlock);
			for (c=1;c<=AST_CAPI_MAX_CONTROLLERS;c++) {
			    if (i->controllers & (1 << c)) { 
				if (capi_controllers[c]->nfreebchannels > 0) {
				    i->controller = c;
				    strncpy(i->dnid,omsn,sizeof(i->dnid)-1);
			    	    tmp = capi_new(i, AST_STATE_DOWN);
			    	    i->PLCI = -1;
				    i->datahandle = 0;
				    i->outgoing = 1;	// this is an outgoing line
				    i->earlyB3 = -1;
	    			    // capi_detect_dtmf(tmp,1);
				    ast_mutex_unlock(&contrlock);
				    ast_mutex_unlock(&iflock);
				    return tmp;
				}
			    }
			}
	    		ast_mutex_unlock(&contrlock);
			ast_log(LOG_ERROR,"no free b channel on controllers (map=%#x)\n",(int) i->controllers);
		    }
		}
		msn = strsep(&stringp,",");
	    }
	    i = i->next;
	}
	ast_mutex_unlock(&iflock);
	ast_log(LOG_NOTICE,"didn't find capi device with outgoing msn = %s. you should check your config!\n",omsn);
	return NULL;
}


struct capi_pipe *find_pipe(int PLCI,int MN) {
    struct capi_pipe *p;
    // find a pipe by PLCI or by MessageNumber (in case this is a CONNECT_CONF)
    ast_mutex_lock(&pipelock);
    p = pipelist;
    if ((p == NULL) && (capi_last_plci != PLCI)){
	if (capidebug) {
	ast_log(LOG_NOTICE,"PLCI doesnt match last pipe (PLCI = %#x)\n",PLCI);
	}
	ast_mutex_unlock(&pipelock);
	return NULL;
    }
    while(p != NULL) {
	if ((p->PLCI == PLCI) || ( (p->PLCI == -1) && (p->i->MessageNumber == MN) ) ){
	    ast_mutex_unlock(&pipelock);
	    return p;
	}
	p = p->next;
    }
    if (capidebug) {
	ast_log(LOG_ERROR,"unable to find a pipe for PLCI = %#x MN = %#x\n",PLCI,MN);
    }
    ast_mutex_unlock(&pipelock);
    return NULL;
}

int pipe_frame(struct capi_pipe *p,struct ast_frame *f) {
	fd_set wfds;
	int written=0;
	struct timeval tv;
	FD_ZERO(&wfds);
	FD_SET(p->fd,&wfds);
	tv.tv_sec = 0;
	tv.tv_usec = 10;
	if ((f->frametype == AST_FRAME_VOICE) && (p->i->doDTMF == 1) && (p->i->vad != NULL)) {
#ifdef UNSTABLE_CVS
	    f = ast_dsp_process(p->c,p->i->vad,f);
#else
	    f = ast_dsp_process(p->c,p->i->vad,f, 0);
#endif
	    if (f->frametype == AST_FRAME_NULL) {
		return 0;
	    }
	}
	// we dont want the monitor thread to block
	if (select(p->fd + 1,NULL,&wfds,NULL,&tv) == 1) {
	    written = write(p->fd,f,sizeof(struct ast_frame));
	    if (written < (signed int) sizeof(struct ast_frame)) {
		ast_log(LOG_ERROR,"wrote %d bytes instead of %d\n",written,sizeof(struct ast_frame));
		return -1;
	    }
	    if (f->frametype == AST_FRAME_VOICE) {
		written = write(p->fd,f->data,f->datalen);
		if (written < f->datalen) {
		    ast_log(LOG_ERROR,"wrote %d bytes instead of %d\n",written,f->datalen);
		    return -1;
		}
	    }
	} else {
	    return 0;
	}
	return -1;
}

static int search_did(struct ast_channel *c)
{
    // Returns 
    // -1 = Failure 
    //  0 = Match
    //  1 = possible match 
    struct ast_capi_pvt *i = c->pvt->pvt;
    char *exten;
    
    if (strlen(i->dnid)<strlen(i->incomingmsn))
	return -1;
	
//    exten = i->dnid + strlen(i->incomingmsn);
    exten = i->dnid;

    if (ast_exists_extension(NULL, c->context, exten, 1, NULL)) {
	    c->priority = 1;
	    strncpy(c->exten, exten, sizeof(c->exten) - 1);
	return 0;
    }

    if (ast_canmatch_extension(NULL, c->context, exten, 1, NULL)) {
	return 1;
    }


    return -1;
}

int pipe_msg(int PLCI,_cmsg *CMSG) {
	struct capi_pipe *p;
	_cmsg CMSG2;
	MESSAGE_EXCHANGE_ERROR  error;
	struct ast_frame fr;
	char b3buf[1024];
	int j;
	int b3len=0;
	char dtmf;
	unsigned dtmflen;
#ifdef CAPI_ES
	int rxavg = 0;
	int txavg = 0;
#endif    

	p = find_pipe(PLCI,CMSG->Messagenumber);
	if (p == NULL) {
	    if (IS_DISCONNECT_IND(CMSG)) {
		DISCONNECT_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber , 0);
		DISCONNECT_RESP_PLCI(&CMSG2) = PLCI;
		if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
		    ast_log(LOG_NOTICE, "error sending DISCONNECT_RESP PLCI=%#x\n",PLCI);
		} else {
		    if (option_verbose > 5) {
			if (capidebug)
    			    ast_verbose(VERBOSE_PREFIX_4 "sent DISCONNECT_RESP PLCI=%#x\n",PLCI);
		    }
		}	    
		return 0;
	    }
	    if (capidebug) {
		ast_log(LOG_NOTICE,"%s",capi_cmsg2str(CMSG));
	    }
	    return -1;
	}

	if (CMSG != NULL) {
	    switch (CMSG->Subcommand) {
		case CAPI_IND:
		    switch (CMSG->Command) {
			case CAPI_DISCONNECT_B3:
//			    ast_log(LOG_NOTICE,"DISCONNECT_B3_IND\n");

			    DISCONNECT_B3_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber, 0);
			    DISCONNECT_B3_RESP_NCCI(&CMSG2) = DISCONNECT_B3_IND_NCCI(CMSG);

			    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
				ast_log(LOG_NOTICE, "error sending DISCONNECT_B3_RESP NCCI=%#x\n",(int)DISCONNECT_B3_IND_NCCI(CMSG));
			    } else {
				if (option_verbose > 5) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent DISCONNECT_B3_RESP NCCI=%#x\n",(int)DISCONNECT_B3_IND_NCCI(CMSG));
				}
			    }
			    if (p->i->state == CAPI_STATE_BCONNECTED) {
				// passive disconnect
				p->i->state = CAPI_STATE_CONNECTED;
			    } else
			    if (p->i->state == CAPI_STATE_DISCONNECTING) {
				// active disconnect
				memset(&CMSG2,0,sizeof(_cmsg));
				DISCONNECT_REQ_HEADER(&CMSG2, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
				DISCONNECT_REQ_PLCI(&CMSG2) = PLCI;

				if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
				    ast_log(LOG_NOTICE, "error sending DISCONNECT_REQ PLCI=%#x\n",PLCI);
			    	} else {
				    if (option_verbose > 5) {
					if (capidebug)
    					    ast_verbose(VERBOSE_PREFIX_4 "sent DISCONNECT_REQ PLCI=%#x\n",PLCI);
				    }
				}				
			    } else 
			    if (p->i->state == CAPI_STATE_ONHOLD) {
				    // no hangup
			    }
			    ast_mutex_lock(&contrlock);
			    if (p->i->controller > 0) {
				capi_controllers[p->i->controller]->nfreebchannels++;
			    }
			    ast_mutex_unlock(&contrlock);
			    break;
			case CAPI_DISCONNECT:
//			    ast_log(LOG_NOTICE,"DISCONNECT_IND\n");
			    DISCONNECT_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber , 0);
			    DISCONNECT_RESP_PLCI(&CMSG2) = PLCI;
/*			    if (p->i->controller > 0) {
				capi_controllers[p->i->controller]->nfreebchannels++;
			    } */

			    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
				ast_log(LOG_NOTICE, "error sending DISCONNECT_RESP PLCI=%#x\n",PLCI);
			    } else {
				if (option_verbose > 5) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent DISCONNECT_RESP PLCI=%#x\n",PLCI);
				}
			    }
			    
			    if (PLCI == p->i->onholdPLCI) {
				// the caller onhold hung up (or ECTed away)
				p->i->onholdPLCI = 0;
				remove_pipe(PLCI);
				return 0;
			    }
			    
			    if (p->i->state == CAPI_STATE_DID) {
				if ((p->c) != NULL) {
				    ast_hangup(p->c);
				} else {    
				    ast_log(LOG_WARNING, "unable to hangup channel on DID. Channel is NULL.\n");
				}
				return 0;				
			    }
			    
			    p->i->state = CAPI_STATE_DISCONNECTED;

			    fr.frametype = AST_FRAME_CONTROL;
			    if (DISCONNECT_IND_REASON(CMSG) == 0x34a2) {
				fr.subclass = AST_CONTROL_BUSY;
			    } else {
				fr.frametype = AST_FRAME_NULL;
			    }
			    fr.datalen = 0;
			    if (pipe_frame(p,(struct ast_frame *)&fr) == -1) {
		// printf("STATE = %#x\n",p->i->state);
				// in this case * did not read our hangup control frame
				// so we must hangup the channel!
				if ( (p->i->state != CAPI_STATE_DISCONNECTED) && (ast_check_hangup(p->c) == 0)) {
				    if (option_verbose > 1) {
    					ast_verbose(VERBOSE_PREFIX_3 "soft hangup by capi\n");
				    }
				    ast_softhangup(p->c,AST_CONTROL_HANGUP);
				} else {
				    // dont ever hangup while hanging up!
//				    ast_log(LOG_NOTICE,"no soft hangup by capi\n");
				}
				return -1;
			    } else {
				return 0;
			    }

/*			    fr.frametype = AST_FRAME_NULL;
			    fr.datalen = 0;
			    pipe_frame(p,(struct ast_frame *)&fr); */
			    break;
			case CAPI_DATA_B3:

			    memcpy(&b3buf[AST_FRIENDLY_OFFSET],(char *)DATA_B3_IND_DATA(CMSG),DATA_B3_IND_DATALENGTH(CMSG));
			    b3len = DATA_B3_IND_DATALENGTH(CMSG);
			
			    // send a DATA_B3_RESP very quickly to free the buffer in capi
			    DATA_B3_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber,0);
			    DATA_B3_RESP_NCCI(&CMSG2) = DATA_B3_IND_NCCI(CMSG);
			    DATA_B3_RESP_DATAHANDLE(&CMSG2) = DATA_B3_IND_DATAHANDLE(CMSG);
			    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
				ast_log(LOG_ERROR,"error sending DATA_B3_RESP (error=%#x)\n",error);
			    } else {
				if (option_verbose > 6) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent DATA_B3_RESP (NCCI=%#x)\n",(int)DATA_B3_IND_NCCI(CMSG));
				}
			    }
#ifdef CAPI_SYNC
			    ast_mutex_lock(&p->i->lockB3in);
			    p->i->B3in++;
			    if (p->i->B3in > AST_CAPI_MAX_B3_BLOCKS) p->i->B3in = AST_CAPI_MAX_B3_BLOCKS;
			    ast_mutex_unlock(&p->i->lockB3in);
#endif			    
#ifdef CAPI_ES
			    if ((p->i->doES == 1)) {
				for (j=0;j<b3len;j++) {
				    b3buf[AST_FRIENDLY_OFFSET + j] = reversebits[(unsigned char)b3buf[AST_FRIENDLY_OFFSET + j]]; 
				    rxavg += abs(capiXLAW2INT( reversebits[(unsigned char)b3buf[AST_FRIENDLY_OFFSET + j]]));
				}
				rxavg = rxavg/j;
				for(j=0;j<ECHO_EFFECTIVE_TX_COUNT;j++) {
				    txavg += p->i->txavg[j];
 				}
				txavg = txavg/j;
			    
				if( (txavg/ECHO_TXRX_RATIO) > rxavg) { 
#ifdef CAPI_ULAW
				    memset(&b3buf[AST_FRIENDLY_OFFSET],255,b3len);
#else
				    memset(&b3buf[AST_FRIENDLY_OFFSET],84,b3len);
#endif
				    if (capidebug) {
				    	ast_log(LOG_NOTICE,"SUPPRESSING ECHOrx=%d, tx=%d\n",rxavg,txavg);
				    }
				}
			    } else {
#ifdef CAPI_GAIN
				for (j=0;j<b3len;j++) {
				    b3buf[AST_FRIENDLY_OFFSET + j] = reversebits[p->i->g.rxgains[(unsigned char)b3buf[AST_FRIENDLY_OFFSET + j]]]; 
				}
#else
				for (j=0;j<b3len;j++) {
				    b3buf[AST_FRIENDLY_OFFSET + j] = reversebits[(unsigned char)b3buf[AST_FRIENDLY_OFFSET + j]]; 
				}
#endif
			    }
#else

#ifdef CAPI_GAIN
			    for (j=0;j<b3len;j++) {
				b3buf[AST_FRIENDLY_OFFSET + j] = reversebits[p->i->g.rxgains[(unsigned char)b3buf[AST_FRIENDLY_OFFSET + j]]]; 
			    }
#else
			    for (j=0;j<b3len;j++) {
				b3buf[AST_FRIENDLY_OFFSET + j] = reversebits[(unsigned char)b3buf[AST_FRIENDLY_OFFSET + j]]; 
			    }
#endif

#endif
			    // just being paranoid ...
		/*	    if (p->c->_state != AST_STATE_UP) {
				ast_setstate(p->c,AST_STATE_UP);
			    } */
			    
			    fr.frametype = AST_FRAME_VOICE;
			    fr.subclass = capi_capability;
			    fr.data = (char *)&b3buf[AST_FRIENDLY_OFFSET];
			    fr.datalen = b3len;
			    fr.samples = b3len;
			    fr.offset = AST_FRIENDLY_OFFSET;
			    fr.mallocd = 0;
#ifdef UNSTABLE_CVS
			    fr.delivery.tv_sec = 0;
			    fr.delivery.tv_usec = 0;
#endif
			    fr.src = NULL;
    			//	    ast_verbose(VERBOSE_PREFIX_3 "DATA_B3_IND (len=%d) fr.datalen=%d fr.subclass=%d\n",(int)DATA_B3_IND_DATALENGTH(CMSG),fr.datalen,fr.subclass);
			    return pipe_frame(p,(struct ast_frame *)&fr);
			    break;
			case CAPI_FACILITY:
			    if (FACILITY_IND_FACILITYSELECTOR(CMSG) == 0x0001) {
			// DTMF received
				if (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[0] != (0xff)) {
				    dtmflen = FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[0];
				    FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG) += 1;
				} else {
				    dtmflen = ((__u16 *) (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG) + 1))[0];
				    FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG) += 3;
				}
				if (dtmflen == 1) {
				    dtmf = (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG))[0];
    				    fr.frametype = AST_FRAME_DTMF;
				    fr.subclass = dtmf;
				    if (option_verbose > 1) {
					if (capidebug)
    					    ast_verbose(VERBOSE_PREFIX_3 "c_dtmf = %c\n",dtmf);
				    }
				    pipe_frame(p,(struct ast_frame *)&fr);
				} 
			    }
			    if (FACILITY_IND_FACILITYSELECTOR(CMSG) == 0x0003) {
			    // sservices
			/*	 ast_log(LOG_NOTICE,"FACILITY_IND PLCI = %#x\n",(int)FACILITY_IND_PLCI(CMSG));
				 ast_log(LOG_NOTICE,"%#x\n",FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[0]);
				 ast_log(LOG_NOTICE,"%#x\n",FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[1]);
				 ast_log(LOG_NOTICE,"%#x\n",FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[2]);
				 ast_log(LOG_NOTICE,"%#x\n",FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[3]);
				 ast_log(LOG_NOTICE,"%#x\n",FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[4]);
				 ast_log(LOG_NOTICE,"%#x\n",FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[5]);  */
				// RETRIEVE
				if ( (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[1] == 0x3) && (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[3] == 0x2)) {
				    p->i->state = CAPI_STATE_CONNECTED;
				    p->i->PLCI = p->i->onholdPLCI;
				    p->i->onholdPLCI = 0;
				}
				if ( (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[1] == 0x2) && (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[3] == 0x2)) {
				    if ((FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[5] != 0) && (FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[4] != 0)) { 
					// reason != 0x0000 == problem
					p->i->onholdPLCI = 0;
					p->i->state = CAPI_STATE_ONHOLD;
    					ast_log(LOG_WARNING, "unable to put PLCI=%#x onhold, REASON = %#x%#x, maybe you need to subscribe for this...\n",(int)FACILITY_IND_PLCI(CMSG),FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[5],FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG)[4]);
				    } else {
					// reason = 0x0000 == call on hold
					p->i->state = CAPI_STATE_ONHOLD;
					if (capidebug) 
    					    ast_log(LOG_NOTICE, "PLCI=%#x put onhold\n",(int)FACILITY_IND_PLCI(CMSG));
				    }
				}
			    }
			    
			    error = FACILITY_RESP(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber,FACILITY_IND_PLCI(CMSG),FACILITY_IND_FACILITYSELECTOR(CMSG),FACILITY_IND_FACILITYINDICATIONPARAMETER(CMSG));

			    if (error != 0) {
			        ast_log(LOG_ERROR,"error sending FACILITY_RESP (error=%#x)\n",error);
			    } else {
				if (option_verbose > 5) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent FACILITY_RESP (PLCI=%#x)\n",(int)FACILITY_IND_PLCI(CMSG));
				}
			    }
			    break;
			case CAPI_INFO:
			    // ast_log(LOG_ERROR,"INFO_IND PLCI=%#x INFO# = %#x\n",PLCI,INFO_IND_INFONUMBER(CMSG));

			    memset(&CMSG2,0,sizeof(_cmsg));
			    error = INFO_RESP(&CMSG2,ast_capi_ApplID,CMSG->Messagenumber,PLCI);
			    if (error != 0) {
			    	ast_log(LOG_ERROR,"error sending INFO_RESP (error=%#x)\n",error);
			    	return -1;
			    } else {
				if (option_verbose > 5) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent INFO_RESP (PLCI=%#x)\n",PLCI);
				}
			    }
/*			    if ((INFO_IND_INFONUMBER(CMSG) >> 8) == 0x00) {
				ast_log(LOG_ERROR,"%#x\n",INFO_IND_INFOELEMENT(CMSG)[0]);
				ast_log(LOG_ERROR,"%#x\n",INFO_IND_INFOELEMENT(CMSG)[1]);
				ast_log(LOG_ERROR,"%#x\n",INFO_IND_INFOELEMENT(CMSG)[2]);
				ast_log(LOG_ERROR,"%#x\n",INFO_IND_INFOELEMENT(CMSG)[3]);
			    } */
#ifndef NEVER_EVER_EARLY_B3_CONNECTS
			    if ((INFO_IND_INFONUMBER(CMSG) == 0x001e) && (p->i->doB3 != AST_CAPI_B3_DONT) && (p->i->earlyB3 == -1) && (p->i->state != CAPI_STATE_BCONNECTED)){
				// ETSI 300 102-1 Progress Indicator
				// we do early B3 Connect
                                if(INFO_IND_INFOELEMENT(CMSG)[0] >= 2) {
				    if(INFO_IND_INFOELEMENT(CMSG)[2] & 0x2) {
					p->i->calledPartyIsISDN = 0;
					// ast_log(LOG_NOTICE,"A N A L O G \n");
				    } else {
					p->i->calledPartyIsISDN = 1;
					// ast_log(LOG_NOTICE,"I S D N\n");
				    }
				    if(INFO_IND_INFOELEMENT(CMSG)[2] & 0x88) {
					 // in-band info available
					p->i->earlyB3 = 1;
					memset(&CMSG2,0,sizeof(_cmsg));
					CONNECT_B3_REQ_HEADER(&CMSG2, ast_capi_ApplID, ast_capi_MessageNumber++,0);
					CONNECT_B3_REQ_PLCI(&CMSG2) = PLCI;
					if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
			    		    ast_log(LOG_ERROR,"error sending early CONNECT_B3_REQ (error=%#x)\n",error);
			    		    return -1;
					} else {
					    if (option_verbose > 1) {
						if (capidebug)
    						    ast_verbose(VERBOSE_PREFIX_4 "sent early CONNECT_B3_REQ (PLCI=%#x)\n",PLCI);
					    }
					}
				    }
				}
			    }
			    // DISCONNECT
			    if ((INFO_IND_INFONUMBER(CMSG) == 0x8045) && (PLCI == p->i->onholdPLCI)) {
				// the caller onhold hung up (or ECTed away)
				// send a disconnect_req , we cannot hangup the channel here!!!
				memset(&CMSG2,0,sizeof(_cmsg));
				DISCONNECT_REQ_HEADER(&CMSG2, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
				DISCONNECT_REQ_PLCI(&CMSG2) = p->i->onholdPLCI;

				if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
				    ast_log(LOG_NOTICE, "error sending DISCONNECT_REQ PLCI=%#x\n",PLCI);
			    	} else {
				    if (option_verbose > 1) {
					if (capidebug)
    					    ast_verbose(VERBOSE_PREFIX_4 "sent DISCONNECT_REQ for onholdPLCI=%#x\n",PLCI);
				    }
				}				
				return 0;
			    }

			    // case 1: B3 on success or no B3 at all
			    if ((INFO_IND_INFONUMBER(CMSG) == 0x8045) && (p->i->doB3 != AST_CAPI_B3_ALWAYS) && (p->i->outgoing == 1)) {
				p->i->earlyB3 = 0; // !!!
				fr.frametype = AST_FRAME_NULL;
				fr.datalen = 0;
				return pipe_frame(p,(struct ast_frame *)&fr);
			    }
			    // case 2: we are doing B3, and receive the 0x8045 after a successful call
			    if ((INFO_IND_INFONUMBER(CMSG) == 0x8045) && (p->i->doB3 != AST_CAPI_B3_DONT) && (p->i->earlyB3 == 0) && (p->i->outgoing == 1)) {
				fr.frametype = AST_FRAME_NULL;
				fr.datalen = 0;
				return pipe_frame(p,(struct ast_frame *)&fr);
			    }
			    // case 3: this channel is an incoming channel! the user hung up!
			    // it is much better to hangup now instead of waiting for a timeout and
			    // network caused DISCONNECT_IND!
			    if ((INFO_IND_INFONUMBER(CMSG) == 0x8045) && (p->i->outgoing == 0)) {
			    // ast_log(LOG_NOTICE,"case 3\n");
				fr.frametype = AST_FRAME_NULL;
				fr.datalen = 0;
				return pipe_frame(p,(struct ast_frame *)&fr);
			    }
			    // case 4 (a.k.a. the italian case): B3 always. call is unsuccessful
			    if ((INFO_IND_INFONUMBER(CMSG) == 0x8045) && (p->i->doB3 == AST_CAPI_B3_ALWAYS) && (p->i->earlyB3 == -1) && (p->i->outgoing == 1)) {
				// wait for the 0x001e (PROGRESS), play audio and wait for a timeout from the network
				return 0;
			    }
#endif
			    // Handle DID digits
			    if ((INFO_IND_INFONUMBER(CMSG) == 0x0070) && p->i->isdnmode && (p->c != NULL)) {
				int search = -1;
				char name[AST_CHANNEL_NAME] = "";
				char *did;

				did = capi_number(INFO_IND_INFOELEMENT(CMSG),1);
				if (strcasecmp(p->i->dnid, did)) {
				    strncat(p->i->dnid, did, sizeof(p->i->dnid)-1);
				}
				
				snprintf(name,sizeof(name),"CAPI[contr%d/%s]/%d",p->i->controller,p->i->dnid,capi_counter++);
				ast_change_name(p->c, name);
				
				search = search_did(p->c);
				if (search != -1) {
				    if (!search) {
					ast_setstate(p->c, AST_STATE_RING);
					// we are alerting (phones ringing)
					capi_alert(p->c); // Do this here after pbx_start the Channel can be destroyed
					if (ast_pbx_start(p->c)) {
					    ast_log(LOG_ERROR,"Unable to start pbx on channel!\n");
					    ast_hangup(p->c);
					} else {
					    if (option_verbose > 2) {
						if (capidebug)
			        		    ast_verbose(VERBOSE_PREFIX_3 "started pbx on channel!\n");
					    }
					}
				    }
				} else {
				    ast_log(LOG_ERROR,"did not find device for msn = %s\n",p->i->dnid);
				    CONNECT_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber, 0);
				    CONNECT_RESP_PLCI(&CMSG2) = PLCI;
				    CONNECT_RESP_REJECT(&CMSG2) = 1; // ignore
				    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
					ast_log(LOG_ERROR,"error sending CONNECT_RESP for PLCI = %#x\n",(int)CONNECT_IND_PLCI(CMSG));
				    } else {
					if (option_verbose > 5) {
					    if (capidebug)
    		    				ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_RESP for PLCI = %#x\n",(int)CONNECT_IND_PLCI(CMSG));
					}
				    }
				    
				    return 0;
				}
			    }
			    if (INFO_IND_INFONUMBER(CMSG) == 0x8001) {
				fr.frametype = AST_FRAME_CONTROL;
				fr.subclass = AST_CONTROL_RINGING;
				return pipe_frame(p,(struct ast_frame *)&fr);
			    }
			    if (INFO_IND_INFONUMBER(CMSG) == 0x800d) {
				fr.frametype = AST_FRAME_CONTROL;
				fr.subclass = AST_CONTROL_PROGRESS;
				return pipe_frame(p,(struct ast_frame *)&fr);
			    }
			    if (INFO_IND_INFONUMBER(CMSG) == 0x74) {
				strncpy(p->i->owner->exten,capi_number(INFO_IND_INFOELEMENT(CMSG),3),sizeof(p->i->owner->exten)-1);
				strncpy(p->i->owner->dnid,capi_number(INFO_IND_INFOELEMENT(CMSG),3),sizeof(p->i->owner->dnid)-1);
				ast_log(LOG_NOTICE,"%s\n",capi_cmsg2str(CMSG));
			    }
			    if (INFO_IND_INFONUMBER(CMSG) == 0x28) {
			//	ast_sendtext(p->i->owner,capi_number(INFO_IND_INFOELEMENT(CMSG),0));
    			//	struct ast_frame ft = { AST_FRAME_TEXT, capi_number(INFO_IND_INFOELEMENT(CMSG),0), };
                        //    	ast_queue_frame(p->i->owner, &ft);
			//	ast_log(LOG_NOTICE,"%s\n",capi_number(INFO_IND_INFOELEMENT(CMSG),0));
			    }
			    break;
			case CAPI_CONNECT_ACTIVE:
//			    ast_log(LOG_NOTICE,"CONNECT_ACTIVE_IND PLCI=%#x\n",(int)CONNECT_ACTIVE_IND_PLCI(CMSG));
			    CONNECT_ACTIVE_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber,0);
			    CONNECT_ACTIVE_RESP_PLCI(&CMSG2) = PLCI;
			    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
			        ast_log(LOG_ERROR,"error sending CONNECT_ACTIVE_RESP (error=%#x)\n",error);
			        return -1;
			    } else {
				if (option_verbose > 5) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_ACTIVE_RESP (PLCI=%#x)\n",PLCI);
				}
			    }
			    // normal processing
			    if (p->i->earlyB3 != 1) {
				p->i->state = CAPI_STATE_CONNECTED;
			    
				// send a CONNECT_B3_REQ
				if (p->i->outgoing == 1) {
				    // outgoing call
				    memset(&CMSG2,0,sizeof(_cmsg));
				    CONNECT_B3_REQ_HEADER(&CMSG2, ast_capi_ApplID, ast_capi_MessageNumber++,0);
				    CONNECT_B3_REQ_PLCI(&CMSG2) = PLCI;
				    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
			    		ast_log(LOG_ERROR,"error sending CONNECT_B3_REQ (error=%#x)\n",error);
			    	        return -1;
				    } else {
					if (option_verbose > 1) {
					    if (capidebug)
    						ast_verbose(VERBOSE_PREFIX_3 "sent CONNECT_B3_REQ (PLCI=%#x)\n",PLCI);
					}
				    }
				} else {
				    // incoming call
				    // RESP already sent ... wait for CONNECT_B3_IND
//				    ast_log(LOG_NOTICE,"waiting for CONNECT_B3_IND\n");
				}
			    } else {
				// special treatment for early B3 connects
				p->i->state = CAPI_STATE_BCONNECTED;
				if (p->c->_state != AST_STATE_UP) {
				    ast_setstate(p->c,AST_STATE_UP);
				}
				p->i->earlyB3 = 0;	// not early anymore
				fr.frametype = AST_FRAME_CONTROL;
				fr.subclass = AST_CONTROL_ANSWER;
				fr.datalen = 0;
				return pipe_frame(p,(struct ast_frame *)&fr);
				
			    }
			    break;
			case CAPI_CONNECT_B3:
			    // then send a CONNECT_B3_RESP
			    memset(&CMSG2,0,sizeof(_cmsg));
			    CONNECT_B3_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber, 0);
			    CONNECT_B3_RESP_NCCI(&CMSG2) = CONNECT_B3_IND_NCCI(CMSG);
			    p->NCCI = CONNECT_B3_IND_NCCI(CMSG);
			    p->i->NCCI = p->NCCI;
			    CONNECT_B3_RESP_REJECT(&CMSG2) = 0;

			    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
				ast_log(LOG_ERROR,"error sending CONNECT_B3_RESP (error=%#x)\n",error);
				return -1;
			    } else {
				if (option_verbose > 5) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_B3_RESP (NCCI=%#x)\n",p->i->NCCI);
				}
			    }
		/*	    if (p->i->controller > 0) {
				capi_controllers[p->i->controller]->nfreebchannels--;
			    } */
			    break; 
			case CAPI_CONNECT_B3_ACTIVE:
//			    ast_log(LOG_NOTICE,"CONNECT_B3_ACTIVE_IND NCCI=%#x\n",p->i->NCCI);
			    // then send a CONNECT_B3__ACTIVERESP

			    CONNECT_B3_ACTIVE_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber, 0);
			    CONNECT_B3_ACTIVE_RESP_NCCI(&CMSG2) = p->i->NCCI;

			    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
				ast_log(LOG_ERROR,"error sending CONNECT_B3_ACTIVE_RESP (error=%#x)\n",error);
			        return -1;
			    } else {
				if (option_verbose > 5) {
				    if (capidebug)
    					ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_B3_ACTIVE_RESP (NCCI=%#x)\n",p->i->NCCI);
				}
			    }

			    ast_mutex_lock(&contrlock);
			    if (p->i->controller > 0) {
				capi_controllers[p->i->controller]->nfreebchannels--;
			    }
			    ast_mutex_unlock(&contrlock);

		    	    p->i->state = CAPI_STATE_BCONNECTED;
			    capi_echo_canceller(p->c,EC_FUNCTION_ENABLE);
			    capi_detect_dtmf(p->c,1);

			    if (p->i->earlyB3 != 1) {
				ast_setstate(p->c,AST_STATE_UP);
				fr.frametype = AST_FRAME_CONTROL;
				fr.subclass = AST_CONTROL_ANSWER;
				fr.datalen = 0;
				return pipe_frame(p,(struct ast_frame *)&fr);
			    }
			    return 0;
			    break;
		    }
		    break;

		case CAPI_CONF:
		    switch (CMSG->Command) {
			case CAPI_FACILITY:
			    if (FACILITY_CONF_FACILITYSELECTOR(CMSG) == 0x3) {
				if ((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(CMSG)[1] == 0x2) && (FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(CMSG)[2] == 0x0)) {
				    if ((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(CMSG)[4] == 0x0) && (FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(CMSG)[5] == 0x0)) {
				    } else {
					p->i->state = CAPI_STATE_BCONNECTED;
					if (capidebug)
				    	    ast_log(LOG_NOTICE,"%s\n",capi_cmsg2str(CMSG));
				    }
				}
			    }
			    break;
			case CAPI_DATA_B3:
//			    ast_log(LOG_NOTICE,"DATA_B3_CONF (NCCI %#x) for DATAHANDLE %#x\n",DATA_B3_CONF_NCCI(CMSG),DATA_B3_CONF_DATAHANDLE(CMSG));
			    break;
			case CAPI_ALERT:
//			    ast_log(LOG_NOTICE,"ALERT_CONF (PLCI=%#x)\n",(int)ALERT_CONF_PLCI(CMSG));
			    p->i->state = CAPI_STATE_ALERTING;
			    if (p->c->_state == AST_STATE_RING) {
				p->c->rings = 1;
			    }
			    break;
			case CAPI_CONNECT:
			    if (option_verbose > 1) {
				if (capidebug)
    				    ast_verbose(VERBOSE_PREFIX_2 "received CONNECT_CONF PLCI = %#x INFO = %#x\n",(int)CONNECT_CONF_PLCI(CMSG),CONNECT_CONF_INFO(CMSG));
			    }
			    if (CONNECT_CONF_INFO(CMSG) == 0) {
				p->i->PLCI = CONNECT_CONF_PLCI(CMSG);
				p->PLCI = p->i->PLCI;
				ast_setstate(p->c,AST_STATE_DIALING);
			    } else {
	// here, something has to be done -->
    				fr.frametype = AST_FRAME_CONTROL;
				fr.subclass = AST_CONTROL_BUSY;
				fr.datalen = 0;
				return pipe_frame(p,(struct ast_frame *)&fr);
			    }
			    break;
			case CAPI_CONNECT_B3:
//			    ast_log(LOG_NOTICE,"received CONNECT_B3_CONF NCCI = %#x INFO = %#x\n",(int)CONNECT_B3_CONF_NCCI(CMSG),CONNECT_B3_CONF_INFO(CMSG));
			    if (CONNECT_B3_CONF_INFO(CMSG) == 0) {
				p->i->NCCI = CONNECT_B3_CONF_NCCI(CMSG);
			    } else {
				p->i->earlyB3 = -1;
				p->i->doB3 = AST_CAPI_B3_DONT;
			    }
			    break;
		    }
		    break;		    
	    }
	}
//	ast_log(LOG_NOTICE,"returning\n");
	return 0;
}

static void capi_handle_msg(_cmsg *CMSG) {
    struct ast_capi_pvt *i;
    char *DNID;
    char *CID;
    char *msn;
    _cmsg CMSG2;
    MESSAGE_EXCHANGE_ERROR  error;
    int PLCI=0,NCCI;
    int NPLAN=0;
    int fds[2];
    int controller=0;
    char buffer[AST_MAX_EXTENSION];
    struct capi_pipe *p;
    char *magicmsn = "*\0";
    char *emptyid = "\0";
    char *emptydnid = "s\0";
    long flags;
#ifdef DEFLECT_ON_CIRCUITBUSY
    int deflect=0;
#endif
    
    switch (CMSG->Subcommand) {
	// indication msgs	
	case CAPI_IND:

	switch (CMSG->Command) {
	    case CAPI_CONNECT:	// only connect_ind are global (not channel specific)
		if (capidebug)
		    ast_log(LOG_NOTICE,"%s\n",capi_cmsg2str(CMSG));
		DNID = capi_number(CONNECT_IND_CALLEDPARTYNUMBER(CMSG),1);
		if ((DNID && *DNID == 0) || !DNID) {
		    DNID = emptydnid;
		}
                NPLAN = (CONNECT_IND_CALLINGPARTYNUMBER(CMSG)[1] & 0x70);
		CID = capi_number(CONNECT_IND_CALLINGPARTYNUMBER(CMSG),2);
		PLCI = CONNECT_IND_PLCI(CMSG);
		controller = PLCI & 0xff;
		if (option_verbose > 1) {
		    if (capidebug)
    			ast_verbose(VERBOSE_PREFIX_2 "CONNECT_IND (PLCI=%#x,DID=%s,CID=%s,CIP=%#x,CONTROLLER=%#x)\n",PLCI,DNID,CID,CONNECT_IND_CIPVALUE(CMSG),controller);
		}
		if(CONNECT_IND_BCHANNELINFORMATION(CMSG))
		if ((CONNECT_IND_BCHANNELINFORMATION(CMSG)[1] == 0x02) && (!capi_controllers[controller]->isdnmode)) {
		    // this is a call waiting CONNECT_IND with BChannelinformation[1] == 0x02
		    // meaning "no B or D channel for this call", since we can't do anything with call waiting now
   		    // just reject it with "user busy"
		    // however...if we are a p2p BRI then the telco switch will allow us to choose the b channel
		    // so it will look like a callwaiting connect_ind to us

                    ast_log(LOG_ERROR,"received a call waiting CONNECT_IND\n");
#ifndef DEFLECT_ON_CIRCUITBUSY
		    CONNECT_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber, 0);
                    CONNECT_RESP_PLCI(&CMSG2) = CONNECT_IND_PLCI(CMSG);
                    CONNECT_RESP_REJECT(&CMSG2) = 3; // user is busy
                    if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
                       ast_log(LOG_ERROR,"error sending CONNECT_RESP for PLCI = %#x\n",(int)CONNECT_IND_PLCI(CMSG));
		    } else {
                       if (option_verbose > 5) {
		    	    if (capidebug)
                        	ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_RESP for PLCI = %#x\n",(int)CONNECT_IND_PLCI(CMSG));
		       }
		    }
                    // no need to pipe this
                    PLCI = 0;
		    break;		
#else
		    deflect = 1;
#endif
		}
		// well...somebody is calling us. let's set up a channel
		ast_mutex_lock(&iflock);
		i = iflist;
		while(i) {
		    //XXX test this!
		    // has no owner
		    if ((!i->owner) && (i->incomingmsn != NULL)){
			strncpy(buffer,i->incomingmsn,sizeof(buffer)-1);
			msn = strtok(buffer,",");
			while (msn != NULL) {
//		    	ast_log(LOG_NOTICE,"msn=%s\n",msn);
			    if (DNID && ((!strcasecmp(msn,DNID)) ||
				 (i->isdnmode && (strlen(msn)<strlen(DNID)) && !strncasecmp(msn, DNID, strlen(msn))) || (!strncasecmp(msn,magicmsn,strlen(msn)))) && 
				(i->controllers & (1 << controller))) {
				if (CID != NULL) {
				    if(NPLAN == CAPI_ETSI_NPLAN_NATIONAL)
					snprintf(i->cid, (sizeof(i->cid)-1), "%s%s%s", i->prefix, capi_national_prefix, CID);
				    else if(NPLAN == CAPI_ETSI_NPLAN_INTERNAT)
					snprintf(i->cid, (sizeof(i->cid)-1), "%s%s%s", i->prefix, capi_international_prefix, CID);
				    else
					snprintf(i->cid, (sizeof(i->cid)-1), "%s%s", i->prefix, CID);
				} else
				    strncpy(i->cid,emptyid,sizeof(i->cid)-1);

				if (DNID != NULL) 
				    strncpy(i->dnid,DNID,sizeof(i->dnid)-1);
				else
				    strncpy(i->dnid,emptydnid,sizeof(i->dnid)-1);
				
				i->controller=controller;
				i->PLCI = PLCI;
				i->MessageNumber = CMSG->Messagenumber;
				if (pipe(fds) == 0) {
				    if (option_verbose > 4) {
    					ast_verbose(VERBOSE_PREFIX_3 "creating pipe for PLCI=%#x msn = %s\n",PLCI,msn);
				    }
				    i->fd = fds[0];
				    flags = fcntl(i->fd,F_GETFL);
				    fcntl(i->fd,F_SETFL,flags | O_SYNC | O_DIRECT);
//				    ast_log(LOG_NOTICE,"i->fd = %d\n",i->fd);
				    p = malloc(sizeof(struct capi_pipe));
				    memset(p, 0, sizeof(struct capi_pipe));
				    p->fd = fds[1];
				    flags = fcntl(i->fd,F_GETFL);
				    fcntl(p->fd,F_SETFL,flags | O_SYNC | O_DIRECT);
//				    ast_log(LOG_NOTICE,"p->fd = %d\n",p->fd);
				    p->PLCI = PLCI;
			    	    p->i = i;
    				    ast_pthread_mutex_init(&(p->lock),NULL);
				    i->mypipe = p;
				    if (i->isdnmode) {
					p->c = capi_new(i,AST_STATE_DOWN);
					i->state = CAPI_STATE_DID;
				    } else {
					p->c = capi_new(i,AST_STATE_RING);
				    }
			    	    p->next = pipelist;
				    pipelist = p;
				// hmmm....
				    ast_mutex_unlock(&iflock);
#ifdef DEFLECT_ON_CIRCUITBUSY
				    if ((deflect == 1) && (i->deflect2)) {
					capi_deflect(p->c,i->deflect2);
				    }
#endif
				    return;
				} else {
				    ast_log(LOG_ERROR,"creating pipe for PLCI=%#x failed\n",PLCI);
				}
				break;
			    } // if strcasecmp
			    msn = strtok(NULL,",");
			} // while strtok
		    } // if
		    i = i->next;
		} // while interface list
		ast_mutex_unlock(&iflock);		// obviously we are not called...so tell capi to ignore this call
		if (capidebug) {
		    ast_log(LOG_ERROR,"did not find device for msn = %s\n",DNID);
		}
		CONNECT_RESP_HEADER(&CMSG2, ast_capi_ApplID, CMSG->Messagenumber, 0);
		CONNECT_RESP_PLCI(&CMSG2) = CONNECT_IND_PLCI(CMSG);
		CONNECT_RESP_REJECT(&CMSG2) = 1; // ignore
		if ((error = _capi_put_cmsg(&CMSG2)) != 0) {
		    ast_log(LOG_ERROR,"error sending CONNECT_RESP for PLCI = %#x\n",(int)CONNECT_IND_PLCI(CMSG));
		} else {
		    if (option_verbose > 5) {
			if (capidebug)
    		    	    ast_verbose(VERBOSE_PREFIX_4 "sent CONNECT_RESP for PLCI = %#x\n",(int)CONNECT_IND_PLCI(CMSG));
		    }
		}
		ast_mutex_lock(&pipelock);
		if (pipelist == NULL) {
		    capi_last_plci = PLCI;
		}
		ast_mutex_unlock(&pipelock);
		// no need to pipe this
		PLCI = 0;
//		ast_mutex_unlock(&iflock);
//		return;
		break;
	    case CAPI_FACILITY:
		PLCI = FACILITY_IND_PLCI(CMSG) & 0xffff;  // this is for you eicon
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"FACILITY_IND PLCI=%#x\n",PLCI);
	    break;
	    case CAPI_INFO:
		PLCI = INFO_IND_PLCI(CMSG);
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"INFO_IND PLCI=%#x INFO# = %#x\n",PLCI,INFO_IND_INFONUMBER(CMSG));
	    break;
	    case CAPI_CONNECT_ACTIVE:
		PLCI = CONNECT_ACTIVE_IND_PLCI(CMSG);
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"CONNECT_ACTIVE_IND PLCI=%#x\n",PLCI);
	    break;
	    case CAPI_CONNECT_B3:
		NCCI = CONNECT_B3_IND_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"CONNECT_B3_IND NCCI=%#x PLCI=%#x\n",NCCI,PLCI);
	    break;
	    case CAPI_CONNECT_B3_ACTIVE:
		NCCI = CONNECT_B3_IND_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"CONNECT_B3_ACTIVE_IND NCCI=%#x PLCI=%#x\n",NCCI,PLCI);
	    break;
	    case CAPI_DATA_B3:
		NCCI = DATA_B3_IND_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
//		ast_log(LOG_ERROR,"DATA_B3_IND NCCI=%#x PLCI=%#x\n",NCCI,PLCI);
	    break;
	    case CAPI_DISCONNECT_B3:
		NCCI = DISCONNECT_B3_IND_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
		if (option_verbose > 1) {
		    if (capidebug)
    			ast_verbose(VERBOSE_PREFIX_2 "DISCONNECT_B3_IND NCCI=%#x\n",NCCI);
		}
	    break;
	    case CAPI_DISCONNECT:
		PLCI = DISCONNECT_IND_PLCI(CMSG);
		if (option_verbose > 1) {
		    if (capidebug)
    			ast_verbose(VERBOSE_PREFIX_2 "DISCONNECT_IND PLCI=%#x REASON=%#x\n",PLCI,DISCONNECT_IND_REASON(CMSG));
		}
	    break;
	    default:
    		ast_log(LOG_ERROR,"Command.Subcommand = %#x.%#x\n",CMSG->Command,CMSG->Subcommand);
	}
	break;
	// confirmation msgs
	case CAPI_CONF:
	switch (CMSG->Command) {
	    case CAPI_FACILITY:
		NCCI = FACILITY_CONF_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
		if (option_verbose > 2) {
    			if (FACILITY_CONF_FACILITYSELECTOR(CMSG) == 6) {
				if (FACILITY_CONF_INFO(CMSG)) 
					ast_verbose (VERBOSE_PREFIX_3 "Error setting up echo canceller (PLCI=%#x, Info=%#04x)\n", PLCI, FACILITY_CONF_INFO(CMSG));
				else
					ast_verbose (VERBOSE_PREFIX_3 "Echo canceller successfully set up (PLCI=%#x)\n",PLCI);
			}
		}
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"FACILITY_CONF NCCI=%#x INFO=%#x\n",(int)FACILITY_CONF_NCCI(CMSG),FACILITY_CONF_INFO(CMSG));
	    break;
	    case CAPI_INFO:
		PLCI = INFO_CONF_PLCI(CMSG);
//		ast_log(LOG_ERROR,"INFO_CONF PLCI=%#x INFO=%#x\n",PLCI,INFO_CONF_INFO(CMSG));
	    break;
	    case CAPI_CONNECT:
		PLCI = CONNECT_CONF_PLCI(CMSG);
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"CONNECT_CONF PLCI=%#x INFO=%#x MN=%#x\n",PLCI,CONNECT_CONF_INFO(CMSG),CMSG->Messagenumber);
	    break;
	    case CAPI_DISCONNECT:
		PLCI = DISCONNECT_CONF_PLCI(CMSG);
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"DISCONNECT_CONF PLCI=%#x INFO=%#x MN=%#x\n",PLCI,DISCONNECT_CONF_INFO(CMSG),CMSG->Messagenumber);
	    break;
	    case CAPI_DISCONNECT_B3:
		NCCI = DISCONNECT_B3_CONF_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"DISCONNECT_B3_CONF NCCI=%#x INFO=%#x MN=%#x\n",NCCI,DISCONNECT_B3_CONF_INFO(CMSG),CMSG->Messagenumber);
	    break;
	    case CAPI_CONNECT_B3:
		NCCI = CONNECT_B3_CONF_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		    ast_log(LOG_ERROR,"CONNECT_B3_CONF PLCI=%#x INFO=%#x MN=%#x\n",PLCI,CONNECT_B3_CONF_INFO(CMSG),CMSG->Messagenumber);
	    break;
	    case CAPI_ALERT:
		PLCI = ALERT_CONF_PLCI(CMSG);
		if (option_verbose > 3) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"ALERT_CONF PLCI=%#x\n",PLCI);
	    break;	    
	    case CAPI_DATA_B3:
		NCCI = DATA_B3_CONF_NCCI(CMSG);
		PLCI = (NCCI << 16) >> 16;
		if (option_verbose > 5) {
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(CMSG));
		}
//		ast_log(LOG_ERROR,"DATA_B3_CONF NCCI=%#x PLCI=%#x\n",NCCI,PLCI);
	    break;
	    default:
    		ast_log(LOG_ERROR,"Command.Subcommand = %#x.%#x\n",CMSG->Command,CMSG->Subcommand);
	}
	break;
    }
    if (PLCI > 0) {
	pipe_msg(PLCI,CMSG);
    }

}





// module stuff, monitor...

static void *do_monitor(void *data) {
   unsigned int Info;
    _cmsg *monCMSG;
    for (;;) {
/*
	if (ast_mutex_lock(&monlock)) {
	    ast_log(LOG_ERROR,"Unable to get monitor lock!\n");
	    return NULL;
	}
	// do some nifty stuff
	ast_mutex_unlock(&monlock);
*/
	monCMSG = malloc(sizeof(_cmsg));
	memset(monCMSG,0,sizeof(_cmsg));
	switch(Info = check_wait_get_cmsg(monCMSG)) {
	    case 0x0000:
    		    if (capidebug)
			ast_verbose(VERBOSE_PREFIX_3 "%s\n",capi_cmsg2str(monCMSG));
		capi_handle_msg(monCMSG);
    		break;
	    case 0x1104:
		// CAPI queue is empty
    		break;
	    default:
		// something is wrong!
		 break;
	} //switch
	free(monCMSG);
    } // for
    // never reached
    return NULL;
}

static int restart_monitor() {
    // stay stopped if wanted
    if (ast_mutex_lock(&monlock)) {
        ast_log(LOG_WARNING,"Unable to get monitor lock!\n");
        return -1;
    }
    if (monitor_thread == pthread_self()) {
	ast_mutex_unlock(&monlock);
        ast_log(LOG_WARNING,"Unable to kill myself!\n");
	return -1;
    }
    
    // restart
    if (ast_pthread_create(&monitor_thread,NULL,do_monitor,NULL) < 0) {
	ast_mutex_unlock(&monlock);
        ast_log(LOG_ERROR,"Unable to start monitor thread!\n");
	return -1;
    }

    return 0;
}

#ifdef CAPI_GAIN
static void capi_gains(struct ast_capi_gains *g,float rxgain,float txgain) {
    int i=0;
    int x=0;
    if (rxgain != 1.0) {
	for (i=0;i<256;i++) {
	    x = (int)(((float)capiXLAW2INT(i)) * rxgain);
	    if (x > 32767) x = 32767;
	    if (x < -32767) x = -32767;
	    g->rxgains[i] = capiINT2XLAW(x);
	}
    } else {
	for (i=0;i<256;i++) {
	    g->rxgains[i] = i;
	}
    }
    if (txgain != 1.0) {
	for (i=0;i<256;i++) {
	    x = (int)(((float)capiXLAW2INT(i)) * txgain);
	    if (x > 32767) x = 32767;
	    if (x < -32767) x = -32767;
	    g->txgains[i] = capiINT2XLAW(x);
	}
    } else {
	for (i=0;i<256;i++) {
	    g->txgains[i] = i;
	}
    }
    
}
#endif
#ifdef DEFLECT_ON_CIRCUITBUSY
int mkif(char *msn,char *incomingmsn,char *context,char *controllerstr,int devices,int softdtmf,int echocancel,int ecoption,int ectail, char *prefix, int isdnmode, int es,float rxgain,float txgain, char *deflect2, char *accountcode, unsigned int callgroup) {
#else
int mkif(char *msn,char *incomingmsn,char *context,char *controllerstr,int devices,int softdtmf,int echocancel,int ecoption,int ectail, char *prefix, int isdnmode, int es,float rxgain,float txgain, char *accountcode, unsigned int callgroup) {
#endif
    struct ast_capi_pvt *tmp;
    int i=0;
    char buffer[100];
    char *contr;
    unsigned long contrmap=0;

    for (i=0;i<devices;i++) {
	tmp = malloc(sizeof(struct ast_capi_pvt));
	memset(tmp, 0, sizeof(struct ast_capi_pvt));
	if (tmp) {
    	    ast_pthread_mutex_init(&(tmp->lock),NULL);
	    strncpy(tmp->context, context, sizeof(tmp->context)-1);
	    strncpy(tmp->msn, msn, sizeof(tmp->msn)-1);
	    strncpy(tmp->incomingmsn, incomingmsn, sizeof(tmp->incomingmsn)-1);
	    strncpy(tmp->prefix, prefix, sizeof(tmp->prefix)-1);
	    strncpy(tmp->accountcode, accountcode, sizeof(tmp->accountcode)-1);
	    
	    strncpy(buffer,controllerstr,sizeof(buffer)-1);
	    contr = strtok(buffer,",");
	    while (contr != NULL) {
		contrmap |= (1 << atoi(contr));
		if (capi_controllers[atoi(contr)]) {
		    capi_controllers[atoi(contr)]->isdnmode = isdnmode;
		//    ast_log(LOG_NOTICE, "contr %d isdnmode %d\n",atoi(contr),isdnmode);
		}
		contr = strtok(NULL,",");
	    }
	    tmp->controllers = contrmap;
	    capi_used_controllers |= contrmap;
	    tmp->controller = 0;
	    tmp->CLIR = 0;
	    tmp->earlyB3 = -1;
	    tmp->onholdPLCI = 0;
	    tmp->doEC = echocancel;
	    tmp->ecOption = ecoption;
	    tmp->ecTail = ectail;
	    tmp->isdnmode = isdnmode;
	    tmp->doES = es;
	    tmp->callgroup = callgroup;
#ifdef CAPI_ES	    
#endif
#ifdef CAPI_GAIN
	    tmp->rxgain = rxgain;
	    tmp->txgain = txgain;
	    capi_gains(&tmp->g,rxgain,txgain);
#endif
#ifdef DEFLECT_ON_CIRCUITBUSY
	    strncpy(tmp->deflect2, deflect2, sizeof(tmp->deflect2)-1);
#endif
#ifndef FORCE_SOFTWARE_DTMF
	    if (softdtmf == 1) {
#endif
		tmp->doDTMF = 1;
#ifndef FORCE_SOFTWARE_DTMF
	    } else {
		tmp->doDTMF = 0;
	    }
#endif
	    tmp->next = iflist;	// prepend
	    iflist = tmp;
    	//	    ast_log(LOG_NOTICE, "ast_capi_pvt(%s,%s,%s,%#x,%d) (%d,%d,%d) (%d)(%f/%f) %d\n",tmp->msn,tmp->incomingmsn,tmp->context,(int)tmp->controllers,devices,tmp->doEC,tmp->ecOption,tmp->ecTail,tmp->doES,tmp->rxgain,tmp->txgain,callgroup);
	    if (option_verbose > 2) {
    		    ast_verbose(VERBOSE_PREFIX_2 "ast_capi_pvt(%s,%s,%s,%d,%d) (%d,%d,%d)\n",tmp->msn,tmp->incomingmsn,tmp->context,tmp->controller,devices,tmp->doEC,tmp->ecOption,tmp->ecTail);
	    }
	    
	} else {
	    return -1;
	}
    }
    return 0;
}

void supported_sservices(struct ast_capi_controller *cp) {
    MESSAGE_EXCHANGE_ERROR error;
    _cmsg	CMSG,CMSG2;
    struct timeval tv;
    char fac[20];

    FACILITY_REQ_HEADER(&CMSG, ast_capi_ApplID, ast_capi_MessageNumber++, 0);
    FACILITY_REQ_CONTROLLER(&CMSG) = cp->controller;
    FACILITY_REQ_FACILITYSELECTOR(&CMSG) = 0x0003; // sservices
    fac[0] = 3;
    fac[1] = 0;
    fac[2] = 0;
    fac[3] = 0;
    FACILITY_REQ_FACILITYREQUESTPARAMETER(&CMSG) = (char *)&fac;
    if ((error= _capi_put_cmsg(&CMSG)) != 0) {
	ast_log(LOG_ERROR,"error sending FACILITY_REQ (error=%#x)\n",error);
    } else {
        if (option_verbose > 5) {
	    ast_verbose(VERBOSE_PREFIX_4 "sent FACILITY_REQ (CONTROLLER=%#x)\n",cp->controller);
	}
    }

    tv.tv_sec = 1;
    tv.tv_usec = 0;
    for (;;){
        error = capi20_waitformessage(ast_capi_ApplID,&tv);
        error = capi_get_cmsg(&CMSG2,ast_capi_ApplID); 
//        error = check_wait_get_cmsg(&CMSG2);
	if (error == 0) {
		if (IS_FACILITY_CONF(&CMSG2)) {
		    if (option_verbose > 5) {
    			ast_verbose(VERBOSE_PREFIX_4 "FACILITY_CONF INFO = %#x\n",FACILITY_CONF_INFO(&CMSG2));
		    }
		    break;
		}
	}
    } 
    // parse supported sservices
    if (FACILITY_CONF_FACILITYSELECTOR(&CMSG2) == 0x0003) {
	// success
	if (FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[4] == 0) {
	    if ((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 1) == 1) {
		cp->holdretrieve = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "HOLD/RETRIEVE\n");
	    } else {
		cp->holdretrieve = 0;	
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 2) >> 1) == 1) {
		cp->terminalportability = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "TERMINAL PORTABILITY\n");
	    } else {
		cp->terminalportability = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 4) >> 2) == 1) {
		cp->ECT = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "ECT\n");
	    } else {
		cp->ECT = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 8) >> 3) == 1) {
		cp->threePTY = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "3PTY\n");
	    } else {
		cp->threePTY = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 16) >> 4) == 1) {
		cp->CF = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "CF\n");
	    } else {
		cp->CF = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 32) >> 5) == 1) {
		cp->CD = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "CD\n");
	    } else {
		cp->CD = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 64) >> 6) == 1) {
		cp->MCID = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "MCID\n");
	    } else {
		cp->MCID = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[6] & 128) >> 7) == 1) {
		cp->CCBS = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "CCBS\n");
	    } else {
		cp->CCBS = 0;
	    }
	    if ((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[7] & 1) == 1) {
		cp->MWI = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "MWI\n");
	    } else {
		cp->MWI = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[7] & 2) >> 1) == 1) {
		cp->CCNR = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "CCNR\n");
	    } else {
		cp->CCNR = 0;
	    }
	    if (((FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[7] & 4) >> 2) == 1) {
		cp->CONF = 1;
		if (option_verbose > 3)
		    ast_verbose(VERBOSE_PREFIX_4 "CONF\n");
	    } else {
		cp->CONF = 0;
	    }
	} else {
	    ast_log(LOG_NOTICE,"supplementary services info  = %#x\n",(short)FACILITY_CONF_FACILITYCONFIRMATIONPARAMETER(&CMSG2)[1]);
	}
    } else  {
	ast_log(LOG_NOTICE,"unexpected FACILITY_SELECTOR = %#x\n",FACILITY_CONF_FACILITYSELECTOR(&CMSG2));
    }
}

static int capi_info(int fd, int argc, char *argv[])
{
	int i=0;
	if (argc != 2)
		return RESULT_SHOWUSAGE;
	for (i=1;i<=capi_num_controllers;i++) {
	    ast_mutex_lock(&contrlock);
	    if (capi_controllers[i] != NULL) {
		ast_cli(fd,"Contr%d: %d B channels total, %d B channels free.\n",i,capi_controllers[i]->nbchannels,capi_controllers[i]->nfreebchannels);
	    }
	    ast_mutex_unlock(&contrlock);
	}
	return RESULT_SUCCESS;
}

static int capi_do_debug(int fd, int argc, char *argv[])
{
	if (argc != 2)
		return RESULT_SHOWUSAGE;
	capidebug = 1;
	ast_cli(fd, "CAPI Debugging Enabled\n");
	return RESULT_SUCCESS;
}

static int capi_no_debug(int fd, int argc, char *argv[])
{
	if (argc != 3)
		return RESULT_SHOWUSAGE;
	capidebug = 0;
	ast_cli(fd, "CAPI Debugging Disabled\n");
	return RESULT_SUCCESS;
}

static char info_usage[] = 
"Usage: capi info\n"
"       Show info about B channels.\n";

static char debug_usage[] = 
"Usage: capi debug\n"
"       Enables dumping of CAPI packets for debugging purposes\n";

static char no_debug_usage[] = 
"Usage: capi no debug\n"
"       Disables dumping of CAPI packets for debugging purposes\n";

static struct ast_cli_entry  cli_info =
	{ { "capi", "info", NULL }, capi_info, "Show CAPI info", info_usage };
static struct ast_cli_entry  cli_debug =
	{ { "capi", "debug", NULL }, capi_do_debug, "Enable CAPI debugging", debug_usage };
static struct ast_cli_entry  cli_no_debug =
	{ { "capi", "no", "debug", NULL }, capi_no_debug, "Disable CAPI debugging", no_debug_usage };


int load_module(void)
{
	struct ast_config *cfg;
	struct ast_variable *v;
	char *config = "capi.conf";
	char msn[AST_MAX_EXTENSION]="";
	char incomingmsn[AST_MAX_EXTENSION]="";
	char context[AST_MAX_EXTENSION]="";
	char prefix[AST_MAX_EXTENSION]="";
	char accountcode[20]="";
	char *empty = "\0";
	char deflect2[AST_MAX_EXTENSION]="";
	char controllerstr[AST_MAX_EXTENSION]="";
	int res = 0;
	int controller=0;
	int softdtmf=0;
	int echocancel=1;
	int ecoption=EC_OPTION_DISABLE_G165;
	int ectail=EC_DEFAULT_TAIL;
	int es=0;
	float rxgain = 1.0;
	float txgain = 1.0;
	int isdnmode = 0;
	unsigned int callgroup=0;
	struct ast_capi_controller *cp;

	cfg = ast_load(config);

	/* We *must* have a config file otherwise stop immediately */
	if (!cfg) {
		ast_log(LOG_ERROR, "Unable to load config %s\n", config);
		return -1;
	}
	if (ast_mutex_lock(&iflock)) {
		ast_log(LOG_ERROR, "Unable to lock interface list???\n");
		return -1;
	}

	strncpy(capi_national_prefix, AST_CAPI_NATIONAL_PREF, sizeof(capi_national_prefix)-1);
	strncpy(capi_international_prefix, AST_CAPI_NATIONAL_PREF, sizeof(capi_national_prefix)-1);
	v = ast_variable_browse(cfg, "general");
	while(v) {
		if (!strcasecmp(v->name, "nationalprefix")) {
		    strncpy(capi_national_prefix, v->value, sizeof(capi_national_prefix)-1);
		} else if (!strcasecmp(v->name, "internationalprefix")) {
			strncpy(capi_international_prefix, v->value, sizeof(capi_international_prefix)-1);
		} else if (!strcasecmp(v->name, "rxgain")) {
			if (sscanf(v->value,"%f",&rxgain) != 1) {
			    ast_log(LOG_ERROR,"invalid rxgain\n");
			}
		} else if (!strcasecmp(v->name, "txgain")) {
			if (sscanf(v->value,"%f",&txgain) != 1) {
			    ast_log(LOG_ERROR,"invalid txgain\n");
			}
		}
		v = v->next;
	}



	
	if (capi20_isinstalled() != 0) {
	    ast_log(LOG_NOTICE,"CAPI not installed!\n");
	    return -1;
	}

	if (capi20_register(AST_CAPI_BCHANS,AST_CAPI_MAX_B3_BLOCKS,AST_CAPI_MAX_B3_BLOCK_SIZE,&ast_capi_ApplID) != 0) {
	    ast_log(LOG_NOTICE,"unable to register application at CAPI!\n");
	    return -1;
	}

	if (Listen(ALL_SERVICES) != 0) {
	    ast_log(LOG_NOTICE,"unable to listen!\n");
	    return -1;
	}

	if (capi20_get_profile(0,(char *)&profile) != 0) {
	    ast_log(LOG_NOTICE,"unable to get CAPI profile!\n");
	    return -1;
	} else {
	    capi_num_controllers = profile.ncontrollers;
	    if (option_verbose > 3)
	    ast_verbose(VERBOSE_PREFIX_3 "This box has %d capi controller(s).\n",capi_num_controllers);
	    for (controller=1;controller<=capi_num_controllers;controller++) {

		memset(&profile,0,sizeof(profile));
		capi20_get_profile(controller,(char *)&profile);
		cp = malloc(sizeof(struct ast_capi_controller));
		cp->controller = controller;
		cp->nbchannels = profile.nbchannels;
		cp->nfreebchannels = profile.nbchannels;
		if ((profile.globaloptions & 8) >> 3 == 1) {
		    if (option_verbose > 3)
			ast_verbose(VERBOSE_PREFIX_3 "CAPI[contr%d] supports DTMF\n",controller);
		    cp->dtmf = 1;
		} else {
		    cp->dtmf = 0;	    
		}
		if (profile.globaloptions2 & 1) {
		    if (option_verbose > 3)
			ast_verbose(VERBOSE_PREFIX_3 "CAPI[contr%d] supports echo cancellation\n",controller);
		    cp->echocancel = 1;
		} else {
		    cp->echocancel = 0;
		}
		if ((profile.globaloptions & 16) >> 4 == 1) {
		    cp->sservices = 1;
		} else {
		    cp->sservices = 0;	    
		}
		capi_controllers[controller] = cp;
		if (cp->sservices == 1) {
		    if (option_verbose > 3)
			ast_verbose(VERBOSE_PREFIX_3 "CAPI[contr%d] supports supplementary services\n",controller);
		    supported_sservices(cp);
		}
	    }
	}

	v = ast_variable_browse(cfg, "interfaces");
	while(v) {
		/* Create the interface list */
		if (!strcasecmp(v->name, "devices")) {
#ifdef DEFLECT_ON_CIRCUITBUSY
			if (mkif(msn,incomingmsn,context,controllerstr,atoi(v->value),softdtmf,echocancel,ecoption,ectail, prefix, isdnmode, es,rxgain,txgain,deflect2,accountcode,callgroup)) {
#else
			if (mkif(msn,incomingmsn,context,controllerstr,atoi(v->value),softdtmf,echocancel,ecoption,ectail, prefix, isdnmode, es,rxgain,txgain,accountcode,callgroup)) {
#endif
			    ast_log(LOG_ERROR,"Error creating interface list\n");
			    return -1;
			}
			es=0; 
			strncpy(deflect2, empty, sizeof(deflect2)-1);
		} else if (!strcasecmp(v->name, "context")) {
			strncpy(context, v->value, sizeof(context)-1);
		} else if (!strcasecmp(v->name, "msn")) {
			strncpy(msn, v->value, sizeof(msn)-1);
		} else if (!strcasecmp(v->name, "incomingmsn")) {
			strncpy(incomingmsn, v->value, sizeof(incomingmsn)-1);
		} else if (!strcasecmp(v->name, "controller")) {
			strncpy(controllerstr, v->value, sizeof(controllerstr)-1);
		} else if (!strcasecmp(v->name, "softdtmf")) {
			softdtmf = atoi(v->value);
		} else if (!strcasecmp(v->name, "echosquelch")) {
			es = atoi(v->value);
		} else if (!strcasecmp(v->name, "callgroup")) {
			callgroup = ast_get_group(v->value);
		} else if (!strcasecmp(v->name, "deflect")) {
			strncpy(deflect2, v->value, sizeof(deflect2)-1);
		} else if (!strcasecmp(v->name, "rxgain")) {
			if (sscanf(v->value,"%f",&rxgain) != 1) {
			    ast_log(LOG_ERROR,"invalid rxgain\n");
			}
		} else if (!strcasecmp(v->name, "txgain")) {
			if (sscanf(v->value,"%f",&txgain) != 1) {
			    ast_log(LOG_ERROR,"invalid txgain\n");
			}
		} else if (!strcasecmp(v->name, "echocancel")) {
			if (!strcasecmp(v->value, "yes") || !strcasecmp(v->value, "1") || !strcasecmp(v->value, "on")) {
				echocancel=1;
				ecoption=EC_OPTION_DISABLE_G165;
			}	
			else if (!strcasecmp(v->value, "no") || !strcasecmp(v->value, "0") || !strcasecmp(v->value, "off")) {
				echocancel=0;
				ecoption=0;
			}	
			else if (!strcasecmp(v->value, "g165") || !strcasecmp(v->value, "g.165")) {
				echocancel=1;
				ecoption=EC_OPTION_DISABLE_G165;
			}	
			else if (!strcasecmp(v->value, "g164") || !strcasecmp(v->value, "g.164")) {
				echocancel=1;
				ecoption=EC_OPTION_DISABLE_G164_OR_G165;
			}	
			else if (!strcasecmp(v->value, "force")) {
				echocancel=1;
				ecoption=EC_OPTION_DISABLE_NEVER;
			}
			else {
				ast_log(LOG_ERROR,"Unknown echocancel parameter \"%s\" -- ignoring\n",v->value);
			}
		} else if (!strcasecmp(v->name, "echotail")) {
			ectail = atoi(v->value);
			if (ectail > 255)
				ectail = 255;
		} else if (!strcasecmp(v->name, "prefix")) {
			strncpy(prefix, v->value, sizeof(prefix)-1);
		} else if (!strcasecmp(v->name, "accountcode")) {
			strncpy(accountcode, v->value, sizeof(accountcode)-1);
		} else if (!strcasecmp(v->name, "isdnmode")) {
			if (!strcasecmp(v->value, "ptp") || !strcasecmp(v->value, "1"))
			    isdnmode = 1;
			else if (!strcasecmp(v->value, "ptm") || !strcasecmp(v->value, "1"))
			    isdnmode = 0;
			else
			    ast_log(LOG_ERROR,"Unknown isdnmode parameter \"%s\" -- ignoring\n",v->value);

		}

		v = v->next;
	}
	ast_destroy(cfg);

	    for (controller=1;controller<=capi_num_controllers;controller++) {
		if (capi_used_controllers & (1 << controller)) {
		    if (ListenOnController(ALL_SERVICES,controller) != 0) {
			ast_log(LOG_ERROR,"Unable to listen on contr%d\n",controller);
		    } else {
			if (option_verbose > 2) 
			    ast_verbose(VERBOSE_PREFIX_3 "listening on contr%d CIPmask = %#x\n",controller,ALL_SERVICES);
		    }
		} else {
		    ast_log(LOG_WARNING,"Unused contr%d\n",controller);
		}
	    }


	ast_mutex_unlock(&iflock);
	
	if (ast_channel_register(type, tdesc, capi_capability, capi_request)) {
		ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
		unload_module();
		return -1;
	}

	ast_cli_register(&cli_info);
	ast_cli_register(&cli_debug);
	ast_cli_register(&cli_no_debug);
	
	restart_monitor();

	return res;
}


int unload_module()
{
	if (capi20_release(ast_capi_ApplID) != 0)
		ast_log(LOG_WARNING,"Unable to unregister from CAPI!\n");
	ast_channel_unregister(type);
	return 0;
}

int usecount()
{
	int res;
	ast_mutex_lock(&usecnt_lock);
	res = usecnt;
	ast_mutex_unlock(&usecnt_lock);
	return res;
}

char *description()
{
	return desc;
}


char *key()
{
	return ASTERISK_GPL_KEY;
}
