/* $Id: ArkCache.cpp,v 1.35 2003/03/23 20:09:25 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkCache.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkLoader.h>
#include <map>

namespace Ark
{

   /// Objects are ordered by their name and their ID...
   class ObjectID
   {
	 friend bool operator < (const ObjectID &a, const ObjectID &b);

	 String m_Name;
	 int m_Type;

      public:
	 ObjectID () :
	    m_Name (),
	    m_Type ( V_UNDEFINED )
	 {}

	 ObjectID (const String& name, int type) :
	    m_Name (name),
	    m_Type (type)
	 {}
   };

   bool operator < (const ObjectID &a, const ObjectID &b)
   {
      int dif = strcmp (a.m_Name.c_str(), b.m_Name.c_str());
      
      if (dif < 0)
	 return true;
      else if (dif == 0)
	 return a.m_Type < b.m_Type;
      else
	 return false;
   }

   typedef std::map< ObjectID, Object* > ObjectList;

   /*===================================== OBJECT "CACHE" IMPLEMENTATION*/

   class CacheP
   {
       typedef ObjectList::iterator OLIterator;
	 friend class Cache;
	 
	 ObjectList m_Objects;
	 
      public:      
	 Object *AddObject (Object *pointer)
	 {
	    assert( pointer );

	    ObjectID id(pointer->Name(), pointer->Type());
	    m_Objects.insert( std::pair< ObjectID, Object* >(id, pointer) );
	    pointer->Ref();

	    return pointer;
	 }

	 Object *FindObject (ObjectType type, const String &name)	
	 {
	    ObjectID id (name,type);
	    OLIterator it = m_Objects.find (id);
	    
	    if (it == m_Objects.end())
	       return 0;

	    return it->second;
	 }

	 void Clear (bool force)
	 {
	    int c = 0;

	    while (c == 0 || (force && !m_Objects.empty()))
	    {
	       for (OLIterator i = m_Objects.begin(); i != m_Objects.end() ;)
	       {
		  //std::cerr << i->second->Dump() << " "
		  //          << i->second->RefCount()
		  //          <<"\n";
		  if (i->second->RefCount() == 1)
		  {
		     //std::cerr << "removing " << i->second->m_Name
		     //          << " from cache\n";
		     i->second->Unref();
		     i->second = 0; // Just to be sure..
		     
		     // if we remove, here is the way to do it the good way
		     // with associative container
		     m_Objects.erase(i++);
		  }
		  else
		  {
		     // we do not remove, increment operator
		     ++i;
		  }
	       }

	       if (c > 100)
	       {
		  Sys()->Warning("In Cache::Clear(force=true): Some objects "
                                 "are  still being used, or there is some "
				 "kind of circular reference (c > 100).");
		  Dump(true);
		  break;
	       }
	       ++c;
	    }
	 }

	 void Dump (bool long_ver)
	 {
	    for (OLIterator i = m_Objects.begin(); i != m_Objects.end(); ++i)
	    {
	       String str = i->second->Dump(long_ver);
	       if (! str.empty() && long_ver)
		  Sys()->Log ("%s (%d ref(s) left)\n", str.c_str(),
			      i->second->RefCount());
	       else
		  Sys()->Log ("%s\n", str.c_str());
	    }	    
	 }

	 void MemReport ()
	 {
	    // Compute total texture memory used.

	    scalar texmem = 0.0;
	    for (OLIterator i = m_Objects.begin(); i != m_Objects.end(); ++i)
	    {
	       if (i->second->Type() != V_TEXTURE) continue;

	       Texture *tex = static_cast<Texture*> (i->second);
	       const Image::Format format = tex->m_Format;
	       const int bpp = Image::GetBytesPerPixel(format);
	       texmem += (scalar) tex->m_Width * tex->m_Height * bpp;
	    }

	    std::cout << "Estimated texture memory usage: "
		 << texmem / (1024.0*1024.0)
		 << " MB" << std::endl;
	 }
   };

   /*=============================================== CONSTRUCTOR/DESTRUCTOR */
 
   Cache::Cache(bool server) :
      m_ColSystem(0),
      m_Server(server),
      m_Objects(new CacheP ())
   {}
   
   Cache::~Cache()
   {
      Clear(true);
      delete m_Objects;

      if (m_ColSystem)
	 delete m_ColSystem;
   }

   void Cache::Clear (bool force)
   {
      assert(m_Objects);
      
      m_Objects->Clear (force);
   }
   
   
   /*================================================== ACCESS FUNCTIONS */
   Object *
   Cache::Get (ObjectType type, const String &file)
   {
      assert (m_Objects);
      
      Object *vis = m_Objects->FindObject (type, file);
      
      if (vis == NULL)
	 vis = m_Objects->AddObject (LoadObject (type, file)); 
      else
	 vis->Ref();
      assert (vis->Type() == type);
      return vis;
   }   
   
   void Cache::Dump ()
   {
      m_Objects->Dump (true);
   }

   void
   Cache::MemReport ()
   {
      m_Objects->MemReport();
   }
   
   /*============================================ OBJECT LOADING & CREATING */
   // Load a generic object.
   Object *Cache::LoadObject  (ObjectType type, const String &name)
   {
     Object *vis = CreateObject(type, name);

     if (vis == 0)
	return 0;

     assert (vis->m_Type == type);

     if (! name.empty() )
     {
	if (!vis->Load (this, name))
	{
	   bool res = Sys()->GetLoaders()->Load(vis, name, this);
	   
	   if (res == false)
	   {
	      delete vis;
	      return 0;
	   }
	}
	else
	{
	   Sys()->Log ("load: %s: %s\n",
		       name.c_str(), "Builtin load method");
	}
	
	vis->PostLoad(this);
     }
     
     return vis;
   }
   
   Object *
   Cache::CreateObject (ObjectType type, const String &name)
   {
      switch (type)
      {
	 case V_TEXTURE:   return new Texture (name);
	 case V_MATERIAL:  return new Material(name);
	 case V_MODEL:     return new Model(name);
	 case V_PARTICLE:  return new ParticleTmpl(name);
	 case V_IMAGE:     return new Image(name, 0, 0, Image::NONE_0);
	 case V_SKIN:      return new Skin(name);
	 case V_SEQUENCE:  return new Sequence(name);
	 default:          return 0;
      }
      
      return 0;
   }

/* namespace Ark */
}
