/*
    Copyright (C) 2001 Paul Davis
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: main.cc,v 1.18 2004/12/08 14:00:46 pauld Exp $
*/

#include <vector>
#include <string>
#include <iostream>

#include <string.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <sys/errno.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>

#include <pbd/atomic.h>
#include <pbd/error.h>
#include <pbd/rt.h>
#include <pbd/textreceiver.h>
#include <pbd/pthread_spinlock.h>
#include <pbd/platform.h>
#include <pbd/platform_factory.h>
#include <pbd/failed_constructor.h>
#include <midi++/mmc.h>
#include <sigc++/signal_system.h>
#include <ardour/ardour.h>
#include <ardour/audioengine.h>

#include "i18n.h"
#include "physical_keyboard.h"
#include "version.h"
#include "ksi.h"

#include "opts.h"

#include <ardour/version.h>
#include <ardour/configuration.h>
#include <ardour/ladspa_plugin.h>

/* our only global variables */

static KSI *ksi = 0;

Transmitter  error (Transmitter::Error);
Transmitter  info (Transmitter::Info);
Transmitter  fatal (Transmitter::Fatal);
Transmitter  warning (Transmitter::Warning);
TextReceiver text_receiver ("ksi");

using namespace SigC;
using namespace KSI_ARDOUR;
using namespace ARDOUR;

ARDOUR::AudioEngine *engine;
ARDOUR::Session *session = 0;

void
die (void)

{
	if (ksi) {
		delete ksi;
	} else {
		endwin ();
	}

	if (engine) {
		delete engine;
	}

	exit (1);
}

void
die_gracefully (void)

{
	if (ksi) {
		ksi->safe_to_save(true);
		delete ksi;
	} else {
		endwin ();
	}

	if (engine) {
		delete engine;
	}
	
	exit (0);
}

int
main (int argc, char *argv[])

{
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	cout << "KSI version " << ksi_version << endl
	     << " running with libardour "
	     << ardour_major_version << '.'
	     << ardour_minor_version << '.' 
	     << ardour_micro_version << '\n'
	     << "Copyright 1999-2002 Paul Davis" 
	     << endl
	     << "ksi comes with ABSOLUTELY NO WARRANTY" << endl
	     << "This is free software, and you are welcome to redistribute it "
	     << endl
	     << "under certain conditions; see the file COPYING for details"
	     << endl;

	text_receiver.listen_to (error);
	text_receiver.listen_to (info);
	text_receiver.listen_to (fatal);
	text_receiver.listen_to (warning);

	if (parse_engine_opts (argc, argv)) {
		exit (1);
	}

	try { 
		engine = new ARDOUR::AudioEngine ("ardour-ksi");
	} 
	
	catch (AudioEngine::NoBackendAvailable& err) {
		error << _("Could not connect to JACK server") <<  endmsg;
		exit (1);
	}
	
	catch (failed_constructor& err) {
		error << _("Could not become JACK client \"ardour-ksi\"") << endmsg;
		exit (1);
	}

	if (ARDOUR::init (*engine, false)) {
		error << _("cannot initialize ardour API") << endmsg;
		exit (1);
	}
	
	try {
		ksi = new KSI (*engine, remote);
	}
	
	catch (failed_constructor& err) {
		error << _("could not create KSI object") << endmsg;
		goto out;
	}

	text_receiver.hangup ();

	ksi->listen_to (error);
	ksi->listen_to (info);
	ksi->listen_to (fatal);
	ksi->listen_to (warning);

	ksi->run ();

  out:
	die_gracefully();

	ARDOUR::cleanup ();

	exit (0);
}

