# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::Widget::ChangeList;

use Glib qw(TRUE FALSE);
use Gtk2;
use Gtk2::Pango;

use base qw(Gtk2::TreeView);

use Arch::Changes qw(:type);

use constant MODEL_COL_TYPES  => qw(
	Glib::String
	Glib::String
	Glib::String
	Glib::String
	Glib::Scalar
);

use constant MODEL_COL_TYPE      => 0;
use constant MODEL_COL_PATH      => 1;
use constant MODEL_COL_OPTION    => 2;
use constant MODEL_COL_SORT_TYPE => 3;
use constant MODEL_COL_NUMBER    => 4;

my @VIEW_COLUMNS = (
	{
		title       => 'Type',
		column      => MODEL_COL_TYPE,
		attributes  => {
		},
		sort_column => MODEL_COL_SORT_TYPE,
	},
	{
		title      => 'Path',
		column     => MODEL_COL_PATH,
		attributes => {
		},
		sort_column => MODEL_COL_PATH,
	},
	{
		title      => 'Options',
		column     => MODEL_COL_OPTION,
		attributes => {
		},

	},
);

my %CHANGE_TYPE_SORT = (
	MODIFY,      0,
	ADD,         1,
	REMOVE,      2,
	RENAME,      3,
	META_MODIFY, 4,
);

sub new ($;$) {
	my $class   = shift;
	my $changes = shift;

	my $self = bless Gtk2::TreeView->new(), $class;
	$self->set_model(Gtk2::ListStore->new(MODEL_COL_TYPES));

	$self->_init();
	$self->show($changes) if ($changes);

	return $self;
}

sub _init ($) {
	my $self = shift;

	$self->set_rules_hint(TRUE);
	$self->set_search_column(MODEL_COL_PATH);

	foreach my $col (@VIEW_COLUMNS) {
		my $column = Gtk2::TreeViewColumn->new_with_attributes(
			$col->{title}, Gtk2::CellRendererText->new(),
			text => $col->{column},
			%{$col->{attributes}},
		);
		$column->set_resizable(TRUE);
		$column->set_sort_column_id($col->{sort_column})
			if exists $col->{sort_column};

		$self->append_column($column);
	}
}

sub get ($$;@) {
	my $self = shift;
	my $iter = shift;

	return $self->get_model()->get($iter, @_);
}

sub get_selected ($;@) {
	my $self = shift;

	my $iter = $self->get_selection()->get_selected();

	return $iter ? $self->get($iter, @_) : undef;
}

sub get_selected_number ($) {
	my $self = shift;

	return $self->get_selected(MODEL_COL_NUMBER);
}

sub clear ($) {
	my $self = shift;

	$self->get_model->clear;
}

sub show ($$) {
	my $self    = shift;
	my $changes = shift;

	$self->clear;

	my $model = $self->get_model;
	if ($changes->count) {
		for (my $i = 0; $i < $changes->count; ++$i) {
			my $change = $changes->get($i);
			
			my $iter = $model->append;
			$model->set($iter,
				MODEL_COL_TYPE,      Arch::Changes->type_string($change),
				MODEL_COL_PATH,      $change->{arguments}->[0],
				MODEL_COL_OPTION,    $change->{arguments}->[1] || '',
				MODEL_COL_SORT_TYPE, $CHANGE_TYPE_SORT{$change->{type}} 
					. ($change->{is_dir} ? '1' : '0')
					. ' ' . $change->{arguments}->[0],
				MODEL_COL_NUMBER,    $i,
			);
		}

		$model->set_sort_column_id(MODEL_COL_SORT_TYPE, 'ascending');

		my $first = $self->get_model()->iter_nth_child(undef, 0);
		$self->get_selection()->select_iter($first);

	} else {
		my $iter = $model->append;
		$model->set($iter,
			MODEL_COL_TYPE,      '',
			MODEL_COL_PATH,      '*** no changes ***',
			MODEL_COL_OPTION,    '',
			MODEL_COL_SORT_TYPE, '',
			MODEL_COL_NUMBER,    undef,
		);
	}
}

1;

__END__
