// ---------------------------------------------------------------------------
// - Librarian.hpp                                                           -
// - aleph engine - librarian class definition                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_LIBRARIAN_HPP
#define  ALEPH_LIBRARIAN_HPP

#ifndef  ALEPH_STRVEC_HPP
#include "Strvec.hpp"
#endif

#ifndef  ALEPH_INPUTMAPPED_HPP
#include "InputMapped.hpp"
#endif

namespace aleph {

  /// The Librarian class is a management class that allows to combine several
  /// files into a single one. Without argument, an empty librarian is created.
  /// With a string argument a librarian is opened for input. Writing a 
  /// librarian is done with the write method using a string as the file name.
  /// The class can be used inside the Aleph interpreter as well.
  /// @author amaury darsch

  class Librarian : public Object {
  private:
    /// the librarian mode
    enum t_mode {INPUT, OUTPUT};
    /// the librarian type
    t_mode d_type;
    /// the librarian name
    String d_lname;
    /// the librarian structure
    struct s_fdesc* p_desc;

  public:
    /// create an empty librarian
    Librarian (void);

    /// create a library by name
    /// @param lname the librarian name
    Librarian (const String& lname);

    /// destroy this librarian
    ~Librarian (void);

    /// @return the class name
    String repr (void) const;

    /// @return the librarian name
    String getname (void) const;

    /// add a new file to this librarian
    /// @param path the file path to add
    void add (const String& path);

    /// @return the number of files in this librarian
    long length (void) const;

    /// @return true if a name exists in the librarian
    bool exists (const String& name) const;

    /// @return a vector of file name in this librarian
    Strvec getlist (void) const;

    /// @return avector of string object in this librarian
    Vector* getstr (void) const;

    /// @return an input mapped file by name
    InputMapped* extract (const String& name) const;

    /// write this librarian to an output file
    /// @param lname the librarian name
    void write (const String& lname) const;

    /// format to an output stream the librarian structure
    /// @param os the output stream to write
    void format (Output& os) const;

    /// @return true if the path is a valid librarian file
    static bool valid (const String& path);

    /// generate a new librarian
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this librarian with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the copy constructor private
    Librarian (const Librarian&);
    // make the assignment operator private
    Librarian& operator = (const Librarian&);
  };
}

#endif
