// ---------------------------------------------------------------------------
// - Pathasio.cpp                                                            -
// - aleph:sio library - file system path functions implementation           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Vector.hpp"
#include "System.hpp"
#include "Pathasio.hpp"
#include "Exception.hpp"
#include "cdir.hpp"
#include "csys.hpp"

namespace aleph {

  // remove a directory by names

  Object* asio_rmdir (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    try {
      long    argc = (argv == nilp) ? 0 : argv->length ();
      for (long i = 0; i < argc; i++)
	System::rmdir (argv->getstring (i));
      delete argv;
      return nilp;
    } catch (...) {
      delete argv;
      throw;
    }
  }

  // remove a file by names

  Object* asio_rmfile (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    try {
      long    argc = (argv == nilp) ? 0 : argv->length ();
      for (long i = 0; i < argc; i++)
	System::rmfile (argv->getstring (i));
      delete argv;
      return nilp;
    } catch (...) {
      delete argv;
      throw;
    }
  }

  // create a temporary name

  Object* asio_tmp_name (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    try {
      long    argc = (argv == nilp) ? 0 : argv->length ();
      if (argc != 0) 
	throw Exception ("argument-error", "too many arguments with tmp-name");
      delete argv;
      return new String (c_tmpname ());
    } catch (...) {
      delete argv;
      throw;
    }
  }

  // create an absolute path name

  Object* asio_absolute_path (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // root directory first
    if (argc == 0) {
      delete argv;
      return new String (c_rootdir ());
    }

    // first argument build root
    String result = c_rootdir ();
    result = result + argv->getstring (0);
    if (argc == 1) {
      delete argv;
      return new String (result);
    }

    // loop with the rest
    for (long i = 1; i < argc; i++)
      result = System::join (result, argv->getstring (i));
    delete argv;
    return new String (result);
  }

  // create a relative path name

  Object* asio_relative_path (Runnable* robj, Nameset* nset, Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // check for at least one argument
    if (argc == 0) {
      delete argv;
      throw Exception ("argument-error", 
		       "missing arguments with absolute-path");
    }

    // first argument starts the path
    String result = argv->getstring (0);

    // loop with the rest
    for (long i = 1; i < argc; i++)
      result = System::join (result, argv->getstring (i));
    delete argv;
    return new String (result);
  }
}
