// ---------------------------------------------------------------------------
// - Collection.hpp                                                          -
// - aleph:odb library - collection class definition                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_COLLECTION_HPP
#define  ALEPH_COLLECTION_HPP

#ifndef  ALEPH_INPUT_HPP
#include "Input.hpp"
#endif

#ifndef  ALEPH_OUTPUT_HPP
#include "Output.hpp"
#endif

#ifndef  ALEPH_TABLE_HPP
#include "Table.hpp"
#endif

namespace aleph {

  /// The Collection class is a collection of tables. Because of the table
  /// format, a collection look like a 3-dimensional array of cells. Like the
  /// table, the collection is defined with a name and a vector of tables.
  /// @author amaury darsch

  class Collection : public Persist {
  private:
    /// the collection name
    String d_name;
    /// the table vector
    Vector d_vtbls;

  public:
    /// create an empty collection
    Collection (void);

    /// create a new collection with a name
    /// @param name the collection name
    Collection (const String& name);

    /// create a new collection with a stream
    /// @param is the input stream to use
    Collection (Input* is);

    /// @return the object name
    String repr (void) const;

    /// write the collection to an output stream
    /// @param os the output stream
    void write (Output& os);

    /// serialize this collection
    /// @param os the output stream
    void wrstream (Output& os) const;

    /// deserialize this collection
    /// @param is the input stream
    void rdstream (Input& os);

    /// @return the collection name
    String getname (void) const;

    /// @return the collection length
    long length (void) const;

    /// add a new table in this collection
    /// @param table the table to add
    void tadd (Table* table);

    /// @return a table by index
    Table* tget (const long index) const;

    /// set a table by index
    /// @param index the table index
    /// @param table the table to set
    void tset (const long index, Table* table);

    /// generate a new collection
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this collection with a set of arguments and a quark
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
    
  private:
    // make the copy constructor private
    Collection (const Collection&);
    // make the assignment operator private
    Collection& operator = (const Collection&);
  };
}

#endif
