// ---------------------------------------------------------------------------
// - Cell.cpp                                                                -
// - aleph:odb library - cell class implementation                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cell.hpp"
#include "Odbsid.hpp"
#include "Vector.hpp"
#include "Output.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace aleph {
  // the cell supported quarks
  static const long QUARK_GET     = String::intern ("get");
  static const long QUARK_SET     = String::intern ("set");
  static const long QUARK_GETNAME = String::intern ("get-name");
  static const long QUARK_SETNAME = String::intern ("set-name");

  // this procedure returns a new cell object for deserialization
  static Serial* mksob (void) {
    return new Cell;
  }
  // register this cell serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_CELL_ID, mksob);

  // create an empty cell
  
  Cell::Cell (void) {
    d_quark = 0;
    p_cobj  = nilp;
    d_const = false;
  }

  // create a cell with a literal
  
  Cell::Cell (Literal* cobj) {
    d_quark = 0;
    Object::iref (p_cobj = cobj);
    d_const = false;
  }

  // create a cell by name with a literal
  
  Cell::Cell (const String& name, Literal* cobj) {
    d_quark = name.toquark ();
    Object::iref (p_cobj = cobj);
    d_const = false;
  }

  // copy construct this cell

  Cell::Cell (const Cell& that) {
    that.rdlock ();
    d_quark = that.d_quark;
    d_const = that.d_const;
    Object::iref (p_cobj = that.p_cobj);
    that.unlock ();
  }

  // destroy this cell

  Cell::~Cell (void) {
    Object::dref (p_cobj);
  }

  // return the class name

  String Cell::repr (void) const {
    return "Cell";
  }

  // match a quark against the cell

  bool Cell::operator ==  (const long quark) const {
    rdlock ();
    bool result = (d_quark == quark);
    unlock ();
    return result;
  }

  // return the cell serial code

  t_byte Cell::serialid (void) const {
    return SERIAL_CELL_ID;
  }

  // serialize this cell

  void Cell::wrstream (Output& os) const {
    rdlock ();
    const String& name = String::qmap (d_quark);
    name.wrstream (os);
    if (p_cobj == nilp) 
      os.write (Serial::SERIAL_NILP_ID);
    else
      p_cobj->serialize (os);
    unlock ();
  }

  // deserialize this cell

  void Cell::rdstream (Input& is) {
    wrlock ();
    String  sval;
    sval.rdstream (is);
    Object* cobj = Serial::deserialize (is);
    d_quark = sval.toquark ();
    p_cobj  = dynamic_cast <Literal*> (cobj);
    if ((cobj != nilp) && (p_cobj == nilp)) {
      unlock ();
      throw  Exception ("type-error", "invalid cell type object", 
			cobj->repr ());
    }
    Object::iref (p_cobj);
    unlock ();
  }

  // return the cell name

  String Cell::getname (void) const {
    rdlock ();
    String result = String::qmap (d_quark);
    unlock ();
    return result;
  }

  // set the cell name

  void Cell::setname (const String& name) {
    wrlock ();
    d_quark = name.toquark ();
    unlock ();
  }

  // return the cell literal value

  Literal* Cell::get (void) const {
    rdlock ();
    Literal* result = dynamic_cast <Literal*> (p_cobj);
    unlock ();
    return result;
  }

  // set the cell value

  void Cell::set (Literal* cobj) {
    wrlock ();
    if (d_const == true) {
      unlock ();
      throw Exception ("const-error", "cell const violation");
    } 
    Object::iref (cobj);
    Object::dref (p_cobj);
    p_cobj = cobj;
    unlock ();
  }

  // create a new cell in a generic way

  Object* Cell::mknew (Vector* argv) {
    // get number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Cell;
    // check for one argument
    if (argc == 1) {
      Object*  lobj = argv->get (0);
      Literal* cobj = dynamic_cast <Literal*> (lobj);
      if ((lobj != nilp) && (cobj == nilp))
	throw Exception ("type-error", "invalid object for cell constructor",
			 Object::repr (lobj));
      return new Cell (cobj);
    }
    // check for two argument
    if (argc == 2) {
      String name = argv->getstring (0);
      Object*  lobj = argv->get (1);
      Literal* cobj = dynamic_cast <Literal*> (lobj);
      if ((lobj != nilp) && (cobj == nilp))
	throw Exception ("type-error", "invalid object for cell constructor",
			 Object::repr (lobj));
      return new Cell (name, cobj);
    }    
    throw Exception ("argument-error", "too many arguments with cell");
  }

  // set an object to this cell

  Object* Cell::vdef (Runnable* robj, Nameset* nset, Object* object) {
    Literal* lobj = dynamic_cast <Literal*> (object);
    if (lobj != nilp) {
      set (lobj);
      return this;
    }
    throw Exception ("type-error", "invalid object with cell vdef",
		     Object::repr (object));
  }

  // apply a cell method with a set of arguments and a quark

  Object* Cell::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETNAME) return new String  (getname ());
      if (quark == QUARK_GET) {
	rdlock ();
	Object* result = get ();
	robj->post (result);
	unlock ();
	return result;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nilp;
      }
      if (quark == QUARK_SET) {
	Object*   obj = argv->get (0);
	Literal* cobj = dynamic_cast <Literal*> (obj);
	if ((obj != nilp) && (cobj == nilp)) 
	  throw Exception ("type-error", "invalid object to set in cell",
			   obj->repr ());
	set (cobj);
	return nilp;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
