// ---------------------------------------------------------------------------
// - Axlopts.hpp                                                             -
// - aleph cross librarian - options class implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "System.hpp"
#include "Axlopts.hpp"
#include "OutputTerm.hpp"

namespace aleph {

  // print a usage message when options are wrongs
  static void usage (void) {
    OutputTerm terr (OutputTerm::ERROR);
  
    terr << "usage: axl [options] files\n";
    terr << "           [-h]              print this help message\n";
    terr << "           [-v]              print version information\n";
    terr << "           [-c]              create a new librarian\n";
    terr << "           [-x]              extract from the librarian\n";
    terr << "           [-s]              get file names from the librarian\n";
    terr << "           [-t]              report librarian contents\n";
    terr << "           [-f  librarian]   set the librarian file name\n";
  }

  // print the aleph system version
  static void version (void) {
    OutputTerm terr (OutputTerm::ERROR);
    terr << "aleph cross librarian " << System::osname () << " revision ";
    terr << System::version () << "\n";
  }

  // create a new option class by parsing the argument array

  Axlopts::Axlopts (int argc, char** argv) {
    // prepare for options
    d_cflg = false;
    d_xflg = false;
    d_info = false;
    d_list = false;
    OutputTerm terr(OutputTerm::ERROR);

    // process options
    char optsep = System::optsep ();
    int  count  = 1;
    while (count < argc) {
      String arg = argv[count];
      // check if we have an option
      if (arg[0] == optsep) {
	// check option length
	if (arg.length () != 2) {
	  terr << "fatal: illegal option " << arg << "\n";
	  System::exit (1);
	}
	// switch between options
	switch (arg[1]) {
	case 'h':
	  // print a usage message
	  usage ();
	  System::exit (0);	  
	case 'v':
	  // print system information
	  version ();
	  System::exit (0);
	case 'c':
	  d_cflg = true;
	  count++;
	  continue;
	case 'x':
	  d_xflg = true;
	  count++;
	  continue;
	case 's':
	  d_info = true;
	  d_list = true;
	  count++;
	  continue;
	case 't':
	  d_info = true;
	  count++;
	  continue;
	case 'f':
	  // check for enough arguments
	  if (++count >= argc) {
	    terr << "fatal: missing file name after option -f\n";
	    System::exit (1);
	  }	
	  // process the option value
	  d_lname = argv[count++];
	  continue;
	default:
	  // invalid option
	  terr << "fatal: illegal option " << arg << "\n";
	  System::exit (1);
	}      
      } else {
	d_path.add (argv[count++]);
	continue;
      }
    }
    // check that the librarian file is set
    if (d_lname.length () == 0) {
      terr << "fatal: missing librarian name\n";
      System::exit (1);
    }
    // check that both extact and create are not set
    if ((d_cflg == true) && (d_xflg == true)) {
      terr << "fatal: both create and extract flags are set\n";
      System::exit (1);
    }
  }
}
