#
# Copyright 2001 by Object Craft P/L, Melbourne, Australia.
#
# LICENCE - see LICENCE file distributed with this software for details.
#
# 
# URI parsing and manipulation tests
#
# $Id: uriparse.py,v 1.1 2004/01/15 05:28:24 andrewm Exp $

import unittest
import albatross

class DummyApp:
    def __init__(self, base_url):
        self.__base_url = base_url

    def base_url(self):
        return self.__base_url

class DummyRequest:
    def __init__(self, request_uri):
        self.__request_uri = request_uri

    def get_uri(self):
        return self.__request_uri

class URIparse(unittest.TestCase):
    def _get_ctx(self, base_url, request_uri):
        ctx = albatross.AppContext(DummyApp(base_url))
        ctx.set_request(DummyRequest(request_uri))
        return ctx

    def base_url(self):
        base_url = 'app.py'
        ctx = self._get_ctx(base_url, None)
        self.assertEqual(ctx.base_url(), base_url)

    def current_url(self):
        ctx = self._get_ctx(None, 'http://localhost/cgi-bin/app.py')
        self.assertEqual(ctx.current_url(), '/cgi-bin/app.py')
    
    def absolute_base_url(self):
        ctx = self._get_ctx('app.py', 'http://localhost/cgi-bin/app.py/poo')
        self.assertEqual(ctx.absolute_base_url(), '/cgi-bin/app.py')

        ctx = self._get_ctx('/cgi-bin/app.py', 
                            'http://localhost/cgi-bin/app.py/poo')
        self.assertEqual(ctx.absolute_base_url(), '/cgi-bin/app.py')

        # Check base_url vs request url mismatch
        ctx = self._get_ctx('/cgi-bin/app.py', 
                            'http://localhost/cgi-bin/whatever.py/poo')
        self.assertRaises(albatross.ApplicationError, ctx.absolute_base_url)

    def redirect_url(self):
        # Redirect within the app - make absolute
        ctx = self._get_ctx('app.py', 'http://localhost/cgi-bin/app.py/poo')
        self.assertEqual(ctx.redirect_url('foo'), 
                         'http://localhost/cgi-bin/app.py/foo')

        # Redirect within the app - other access schemes (https)
        ctx = self._get_ctx('app.py', 'https://localhost/cgi-bin/app.py/poo')
        self.assertEqual(ctx.redirect_url('foo'), 
                         'https://localhost/cgi-bin/app.py/foo')

        # Redirect outside the app, leave alone - absolute URI
        ctx = self._get_ctx('app.py', 'http://localhost/cgi-bin/app.py/poo')
        self.assertEqual(ctx.redirect_url('http://localhost/foo.html'), 
                         'http://localhost/foo.html')

        # Redirect outside the app, absolute PATH
        ctx = self._get_ctx('app.py', 'http://localhost/cgi-bin/app.py/poo')
        self.assertEqual(ctx.redirect_url('/foo.html'), 
                         'http://localhost/foo.html')


class URIparseSuite(unittest.TestSuite):
    test_list = (
        'base_url',
        'current_url',
        'absolute_base_url',
        'redirect_url',
    )
    def __init__(self):
        unittest.TestSuite.__init__(self, map(URIparse, self.test_list))

def suite():
    return URIparseSuite()

if __name__ == '__main__':
    unittest.main(defaultTest='suite')
