#
# Copyright 2004 by Object Craft P/L, Melbourne, Australia.
#
# LICENCE - see LICENCE file distributed with this software for details.
#

import base64
try:
    import zlib
    have_zlib = 1
except ImportError:
    have_zlib = 0

from albatross.context import SessionBase, NameRecorderMixin
from albatross.app import AppContext
from albatross.common import *

class BranchingSessionMixin(SessionBase):
    def __init__(self):
        SessionBase.__init__(self)
        self.__sesid = None

    def sesid(self):
        if self.__sesid is None:
            self.__sesid = self.app.new_session()
        return self.__sesid

    def load_session(self):
        if not self.request.has_field('__albsessid__'):
            return
        sesid = self.request.field_value('__albsessid__')
        if len(sesid) < 8 or len(sesid) > 256 or not sesid.isalnum():
            raise SecurityError('Hostile session id in __albsessid__ field: %r'%
                                sesid)
        text = self.app.get_session(sesid)
        if not text:
            if self.request.get_method().upper() != 'GET':
                raise SessionExpired('Session expired')
        self.__sesid = None             # Force new sesid
        if text:
            text = base64.decodestring(text)
            try:
                if have_zlib:
                    text = zlib.decompress(text)
                self.decode_session(text)
            except:
                self.app.del_session(sesid)
                raise

    def save_session(self):
        if self.should_save_session():
            text = self.encode_session()
            if have_zlib:
                text = zlib.compress(text)
            text = base64.encodestring(text)
            self.app.put_session(self.sesid(), text)

    def remove_session(self):
        SessionBase.remove_session(self)
        if self.__sesid is not None:
            self.app.del_session(self.__sesid)
            self.__sesid = None

    def form_close(self):
        if self.should_save_session():
            self.write_content('<input type="hidden" name="__albsessid__" value="%s" />\n' % self.sesid())

class BranchingSessionContext(AppContext, 
                              NameRecorderMixin, 
                              BranchingSessionMixin):
    def __init__(self, app):
        AppContext.__init__(self, app)
        NameRecorderMixin.__init__(self)
        BranchingSessionMixin.__init__(self)

    def form_close(self):
        use_multipart_enc = NameRecorderMixin.form_close(self)
        BranchingSessionMixin.form_close(self)
        return use_multipart_enc
