/**********************************************************************
 ** Marker class: These methods manipulate a marker class in the mud. A
 **               marker class is an object that is non-gettable and that
 **               supplies an examine mainly. It also provides an object at
 **               a location for manipulation, like pushing it
 **
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef MARKER_C
#define MARKER_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "builder.h"
#include "mudobject.h"
#include "item.h"
#include "objtype.h"
#include "marker.h"
#include "flags.h"
#include "player.h"

/***********************************************************************
 ** Marker (constructor) - creates the item
 **
 ** Parameters: the_name - the name of the marker
 **             the_area - the area the marker belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Marker::Marker(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_MARKER;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   set_loc_str("none");
}

/***********************************************************************
 ** ~Marker (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Marker::~Marker()
{
}


/***********************************************************************
 ** write_object - writes the location to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Marker::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nmarker %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());
  
   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
}

/***********************************************************************
 ** describe - describes the marker to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Marker::describe(Builder *the_builder)
{

   the_builder->send_bldr("\n&+GMarker: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_builder->send_bldr("&+GClones: \t&+g%s&*\n", get_clones());
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_builder->send_bldr("&+GLocation: \t&+M%s&*\n", get_location());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_builder->send_bldr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_builder->send_bldr("&+GDesc:&*\n%s\n", get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the marker to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Marker::describe(Player *the_player)
{

   the_player->send_plr("\n&+GMarker: \t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_player->send_plr("&+GClones: \t&+g%s&*\n", get_clones());
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_player->send_plr("&+GLocation: \t&+M%s&*\n", get_location());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_player->send_plr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_player->send_plr("&+GDesc:&*\n%s\n", get_desc());
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   list_specials(the_player);

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Marker::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a key.\n"
                  "   title, altnames, clones, guards, desc, itemflags,\n"
                  "   location, capacity and specials\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "altnames",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "clones",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "capacity",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_capacity(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "guards",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "itemflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "location", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_loc(the_parsed, the_builder);
   }


   the_builder->send_bldr("The attribute '%s' is not a marker attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}




/***********************************************************************
 ** copy_object - copies the marker to an object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Marker::copy_object(Entity *copy_obj)
{
   Marker *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_MARKER)
      return 0;

   copy_from = (Marker *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the item attributes ******/
   copy_item_attrib((Item *) copy_from);

   /****** set the marker attributes ******/
   /* none yet */

   return 1;
}


/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Marker *Marker::operator = (Marker *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Marker::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Marker::get_mem_size_dynamic()
{
   int  size = 0;

   size += get_mem_size_item();
   size += get_mem_size_mudobj(); 
   size += get_mem_size_entity();

   return size;
}


#endif













