/**
 * ========================================
 * JFreeReport : a free Java report library
 * ========================================
 *
 * Project Info:  http://www.object-refinery.com/jfreereport/index.html
 * Project Lead:  Thomas Morgner (taquera@sherito.org);
 *
 * (C) Copyright 2000-2002, by Simba Management Limited and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * The Original Code is 'iText, a free JAVA-PDF library'.
 *
 * The Initial Developer of the Original Code is Bruno Lowagie. Portions created by
 * the Initial Developer are Copyright (C) 1999, 2000, 2001, 2002 by Bruno Lowagie.
 * All Rights Reserved.
 * Co-Developer of the code is Paulo Soares. Portions created by the Co-Developer
 * are Copyright (C) 2000, 2001, 2002 by Paulo Soares. All Rights Reserved.
 *
 * Contributor(s): all the names of the contributors are added in the source code
 * where applicable.
 *
 */
package com.jrefinery.report.ext.barcode;

import com.jrefinery.report.ElementAlignment;
import com.jrefinery.report.ext.barcode.DefaultFontMetrics;
import com.jrefinery.report.targets.FontDefinition;

import java.awt.Color;
import java.awt.Font;
import java.awt.Image;
import java.awt.geom.Dimension2D;

/**
 * Base class containing properties and methods commom to all
 * barcode types.
 *
 * @author Paulo Soares (psoares@consiste.pt)
 */
public abstract class Barcode implements Cloneable
{
  /** A type of barcode */
//  public static final int EAN13 = 1;
  /** A type of barcode */
//  public static final int EAN8 = 2;
  /** A type of barcode */
//  public static final int UPCA = 3;
  /** A type of barcode */
//  public static final int UPCE = 4;
  /** A type of barcode */
//  public static final int SUPP2 = 5;
  /** A type of barcode */
//  public static final int SUPP5 = 6;
  /** A type of barcode */
//  public static final int POSTNET = 7;
  /** A type of barcode */
//  public static final int PLANET = 8;
  /** A type of barcode */
//  public static final int CODE128 = 9;
  /** A type of barcode */
//  public static final int CODE128_UCC = 10;
  /** A type of barcode */
//  public static final int CODE128_RAW = 10;

  public static final FontDefinition DEFAULT_FONT = new FontDefinition("SansSerif", 10);

  /** The minimum bar width. (was X)
   */
  private float minWidth;

  /** The text font. <CODE>null</CODE> if no text.
   */
  private FontDefinition font;

  /**
   * If positive, the text distance under the bars. If zero or negative,
   * the text distance above the bars.
   */
  private float baseline;

  /**
   * The height of the bars.
   */
  private float barHeight;

  /**
   * The text alignment. Can be <CODE>Element.ALIGN_LEFT</CODE>,
   * <CODE>Element.ALIGN_CENTER</CODE> or <CODE>Element.ALIGN_RIGHT</CODE>.
   */
  private ElementAlignment textAlignment;

  /**
   * The optional checksum generation.
   */
  private boolean generateChecksum;

  /**
   * Shows the generated checksum in the the text.
   */
  private boolean displayChecksumInText;

  /**
   * The code to generate.
   */
  private String code;

  public Barcode()
  {
    setMinWidth(0.8f);
    setFont(DEFAULT_FONT);
    int size = DEFAULT_FONT.getFontSize();
    setBaseline(size);
    setBarHeight(size * 3);
    setTextAlignment(ElementAlignment.CENTER);
    generateChecksum = true;
  }

  /**
   * Gets the minimum bar width.
   * @return the minimum bar width
   */
  public float getMinWidth()
  {
    return minWidth;
  }

  /** Sets the minimum bar width.
   * @param minWidth the minimum bar width
   */
  public void setMinWidth(float minWidth)
  {
    this.minWidth = minWidth;
  }

  /** Gets the text font. <CODE>null</CODE> if no text.
   * @return the text font. <CODE>null</CODE> if no text
   */
  public FontDefinition getFont()
  {
    return font;
  }

  /** Sets the text font. <CODE>null</CODE> if no text.
   * @param font the text font. <CODE>null</CODE> if no text
   */
  public void setFont(FontDefinition font)
  {
    this.font = font;
  }

  /** Gets the size of the text.
   * @return the size of the text
   *
  public float getTextSize()
  {
    return textSize;
  }
*/
  /** Sets the size of the text.
   * @param size the size of the text
  public void setTextSize(float size)
  {
    this.textSize = size;
  }
   */

  /** Gets the text baseline.
   * If positive, the text distance under the bars. If zero or negative,
   * the text distance above the bars.
   * @return the baseline.
   */
  public float getBaseline()
  {
    return baseline;
  }

  /** Sets the text baseline.
   * If positive, the text distance under the bars. If zero or negative,
   * the text distance above the bars.
   * @param baseline the baseline.
   */
  public void setBaseline(float baseline)
  {
    this.baseline = baseline;
  }

  /** Gets the height of the bars.
   * @return the height of the bars
   */
  public float getBarHeight()
  {
    return barHeight;
  }

  /** Sets the height of the bars.
   * @param barHeight the height of the bars
   */
  public void setBarHeight(float barHeight)
  {
    this.barHeight = barHeight;
  }

  /** Gets the text alignment. Can be <CODE>Element.ALIGN_LEFT</CODE>,
   * <CODE>Element.ALIGN_CENTER</CODE> or <CODE>Element.ALIGN_RIGHT</CODE>.
   * @return the text alignment
   */
  public ElementAlignment getTextAlignment()
  {
    return textAlignment;
  }

  /** Sets the text alignment. Can be <CODE>Element.ALIGN_LEFT</CODE>,
   * <CODE>Element.ALIGN_CENTER</CODE> or <CODE>Element.ALIGN_RIGHT</CODE>.
   * @param textAlignment the text alignment
   */
  public void setTextAlignment(ElementAlignment textAlignment)
  {
    this.textAlignment = textAlignment;
  }

  /** Gets the optional checksum generation.
   * @return the optional checksum generation
   */
  public boolean isGenerateChecksum()
  {
    return generateChecksum;
  }

  /** Setter for property generateChecksum.
   * @param generateChecksum New value of property generateChecksum.
   */
  public void setGenerateChecksum(boolean generateChecksum)
  {
    this.generateChecksum = generateChecksum;
  }

  /** Gets the property to show the generated checksum in the the text.
   * @return value of property checksumText
   */
  public boolean isDisplayChecksumText()
  {
    return displayChecksumInText;
  }

  /** Sets the property to show the generated checksum in the the text.
   * @param checksumText new value of property checksumText
   */
  public void setDisplayChecksumText(boolean checksumText)
  {
    this.displayChecksumInText = checksumText;
  }

  /** Gets the code to generate.
   * @return the code to generate
   */
  public String getCode()
  {
    return code;
  }

  /** Sets the code to generate.
   * @param code the code to generate
   */
  public void setCode(String code)
  {
    this.code = code;
  }

  /**
   * Gets the maximum area that the barcode and the text, if
   * any, will occupy.
   * @return the size the barcode occupies.
   */
  public abstract Dimension2D getBarcodeSize();

  /** Places the barcode in a <CODE>PdfContentByte</CODE>. The
   * barcode is always placed at coodinates (0, 0). Use the
   * translation matrix to move it elsewhere.<p>
   * The bars and text are written in the following colors:<p>
   * <P><TABLE BORDER=1>
   * <TR>
   *    <TH><P><CODE>barColor</CODE></TH>
   *    <TH><P><CODE>textColor</CODE></TH>
   *    <TH><P>Result</TH>
   *    </TR>
   * <TR>
   *    <TD><P><CODE>null</CODE></TD>
   *    <TD><P><CODE>null</CODE></TD>
   *    <TD><P>bars and text painted with current fill color</TD>
   *    </TR>
   * <TR>
   *    <TD><P><CODE>barColor</CODE></TD>
   *    <TD><P><CODE>null</CODE></TD>
   *    <TD><P>bars and text painted with <CODE>barColor</CODE></TD>
   *    </TR>
   * <TR>
   *    <TD><P><CODE>null</CODE></TD>
   *    <TD><P><CODE>textColor</CODE></TD>
   *    <TD><P>bars painted with current color<br>text painted with <CODE>textColor</CODE></TD>
   *    </TR>
   * <TR>
   *    <TD><P><CODE>barColor</CODE></TD>
   *    <TD><P><CODE>textColor</CODE></TD>
   *    <TD><P>bars painted with <CODE>barColor</CODE><br>text painted with <CODE>textColor</CODE></TD>
   *    </TR>
   * </TABLE>
   *
   * @param cb the <CODE>PdfContentByte</CODE> where the barcode will be placed
   * @param barColor the color of the bars. It can be <CODE>null</CODE>
   * @param textColor the color of the text. It can be <CODE>null</CODE>
   * @return the dimensions the barcode occupies
   */
  //public abstract Rectangle placeBarcode(PdfContentByte cb, Color barColor, Color textColor);

  /** Creates an <CODE>Image</CODE> with the barcode.
   * @param barColor the color of the bars. It can be <CODE>null</CODE>
   * @param textColor the color of the text. It can be <CODE>null</CODE>
   * @return the <CODE>Image</CODE>
   */
  public abstract Image createImageWithBarcode(Color barColor, Color textColor);

  protected float getFontDescent(Font font)
  {
    DefaultFontMetrics fm = new DefaultFontMetrics(font);

    // correction factor..
    double cFact = font.getSize2D() / fm.getHeight();
    float baseline = (float) fm.getAscent();
    float correctedBaseline = (float) (baseline * cFact);

    float l = (font.getSize2D() + correctedBaseline) / 2.0f;
    return l;
  }

  protected float getFontAscent (Font font)
  {
    DefaultFontMetrics fm = new DefaultFontMetrics(font);

    // correction factor..
    double cFact = font.getSize2D() / fm.getHeight();
    float baseline = (float) fm.getAscent();
    float correctedBaseline = (float) (baseline * cFact);
    return correctedBaseline;
  }

  public Object clone () throws CloneNotSupportedException
  {
    return super.clone();
  }
}
