/***************************************************************************
                          ai_group.c  -  description
                             -------------------
    begin                : Fri Jan 19 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "lgeneral.h"
#include "unit.h"
#include "action.h"
#include "map.h"
#include "ai_group.h"

/*
====================================================================
Externals
====================================================================
*/
extern Player *cur_player;
extern List *units, *avail_units;
extern int map_w, map_h;
extern Map_Tile **map;
extern Mask_Tile **mask;
extern int trgt_type_count;
extern int cur_weather;

/*
====================================================================
LOCALS
====================================================================
*/

/* OH NO, A HACK! */
extern int ai_get_dist( Unit *unit, int x, int y, int type, int *dx, int *dy, int *dist );

/*
====================================================================
Check the surrounding of a tile and apply the eval function to
it. Results may be stored in the context.
If 'eval_func' returns False the evaluation is broken up.
====================================================================
*/
typedef struct {
    int x, y;
} AI_Pos;
static AI_Pos *ai_create_pos( int x, int y )
{
    AI_Pos *pos = calloc( 1, sizeof( AI_Pos ) );
    pos->x = x; pos->y = y;
    return pos;
}
static void ai_eval_hexes( int x, int y, int range, int(*eval_func)(int,int,void*), void *ctx )
{
    List *list = list_create( LIST_NO_AUTO_DELETE, LIST_NO_CALLBACK );
    AI_Pos *pos;
    int i, nx, ny;
    /* gather and handle all positions by breitensuche.
       use AUX mask to mark visited positions */
    map_clear_mask( F_AUX );
    list_add( list, ai_create_pos( x, y ) ); mask[x][y].aux = 1;
    list_reset( list );
    while ( list->count > 0 ) {
        pos = list_dequeue( list );
        if ( !eval_func( pos->x, pos->y, ctx ) ) {
            free( pos );
            break;
        }
        for ( i = 0; i < 6; i++ )
            if ( get_close_hex_pos( pos->x, pos->y, i, &nx, &ny ) )
                if ( !mask[nx][ny].aux && get_dist( x, y, nx, ny ) <= range ) {
                    list_add( list, ai_create_pos( nx, ny ) );
                    mask[nx][ny].aux = 1;
                }
        free( pos );
    }
    list_delete( list );
}

/*
====================================================================
Check if there is an unspotted tile or an enemy within range 3.
====================================================================
*/
typedef struct {
    Player *player;
    int unsafe;
} MountCtx;
static int hex_is_safe( int x, int y, void *_ctx )
{
    MountCtx *ctx = _ctx;
    if ( !mask[x][y].spot ) {
        ctx->unsafe = 1;
        return 0;
    }
    if ( map[x][y].g_unit && !player_is_ally( ctx->player, map[x][y].player ) ) {
        ctx->unsafe = 1;
        return 0;
    }
    return 1;
}
static int ai_unsafe_mount( Unit *unit, int x, int y ) 
{
    MountCtx ctx = { unit->player, 0 };
    ai_eval_hexes( x, y, 3, hex_is_safe, &ctx );
    return ctx.unsafe;
}

/*
====================================================================
Count the number of defensive supporters.
====================================================================
*/
typedef struct {
    Unit *unit;
    int count;
} DefCtx;
static int hex_df_unit( int x, int y, void *_ctx )
{
    DefCtx *ctx = _ctx;
    if ( map[x][y].g_unit ) {
        if ( ctx->unit->sel_prop->flags & FLYING ) {
            if ( map[x][y].g_unit->sel_prop->flags & AIR_DEFENSE )
                ctx->count++;
        }
        else {
            if ( map[x][y].g_unit->sel_prop->flags & ARTILLERY )
                ctx->count++;
        }
    }
    return 1;
}
static void ai_count_df_units( Unit *unit, int x, int y, int *result )
{
    DefCtx ctx = { unit, 0 };
    *result = 0;
    if ( unit->sel_prop->flags & ARTILLERY )
        return;
    ai_eval_hexes( x, y, 3, hex_df_unit, &ctx );
    /* only three defenders are taken in account */
    if ( *result > 3 )
        *result = 3;
}

/*
====================================================================
Gather all valid targets of a unit.
====================================================================
*/
typedef struct {
    Unit *unit;
    List *targets;
} GatherCtx;
static int hex_add_targets( int x, int y, void *_ctx )
{
    GatherCtx *ctx = _ctx;
    if ( mask[x][y].spot ) {
        if ( map[x][y].a_unit && unit_check_attack( ctx->unit, map[x][y].a_unit, UNIT_ACTIVE_ATTACK ) )
            list_add( ctx->targets, map[x][y].a_unit );
        if ( map[x][y].g_unit && unit_check_attack( ctx->unit, map[x][y].g_unit, UNIT_ACTIVE_ATTACK ) )
            list_add( ctx->targets, map[x][y].g_unit );
    }
    return 1;
}
static List* ai_gather_targets( Unit *unit, int x, int y )
{
    GatherCtx ctx;
    ctx.unit = unit;
    ctx.targets= list_create( LIST_NO_AUTO_DELETE, LIST_NO_CALLBACK );
    ai_eval_hexes( x, y, unit->sel_prop->rng + 1, hex_add_targets, &ctx );
    return ctx.targets;
}


/*
====================================================================
Evaluate a unit's attack against target.
  score_base: basic score for attacking
  score_rugged: score added for each rugged def point (0-100)
                of target
  score_kill: score unit receives for each (expected) point of
              strength damage done to target
  score_loss: score that is substracted per strength point
              unit is expected to loose
The final score is stored to 'result' and True if returned if the
attack may be performed else False.
====================================================================
*/
static int unit_evaluate_attack( Unit *unit, Unit *target, int score_base, int score_rugged, int score_kill, int score_loss, int *result )
{
    int unit_dam = 0, target_dam = 0, rugged_def = 0;
    if ( !unit_check_attack( unit, target, UNIT_ACTIVE_ATTACK ) ) return 0;
    unit_get_expected_losses( unit, target, &unit_dam, &target_dam );
    if ( unit_check_rugged_def( unit, target ) )
        rugged_def = unit_get_rugged_def_chance( unit, target );
    if ( rugged_def < 0 ) rugged_def = 0;
    *result = score_base + rugged_def * score_rugged + target_dam * score_kill + unit_dam * score_loss;
    /* if target is a df unit give a small bonus */
    if ( target->sel_prop->flags & ARTILLERY || target->sel_prop->flags & AIR_DEFENSE )
        *result += score_kill;
    return 1;
}

/*
====================================================================
Get the best target for unit if any.
====================================================================
*/
static int ai_get_best_target( Unit *unit, int x, int y, AI_Group *group, Unit **target, int *score )
{
    int old_x = unit->x, old_y = unit->y;
    int result;
    Unit *entry;
    List *targets;
    int score_atk_base, score_rugged, score_kill, score_loss;
    
    /* scores */
    score_atk_base = ( group->order + 2 ) * 10;
    score_rugged   = -2;
    score_kill     = 20; /*( group->order + 3 ) * 10;*/
    score_loss     = ( 2 - group->order ) * -10;
    
    unit->x = x; unit->y = y;
    *target = 0; *score = -999999;
    /* if the transporter is needed attacking is suicide */
    if ( mask[x][y].mount && unit->trsp_prop.id )
        return 0;
    /* gather targets */
    targets = ai_gather_targets( unit, x, y );
    /* get best target */
    if ( targets ) {
        list_reset( targets ); 
        while ( ( entry = list_next( targets ) ) )
            if ( unit_evaluate_attack( unit, entry, score_atk_base, score_rugged, score_kill, score_loss, &result ) ) {
                if ( result > *score ) {
                    *target = entry;
                    *score = result;
                }
            }
        list_delete( targets );
    }
    unit->x = old_x; unit->y = old_y;
    return (*target) != 0;
}

/*
====================================================================
Evaluate position for a unit by checking the group context. 
Return True if this evaluation is valid. The results are stored
to 'eval'.
====================================================================
*/
typedef struct {
    Unit *unit; /* unit that's checked */
    AI_Group *group;
    int x, y; /* position that was evaluated */
    int mov_score; /* result for moving */
    Unit *target; /* if set atk_result is relevant */
    int atk_score; /* result including attack evaluation */
} AI_Eval;
static AI_Eval *ai_create_eval( Unit* unit, AI_Group *group, int x, int y )
{
    AI_Eval *eval = calloc( 1, sizeof( AI_Eval ) );
    eval->unit = unit; eval->group = group;
    eval->x = x; eval->y = y;
    return eval;
}
int ai_evaluate_hex( AI_Eval *eval )
{
    int result;
    int i, nx, ny, ox, oy,odist;
    eval->target = 0;
    eval->mov_score = eval->atk_score = 0;
    /* terrain modifications which only apply for ground units */
    if ( !(eval->unit->sel_prop->flags & FLYING ) ) {
        /* entrenchment bonus. infantry receives more than others. */
        eval->mov_score += 5 * ((eval->unit->sel_prop->flags&INFANTRY)?2:1) *
                           ( map[eval->x][eval->y].terrain->min_entr + 
                             map[eval->x][eval->y].terrain->min_entr ) / 2;
        /* if the unit looses initiative on this terrain we give a malus */
        if ( map[eval->x][eval->y].terrain->max_ini < eval->unit->sel_prop->ini )
            eval->mov_score -= 5 * ( eval->unit->sel_prop->ini - 
                                      map[eval->x][eval->y].terrain->max_ini );
        /* rivers should be avoided */
        if ( map[eval->x][eval->y].terrain->flags[cur_weather] & RIVER )
            eval->mov_score -= 100;
        /* inf_close_def will benefit an infantry while disadvantaging
           other units */
        if ( map[eval->x][eval->y].terrain->flags[cur_weather] & INF_CLOSE_DEF ) {
            if ( eval->unit->sel_prop->flags & INFANTRY )
                eval->mov_score += 50;
            else
                eval->mov_score -= 20;
        }
        /* if this is a mount position and an enemy or fog is less than
           3 tiles away we give a malus */
        if ( mask[eval->x][eval->y].mount )
            if ( ai_unsafe_mount( eval->unit, eval->x, eval->y ) )
                eval->mov_score -= 150;
        /* conquering a flag gives a bonus */
        if ( map[eval->x][eval->y].player )
            if ( !player_is_ally( eval->unit->player, map[eval->x][eval->y].player ) )
                if ( map[eval->x][eval->y].g_unit == 0 ) {
                    eval->mov_score += 100;
                    if ( map[eval->x][eval->y].obj )
                        eval->mov_score += 100;
                }
        /* if this position allows debarking or is just one hex away
           this tile receives a big bonus. */
        if ( eval->unit->embark == EMBARK_SEA ) {
            if ( map_check_unit_debark( eval->unit, eval->x, eval->y, EMBARK_SEA ) )
                eval->mov_score += 400;
            else
                for ( i = 0; i < 6; i++ )
                    if ( get_close_hex_pos( eval->x, eval->y, i, &nx, &ny ) )
                        if ( map_check_unit_debark( eval->unit, nx, ny, EMBARK_SEA ) ) {
                            eval->mov_score += 200;
                            break;
                        }
        }
    }
    /* each group has a 'center of interest'. getting closer
       to this center is honored. */
    if ( eval->group->x == -1 ) {
        /* proceed to the nearest flag */
        if ( !(eval->unit->sel_prop->flags & FLYING ) ) {
            if ( eval->group->order > 0 ) {
                if ( ai_get_dist( eval->unit, eval->x, eval->y, AI_FIND_ENEMY_OBJ, &ox, &oy, &odist ) )
                    eval->mov_score -= odist * 10;
            }
            else
                if ( eval->group->order < 0 ) {
                    if ( ai_get_dist( eval->unit, eval->x, eval->y, AI_FIND_OWN_OBJ, &ox, &oy, &odist ) )
                        eval->mov_score -= odist * 10;
                }
        }
    }
    else
        eval->mov_score -= 10 * get_dist( eval->x, eval->y, eval->group->x, eval->group->y );
    /* defensive support */
    ai_count_df_units( eval->unit, eval->x, eval->y, &result );
    eval->mov_score += result * 10;
    /* check for the best target and save the result to atk_score */
    eval->atk_score = eval->mov_score;
    if ( !mask[eval->x][eval->y].mount )
    if ( !( eval->unit->sel_prop->flags & ATTACK_FIRST ) || 
          ( eval->unit->x == eval->x && eval->unit->y == eval->y ) )
    if ( ai_get_best_target( eval->unit, eval->x, eval->y, eval->group, &eval->target, &result ) )
        eval->atk_score += result;
    return 1;
}

/*
====================================================================
Choose and store the best tactical action of a unit (found by use of
ai_evaluate_hex). If there is none AI_SUPPLY is stored.
====================================================================
*/
void ai_handle_unit( Unit *unit, AI_Group *group )
{
    int x, y, nx, ny, i, action = 0;
    List *list = list_create( LIST_NO_AUTO_DELETE, LIST_NO_CALLBACK );
    AI_Eval *eval;
    Unit *target = 0;
    int score = -999999;
    /* get move mask */
    map_get_unit_move_mask( unit );
    x = unit->x; y = unit->y; target = 0;
    /* evaluate all positions */
    list_add( list, ai_create_eval( unit, group, unit->x, unit->y ) );
    while ( list->count > 0 ) {
        eval = list_dequeue( list );
        if ( ai_evaluate_hex( eval ) ) {
            /* movement evaluation */
            if ( eval->mov_score > score ) {
                score = eval->mov_score;
                target = 0;
                x = eval->x; y = eval->y;
            }
            /* movement + attack evaluation */
            if ( eval->target && eval->atk_score > score ) {
                score = eval->atk_score;
                target = eval->target;
                x = eval->x; y = eval->y;
            }
        }
        /* store next hex tiles */
        for ( i = 0; i < 6; i++ )
            if ( get_close_hex_pos( eval->x, eval->y, i, &nx, &ny ) )
                if ( ( mask[nx][ny].in_range && !mask[nx][ny].blocked ) || mask[nx][ny].sea_embark ) {
                    mask[nx][ny].in_range = 0; 
                    mask[nx][ny].sea_embark = 0;
                    list_add( list, ai_create_eval( unit, group, nx, ny ) );
                }
        free( eval );
    }
    list_delete( list );
    /* check result and store appropiate action */
    if ( unit->x != x || unit->y != y ) {
        if ( map_check_unit_debark( unit, x, y, EMBARK_SEA ) ) {
            action_queue_debark_sea( unit, x, y ); action = 1;
#ifdef DEBUG_AI
            printf( "%s debarks at %i,%i\n", unit->name, x, y );
#endif
        }
        else {
            action_queue_move( unit, x, y ); action = 1;
#ifdef DEBUG_AI
            printf( "%s moves to %i,%i\n", unit->name, x, y );
#endif
        }
    }
    if ( target ) {
        action_queue_attack( unit, target ); action = 1;
#ifdef DEBUG_AI
        printf( "%s attacks %s\n", unit->name, target->name );
#endif
    }
    if ( !action ) {
        action_queue_supply( unit );
#ifdef DEBUG_AI
        printf( "%s supplies\n", unit->name );
#endif
    }
}

/*
====================================================================
PUBLICS
====================================================================
*/

/*
====================================================================
Create/Delete a group
====================================================================
*/
AI_Group *ai_group_create( int order, int x, int y )
{
    AI_Group *group = calloc( 1, sizeof( AI_Group ) );
    group->order = order;
    group->x = x; group->y = y;
    group->units = list_create( LIST_NO_AUTO_DELETE, LIST_NO_CALLBACK );
    list_reset( group->units );
    return group;
}
void ai_group_add_unit( AI_Group *group, Unit *unit )
{
    list_add( group->units, unit );
}
void ai_group_delete( void *ptr )
{
    AI_Group *group = ptr;
    if ( group ) {
        if ( group->units )
            list_delete( group->units );
        free( group );
    }
}
/*
====================================================================
Handle next unit of a group to follow order. Stores all nescessary 
unit actions. If group is completely handled it returns False.
====================================================================
*/
int ai_group_handle_next_unit( AI_Group *group )
{
    Unit *unit = list_next( group->units );
    if ( unit == 0 ) return 0;
    ai_handle_unit( unit, group );
        /* artillery shoots first */
        /* bombers*/
        /* fighters */
        /* normal units */
        /* artillery moves */
    return 1;
}
