/****************************************************************************
/****************************************************************************
 *                                                                          *
 * Copyright 1999-2005 ATI Technologies Inc., Markham, Ontario, CANADA.     *
 * All Rights Reserved.                                                     *
 *                                                                          *
 * Your use and or redistribution of this software in source and \ or       *
 * binary form, with or without modification, is subject to: (i) your       *
 * ongoing acceptance of and compliance with the terms and conditions of    *
 * the ATI Technologies Inc. software End User License Agreement; and (ii)  *
 * your inclusion of this notice in any version of this software that you   *
 * use or redistribute.  A copy of the ATI Technologies Inc. software End   *
 * User License Agreement is included with this software and is also        *
 * available by contacting ATI Technologies Inc. at http://www.ati.com      *
 *                                                                          *
 ****************************************************************************/
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/pagemap.h>
#include <linux/miscdevice.h>
#include <linux/pm.h>
#include <linux/agp_backend.h>
#include <linux/agpgart.h>
#include <linux/vmalloc.h>
#include <asm/io.h>
#include <linux/module.h>
#include <linux/kmod.h>
#include "agp.h"
#include "firegl_agp_module.h"
#include <linux/version.h>


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,10)
#define	firegl_pci_find_class(class, from) pci_get_class(class,from)
#else
#define	firegl_pci_find_class(class, from) pci_find_class(class,from)
#endif

typedef struct {
	void			(*free_memory)(struct agp_memory *);
	struct agp_memory *	(*allocate_memory)(size_t, u32);
	int			(*bind_memory)(struct agp_memory *, off_t);
	int			(*unbind_memory)(struct agp_memory *);
	void			(*enable)(u32);
	int			(*acquire)(void);
	void			(*release)(void);
	int			(*copy_info)(struct agp_kern_info *);
	struct agp_memory * (*agp_allocate_memory_phys_list)(size_t, u32, unsigned long *);
} fglrx_agp_t;

static const fglrx_agp_t fglrx_agp = {
	&agp_free_memory,
	&agp_allocate_memory,
	&agp_bind_memory,
	&agp_unbind_memory,
	&agp_enable,
	&agp_backend_acquire,
	&agp_backend_release,
	&agp_copy_info,
	&agp_allocate_memory_phys_list
};


static struct pci_dev *dev = NULL;

typedef struct pci_device_id_agp {
        __u32 vendor, device;           /* Vendor and device ID or PCI_ANY_ID*/
        __u32 subvendor, subdevice;     /* Subsystem ID's or PCI_ANY_ID */
        __u32 class, class_mask;        /* (class,subclass,prog-if) triplet */
        kernel_ulong_t driver_data;     /* Data private to the driver */
} pci_device_id_agp;

void _X(pci_unregister_driver)(struct pci_driver* driver)
{
}

int _X(pci_module_init)(struct pci_driver* driver)
{
    int i = 0;

    while (driver->id_table[i].vendor)
    {
        if (driver->id_table[i].vendor == dev->vendor)
        {
            if ((driver->id_table[i].device == dev->device) ||
                (driver->id_table[i].device == PCI_ANY_ID))
            {
                if (!driver->probe(dev, &driver->id_table[i])){
                    return 0;
                }
            }
        }
        i++;
    }
   printk(KERN_INFO "failed pci_module_init \n");
   firegl_agp_exit();
    return -ENODEV;
}

typedef int (*agp_init_func)(void);
typedef void (*agp_exit_func)(void);


struct 
{
    agp_init_func fgl_agp_init_func;
    agp_exit_func fgl_agp_exit_func;
}
fgl_agp_module_funcs[] =
{
    {fgl_agp_intel_init,       fgl_agp_intel_cleanup},
//    {fgl_agp_ali_init,         fgl_agp_ali_cleanup},
#ifndef CONFIG_GART_IOMMU
//    {fgl_agp_amd64_init,       fgl_agp_amd64_cleanup},
#endif
//    {fgl_agp_amdk7_init,       fgl_agp_amdk7_cleanup},
//    {fgl_agp_ati_init,         fgl_agp_ati_cleanup},
//    {fgl_agp_efficeon_init,    fgl_agp_efficeon_cleanup},
//    {fgl_agp_intelmch_init,    fgl_agp_intelmch_cleanup},
//    {fgl_agp_nvidia_init,      fgl_agp_nvidia_cleanup},
//    {fgl_agp_sis_init,         fgl_agp_sis_cleanup},
//    {fgl_agp_serverworks_init, fgl_agp_serverworks_cleanup},
//    {fgl_agp_via_init,         fgl_agp_via_cleanup},
    {NULL,                     NULL},
};


static int firegl_agp_probe(void)
{
    int i = 0;
	
    printk(KERN_INFO "fglrx_agp is probing for an AGP device\n");

    if ((dev = firegl_pci_find_class(PCI_CLASS_BRIDGE_HOST << 8, dev)) == NULL)
        return -ENODEV;
    printk (KERN_INFO "Device found = %x\n", dev->vendor);


#if 1
#ifdef __x86_64__
#ifdef CONFIG_GART_IOMMU
    printk (KERN_INFO "AMD64 AGPGART is used by IOMMU, cannot use FireGL build-in driver.\n");
    return -ENODEV;    
#else // CONFIG_GART_IOMMU
    if (fgl_agp_amd64_init() == 0) return 0;
#endif // CONFIG_GART_IOMMU
#else // __x86_64__
    while (fgl_agp_module_funcs[i].fgl_agp_init_func)
    {
        if (fgl_agp_module_funcs[i].fgl_agp_init_func() == 0) return 0;
        i++;
    }
#endif // __x86_64__
#else
    while (fgl_agp_module_funcs[i].fgl_agp_init_func)
    {
        if (fgl_agp_module_funcs[i].fgl_agp_init_func() == 0) return 0;
        i++;
    }
#endif

    printk (KERN_INFO "firegl_agp_probe failed\n");
    return -ENODEV;
}


extern void _X(agp_cleanup)(void)
{
    agp_remove_bridge(agp_bridge);
}


extern void agp_free_memory_phys_list(struct agp_memory * curr)
{
    int i;

    if ((agp_bridge->type == NOT_SUPPORTED) || (curr == NULL)) {
        return;
    }

    if (curr->is_bound == TRUE) {
        agp_unbind_memory(curr);
    }

    if (curr->type != 0) return;

    if (curr->page_count != 0) {
        for (i = 0; i < curr->page_count; i++) {
            curr->memory[i] &= ~(0x00000fff);
        }
    }

    agp_free_key(curr->key);    // release the agp handle
    vfree(curr->memory);        // release the table with the phys adresses
    kfree(curr);                // release the agp_memory structure itself

    //module_put(THIS_MODULE);
}

#define ENTRIES_PER_PAGE                (PAGE_SIZE / sizeof(unsigned long))
#define AGP_ALLOC_TYPE_REGULAR     0x00000000
#define AGP_ALLOC_TYPE_PHYS_LIST   0x00000001

extern void _X(agp_free_memory)(struct _X(agp_memory) * curr)
{
    if ((agp_bridge->type == NOT_SUPPORTED) || (curr == NULL)) {
        return;
    }
    if (curr->is_bound == TRUE) {
        agp_unbind_memory(curr);
    }

    // does this memory need release trough the alternate slot?
    if (curr->alloc_type == AGP_ALLOC_TYPE_PHYS_LIST)
    {
        agp_free_memory_phys_list(curr);
        return;
    }

    agp_free_memory(curr);
}

extern struct _X(agp_memory) *_X(agp_allocate_memory_phys_list) (size_t page_count, u32 type, unsigned long *phys_addr)
{
    int scratch_pages;
    struct agp_memory *new;
    int i;

    if (agp_bridge->type == NOT_SUPPORTED) return NULL;

    if ((atomic_read(&agp_bridge->current_memory_agp) + page_count) >
        agp_bridge->max_memory_agp) {
        return NULL;
    }

    if (type != 0) return NULL;    // types are not yet supported for this call

    /* We always increase the module count, since free auto-decrements it
     */

    /* allocate a "new" agp_memory structure for the requested pages */
    scratch_pages = (page_count + ENTRIES_PER_PAGE - 1) / ENTRIES_PER_PAGE;
    new = agp_create_memory(scratch_pages);
    if (new == NULL) {
        //module_put(THIS_MODULE);
        return NULL;
    }
    new->alloc_type = AGP_ALLOC_TYPE_PHYS_LIST;

    /* convert/copy the physical adresses into the "new" agp_memory table */
    for (i = 0; i < page_count; i++) 
    {
        new->memory[i] = agp_bridge->driver->mask_memory( phys_addr[i], type);
    }
    new->page_count += page_count;


    return new;
}
/* Generic agp page allocation */
void* firegl_agp_alloc_pages(void)
{
	struct page * page;
	
	page = alloc_page(GFP_KERNEL);
	if (page == NULL)
		return NULL;

	map_page_into_agp(page);

	get_page(page);
	SetPageLocked(page);
	atomic_inc(&agp_bridge->current_memory_agp);
	return page_address(page);
}
void firegl_agp_generic_destroy_page(void *addr)
{
	struct page *page;

	if (addr == NULL)
		return;

	page = virt_to_page(addr);
	unmap_page_from_agp(page);
	put_page(page);
	unlock_page(page);
	free_page((unsigned long)addr);
	atomic_dec(&agp_bridge->current_memory_agp);
}

static char *fglrx_agp_string = "From fglrx_agp.ko: Inter module comm successful with fglrx_agp.ko";

static const fglrx_agp_t *fglrx_agp_stub = NULL;

/* init_module is called when insmod is used to load the module */
static int __init firegl_agp_init(void)
{
    int ret_val;

    printk(KERN_INFO "Fire GL agpgart support\n");
    inter_module_register("im_string", THIS_MODULE, fglrx_agp_string);
    inter_module_register("im_fglrx_agp", THIS_MODULE, &fglrx_agp);

    //Check for kernel AGP
    fglrx_agp_stub = (fglrx_agp_t *)inter_module_get_request("drm_agp", "drm_agp");
    
    //If it's there's a kernel AGP don't configure agpgart. 
    if (fglrx_agp_stub)
    {
	    printk(KERN_INFO "Kernel AGP detected\n");
	    return 0;
    }

    ret_val = firegl_agp_probe();
    if (ret_val) 
    {
        agp_bridge->type = NOT_SUPPORTED;
        return ret_val;
    }

    return 0;
}

/* cleanup_module is called when rmmod is used to unload the module */
void __exit firegl_agp_exit(void)
{
	inter_module_unregister("im_string");
	inter_module_unregister("im_fglrx_agp");

    //release use of data from drm_agp module
	if (fglrx_agp_stub) {
		inter_module_put("drm_agp");
	}
	
	printk(KERN_INFO "Unloading fglrx_agp\n");
}

module_init(firegl_agp_init);
module_exit(firegl_agp_exit);
