/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qtextedit.h>
#include <qsyntaxhighlighter.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qstatusbar.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <qcursor.h>
#include <qregexp.h>
#include <qclipboard.h>
#include <qtoolbutton.h>
#include "domain.h"
#include "mainimpl.h"
#include "git.h"
#include "annotate.h"
#include "filecontent.h"

#define MAX_LINE_NUM 5

class FileHighlighter : public QSyntaxHighlighter {
public:
	FileHighlighter(QTextEdit* te, FileContent* fc) : QSyntaxHighlighter(te), f(fc) {};
	int highlightParagraph(const QString& p, int) {

		if (!f->isRangeFilterActive)
			setFormat(0, p.length(), textEdit()->currentFont());

		if (f->isShowAnnotate)
			setFormat(0, f->annoLen + MAX_LINE_NUM, Qt::lightGray);
		else
			setFormat(0, MAX_LINE_NUM, Qt::lightGray);

		if (f->isRangeFilterActive && f->rangeInfo->start != 0)
			if (f->rangeInfo->start - 1 <= currentParagraph() &&
			    f->rangeInfo->end - 1 >= currentParagraph()) {

				QFont fn(textEdit()->currentFont());
				fn.setBold(true);
				setFormat(0, p.length(), fn, Qt::blue);
			}
		return 0;
	}
private:
	FileContent* f;
};

FileContent::FileContent(Domain* dm, Git* g, QTextEdit* f) :
                         d(dm), git(g), ft(f) {

	st = &(d->st);

	// init native types
	isShowAnnotate = true;
	isFileAvailable = isRangeFilterActive = false;
	isAnnotateAvailable = isAnnotationAppended = false;
	annoLen = 0;
	line = 1;

	rangeInfo = new RangeInfo();
	fileHighlighter = new FileHighlighter(ft, this);

	connect(ft, SIGNAL(doubleClicked(int,int)),this, SLOT(on_doubleClicked(int,int)));
	connect(git, SIGNAL(annotateReady(Annotate*, const QString&, bool, const QString&)),
	        this, SLOT(on_annotateReady(Annotate*,const QString&,bool,const QString&)));
}

FileContent::~FileContent() {

	clear();
	delete fileHighlighter;
	delete rangeInfo;
}

void FileContent::clearAnnotate() {

	git->cancelAnnotate(annotateObj);
	annotateObj = NULL;
	annoLen = 0;
	curAnn = NULL;
	isAnnotateAvailable = false;
	emit annotationAvailable(false);
}

void FileContent::clearText(bool emitSignal) {

	git->cancelProcess(proc);
	ft->clear();
	fileData = pending = "";
	isFileAvailable = isAnnotationAppended = false;
	line = 1;
	if (emitSignal)
		emit fileAvailable(false);
}

void FileContent::clear() {

	clearText();
	clearAnnotate();
}

void FileContent::update(bool force) {

	bool shaChanged = (st->sha(true) != st->sha(false));
	bool fileNameChanged = (st->fileName(true) != st->fileName(false));

	if (!fileNameChanged && !shaChanged && !force)
		return;

	saveScreenState();

	if (fileNameChanged)
		clear();
	else
		clearText();

	lookupAnnotation(); // before file loading
	proc = git->getFile(st->fileName(), st->sha(), this, NULL); // non blocking

	if (curAnn) {
		// call seekPosition() while loading the file so to shadow the compute time
		if (ss.hasSelectedText)
			ss.isValid = (annotateObj->seekPosition(ss.paraFrom, ss.paraTo,
			              st->sha(false), st->sha()));
		else
			ss.isValid = (annotateObj->seekPosition(ss.topPara, ss.topPara,
			              st->sha(false), st->sha()));
	} else
		ss.isValid = false;
}

void FileContent::startAnnotate() {

	annotateObj = git->startAnnotate(d); // non blocking
}

uint FileContent::annotateLength(const FileAnnotation* annFile) {

	uint maxLen = 0;
	loopList(it, annFile->lines)
		if ((*it).length() > maxLen)
			maxLen = (*it).length();

	return maxLen;
}

bool FileContent::getRange(SCRef sha, RangeInfo* r) {

	if (annotateObj)
		return  annotateObj->getRange(sha, r);

	return false;
}

void FileContent::goToAnnotation(int revId) {

	if (!isAnnotationAppended || !curAnn || revId == 0)
		return;

	const QString firstLine(QString::number(revId) + ".");
	int idx = 0;
	QStringList::const_iterator it(curAnn->lines.constBegin());

	for ( ; it != curAnn->lines.constEnd(); ++it, ++idx)
		if ((*it).stripWhiteSpace().startsWith(firstLine))
			break;

	if (it == curAnn->lines.constEnd())
		return;

	ft->setSelection(idx, 0, idx, (*it).length());
}

bool FileContent::goToRangeStart() {

	if (!isRangeFilterActive || rangeInfo->start == 0 || !curAnn)
		return false;

	// scroll the viewport so that range is at top
	ft->setCursorPosition(rangeInfo->start - 2, 0);
	int t = ft->paragraphRect(rangeInfo->start - 2).top();
	ft->setContentsPos(0, t);
	return true;
}

void FileContent::copySelection() {

	if (!ft->hasSelectedText())
		return;

	int headLen = annoLen + MAX_LINE_NUM + 1; // 1 is for space after line number
	if (!isShowAnnotate)
		headLen -= annoLen;

	int indexFrom, dummy;
	ft->getSelection(&dummy, &indexFrom, &dummy, &dummy);

	QString sel(ft->selectedText());
	QClipboard* cb = QApplication::clipboard();

	// check if the selection starts in the middle of an header
	if (indexFrom < headLen) {
		sel.remove(0, headLen - indexFrom);
		if (sel.isEmpty()) { // an header part, like the author name, was selected
			cb->setText(ft->selectedText(), QClipboard::Clipboard);
			return;
		}
	}
	QRegExp re(QRegExp("\n.{0," + QString::number(headLen) + "}"));
	sel.replace(re, "\n");
	cb->setText(sel, QClipboard::Clipboard);
}

bool FileContent::rangeFilter(bool b) {

	isRangeFilterActive = false;

	if (b) {
		if (!annotateObj) {
			dbs("ASSERT in rangeFilter: annotateObj not available");
			return false;
		}
		int indexFrom, paraFrom, paraTo, dummy;
		ft->getSelection(&paraFrom, &indexFrom, &paraTo, &dummy);

		QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
		EM_PROCESS_EVENTS_NO_INPUT;

		QString anc = annotateObj->computeRanges(st->sha(), ++paraFrom, ++paraTo);

		QApplication::restoreOverrideCursor();

		if (anc.isEmpty())
			return false;

		if (annotateObj->getRange(anc, rangeInfo)) {
			isRangeFilterActive = true;
			fileHighlighter->rehighlight();
			int st = rangeInfo->start - 1;
			ft->setSelection(st, 0, st, 0); // clear selection
			goToRangeStart();
			return true;
		}
	} else {
		fileHighlighter->rehighlight();
		ft->setSelection(rangeInfo->start - 1, 0, rangeInfo->end, 0);
		rangeInfo->clear();
	}
	return false;
}

bool FileContent::lookupAnnotation() {

	if (st->sha().isEmpty() || st->fileName().isEmpty() ||
		!isAnnotateAvailable || !annotateObj)
		return false;

	curAnn = git->lookupAnnotation(annotateObj, st->fileName(), st->sha());

	if (curAnn)
		annoLen = annotateLength(curAnn);
	else {
		dbp("ASSERT in lookupAnnotation: no annotation for %1", st->fileName());
		clearAnnotate();
	}
	return (curAnn != NULL);
}

void FileContent::saveScreenState() {

	ss.isValid = true;
	ss.topPara = ft->paragraphAt(QPoint(ft->contentsX(), ft->contentsY()));
	ss.hasSelectedText = ft->hasSelectedText();
	if (ss.hasSelectedText) {
		ft->getSelection(&ss.paraFrom, &ss.indexFrom, &ss.paraTo, &ss.indexTo);
		ss.annoLen = annoLen;
		ss.isShowAnnotate = isShowAnnotate;
	}
}

void FileContent::restoreScreenState() {

	if (!ss.isValid)
		return;

	if (ss.hasSelectedText) {
		// index without previous annotation
		ss.indexFrom -= (ss.isShowAnnotate) ? ss.annoLen : 0;
		ss.indexTo -= (ss.isShowAnnotate) ? ss.annoLen : 0;

		// index with current annotation
		ss.indexFrom += (isShowAnnotate) ? annoLen : 0;
		ss.indexTo += (isShowAnnotate) ? annoLen : 0;
		ss.indexFrom = QMAX(ss.indexFrom, 0);
		ss.indexTo = QMAX(ss.indexTo, 0);
		ft->setSelection(ss.paraFrom, ss.indexFrom, ss.paraTo, ss.indexTo); // slow

	} else if (ss.topPara != 0) {
		int t = ft->paragraphRect(ss.topPara).bottom(); // slow for big files
		ft->setContentsPos(0, t);
	}
	ss.isShowAnnotate = isShowAnnotate; // leave ss in a consistent state
	ss.annoLen = annoLen;
}

void FileContent::addAnnotation() {
// annotation was not ready at the beginning of file content reading,
// so feed again on_procDataReady() to add also annotation data

	if (!isFileAvailable || !isAnnotateAvailable ||
		(isAnnotationAppended == isShowAnnotate))
		return;

	const QString tmp(fileData);

	isShowAnnotate = !isShowAnnotate;
	saveScreenState();
	isShowAnnotate = !isShowAnnotate;

	clearText(false);
	on_procDataReady(tmp);
	on_eof(false); // will call restoreScreenState()
}

// ************************************ SLOTS ********************************

void FileContent::setShowAnnotate(bool b)  {

	isShowAnnotate = b;
	addAnnotation();
}

void FileContent::on_annotateReady(Annotate* readyAnn, const QString& fileName,
						   bool ok, const QString& msg) {
	if (!ok) {
		d->m()->statusBar()->message("Sorry, annotation not available for this file.");
		return;
	}
	if (readyAnn != annotateObj) {
		dbs("ASSERT in on_annotateReady: arrived different annotation from expected");
		return;
	}
	if (st->fileName() != fileName) {
		dbs("ASSERT arrived annotation of a different file");
		return;
	}
	d->m()->statusBar()->message(msg, 7000);

	isAnnotateAvailable = true;
	lookupAnnotation();
	emit annotationAvailable(true);
}

void FileContent::on_eof(bool emitSignal) {

	if (!pending.isEmpty())
		on_procDataReady(QString("\n")); // No newline at end of file, fake one

	if (!pending.isEmpty())
		dbs("ASSERT in FileContent::on_eof(): there is still pending data");

	pending = "";
	isFileAvailable = true;
	line = 1;

	if (ss.isValid)
		restoreScreenState(); // could be slow for big files

	QApplication::restoreOverrideCursor();
	if (emitSignal)
		emit fileAvailable(true);
}

void FileContent::on_doubleClicked(int para, int) {

	QString id(ft->text(para));
	id = id.section('.', 0, 0, QString::SectionSkipEmpty);
	emit revIdSelected(id.toInt());
}

void FileContent::on_procDataReady(const QString& fileChunk) {
// called to add data to file view

	if (fileData.isEmpty()) // optimize common case, the whole file in one big chunk
		fileData = fileChunk;
	else
		fileData.append(fileChunk);

	if (pending.isEmpty()) // optimize common case, pending is almost always empty
		pending = fileChunk;
	else
		pending.append(fileChunk); // add to previous half lines

	int nextEOL = pending.find('\n');
	int lastEOL = -1;
	int X = ft->contentsX();
	int Y = ft->contentsY();
	uint annLineNum = 0;

	if (ft->text().isEmpty() && isShowAnnotate) { // at the beginning only

		// check if it is possible to add annotation while appending data
		// if annotation is not available we will defer this in a separated
		// step, calling addAnnotation() at proper time
		isAnnotationAppended = (curAnn != NULL);
	}
	QStringList::const_iterator it;
	if (isAnnotationAppended && curAnn) { // curAnn could disappear while loading
		it = curAnn->lines.at(line - 1);
		annLineNum = curAnn->lines.count();
	}
	QString newStuff;
	while (nextEOL != -1) {

		// first add annotation...
		if (isAnnotationAppended && curAnn) {

			if (line > annLineNum) {
				dbs("ASSERT bad annotate in FileContent::on_procDataReady");
				clearAnnotate();
				return;
			}
			newStuff.append(QString("%1").arg(*it, -annoLen)); // left-aligned author
			++it;
		}
		// ...then line number...
		newStuff.append(QString("%1 ").arg(line, MAX_LINE_NUM));

		// ...then content
		newStuff.append(pending.mid(lastEOL + 1, nextEOL - lastEOL));
		lastEOL = nextEOL;
		nextEOL = pending.find('\n', lastEOL + 1);
		++line;
	}
	if (lastEOL != -1)
		pending.remove(0, lastEOL + 1);

	// check for partial paragraphs. In patch loading sometimes happen.
	// if it is the case also here we may have to add some more logic
	if (!ft->text().isEmpty() && !ft->text().endsWith("\n"))
		dbp("ASSERT partial file paragraph arrived <%1>", newStuff.left(30));

	newStuff.prepend(ft->text());
	ft->setText(newStuff); // much faster then ft->append()
	ft->setContentsPos(X, Y);
}
