#!/usr/bin/env python
# arch-tag: a5efd571-3cd6-4ab4-89bb-44c370b1bf4e
# Copyright (C) 2004 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for pybaz.Patchlog.
"""

import math
import time

import pybaz as arch
from pybaz.pathname import DirName, FileName, PathName
import framework
from framework import TestCase
import fixtures

from test_arch import explicit_files_import_listing


def tag_file(name):
    name = FileName(name)
    return name.dirname()/'.arch-ids'/name.basename()+'.id'

def sorted(sequence):
    sequence = list(sequence)
    sequence.sort()
    return sequence

class TestPatchlogRepr(framework.NewTestCase):
    def setUp(self):
        framework.NewTestCase.setUp(self)
        self.revision = 'jdoe@example.com/cat--brn--0--patch-1'

    def test(self):
        log = arch.Patchlog(self.revision)
        self.assertEqual(repr(log), "Patchlog(%r)" % self.revision)
        log = arch.Patchlog(self.revision, tree='/foo/bar')
        self.assertEqual(repr(log),
                         "Patchlog(%r, tree='/foo/bar')" % self.revision)
        log = arch.Patchlog(self.revision, fromlib=1)
        self.assertEqual(repr(log),
                         "Patchlog(%r, fromlib=True)" % self.revision)

class TestPatchlog(framework.NewTestCase):
    fixture = fixtures.HistoryFixture(
        dict()) # just an empty base-0

    def test_missing(self):
        """Patchlog getitem on a missing header returns None."""
        log = self.fixture.version['base-0'].patchlog
        self.assertEqual(None, log['Not-Present-Header'])

    def test_items(self):
        """Patchlog.items() and __getitem__ are consistent."""
        log = self.fixture.version['base-0'].patchlog
        for k, v in log.items():
            self.assertEqual(log[k], v)

    def test_revision(self):
        """Patchlog.revision works."""
        rvsn = self.fixture.version['base-0']
        self.assertEqual(rvsn, rvsn.patchlog.revision)

    def test_summary(self):
        """Patchlog.summary works."""
        log = self.fixture.version['base-0'].patchlog
        self.assertEqual("initial import", log.summary)

    def test_summary_from_tree(self):
        """Patchlo.summary works when reading from tree."""
        revision = self.fixture.version['base-0']
        log = arch.Patchlog(revision, tree=self.fixture.tree)
        self.assertEqual("initial import", log.summary)



class TestDescription(framework.NewTestCase):
    fixture = fixtures.WorkingTreeFixture()

    def test(self):
        """Patchlog.description works."""
        tree = self.fixture.tree
        msg = tree.log_message()
        description = "Some text\nSpanning two lines\n"
        msg.description = description
        tree.import_(msg)
        log = self.fixture.version['base-0'].patchlog
        self.assertEqual(log.description, description)


class TestDate(framework.NewTestCase):
    fixture = fixtures.WorkingTreeFixture()

    def test(self):
        """Patchlog.date is plausible."""
        # date has one second precision
        before = math.floor(time.mktime(time.gmtime()))
        self.fixture.tree.import_()
        after = time.mktime(time.gmtime())
        log = self.fixture.version['base-0'].patchlog
        logdate = time.mktime(log.date)
        if not (before <= logdate <= after):
            self.fail("Wrong date ordering: %r <= %r <= %r"
                      % (before, logdate, after))


class TestCreator(framework.NewTestCase):
    fixture = fixtures.HistoryFixture(dict()) # empty import

    def test(self):
        """Patchlog.creator works."""
        log = self.fixture.version['base-0'].patchlog
        self.assertEqual(log.creator, self.fixture.my_id)


class TestContinuationOf(framework.NewTestCase):
    fixture = fixtures.HistoryFixture(dict()) # empty import

    def test(self):
        """Patchlog.continuation_of works."""
        version = self.fixture.version
        version['base-0'].make_continuation(version)
        base0 = self.fixture.version['base-0']
        self.assertEqual(base0.patchlog.continuation_of, None)
        patch1 = self.fixture.version['patch-1']
        self.assert_(arch.factory.isRevision(patch1.patchlog.continuation_of))
        self.assertEqual(patch1.patchlog.continuation_of, base0)


class NewPatchesFixture(fixtures.HistoryFixture):

    def __init__(self):
        history = (dict(), dict())
        super(NewPatchesFixture, self).__init__(*history)

    def setUp(self):
        super(NewPatchesFixture, self).setUp()
        self.version['base-0'].make_continuation(self.version)
        self.tree.update()
        self.version['patch-1'].apply(self.tree)
        msg = self.tree.log_message()
        msg['Summary'] = 'revision 3 (replay)'
        self.tree.commit(msg)


class TestNewPatches(framework.NewTestCase):
    fixture = NewPatchesFixture()

    def test_new_patches(self):
        """Patchlog.new_patches works."""
        version = self.fixture.version
        new_patches = version['patch-1'].patchlog.new_patches
        self.assertEqual(len(new_patches), 1)
        self.assert_(arch.factory.isRevision(new_patches[0]))
        self.assertEqual(new_patches[0], version['patch-1'])
        new_patches = version['patch-3'].patchlog.new_patches
        self.assertEqual(len(new_patches), 2)
        self.assert_(arch.factory.isRevision(new_patches[0]))
        self.assert_(arch.factory.isRevision(new_patches[1]))
        self.assertEqual(sorted([str(P) for P in new_patches]),
                         [str(version['patch-1']), str(version['patch-3'])])

    def test_merged_patches(self):
        """Patchlog.merged_patches works."""
        version = self.fixture.version
        merged = version['patch-1'].patchlog.merged_patches
        self.assertEqual(merged, [])
        merged = version['patch-3'].patchlog.merged_patches
        self.assertEqual(len(merged), 1)
        self.assert_(arch.factory.isRevision(merged[0]))
        self.assertEqual(merged, [version['patch-1']])


class TestRename(framework.NewTestCase):
    fixture = fixtures.HistoryFixture(
        {'%add': ['foo', 'with space']},
        {'%mv': [['foo', 'bar'], ['with space', 'another space']]})

    def test(self):
        """Patchlog.renamed_files works."""
        log = self.fixture.version['patch-1'].patchlog
        expected = {'foo':'bar', tag_file('foo'): tag_file('bar'),
                    'with space':'another space',
                    tag_file('with space'): tag_file('another space')}
        self.assertEqual(expected, log.renamed_files)


class TestNewFiles(framework.NewTestCase):
    fixture = fixtures.HistoryFixture(
        {'%add': ['foo bar']})

    def test(self):
        """Patchlog.new_files works."""
        log = self.fixture.version['base-0'].patchlog
        expected = explicit_files_import_listing([FileName('foo bar')])
        self.assertEqual(expected, log.new_files)


class TestModifiedFiles(framework.NewTestCase):
    fixture = fixtures.HistoryFixture(
        {'%add': ['foo bar']},
        {'foo bar': 'modified'})

    def test(self):
        """Patchlog.modified_files works."""
        log = self.fixture.version['patch-1'].patchlog
        self.assertEqual(['foo bar'], log.modified_files)


class TestRemovedFiles(framework.NewTestCase):
    fixture = fixtures.HistoryFixture(
        {'%add': ['foo bar']},
        {'%rm': ['foo bar']})

    def test(self):
        """Patchlog.removed_files works."""
        log = self.fixture.version['patch-1'].patchlog
        self.assertEqual(sorted(['foo bar', tag_file('foo bar')]),
                         sorted(log.removed_files))


framework.register(__name__)
