/*

This is a program to print the status of an LSI 1030 RAID or LSI
SAS1064 SCSI controller.

Copyright (C) 2004 CNET Networks, Inc.
Copyright (C) 2005-2006 Roberto Nibali

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
USA.

*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>

#include <sys/ioctl.h>
#include <linux/compiler.h>
#include "mpt-status.h"

#define ARG_M_A 0x0001

static int m = 0;
static int quiet_mode = 0;
static int verbose_mode = 0;
static int debug = 0;
static int debug_level = 0;
static int auto_load = 0;
static int probe_id = 0;

static int sel;
static const struct option long_options[] = {
	{ "autoload",		no_argument,       &sel,  ARG_M_A },
	{ "controller",		required_argument, NULL, 'u' },
	{ "debug",		optional_argument, NULL, 'd' },
	{ "help",		no_argument,       NULL, 'h' },
	{ "status_only",	no_argument,       NULL, 's' },
	{ "probe_id",		no_argument,	   NULL, 'p' },
	{ "quiet",		no_argument,       NULL, 'q' },
	{ "set_id",		required_argument, NULL, 'i' },
	{ "verbose",            no_argument,       NULL, 'v' },
	{ "version",            no_argument,       NULL, 'V' },
	{ 0,                    no_argument,       NULL,  0  },
};
static const char* const short_options  = "d:hi:pqsu:vV";
static const char* const usage_template =
  "Usage: %s [ options ]\n"
  "\n"
  "      --autoload             This will try to automatically load mptctl\n"
  "  -u, --controller <int>     Set the IOC unit (controller)\n"
  "  -d, --debug [<int>]        Enable debugging and set level optionally\n"
  "                             NOTE: This is not implemented yet\n"
  "  -h, --help                 Print this help information page\n"
  "  -i, --set_id <int>         Set id of primary device (check README)\n"
  "  -p, --probe_id		Use this to probe SCSI id's when not on id 0\n"
  "  -q, --quiet                Do not display any warnings or boring info\n"
  "  -s, --status_only		Only print the status information. This can\n"
  "                             be used for easy scripting\n"
  "  -v, --verbose              Print verbose information, such as warnings\n"
  "  -V, --version              Print version information\n"
  "\n"
  "   You can write ``-o arg'' or ``--option arg'' or ``--option=arg''\n"
  "   Note that for some parameters only the long format is supported.\n"
  "\n";

static int print_status_only = PRINT_STATUS_ONLY;
static int id_of_primary_device = 0;
static int ioc_unit = 0;

static void print_usage(const char *progname) {
        printf(usage_template, progname);
}

static void print_version(void) {
	printf("Version: %s\n", VERSION);
}

/*
// 32bit version
static void *read_page(U8 pagetype, U8 pagewhich, U8 pagenumber,
			U32 address, U8 version) {
*/
// original
static void *read_page(unsigned int pagetype, unsigned int pagewhich,
			unsigned int pagenumber, unsigned int address,
			unsigned int version) {
/*
// 64bit version 
static void *read_page(uint8_t pagetype, uint8_t pagewhich,
			uint8_t pagenumber, uint32_t address,
			uint8_t version) {
*/
	char command[REALLYBIG];
	char reply[REALLYBIG];
	char sense[REALLYBIG];
	char *in;
	unsigned int warnings;
	struct mpt_ioctl_command *cmd;
	Config_t *config_request;
	ConfigReply_t *config_reply;
	unsigned short div = sizeof(char *); /* this is fishy */

	if (NULL == (in = calloc(REALLYBIG, 1))) {
		perror("calloc");
		exit(EXIT_FAILURE);
	}

	warnings = 0;
	cmd = (struct mpt_ioctl_command *)command;
	config_request = (Config_t *) cmd->MF;
	config_reply = (ConfigReply_t *) reply;

	memset((void *)command, 0, sizeof(command));
	memset((void *)reply, 0, sizeof(reply));
	memset((void *)sense, 0, sizeof(sense));
	memset((void *)in, 0, sizeof(in));

	cmd->hdr.iocnum = ioc_unit;
	cmd->hdr.port = 0;
	cmd->hdr.maxDataSize = BIG;
	cmd->timeout = 10;
	cmd->dataSgeOffset = (((char *)&config_request->PageBufferSGE) -
	     ((char *)config_request)) / div;
	/*
	printf("SGE ptr: %p\n", (char *)&config_request->PageBufferSGE);
	printf("conf ptr: %p\n", (char *)config_request);
	printf("dataSgeOffset: %d\n", cmd->dataSgeOffset);
	*/
	cmd->senseDataPtr = (void *)sense;
	cmd->replyFrameBufPtr = (void *)config_reply;
	cmd->dataInBufPtr = (void *)in;
	cmd->maxSenseBytes = BIG;
	cmd->maxReplyBytes = BIG;
	cmd->dataInSize = BIG;

	config_request->Function = MPI_FUNCTION_CONFIG;
	config_request->Action = MPI_CONFIG_ACTION_PAGE_HEADER;
	config_request->Header.PageType = pagetype;
	config_request->Header.PageNumber = pagenumber;
	config_request->Header.PageLength = 0;
	config_request->Header.PageVersion = 0;
	config_request->PageAddress = address;

	if (-1 == ioctl(m, MPTCOMMAND, cmd)) {
		perror("MPI_FUNCTION_CONFIG (get page header)");
		exit(EXIT_FAILURE);
	}

	if (version != config_reply->Header.PageVersion) {
		if (verbose_mode > 0) {
			fprintf(stderr,
				"Version mismatch on pagetype %u, page number %u: expected %d, got %d\n",
				pagetype, pagenumber, version,
				config_reply->Header.PageVersion);
		}
		/* Enable the following to automatically output a warning */
		//warnings++;
	}

	config_request->Action = pagewhich;
	config_request->Header.PageLength = config_reply->Header.PageLength;

	/*
	 * Is this check really necessary (255 is max anyway)? --ratz
	 * config_reply->Header.PageLength is U8.
	if (config_reply->Header.PageLength > UINT8_MAX) {
		fprintf(stderr, "reply too big\n");
		exit(EXIT_FAILURE);
	}
	 */

	if (-1 == ioctl(m, MPTCOMMAND, cmd)) {
		perror("MPI_FUNCTION_CONFIG (get page)");
		exit(EXIT_FAILURE);
	}
	if (warnings > 0 && quiet_mode == 0 && verbose_mode != 0) {
		fprintf(stderr, "\nHouston, we have got a problem!"
				" If you're interested in the warnings or you"
				" were asked to send verbose information, add"
				" the '-v' switch to your command line\n");
	}
	return in;
}


static int __probe_scsi_id() {
	int scsi_id;
	int ioc = 0;
	RaidVolumePage0_t *page;

	for (scsi_id = 0; scsi_id < 16; scsi_id++) {
		page = read_page(MPI_CONFIG_PAGETYPE_RAID_VOLUME,
			 MPI_CONFIG_ACTION_PAGE_READ_CURRENT,
			 ioc,
			 scsi_id,
			 MPI_RAIDVOLPAGE0_PAGEVERSION);
		if (0 != page->NumPhysDisks) {
			/* found the first scsi_id */
			return scsi_id;
		}
	}
	/* nothing found */
	return -1;
}

static void print_information() {
	RaidVolumePage0_t *page;
	RaidPhysDiskPage0_t *phys;
	//IOCPage5_t *ioc_hotspare;
	char *VolumeTypes[] = { "IS", "IME", "IM" };
	int i;
	int ioc = 0;

	page = read_page(MPI_CONFIG_PAGETYPE_RAID_VOLUME,
			 MPI_CONFIG_ACTION_PAGE_READ_CURRENT,
			 ioc,
			 id_of_primary_device,
			 MPI_RAIDVOLPAGE0_PAGEVERSION);
	if (0 == page->NumPhysDisks) {
		if (probe_id > 0) {
			id_of_primary_device = __probe_scsi_id();
			if (-1 == id_of_primary_device) {
				printf("Nothing found, contact the author\n");
				exit(EXIT_FAILURE);
			} else {
				printf("Found SCSI id=%d, use ''mpt-status "
				       "-i %d`` to get more information.\n",
					id_of_primary_device,
					id_of_primary_device);
				exit(EXIT_SUCCESS);
			}
		} else {
			printf("%s\n", wrong_scsi_id);
			exit(EXIT_FAILURE);
		}
	}
	if (1 == print_status_only) {
		printf("vol_id %d", page->VolumeID);
	} else {
		printf("ioc%d vol_id %d type %s, %d phy, %d GB",
		       page->VolumeIOC,
		       page->VolumeID,
		       page->VolumeType < sizeof(VolumeTypes) ?
				VolumeTypes[page->VolumeType] : "unknown",
		       page->NumPhysDisks,
		       page->MaxLBA / (2 * 1024 * 1024));
		printf(", state");
	}
	if (page->VolumeStatus.State == MPI_RAIDVOL0_STATUS_STATE_OPTIMAL)
		printf(" OPTIMAL");
	if (page->VolumeStatus.State & MPI_RAIDVOL0_STATUS_STATE_DEGRADED)
		printf(" DEGRADED");
	if (page->VolumeStatus.State & MPI_RAIDVOL0_STATUS_STATE_FAILED)
		printf(" FAILED");
	if (1 != print_status_only) {
		printf(", flags");
		if (page->VolumeStatus.Flags != 0) {
			if (page->VolumeStatus.Flags &
			    MPI_RAIDVOL0_STATUS_FLAG_ENABLED)
				printf(" ENABLED");
			if (page->VolumeStatus.Flags &
			    MPI_RAIDVOL0_STATUS_FLAG_QUIESCED)
				printf(" QUIESCED");
			if (page->VolumeStatus.Flags &
			    MPI_RAIDVOL0_STATUS_FLAG_RESYNC_IN_PROGRESS)
				printf(" RESYNC_IN_PROGRESS");
			if (page->VolumeStatus.Flags &
			    MPI_RAIDVOL0_STATUS_FLAG_VOLUME_INACTIVE)
				printf(" VOLUME_INACTIVE");
		} else {
			printf(" NONE");
		}
	}
	printf("\n");

	/* go through all physical disk of a found array */
	for (i = 0; i < page->NumPhysDisks; ++i) {
		char vendor[BIG];
		char productid[BIG];
		char rev[BIG];
		phys = read_page(MPI_CONFIG_PAGETYPE_RAID_PHYSDISK,
				 MPI_CONFIG_ACTION_PAGE_READ_CURRENT,
				 0,
				 page->PhysDisk[i].PhysDiskNum,
				 MPI_RAIDPHYSDISKPAGE0_PAGEVERSION);
		memset(vendor, 0, sizeof(vendor));
		memset(productid, 0, sizeof(productid));
		memset(rev, 0, sizeof(rev));
		/* the cast to char * is new for 1.1.5, maybe this breaks
		   something subtle since I haven't sanitized the headers yet
		*/
		strncpy(vendor, (char *) phys->InquiryData.VendorID,
			sizeof(phys->InquiryData.VendorID));
		strncpy(productid, (char *) phys->InquiryData.ProductID,
			sizeof(phys->InquiryData.ProductID));
		strncpy(rev, (char *) phys->InquiryData.ProductRevLevel,
			sizeof(phys->InquiryData.ProductRevLevel));

		if (1 == print_status_only) {
			printf("phys_id %d", phys->PhysDiskNum);
		} else {
			printf("ioc%d phy %d scsi_id %d %s %s %s, %d GB",
			       phys->PhysDiskIOC,
			       phys->PhysDiskNum,
			       phys->PhysDiskID,
			       vendor,
			       productid,
			       rev, phys->MaxLBA / (2 * 1024 * 1024));
			printf(", state");
		}
		if (phys->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_ONLINE)
			printf(" ONLINE");
		if (phys->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_MISSING)
			printf(" MISSING");
		if (phys->PhysDiskStatus.State ==
		    MPI_PHYSDISK0_STATUS_NOT_COMPATIBLE)
			printf(" NOT_COMPATIBLE");
		if (phys->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_FAILED)
			printf(" FAILED");
		if (phys->PhysDiskStatus.State ==
		    MPI_PHYSDISK0_STATUS_INITIALIZING)
			printf(" INITIALIZING");
		if (phys->PhysDiskStatus.State ==
		    MPI_PHYSDISK0_STATUS_OFFLINE_REQUESTED)
			printf(" OFFLINE_REQUESTED");
		if (phys->PhysDiskStatus.State ==
		    MPI_PHYSDISK0_STATUS_FAILED_REQUESTED)
			printf(" FAILED_REQUESTED");
		if (phys->PhysDiskStatus.State ==
		    MPI_PHYSDISK0_STATUS_OTHER_OFFLINE)
			printf(" OTHER_OFFLINE");
		if (1 != print_status_only) {
			printf(", flags");
			if (phys->PhysDiskStatus.Flags) {
				if (phys->PhysDiskStatus.Flags &
				    MPI_PHYSDISK0_STATUS_FLAG_OUT_OF_SYNC)
					printf(" OUT_OF_SYNC");
				if (phys->PhysDiskStatus.Flags &
				    MPI_PHYSDISK0_STATUS_FLAG_QUIESCED)
					printf(" QUIESCED");
			} else {
				printf(" NONE");
			}
		}
		printf("\n");
	}

	/* go through all hotspares
	ioc_hotspare = read_page(MPI_CONFIG_PAGETYPE_RAID_VOLUME,
			 MPI_CONFIG_ACTION_PAGE_READ_CURRENT,
			 0,
			 id_of_primary_device,
			 MPI_IOCPAGE5_PAGEVERSION);

	//printf("Number of hotspares: %d\n", ioc_hotspare->NumHotSpares);
	for (i = 0; i < ioc_hotspare->NumHotSpares; ++i) {
		Ioc5HotSpare_t *hotspare;

		hotspare = read_page(MPI_CONFIG_PAGETYPE_RAID_PHYSDISK,
			 MPI_CONFIG_ACTION_PAGE_READ_CURRENT,
			 0,
			 ioc_hotspare->HotSpare[i].PhysDiskNum,
			 MPI_IOCPAGE5_PAGEVERSION);
		printf("Hotspare: phys_id %d state: %s\n",
			hotspare->PhysDiskNum,
			hotspare->Flags == MPI_IOC_PAGE_5_HOT_SPARE_ACTIVE ?
			"ACTIVE" : "INACTIVE");
	}
	*/
}

static void do_init(void) {
	int save_errno;

	if (auto_load > 0) {
		system("/sbin/modprobe mptctl");
	}
	m = open("/dev/mptctl", O_RDWR);
	if (-1 == m) {
		save_errno = errno;
		perror("open /dev/mptctl");
		if (save_errno == EACCES) {
			fprintf(stderr,
				"  You need to be root to run this program\n");
		} else if (save_errno == ENOENT) {
			fprintf(stderr, "  Try: mknod /dev/mptctl c 10 220\n");
		} else if (save_errno == ENODEV) {
			fprintf(stderr,
				"  Are you sure your controller is supported by mptlinux?\n");
		}
		fprintf(stderr, "Also make sure mptctl is loaded into the kernel\n");
		exit(EXIT_FAILURE);
	}
}

int main(int argc, char *argv[]) {
	int next_option;
	int option_index;
	char *progname = argv[0];

	do {
		next_option = getopt_long(argc, argv,
			short_options, long_options, &option_index);
		switch (next_option) {
		case 'd':
			debug = 1;
			debug_level = strtoul(optarg, NULL, 10);
			break;
		case 'i':
			id_of_primary_device = strtoul(optarg, NULL, 10);
			break;
		case 'h':
			print_usage(progname);
			exit(EXIT_SUCCESS);
			break;
		case 'p':
			probe_id = 1;
			break;
		case 'q':
			quiet_mode = 1;
			break;
		case 's':
			print_status_only = 1;
			break;
		case 'u':
			ioc_unit = strtoul(optarg, NULL, 10);
			break;
		case 'v':
			verbose_mode = 1;
			break;
		case 'V':
			print_version();
			exit(EXIT_SUCCESS);
			break;
		case  0:
			if (sel == ARG_M_A) {
				auto_load++;
			}
			break;
		case -1:
			// Done with options
			break;
		default:
			print_usage(progname);
			exit(EXIT_SUCCESS);
			break;
		}
	} while (next_option != -1);

	do_init();
	print_information();
	exit(EXIT_SUCCESS);
}
