#ifndef _LIB_COMBO_H_
#define _LIB_COMBO_H_

#include <gtk/gtk.h>
#include <gmodule.h>
#include <time.h>

typedef struct history_dbh_t{
	int hits;
	time_t last_hit;
	char path[256];
}history_dbh_t;
    


typedef struct _xfc_combo_info_t xfc_combo_info_t;
struct _xfc_combo_info_t{
	GtkCombo 	*combo;  
	GtkEntry 	*entry;  
	gchar 		*active_dbh_file; 
	gpointer	cancel_user_data; 
	gpointer	activate_user_data;
	void            (*cancel_func)(GtkEntry *entry,gpointer cancel_user_data);
	void            (*activate_func)(GtkEntry *entry,gpointer activate_user_data);
	/* 
	 * This is private (ro): */
	
	GList 		*list;
	GList 		*limited_list;
	GList 		*old_list;
};


typedef struct _xfc_combo_functions xfc_combo_functions;

struct _xfc_combo_functions {
 /* exported: */
 gboolean 	(*xfc_is_in_history)(char *path2dbh_file,char *path2find);
 gboolean 	(*xfc_set_combo)(xfc_combo_info_t *combo_info, char *token);
 void 		(*xfc_set_blank)(xfc_combo_info_t *combo_info);
 void 		(*xfc_set_entry)(xfc_combo_info_t *combo_info,char *entry_string);
 void 		(*xfc_save_to_history)(char *path2dbh_file,char *path);
 void 		(*xfc_remove_from_history)(char *path2dbh_file,char *path2remove);
 void 		(*xfc_read_history)(xfc_combo_info_t *combo_info, gchar *path2dbh_file);
 void 		(*xfc_clear_history)(xfc_combo_info_t *combo_info);
 xfc_combo_info_t *(*xfc_init_combo)(GtkCombo *combo);
 xfc_combo_info_t *(*xfc_destroy_combo)(xfc_combo_info_t *combo_info);
 /* imported (or null): */
 int		(*extra_key_completion)(gpointer extra_key_data);
 gpointer	extra_key_data;
};
#define XFC_is_in_history (*(load_xfc()->xfc_is_in_history))
#define XFC_set_combo (*(load_xfc()->xfc_set_combo))
#define XFC_set_blank (*(load_xfc()->xfc_set_blank))
#define XFC_set_entry (*(load_xfc()->xfc_set_entry))
#define XFC_save_to_history (*(load_xfc()->xfc_save_to_history))
#define XFC_remove_from_history (*(load_xfc()->xfc_remove_from_history))
#define XFC_read_history (*(load_xfc()->xfc_read_history))
#define XFC_clear_history (*(load_xfc()->xfc_clear_history))
#define XFC_init_combo (*(load_xfc()->xfc_init_combo))
#define XFC_destroy_combo (*(load_xfc()->xfc_destroy_combo))


/**
 * xfc_is_in_history:
 * @dbh_file: path to dbh file. This is a null terminated string.
 * @path: This is the string to look for in
 * the dbh file. It may be a path, but also can be any null terminated string.
 *
 * Call: if(xfc_is_in_history(path2dbh_file,path2find){do_something}
 *
 * Returns: TRUE if operation was successful and the string was found
 * in the dbh file. If string is not found, returns FALSE. Other possible
 * reasons for returning FALSE may include that @dbh_file is not a valid 
 * path or not a valid dbh_file, ergo, the string is not found either.
 *
 **/

G_MODULE_EXPORT
gboolean 
xfc_is_in_history(char *dbh_file,char *path);

/**
 * xfc_set_combo:
 *  @combo_info: the autocompletion combo object
 *  @token: the string with which to do autocompletion to produce a
 *  reduced glist for the combo popup. All elements of the popup will
 *  be related to @token by autocompletion 
 * 
 *  sets the glist for the combo, reduced by autocompletion done with @token.
 * summary of keystoke interpretations:
 * control-tab moves to next slash or end of string,
 * tab cycles to the next history element that matches,
 * control-del removes the item from the history.
 *
 * Call: if(xfc_set_combo(combo_info, token)){do_something}
 *
 * Returns: TRUE is autocompletion is possible. FALSE is autocompletion
 * produces the null set.

 **/

G_MODULE_EXPORT
gboolean 
xfc_set_combo(xfc_combo_info_t *combo_info, char *token);

/**
 * xfc_set_blank: 
 * @combo_info: the autocompletion combo object.
 *
 * sets the initial glist for the combo, and puts a blank
 * as the value for the combo entry. The initial list will be made up of
 * the elements with the greatest hit count.
 *
 * Call:  xfc_set_blank(combo_info);
 *
 * Returns: nothing.
 **/

G_MODULE_EXPORT
void 
xfc_set_blank(xfc_combo_info_t *combo_info);

/**
 * xfc_set_entry: 
 * @combo_info: the autocompletion combo object.
 * @top: the null terminated atring to write into the combo entry.
 *
 * sets the initial glist for the combo, and puts a string
 * as the value for the combo entry. The initial list will be made up of
 * the elements with the greatest hit count. The glist is not reduced by
 * the value of @top (use xfce_set_combo() for that).
 *
 * Call:  xfc_set_entry(combo_info, entry_string);
 *
 *
 * Returns: nothing.
 **/

G_MODULE_EXPORT
void 
xfc_set_entry(xfc_combo_info_t *combo_info,char *top);

/**
 *  @xfc_save_to_history: This saves path to dbhashtable or increments
 *  the hit counter if @path is already in the dbhashtable.
 *
 *  @dbh_file: null terminated string pointing to the path of the 
 *  dbh file.
 *  @path: null terminated string which will be added to the dbh file
 *  (if not already in the dbh file), or that will have its hit counter
 *  incremented (if already in dbh file).
 *
 * Call: xfc_save_to_history(path2dbh_file,path);
 *
 *  Returns: nothing.
 **/ 
	
G_MODULE_EXPORT 
void 
xfc_save_to_history(char *dbh_file,char *path);

/**
 * xfc_remove_from_history:
 *  @dbh_file: null terminated string pointing to the path of the 
 *  dbh file.
 *  @path: null terminated string which will be removed from the dbh file
 * 
 *  This removes @path from the dbhashtable, if it exists.
 * 
 *  Call: xfc_remove_from_history(path2dbh_file, path2remove);
 *  
 *  Returns: nothing.
 **/

G_MODULE_EXPORT
void 
xfc_remove_from_history(char *dbh_file,char *path);

/**
 * xfc_init_combo: 
 * @combo: the GtkCombo to associate the autocompletion combo with.
 *
 * Creates a autocompletion combo object from a GtkCombo.
 * This should be called before any other function. It will allocate the
 * necessary memory for the xfc_combo object and initialize variables.
 * 
 * Call (example): combo_info = xfc_init_combo(combo_info);
 * 
 * Returns: pointer to newly created xfc_combo object or NULL if fails.
 * The combo object returned should be freed with xfc_destroy_combo()
 * when no longer required.
 * 
 **/

G_MODULE_EXPORT
xfc_combo_info_t *
xfc_init_combo(GtkCombo *combo);

/**
 * xfc_destroy_combo:
 * @combo_info: combo object to free.
 *
 * Destroy and free memory used by a combo autocompletion
 * object.
 * 
 * Call (example): 
 *      combo_info = xfc_destroy_combo(combo_info);
 * 
 * Returns: NULL. 
 **/

G_MODULE_EXPORT
xfc_combo_info_t *
xfc_destroy_combo(xfc_combo_info_t *combo_info);

/**
 * xfc_read_history:
 * @combo_info: The autocompletion combo object.
 * @dbh_file: null terminated string with a valid path to the dbh file
 * to be used.
 * 
 * This reads the dbhashtable. This is to be done before all other functions
 * except xfc_init_combo() and need be done only once ---except if you want to
 * switch glists on the same combo. See xffm/libs/input.c for an example
 * on how to use multiple lists on the same combo. 
 * 
 * Call (example): xfc_read_history(combo_info, path2dbh_file);
 *
 * Returns: nothing.
 **/

G_MODULE_EXPORT
void 
xfc_read_history(xfc_combo_info_t *combo_info, gchar *dbh_file);

/**
 * xfc_clear_history: 
 * @combo_info: The autocompletion combo object
 *
 * This clears the glist associated with the combo. 
 * Only need it if you
 * plan to use multiple glists on the same combo (see xffm/libs/input.c 
 * for an example) to avoid memory leaks. Normally not necessary since
 * it is implied by xfce_destroy_combo().
 *
 * Call (example): xfc_clear_history(combo_info);
 *
 * Returns: nothing.
 **/ 

G_MODULE_EXPORT
void xfc_clear_history(xfc_combo_info_t *combo_info);

#endif

