/**
 * ===========================================
 * LibFonts : a free Java font reading library
 * ===========================================
 *
 * Project Info:  http://reporting.pentaho.org/libfonts/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * ITextFontStorage.java
 * ------------
 */

package org.pentaho.reporting.libraries.fonts.itext;

import org.pentaho.reporting.libraries.fonts.cache.FirstLevelFontCache;
import org.pentaho.reporting.libraries.fonts.registry.FontContext;
import org.pentaho.reporting.libraries.fonts.registry.FontIdentifier;
import org.pentaho.reporting.libraries.fonts.registry.FontKey;
import org.pentaho.reporting.libraries.fonts.registry.FontMetrics;
import org.pentaho.reporting.libraries.fonts.registry.FontRegistry;
import org.pentaho.reporting.libraries.fonts.registry.FontStorage;
import org.pentaho.reporting.libraries.base.util.DebugLog;

/**
 * Creation-Date: 22.07.2007, 17:54:43
 *
 * @author Thomas Morgner
 */
public class ITextFontStorage implements FontStorage
{
  private static class EncodingFontContextWrapper implements FontContext
  {
    private FontContext context;
    private String defaultEncoding;

    private EncodingFontContextWrapper(final FontContext context,
                                       final String defaultEncoding)
    {
      this.context = context;
      this.defaultEncoding = defaultEncoding;
    }

    public String getEncoding()
    {
      return defaultEncoding;
    }

    public boolean isEmbedded()
    {
      return context.isEmbedded();
    }

    public boolean isAntiAliased()
    {
      return context.isAntiAliased();
    }

    public boolean isFractionalMetrics()
    {
      return context.isFractionalMetrics();
    }

    public double getFontSize()
    {
      return context.getFontSize();
    }
  }

  private static class EncodingFontKey extends FontKey
  {
    private String encoding;

    public EncodingFontKey(final FontIdentifier identifier,
                           final boolean aliased,
                           final boolean fractional,
                           final double fontSize,
                           final String encoding)
    {
      super(identifier, aliased, fractional, fontSize);
      this.encoding = encoding;
    }

    public EncodingFontKey()
    {
    }

    public String getEncoding()
    {
      return encoding;
    }

    public void setEncoding(final String encoding)
    {
      this.encoding = encoding;
    }

    public boolean equals(final Object o)
    {
      if (this == o)
      {
        return true;
      }
      if (o == null || getClass() != o.getClass())
      {
        return false;
      }
      if (!super.equals(o))
      {
        return false;
      }

      final EncodingFontKey that = (EncodingFontKey) o;

      if (encoding != null ? !encoding.equals(that.encoding) : that.encoding != null)
      {
        return false;
      }

      return true;
    }

    public int hashCode()
    {
      int result = super.hashCode();
      result = 31 * result + (encoding != null ? encoding.hashCode() : 0);
      return result;
    }
  }

  private ITextFontRegistry registry;
  private ITextFontMetricsFactory metricsFactory;
  private EncodingFontKey lookupKey;
  private FirstLevelFontCache knownMetrics;
  private String defaultEncoding;
  private int hits;
  private int misses;

  public ITextFontStorage()
  {
    this.lookupKey = new EncodingFontKey();

    this.registry = new ITextFontRegistry();
    this.registry.initialize();

    this.knownMetrics = new FirstLevelFontCache(registry.getSecondLevelCache());
    this.metricsFactory = (ITextFontMetricsFactory) registry.createMetricsFactory();
  }

  public ITextFontStorage(final ITextFontRegistry registry)
  {
    this(registry, null);
  }

  public ITextFontStorage(final ITextFontRegistry registry,
                          final String encoding)
  {
    this.lookupKey = new EncodingFontKey();
    this.knownMetrics = new FirstLevelFontCache(registry.getSecondLevelCache());
    this.registry = registry;
    this.defaultEncoding = encoding;
    this.metricsFactory = (ITextFontMetricsFactory) registry.createMetricsFactory();
  }

  public FontRegistry getFontRegistry()
  {
    return registry;
  }

  public FontMetrics getFontMetrics(final FontIdentifier rawRecord,
                                    final FontContext context)
  {
    if (rawRecord == null)
    {
      throw new NullPointerException();
    }
    if (context == null)
    {
      throw new NullPointerException();
    }

    final String effectiveEncoding;
    final String contextEncoding = context.getEncoding();
    if (contextEncoding == null)
    {
      effectiveEncoding = defaultEncoding;
    }
    else
    {
      effectiveEncoding = contextEncoding;
    }

    lookupKey.setAliased(context.isAntiAliased());
    lookupKey.setFontSize(context.getFontSize());
    lookupKey.setIdentifier(rawRecord);
    lookupKey.setFractional(context.isFractionalMetrics());
    lookupKey.setEncoding(effectiveEncoding);

    final FontMetrics cachedMetrics = knownMetrics.getFontMetrics(lookupKey);
    if (cachedMetrics != null)
    {
      hits += 1;
      return cachedMetrics;
    }

    misses += 1;

    final EncodingFontContextWrapper contextWrapper = new EncodingFontContextWrapper(context, effectiveEncoding);
    final FontMetrics metrics = metricsFactory.createMetrics(rawRecord, contextWrapper);
    final EncodingFontKey key = new EncodingFontKey(rawRecord, context.isAntiAliased(),
        context.isFractionalMetrics(), context.getFontSize(), effectiveEncoding);
    knownMetrics.putFontMetrics(key, metrics);
    return metrics;
  }

  public void commit()
  {
    DebugLog.log("Font-Storage: hits=" + hits + ", misses=" + misses);
    metricsFactory.close();
    knownMetrics.commit();
  }
}
