/*
 * Decompiled with CFR 0.152.
 */
package io.agroal.pool;

import io.agroal.api.AgroalDataSource;
import io.agroal.api.AgroalDataSourceListener;
import io.agroal.api.AgroalPoolInterceptor;
import io.agroal.api.cache.Acquirable;
import io.agroal.api.cache.ConnectionCache;
import io.agroal.api.configuration.AgroalConnectionPoolConfiguration;
import io.agroal.api.transaction.TransactionAware;
import io.agroal.api.transaction.TransactionIntegration;
import io.agroal.pool.ConnectionFactory;
import io.agroal.pool.ConnectionHandler;
import io.agroal.pool.DefaultMetricsRepository;
import io.agroal.pool.MetricsRepository;
import io.agroal.pool.Pool;
import io.agroal.pool.util.InterceptorHelper;
import io.agroal.pool.util.ListenerHelper;
import io.agroal.pool.util.PriorityScheduledExecutor;
import io.agroal.pool.util.StampedCopyOnWriteArrayList;
import io.agroal.pool.util.XAConnectionAdaptor;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.TransferQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAccumulator;
import java.util.concurrent.atomic.LongAdder;
import java.util.stream.Collectors;
import javax.sql.XAConnection;

public final class ConnectionPool
implements Pool {
    private static final AtomicInteger HOUSEKEEP_COUNT;
    private static final ConnectionHandler TRANSFER_POISON;
    private static final long ONE_SECOND;
    private final AgroalConnectionPoolConfiguration configuration;
    private final AgroalDataSourceListener[] listeners;
    private final StampedCopyOnWriteArrayList<ConnectionHandler> allConnections;
    private final TransferQueue<ConnectionHandler> handlerTransferQueue = new LinkedTransferQueue<ConnectionHandler>();
    private final ConnectionFactory connectionFactory;
    private final PriorityScheduledExecutor housekeepingExecutor;
    private final TransactionIntegration transactionIntegration;
    private final boolean borrowValidationEnabled;
    private final boolean idleValidationEnabled;
    private final boolean leakEnabled;
    private final boolean validationEnabled;
    private final boolean reapEnabled;
    private final boolean recoveryEnabled;
    private final LongAccumulator maxUsed = new LongAccumulator(Math::max, Long.MIN_VALUE);
    private final LongAdder activeCount = new LongAdder();
    private MetricsRepository metricsRepository;
    private ConnectionCache localCache;
    private List<AgroalPoolInterceptor> interceptors;

    public ConnectionPool(AgroalConnectionPoolConfiguration configuration, AgroalDataSourceListener ... listeners) {
        this.configuration = configuration;
        this.listeners = listeners;
        this.allConnections = new StampedCopyOnWriteArrayList<ConnectionHandler>(ConnectionHandler.class);
        this.localCache = configuration.connectionCache();
        this.connectionFactory = new ConnectionFactory(configuration.connectionFactoryConfiguration(), listeners);
        this.housekeepingExecutor = new PriorityScheduledExecutor(1, "agroal-" + HOUSEKEEP_COUNT.incrementAndGet(), listeners);
        this.transactionIntegration = configuration.transactionIntegration();
        this.borrowValidationEnabled = configuration.validateOnBorrow();
        this.idleValidationEnabled = !configuration.validateOnBorrow() && !configuration.idleValidationTimeout().isZero();
        this.leakEnabled = !configuration.leakTimeout().isZero();
        this.validationEnabled = !configuration.validationTimeout().isZero();
        this.reapEnabled = !configuration.reapTimeout().isZero();
        this.recoveryEnabled = configuration.recoveryEnable();
    }

    private TransactionIntegration.ResourceRecoveryFactory getResourceRecoveryFactory() {
        return this.connectionFactory.hasRecoveryCredentials() || !this.configuration.connectionFactoryConfiguration().poolRecovery() ? this.connectionFactory : this;
    }

    @Override
    public void init() {
        if (this.configuration.acquisitionTimeout().compareTo(this.configuration.connectionFactoryConfiguration().loginTimeout()) < 0) {
            ListenerHelper.fireOnWarning(this.listeners, "Login timeout should be smaller than acquisition timeout");
        }
        if (this.leakEnabled) {
            this.housekeepingExecutor.schedule(new LeakTask(), this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.validationEnabled) {
            this.housekeepingExecutor.schedule(new ValidationTask(), this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.reapEnabled) {
            this.housekeepingExecutor.schedule(new ReapTask(), this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.recoveryEnabled) {
            this.transactionIntegration.addResourceRecoveryFactory(this.getResourceRecoveryFactory());
        }
        if (this.configuration.initialSize() < this.configuration.minSize()) {
            ListenerHelper.fireOnInfo(this.listeners, "Initial size smaller than min. Connections will be created when necessary");
        } else if (this.configuration.initialSize() > this.configuration.maxSize()) {
            ListenerHelper.fireOnInfo(this.listeners, "Initial size bigger than max. Connections will be destroyed as soon as they return to the pool");
        }
        for (int n = this.configuration.initialSize(); n > 0; --n) {
            this.housekeepingExecutor.executeNow(new CreateConnectionTask().initial());
        }
    }

    @Override
    public AgroalConnectionPoolConfiguration getConfiguration() {
        return this.configuration;
    }

    @Override
    public int defaultJdbcIsolationLevel() {
        return this.connectionFactory.defaultJdbcIsolationLevel();
    }

    @Override
    public AgroalDataSourceListener[] getListeners() {
        return this.listeners;
    }

    @Override
    public List<AgroalPoolInterceptor> getPoolInterceptors() {
        return Collections.unmodifiableList(this.interceptors);
    }

    @Override
    public void setPoolInterceptors(Collection<? extends AgroalPoolInterceptor> list) {
        if (list.stream().anyMatch(i -> i.getPriority() < 0)) {
            throw new IllegalArgumentException("Negative priority values on AgroalPoolInterceptor are reserved.");
        }
        if (list.isEmpty() && (this.interceptors == null || this.interceptors.isEmpty())) {
            return;
        }
        this.interceptors = list.stream().sorted(AgroalPoolInterceptor.DEFAULT_COMPARATOR).collect(Collectors.toList());
        this.interceptors.forEach(interceptor -> ListenerHelper.fireOnPoolInterceptor(this.listeners, interceptor));
    }

    @Override
    public void flushPool(AgroalDataSource.FlushMode mode) {
        if (mode == AgroalDataSource.FlushMode.LEAK && !this.leakEnabled) {
            ListenerHelper.fireOnWarning(this.listeners, "Flushing leak connections with no specified leak timeout.");
            return;
        }
        this.housekeepingExecutor.execute(new FlushTask(mode));
    }

    @Override
    public void close() {
        if (this.recoveryEnabled) {
            this.transactionIntegration.removeResourceRecoveryFactory(this.getResourceRecoveryFactory());
        }
        for (Runnable task : this.housekeepingExecutor.shutdownNow()) {
            if (!(task instanceof DestroyConnectionTask)) continue;
            task.run();
        }
        for (ConnectionHandler handler : this.allConnections) {
            handler.setState(ConnectionHandler.State.FLUSH);
            new DestroyConnectionTask(handler).run();
        }
        this.allConnections.clear();
        this.activeCount.reset();
        while (this.handlerTransferQueue.tryTransfer(TRANSFER_POISON)) {
        }
    }

    public boolean isRecoverable() {
        return this.connectionFactory.isRecoverable();
    }

    public XAConnection getRecoveryConnection() throws SQLException {
        long stamp = this.beforeAcquire();
        this.checkMultipleAcquisition();
        ConnectionHandler checkedOutHandler = null;
        try {
            do {
                if ((checkedOutHandler = (ConnectionHandler)this.localCache.get()) != null) continue;
                checkedOutHandler = this.handlerFromSharedCache();
            } while (this.borrowValidationEnabled && !this.borrowValidation(checkedOutHandler) || this.idleValidationEnabled && !this.idleValidation(checkedOutHandler));
            if (this.metricsRepository.collectPoolMetrics()) {
                this.activeCount.increment();
            }
            InterceptorHelper.fireOnConnectionAcquiredInterceptor(this.interceptors, checkedOutHandler);
            this.afterAcquire(stamp, checkedOutHandler, false);
            return checkedOutHandler.xaConnectionWrapper();
        }
        catch (Throwable t) {
            if (checkedOutHandler != null) {
                checkedOutHandler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.CHECKED_IN);
            }
            throw t;
        }
    }

    private long beforeAcquire() throws SQLException {
        ListenerHelper.fireBeforeConnectionAcquire(this.listeners);
        if (this.housekeepingExecutor.isShutdown()) {
            throw new SQLException("This pool is closed and does not handle any more connections!");
        }
        return this.metricsRepository.beforeConnectionAcquire();
    }

    private void checkMultipleAcquisition() throws SQLException {
        if (this.configuration.multipleAcquisition() != AgroalConnectionPoolConfiguration.MultipleAcquisitionAction.OFF) {
            for (ConnectionHandler handler : this.allConnections) {
                if (handler.getHoldingThread() != Thread.currentThread()) continue;
                switch (this.configuration.multipleAcquisition()) {
                    case STRICT: {
                        throw new SQLException("Acquisition of multiple connections by the same Thread.");
                    }
                    case WARN: {
                        ListenerHelper.fireOnWarning(this.listeners, "Acquisition of multiple connections by the same Thread. This can lead to pool exhaustion and eventually a deadlock!");
                    }
                }
                break;
            }
        }
    }

    @Override
    public Connection getConnection() throws SQLException {
        long stamp = this.beforeAcquire();
        ConnectionHandler checkedOutHandler = this.handlerFromTransaction();
        if (checkedOutHandler != null) {
            this.transactionIntegration.associate((TransactionAware)checkedOutHandler, checkedOutHandler.getXaResource());
            this.afterAcquire(stamp, checkedOutHandler, true);
            return checkedOutHandler.connectionWrapper();
        }
        this.checkMultipleAcquisition();
        try {
            do {
                if ((checkedOutHandler = (ConnectionHandler)this.localCache.get()) != null) continue;
                checkedOutHandler = this.handlerFromSharedCache();
            } while (this.borrowValidationEnabled && !this.borrowValidation(checkedOutHandler) || this.idleValidationEnabled && !this.idleValidation(checkedOutHandler));
            this.transactionIntegration.associate((TransactionAware)checkedOutHandler, checkedOutHandler.getXaResource());
            if (this.metricsRepository.collectPoolMetrics()) {
                this.activeCount.increment();
            }
            InterceptorHelper.fireOnConnectionAcquiredInterceptor(this.interceptors, checkedOutHandler);
            this.afterAcquire(stamp, checkedOutHandler, true);
            return checkedOutHandler.connectionWrapper();
        }
        catch (Throwable t) {
            if (checkedOutHandler != null) {
                checkedOutHandler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.CHECKED_IN);
            }
            throw t;
        }
    }

    private ConnectionHandler handlerFromTransaction() throws SQLException {
        return (ConnectionHandler)this.transactionIntegration.getTransactionAware();
    }

    private ConnectionHandler handlerFromSharedCache() throws SQLException {
        long remaining = this.configuration.acquisitionTimeout().toNanos();
        long deadline = remaining > 0L ? System.nanoTime() + remaining : Long.MAX_VALUE;
        Future<ConnectionHandler> task = null;
        try {
            int i = 0;
            while (true) {
                ConnectionHandler handler;
                if (this.allConnections.size() < this.configuration.minSize()) {
                    task = this.housekeepingExecutor.executeNow(new CreateConnectionTask());
                }
                if (i == 0 && this.handlerTransferQueue.hasWaitingConsumer()) {
                    long timeout = Long.min(ONE_SECOND, remaining > 0L ? remaining * 9L / 10L : Long.MAX_VALUE);
                    ListenerHelper.fireBeforePoolBlock(this.listeners, timeout);
                    handler = (ConnectionHandler)this.handlerTransferQueue.poll(timeout, TimeUnit.NANOSECONDS);
                    if (handler != null && handler.acquire()) {
                        return handler;
                    }
                }
                for (ConnectionHandler handler2 : this.allConnections) {
                    if (!handler2.acquire()) continue;
                    return handler2;
                }
                if (task == null && this.allConnections.size() < this.configuration.maxSize()) {
                    task = this.housekeepingExecutor.executeNow(new CreateConnectionTask());
                }
                if (task == null) {
                    ConnectionHandler handler3 = this.waitAvailableHandler(deadline);
                    if (handler3.acquire()) {
                        return handler3;
                    }
                } else {
                    long timeout = deadline - System.nanoTime();
                    ListenerHelper.fireBeforePoolBlock(this.listeners, timeout);
                    handler = task.get(timeout, TimeUnit.NANOSECONDS);
                    if (handler != null && handler.acquire()) {
                        return handler;
                    }
                    task = null;
                }
                ++i;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new SQLException("Interrupted while acquiring");
        }
        catch (ExecutionException e) {
            throw this.unwrapExecutionException(e);
        }
        catch (CancellationException | RejectedExecutionException e) {
            throw new SQLException("Can't create new connection as the pool is shutting down", e);
        }
        catch (TimeoutException e) {
            task.cancel(true);
            for (ConnectionHandler handler : this.allConnections) {
                if (!handler.acquire()) continue;
                return handler;
            }
            throw new SQLException("Acquisition timeout while waiting for new connection", e);
        }
    }

    private ConnectionHandler waitAvailableHandler(long deadline) throws InterruptedException, SQLException {
        long timeout = deadline - System.nanoTime();
        ListenerHelper.fireBeforePoolBlock(this.listeners, timeout);
        ConnectionHandler handler = (ConnectionHandler)this.handlerTransferQueue.poll(timeout, TimeUnit.NANOSECONDS);
        if (handler == null) {
            throw new SQLException("Sorry, acquisition timeout!");
        }
        if (handler == TRANSFER_POISON) {
            throw new CancellationException();
        }
        return handler;
    }

    private SQLException unwrapExecutionException(ExecutionException ee) {
        try {
            throw ee.getCause();
        }
        catch (Error | RuntimeException re) {
            throw re;
        }
        catch (SQLException se) {
            return se;
        }
        catch (Throwable t) {
            return new SQLException("Exception while creating new connection", t);
        }
    }

    private boolean idleValidation(ConnectionHandler handler) {
        if (!handler.isIdle(this.configuration.idleValidationTimeout())) {
            return true;
        }
        return this.borrowValidation(handler);
    }

    private boolean borrowValidation(ConnectionHandler handler) {
        if (handler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.VALIDATION)) {
            return this.performValidation(handler, ConnectionHandler.State.CHECKED_OUT);
        }
        return false;
    }

    private boolean performValidation(ConnectionHandler handler, ConnectionHandler.State targetState) {
        ListenerHelper.fireBeforeConnectionValidation(this.listeners, handler);
        if (handler.isValid() && handler.setState(ConnectionHandler.State.VALIDATION, targetState)) {
            ListenerHelper.fireOnConnectionValid(this.listeners, handler);
            this.handlerTransferQueue.tryTransfer(handler);
            return true;
        }
        this.removeFromPool(handler);
        this.metricsRepository.afterConnectionInvalid();
        ListenerHelper.fireOnConnectionInvalid(this.listeners, handler);
        return false;
    }

    private void afterAcquire(long metricsStamp, ConnectionHandler checkedOutHandler, boolean verifyEnlistment) throws SQLException {
        this.metricsRepository.afterConnectionAcquire(metricsStamp);
        ListenerHelper.fireOnConnectionAcquired(this.listeners, checkedOutHandler);
        if (verifyEnlistment && !checkedOutHandler.isEnlisted()) {
            switch (this.configuration.transactionRequirement()) {
                case STRICT: {
                    this.returnConnectionHandler(checkedOutHandler);
                    throw new SQLException("Connection acquired without transaction.");
                }
                case WARN: {
                    ListenerHelper.fireOnWarning(this.listeners, new SQLException("Connection acquired without transaction."));
                }
            }
        }
        if (this.leakEnabled || this.reapEnabled) {
            checkedOutHandler.touch();
        }
        if (this.leakEnabled || this.configuration.multipleAcquisition() != AgroalConnectionPoolConfiguration.MultipleAcquisitionAction.OFF) {
            if (checkedOutHandler.getHoldingThread() != null && checkedOutHandler.getHoldingThread() != Thread.currentThread()) {
                Throwable warn = new Throwable("Shared connection between threads '" + checkedOutHandler.getHoldingThread().getName() + "' and '" + Thread.currentThread().getName() + "'");
                warn.setStackTrace(checkedOutHandler.getHoldingThread().getStackTrace());
                ListenerHelper.fireOnWarning(this.listeners, warn);
            }
            checkedOutHandler.setHoldingThread(Thread.currentThread());
            if (this.configuration.enhancedLeakReport()) {
                checkedOutHandler.setAcquisitionStackTrace(Thread.currentThread().getStackTrace());
            }
        }
    }

    @Override
    public void returnConnectionHandler(ConnectionHandler handler) throws SQLException {
        int currentSize;
        ListenerHelper.fireBeforeConnectionReturn(this.listeners, handler);
        if (this.leakEnabled) {
            handler.setHoldingThread(null);
            if (this.configuration.enhancedLeakReport()) {
                handler.setAcquisitionStackTrace(null);
            }
        }
        if (this.idleValidationEnabled || this.reapEnabled) {
            handler.touch();
        }
        try {
            if (!this.transactionIntegration.disassociate((TransactionAware)handler)) {
                return;
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        if (this.metricsRepository.collectPoolMetrics()) {
            this.activeCount.decrement();
        }
        if ((currentSize = this.allConnections.size()) > this.configuration.maxSize() && currentSize > this.configuration.minSize() || this.configuration.flushOnClose()) {
            handler.setState(ConnectionHandler.State.FLUSH);
            this.removeFromPool(handler);
            this.metricsRepository.afterConnectionReap();
            ListenerHelper.fireOnConnectionReap(this.listeners, handler);
            return;
        }
        try {
            handler.resetConnection();
        }
        catch (SQLException sqlException) {
            ListenerHelper.fireOnWarning(this.listeners, sqlException);
        }
        this.localCache.put((Acquirable)handler);
        InterceptorHelper.fireOnConnectionReturnInterceptor(this.interceptors, handler);
        if (handler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.CHECKED_IN)) {
            this.handlerTransferQueue.tryTransfer(handler);
            this.metricsRepository.afterConnectionReturn();
            ListenerHelper.fireOnConnectionReturn(this.listeners, handler);
        } else {
            this.removeFromPool(handler);
            this.metricsRepository.afterConnectionFlush();
            ListenerHelper.fireOnConnectionFlush(this.listeners, handler);
        }
    }

    private void removeFromPool(ConnectionHandler handler) {
        this.allConnections.remove(handler);
        this.housekeepingExecutor.execute(new FillTask());
        this.housekeepingExecutor.execute(new DestroyConnectionTask(handler));
    }

    public void onMetricsEnabled(boolean metricsEnabled) {
        this.metricsRepository = metricsEnabled ? new DefaultMetricsRepository(this) : new MetricsRepository.EmptyMetricsRepository();
    }

    @Override
    public MetricsRepository getMetrics() {
        return this.metricsRepository;
    }

    @Override
    public long activeCount() {
        return this.activeCount.sum();
    }

    @Override
    public long availableCount() {
        return (long)this.allConnections.size() - this.activeCount.sum();
    }

    @Override
    public long maxUsedCount() {
        return this.maxUsed.get();
    }

    @Override
    public void resetMaxUsedCount() {
        this.maxUsed.reset();
    }

    @Override
    public long awaitingCount() {
        return this.handlerTransferQueue.getWaitingConsumerCount();
    }

    @Override
    public boolean isHealthy(boolean newConnection) throws SQLException {
        ConnectionHandler healthHandler;
        Future<ConnectionHandler> task = null;
        if (newConnection) {
            try {
                while (!(healthHandler = (task = this.housekeepingExecutor.executeNow(new CreateConnectionTask().initial())).get(this.configuration.acquisitionTimeout().isZero() ? Long.MAX_VALUE : this.configuration.acquisitionTimeout().toNanos(), TimeUnit.NANOSECONDS)).setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.VALIDATION)) {
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new SQLException("Interrupted while acquiring");
            }
            catch (ExecutionException ee) {
                throw this.unwrapExecutionException(ee);
            }
            catch (CancellationException | RejectedExecutionException e) {
                throw new SQLException("Can't create new connection as the pool is shutting down", e);
            }
            catch (TimeoutException e) {
                task.cancel(true);
                throw new SQLException("Acquisition timeout on health check");
            }
        } else {
            healthHandler = this.handlerFromSharedCache();
            healthHandler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.VALIDATION);
        }
        return this.performValidation(healthHandler, ConnectionHandler.State.CHECKED_IN);
    }

    static {
        try {
            TRANSFER_POISON = new ConnectionHandler(new XAConnectionAdaptor(null), null);
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
        HOUSEKEEP_COUNT = new AtomicInteger();
        ONE_SECOND = TimeUnit.SECONDS.toNanos(1L);
    }

    private final class LeakTask
    implements Runnable {
        private LeakTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
            for (ConnectionHandler handler : ConnectionPool.this.allConnections) {
                ConnectionPool.this.housekeepingExecutor.execute(new LeakConnectionTask(handler));
            }
        }

        private class LeakConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            LeakConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionLeak(ConnectionPool.this.listeners, this.handler);
                if (this.handler.isLeak(ConnectionPool.this.configuration.leakTimeout())) {
                    ConnectionPool.this.metricsRepository.afterLeakDetection();
                    ListenerHelper.fireOnConnectionLeak(ConnectionPool.this.listeners, this.handler);
                }
            }
        }
    }

    private final class ValidationTask
    implements Runnable {
        private ValidationTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
            for (ConnectionHandler handler : ConnectionPool.this.allConnections) {
                ConnectionPool.this.housekeepingExecutor.execute(new ValidateConnectionTask(handler));
            }
        }

        private class ValidateConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            ValidateConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                if (this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.VALIDATION)) {
                    ConnectionPool.this.performValidation(this.handler, ConnectionHandler.State.CHECKED_IN);
                }
            }
        }
    }

    private final class ReapTask
    implements Runnable {
        private ReapTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
            ConnectionPool.this.localCache.reset();
            for (ConnectionHandler handler : ConnectionPool.this.allConnections) {
                ConnectionPool.this.housekeepingExecutor.execute(new ReapConnectionTask(handler));
            }
        }

        private class ReapConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            ReapConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionReap(ConnectionPool.this.listeners, this.handler);
                if (ConnectionPool.this.allConnections.size() > ConnectionPool.this.configuration.minSize() && this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) {
                    if (this.handler.isIdle(ConnectionPool.this.configuration.reapTimeout())) {
                        ConnectionPool.this.removeFromPool(this.handler);
                        ConnectionPool.this.metricsRepository.afterConnectionReap();
                        ListenerHelper.fireOnConnectionReap(ConnectionPool.this.listeners, this.handler);
                    } else {
                        this.handler.setState(ConnectionHandler.State.CHECKED_IN);
                    }
                }
            }
        }
    }

    private final class CreateConnectionTask
    implements Callable<ConnectionHandler> {
        private boolean initial;

        private CreateConnectionTask() {
        }

        private CreateConnectionTask initial() {
            this.initial = true;
            return this;
        }

        @Override
        public ConnectionHandler call() throws SQLException {
            if (!this.initial && ConnectionPool.this.allConnections.size() >= ConnectionPool.this.configuration.maxSize()) {
                return null;
            }
            ListenerHelper.fireBeforeConnectionCreation(ConnectionPool.this.listeners);
            long metricsStamp = ConnectionPool.this.metricsRepository.beforeConnectionCreation();
            try {
                ConnectionHandler handler = new ConnectionHandler(ConnectionPool.this.connectionFactory.createConnection(), ConnectionPool.this);
                ConnectionPool.this.metricsRepository.afterConnectionCreation(metricsStamp);
                if (!ConnectionPool.this.configuration.maxLifetime().isZero()) {
                    handler.setMaxLifetimeTask(ConnectionPool.this.housekeepingExecutor.schedule(new FlushTask(AgroalDataSource.FlushMode.GRACEFUL, handler), ConnectionPool.this.configuration.maxLifetime().toNanos(), TimeUnit.NANOSECONDS));
                }
                ListenerHelper.fireOnConnectionCreation(ConnectionPool.this.listeners, handler);
                InterceptorHelper.fireOnConnectionCreateInterceptor(ConnectionPool.this.interceptors, handler);
                handler.setState(ConnectionHandler.State.CHECKED_IN);
                ConnectionPool.this.allConnections.add(handler);
                if (ConnectionPool.this.metricsRepository.collectPoolMetrics()) {
                    ConnectionPool.this.maxUsed.accumulate(ConnectionPool.this.allConnections.size());
                }
                ListenerHelper.fireOnConnectionPooled(ConnectionPool.this.listeners, handler);
                ConnectionPool.this.handlerTransferQueue.tryTransfer(handler);
                return handler;
            }
            catch (SQLException e) {
                ListenerHelper.fireOnWarning(ConnectionPool.this.listeners, e);
                throw e;
            }
            catch (Throwable t) {
                ListenerHelper.fireOnWarning(ConnectionPool.this.listeners, "Failed to create connection due to " + t.getClass().getSimpleName());
                throw t;
            }
        }
    }

    private final class FlushTask
    implements Runnable {
        private final AgroalDataSource.FlushMode mode;
        private final ConnectionHandler handler;

        FlushTask(AgroalDataSource.FlushMode mode) {
            this.mode = mode;
            this.handler = null;
        }

        FlushTask(AgroalDataSource.FlushMode mode, ConnectionHandler handler) {
            this.mode = mode;
            this.handler = handler;
        }

        @Override
        public void run() {
            for (ConnectionHandler ch : this.handler != null ? Collections.singleton(this.handler) : ConnectionPool.this.allConnections) {
                ListenerHelper.fireBeforeConnectionFlush(ConnectionPool.this.listeners, ch);
                this.flush(this.mode, ch);
            }
            this.afterFlush(this.mode);
        }

        private void flush(AgroalDataSource.FlushMode mode, ConnectionHandler handler) {
            switch (mode) {
                case ALL: {
                    handler.setState(ConnectionHandler.State.FLUSH);
                    this.flushHandler(handler);
                    break;
                }
                case GRACEFUL: {
                    if (handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) {
                        this.flushHandler(handler);
                        break;
                    }
                    if (handler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.FLUSH) || !handler.isAcquirable()) break;
                    ConnectionPool.this.housekeepingExecutor.execute(this);
                    break;
                }
                case LEAK: {
                    if (!handler.isLeak(ConnectionPool.this.configuration.leakTimeout()) || !handler.setState(ConnectionHandler.State.CHECKED_OUT, ConnectionHandler.State.FLUSH)) break;
                    this.flushHandler(handler);
                    break;
                }
                case IDLE: {
                    if (ConnectionPool.this.allConnections.size() <= ConnectionPool.this.configuration.minSize() || !handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) break;
                    this.flushHandler(handler);
                    break;
                }
                case INVALID: {
                    ListenerHelper.fireBeforeConnectionValidation(ConnectionPool.this.listeners, handler);
                    if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.VALIDATION)) break;
                    if (handler.isValid() && handler.setState(ConnectionHandler.State.VALIDATION, ConnectionHandler.State.CHECKED_IN)) {
                        ListenerHelper.fireOnConnectionValid(ConnectionPool.this.listeners, handler);
                        break;
                    }
                    handler.setState(ConnectionHandler.State.VALIDATION, ConnectionHandler.State.FLUSH);
                    ListenerHelper.fireOnConnectionInvalid(ConnectionPool.this.listeners, handler);
                    this.flushHandler(handler);
                    break;
                }
            }
        }

        private void flushHandler(ConnectionHandler handler) {
            ConnectionPool.this.allConnections.remove(handler);
            ConnectionPool.this.metricsRepository.afterConnectionFlush();
            ListenerHelper.fireOnConnectionFlush(ConnectionPool.this.listeners, handler);
            ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(handler));
        }

        private void afterFlush(AgroalDataSource.FlushMode mode) {
            switch (mode) {
                case ALL: 
                case GRACEFUL: 
                case LEAK: 
                case INVALID: 
                case FILL: {
                    ConnectionPool.this.housekeepingExecutor.execute(new FillTask());
                    break;
                }
                case IDLE: {
                    break;
                }
                default: {
                    ListenerHelper.fireOnWarning(ConnectionPool.this.listeners, "Unsupported Flush mode " + String.valueOf(mode));
                }
            }
        }
    }

    private final class DestroyConnectionTask
    implements Runnable {
        private final ConnectionHandler handler;

        DestroyConnectionTask(ConnectionHandler handler) {
            this.handler = handler;
        }

        @Override
        public void run() {
            ListenerHelper.fireBeforeConnectionDestroy(ConnectionPool.this.listeners, this.handler);
            try {
                InterceptorHelper.fireOnConnectionDestroyInterceptor(ConnectionPool.this.interceptors, this.handler);
                this.handler.closeConnection();
            }
            catch (SQLException e) {
                ListenerHelper.fireOnWarning(ConnectionPool.this.listeners, e);
            }
            ConnectionPool.this.metricsRepository.afterConnectionDestroy();
            ListenerHelper.fireOnConnectionDestroy(ConnectionPool.this.listeners, this.handler);
        }
    }

    private final class FillTask
    implements Runnable {
        private FillTask() {
        }

        @Override
        public void run() {
            for (int n = ConnectionPool.this.configuration.minSize() - ConnectionPool.this.allConnections.size(); n > 0; --n) {
                ConnectionPool.this.housekeepingExecutor.executeNow(new CreateConnectionTask());
            }
        }
    }
}

