﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/LifecyclePolicyDetail.h>
#include <aws/imagebuilder/model/LifecyclePolicyResourceSelection.h>
#include <aws/imagebuilder/model/LifecyclePolicyResourceType.h>
#include <aws/imagebuilder/model/LifecyclePolicyStatus.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class UpdateLifecyclePolicyRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API UpdateLifecyclePolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateLifecyclePolicy"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the lifecycle policy resource.</p>
   */
  inline const Aws::String& GetLifecyclePolicyArn() const { return m_lifecyclePolicyArn; }
  inline bool LifecyclePolicyArnHasBeenSet() const { return m_lifecyclePolicyArnHasBeenSet; }
  template <typename LifecyclePolicyArnT = Aws::String>
  void SetLifecyclePolicyArn(LifecyclePolicyArnT&& value) {
    m_lifecyclePolicyArnHasBeenSet = true;
    m_lifecyclePolicyArn = std::forward<LifecyclePolicyArnT>(value);
  }
  template <typename LifecyclePolicyArnT = Aws::String>
  UpdateLifecyclePolicyRequest& WithLifecyclePolicyArn(LifecyclePolicyArnT&& value) {
    SetLifecyclePolicyArn(std::forward<LifecyclePolicyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional description for the lifecycle policy.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateLifecyclePolicyRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the lifecycle policy resource is enabled.</p>
   */
  inline LifecyclePolicyStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(LifecyclePolicyStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline UpdateLifecyclePolicyRequest& WithStatus(LifecyclePolicyStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or Amazon Resource Name (ARN) of the IAM role that Image Builder
   * uses to update the lifecycle policy.</p>
   */
  inline const Aws::String& GetExecutionRole() const { return m_executionRole; }
  inline bool ExecutionRoleHasBeenSet() const { return m_executionRoleHasBeenSet; }
  template <typename ExecutionRoleT = Aws::String>
  void SetExecutionRole(ExecutionRoleT&& value) {
    m_executionRoleHasBeenSet = true;
    m_executionRole = std::forward<ExecutionRoleT>(value);
  }
  template <typename ExecutionRoleT = Aws::String>
  UpdateLifecyclePolicyRequest& WithExecutionRole(ExecutionRoleT&& value) {
    SetExecutionRole(std::forward<ExecutionRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of image resource that the lifecycle policy applies to.</p>
   */
  inline LifecyclePolicyResourceType GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  inline void SetResourceType(LifecyclePolicyResourceType value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = value;
  }
  inline UpdateLifecyclePolicyRequest& WithResourceType(LifecyclePolicyResourceType value) {
    SetResourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration details for a lifecycle policy resource.</p>
   */
  inline const Aws::Vector<LifecyclePolicyDetail>& GetPolicyDetails() const { return m_policyDetails; }
  inline bool PolicyDetailsHasBeenSet() const { return m_policyDetailsHasBeenSet; }
  template <typename PolicyDetailsT = Aws::Vector<LifecyclePolicyDetail>>
  void SetPolicyDetails(PolicyDetailsT&& value) {
    m_policyDetailsHasBeenSet = true;
    m_policyDetails = std::forward<PolicyDetailsT>(value);
  }
  template <typename PolicyDetailsT = Aws::Vector<LifecyclePolicyDetail>>
  UpdateLifecyclePolicyRequest& WithPolicyDetails(PolicyDetailsT&& value) {
    SetPolicyDetails(std::forward<PolicyDetailsT>(value));
    return *this;
  }
  template <typename PolicyDetailsT = LifecyclePolicyDetail>
  UpdateLifecyclePolicyRequest& AddPolicyDetails(PolicyDetailsT&& value) {
    m_policyDetailsHasBeenSet = true;
    m_policyDetails.emplace_back(std::forward<PolicyDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Selection criteria for resources that the lifecycle policy applies to.</p>
   */
  inline const LifecyclePolicyResourceSelection& GetResourceSelection() const { return m_resourceSelection; }
  inline bool ResourceSelectionHasBeenSet() const { return m_resourceSelectionHasBeenSet; }
  template <typename ResourceSelectionT = LifecyclePolicyResourceSelection>
  void SetResourceSelection(ResourceSelectionT&& value) {
    m_resourceSelectionHasBeenSet = true;
    m_resourceSelection = std::forward<ResourceSelectionT>(value);
  }
  template <typename ResourceSelectionT = LifecyclePolicyResourceSelection>
  UpdateLifecyclePolicyRequest& WithResourceSelection(ResourceSelectionT&& value) {
    SetResourceSelection(std::forward<ResourceSelectionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateLifecyclePolicyRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_lifecyclePolicyArn;

  Aws::String m_description;

  LifecyclePolicyStatus m_status{LifecyclePolicyStatus::NOT_SET};

  Aws::String m_executionRole;

  LifecyclePolicyResourceType m_resourceType{LifecyclePolicyResourceType::NOT_SET};

  Aws::Vector<LifecyclePolicyDetail> m_policyDetails;

  LifecyclePolicyResourceSelection m_resourceSelection;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_lifecyclePolicyArnHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_executionRoleHasBeenSet = false;
  bool m_resourceTypeHasBeenSet = false;
  bool m_policyDetailsHasBeenSet = false;
  bool m_resourceSelectionHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
