/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"glegcontainer.h"
#include	"gleprivate.h"



/* --- prototypes --- */
static void  gle_gcontainer_class_init		(GleGContainerClass*class);
static void  gle_gcontainer_init		(GleGContainer	*gcontainer);
static void  gle_gcontainer_do_initialize	(GleGObject	*gobject);
static void  gle_gcontainer_do_pre_destroy	(GleGObject	*gobject);
static void  gle_gcontainer_do_foreign_associated (GleGObject	*gobject);
static void  gle_gcontainer_do_pre_disassociate	(GleGObject	*gobject);
static void  gle_gcontainer_do_add_child	(GleGContainer	*gcontainer,
						 GleGWidget	*gchild);
static void  gle_gcontainer_do_remove_child	(GleGContainer	*gcontainer,
						 GleGWidget	*gchild);


/* --- variables --- */
static GleGWidgetClass	*parent_class = NULL;


/* --- functions --- */
GtkType
gle_gcontainer_get_type (void)
{
  static GtkType gcontainer_type = 0;
  
  if (!gcontainer_type)
    {
      GtkTypeInfo gcontainer_info =
      {
	"GleGContainer",
	sizeof (GleGContainer),
	sizeof (GleGContainerClass),
	(GtkClassInitFunc) gle_gcontainer_class_init,
	(GtkObjectInitFunc) gle_gcontainer_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) NULL,
      };
      
      gcontainer_type = gtk_type_unique (GLE_TYPE_GWIDGET, &gcontainer_info);
      gtk_type_set_chunk_alloc (gcontainer_type, GLE_GCONTAINERS_PREALLOC);
    }
  
  return gcontainer_type;
}

static void
gle_gcontainer_class_init (GleGContainerClass *class)
{
  GleGObjectClass *gobject_class;
  GleGWidgetClass *gwidget_class;
  
  parent_class = gtk_type_class (GLE_TYPE_GWIDGET);
  gobject_class = (GleGObjectClass*) class;
  gwidget_class = (GleGWidgetClass*) class;
  
  gobject_class->initialize = gle_gcontainer_do_initialize;
  gobject_class->pre_destroy = gle_gcontainer_do_pre_destroy;
  gobject_class->foreign_associated = gle_gcontainer_do_foreign_associated;
  gobject_class->pre_disassociate = gle_gcontainer_do_pre_disassociate;

  class->add_child = gle_gcontainer_do_add_child;
  class->remove_child = gle_gcontainer_do_remove_child;
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGContainerClass, add_child);
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGContainerClass, remove_child);
}

static void
gle_gcontainer_init (GleGContainer *gcontainer)
{
  gcontainer->children = NULL;
  gcontainer->internal_children = NULL;
}

static void
gle_gcontainer_do_initialize (GleGObject *gobject)
{
  GleGContainer	 *gcontainer;
  
  gcontainer = GLE_GCONTAINER (gobject);
  
  GLE_GOBJECT_CLASS (parent_class)->initialize (gobject);
}

void
gle_gcontainer_do_pre_destroy (GleGObject *gobject)
{
  GleGContainer *gcontainer;
  
  gcontainer = GLE_GCONTAINER (gobject);
  
  while (gcontainer->children)
    gle_gwidget_destroy (gcontainer->children->data);
  while (gcontainer->internal_children)
    gle_gwidget_destroy (gcontainer->internal_children->data);
  
  /* chain parent class' handler
   */
  GLE_GOBJECT_CLASS (parent_class)->pre_destroy (gobject);
}

static void
gchild_add_cb (GtkWidget *widget,
	       gpointer	  data)
{
  GleGContainer *gcontainer = data;
  
  /* we silently ignore the case where container==NULL, because
   * this can happen if the parent isn't yet constructed
   */
  if (widget)
    {
      GleGWidget *gwidget;

      gwidget = gle_widget_get_gwidget (widget);
      if (gwidget)
	gle_gcontainer_add_child (gcontainer, gwidget);
    }
}

static void
gle_gcontainer_do_foreign_associated (GleGObject *gobject)
{
  GleGContainer *gcontainer;
  GtkContainer *container;
  
  gcontainer = GLE_GCONTAINER (gobject);
  
  GLE_GOBJECT_CLASS (parent_class)->foreign_associated (gobject);

  container = GLE_GCONTAINER_CONTAINER (gcontainer);
  
  gtk_container_forall (container, gchild_add_cb, gcontainer);
}

static void
gle_gcontainer_do_pre_disassociate (GleGObject *gobject)
{
  GleGContainer *gcontainer;
  GList *list;
  
  gcontainer = GLE_GCONTAINER (gobject);

  GLE_GOBJECT_CLASS (parent_class)->pre_disassociate (gobject);

  for (list = gcontainer->children; list; list = list->next)
    {
      GleGObject *gchild;

      gchild = list->data;
      if (GLE_GOBJECT_IS_INSTANTIATED (gchild))
	gle_gobject_disassociate (gchild);
    }
  for (list = gcontainer->internal_children; list; list = list->next)
    {
      GleGObject *gchild;

      gchild = list->data;
      if (GLE_GOBJECT_IS_INSTANTIATED (gchild))
	gle_gobject_disassociate (gchild);
    }
}

void
gle_gcontainer_add_child (GleGContainer *gcontainer,
			  GleGWidget    *gchild)
{
  g_return_if_fail (gcontainer != NULL);
  g_return_if_fail (GLE_IS_GCONTAINER (gcontainer));
  g_return_if_fail (gchild != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gchild));
  g_return_if_fail (gchild->parent == NULL);
  g_return_if_fail (g_list_find (gcontainer->children, gchild) == NULL);

  gle_gwidget_ref (GLE_GWIDGET (gcontainer));
  gle_gwidget_ref (gchild);

  GLE_GCONTAINER_GET_CLASS (gcontainer)->add_child (gcontainer, gchild);

  GLE_GOBJECT_NOTIFY (gcontainer, add_child, gchild, NULL);

  gle_gwidget_unref (gchild);
  gle_gwidget_unref (GLE_GWIDGET (gcontainer));
}

static void
gle_gcontainer_do_add_child (GleGContainer *gcontainer,
			     GleGWidget    *gchild)
{
  gcontainer->children = g_list_prepend (gcontainer->children, gchild);

  gle_gwidget_set_parent (gchild, GLE_GWIDGET (gcontainer));
}

void
gle_gcontainer_remove_child (GleGContainer *gcontainer,
			     GleGWidget    *gchild)
{
  g_return_if_fail (gcontainer != NULL);
  g_return_if_fail (GLE_IS_GCONTAINER (gcontainer));
  g_return_if_fail (gchild != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gchild));
  g_return_if_fail (gchild->parent == GLE_GWIDGET (gcontainer));
  
  gle_gwidget_ref (GLE_GWIDGET (gcontainer));
  gle_gwidget_ref (gchild);

  GLE_GCONTAINER_GET_CLASS (gcontainer)->remove_child (gcontainer, gchild);

  GLE_GOBJECT_NOTIFY (gcontainer, remove_child, gchild, NULL);

  gle_gwidget_unref (gchild);
  gle_gwidget_unref (GLE_GWIDGET (gcontainer));
}

static void
gle_gcontainer_do_remove_child (GleGContainer *gcontainer,
				GleGWidget    *gchild)
{
  gcontainer->children = g_list_remove (gcontainer->children, gchild);
  GLE_GOBJECT_UNSET_FLAG (gchild, GLE_GWIDGET_INTERNAL);
  gle_gwidget_set_parent (gchild, NULL);
}

static void
gchildren_force_cb (GtkWidget *widget,
		    gpointer   data)
{
  GleGWidget *gwidget;

  gwidget = gle_widget_force_gwidget (widget);
  if (!gwidget->parent)
    {
      GLE_GOBJECT_SET_FLAG (gwidget, GLE_GWIDGET_INTERNAL);
      gle_gcontainer_add_child (data, gwidget);
    }
  if (GLE_IS_GCONTAINER (gwidget))
    gle_gcontainer_force_gchildren (GLE_GCONTAINER (gwidget));
}

void
gle_gcontainer_force_gchildren (GleGContainer *gcontainer)
{
  g_return_if_fail (gcontainer != NULL);
  g_return_if_fail (GLE_IS_GCONTAINER (gcontainer));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gcontainer));

  gcontainer->children = g_list_reverse (gcontainer->children);
  gtk_container_forall (GLE_GCONTAINER_CONTAINER (gcontainer), gchildren_force_cb, gcontainer);
  gcontainer->children = g_list_reverse (gcontainer->children);
}

static void
find_composite_cb (GtkWidget *widget,
		   gpointer   data_p)
{
  gpointer *data = data_p;

  if (data[1] == NULL && GTK_WIDGET_COMPOSITE_CHILD (widget))
    {
      gchar *name;

      name = gtk_widget_get_composite_name (widget);
      if (name && strcmp (name, data[0]) == 0)
	data[1] = widget;
      g_free (name);
    }
}

GtkWidget*
gle_gcontainer_find_composite (GleGContainer  *gcontainer,
			       const gchar    *name)
{
  gpointer data[2];

  g_return_val_if_fail (gcontainer != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GCONTAINER (gcontainer), NULL);
  g_return_val_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gcontainer), NULL);
  g_return_val_if_fail (name != NULL, NULL);

  data[0] = (gchar*) name;
  data[1] = NULL;
  gtk_container_forall (GLE_GCONTAINER_CONTAINER (gcontainer), find_composite_cb, &data);

  return data[1];
}

#if 0
{
  GSList *i_children, *r_children, *slist, *last;
  
  i_children = NULL;
  gtk_container_forall (GLE_GCONTAINER_CONTAINER (gcontainer), children_list, &i_children);
  i_children = g_slist_reverse (i_children);
  r_children = NULL;
  gtk_container_forall (GLE_GCONTAINER_CONTAINER (gcontainer), children_list, &r_children);
  r_children = g_slist_reverse (r_children);

  /* sort out the real children */
  for (slist = r_children; slist; slist = slist->next)
    {
      GSList *tmp;

      tmp = g_slist_find (i_children, slist->data);
      if (!tmp)
	{
	  g_warning ("inconsistent foreach vs. forall behaviour encountered for `%s'",
		     gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gcontainer)));
	  continue;
	}
      tmp->data = NULL;
    }
  /* now remove the invalidated links */
  last = NULL;
  slist = i_children;
  while (slist)
    {
      GSList *next;

      next = slist->next;
      
      if (!slist->data)
	{
	  if (last)
	    last->next = next;
	  else
	    i_children = next;
	  g_slist_free_1 (slist);
	}
      else
	last = slist;

      slist = next;
    }

  g_slist_free (i_children);
  g_slist_free (r_children);
}
#endif
