﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/EdgeOutputConfig.h>
#include <aws/sagemaker/model/Tag.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateDeviceFleetRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateDeviceFleetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDeviceFleet"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the fleet that the device belongs to.</p>
   */
  inline const Aws::String& GetDeviceFleetName() const { return m_deviceFleetName; }
  inline bool DeviceFleetNameHasBeenSet() const { return m_deviceFleetNameHasBeenSet; }
  template <typename DeviceFleetNameT = Aws::String>
  void SetDeviceFleetName(DeviceFleetNameT&& value) {
    m_deviceFleetNameHasBeenSet = true;
    m_deviceFleetName = std::forward<DeviceFleetNameT>(value);
  }
  template <typename DeviceFleetNameT = Aws::String>
  CreateDeviceFleetRequest& WithDeviceFleetName(DeviceFleetNameT&& value) {
    SetDeviceFleetName(std::forward<DeviceFleetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that has access to Amazon Web Services
   * Internet of Things (IoT).</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateDeviceFleetRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the fleet.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateDeviceFleetRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The output configuration for storing sample data collected by the fleet.</p>
   */
  inline const EdgeOutputConfig& GetOutputConfig() const { return m_outputConfig; }
  inline bool OutputConfigHasBeenSet() const { return m_outputConfigHasBeenSet; }
  template <typename OutputConfigT = EdgeOutputConfig>
  void SetOutputConfig(OutputConfigT&& value) {
    m_outputConfigHasBeenSet = true;
    m_outputConfig = std::forward<OutputConfigT>(value);
  }
  template <typename OutputConfigT = EdgeOutputConfig>
  CreateDeviceFleetRequest& WithOutputConfig(OutputConfigT&& value) {
    SetOutputConfig(std::forward<OutputConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Creates tags for the specified fleet.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDeviceFleetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDeviceFleetRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether to create an Amazon Web Services IoT Role Alias during device fleet
   * creation. The name of the role alias generated will match this pattern:
   * "SageMakerEdge-{DeviceFleetName}".</p> <p>For example, if your device fleet is
   * called "demo-fleet", the name of the role alias will be
   * "SageMakerEdge-demo-fleet".</p>
   */
  inline bool GetEnableIotRoleAlias() const { return m_enableIotRoleAlias; }
  inline bool EnableIotRoleAliasHasBeenSet() const { return m_enableIotRoleAliasHasBeenSet; }
  inline void SetEnableIotRoleAlias(bool value) {
    m_enableIotRoleAliasHasBeenSet = true;
    m_enableIotRoleAlias = value;
  }
  inline CreateDeviceFleetRequest& WithEnableIotRoleAlias(bool value) {
    SetEnableIotRoleAlias(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_deviceFleetName;

  Aws::String m_roleArn;

  Aws::String m_description;

  EdgeOutputConfig m_outputConfig;

  Aws::Vector<Tag> m_tags;

  bool m_enableIotRoleAlias{false};
  bool m_deviceFleetNameHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_outputConfigHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_enableIotRoleAliasHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
