// SPDX-FileCopyrightText: Copyright (c) 2008-2013, NVIDIA Corporation. All rights reserved.
// SPDX-License-Identifier: Apache-2.0

/*! \file set_operations.h
 *  \brief Sequential implementation of set operation functions.
 */

#pragma once

#include <thrust/detail/config.h>

#if defined(_CCCL_IMPLICIT_SYSTEM_HEADER_GCC)
#  pragma GCC system_header
#elif defined(_CCCL_IMPLICIT_SYSTEM_HEADER_CLANG)
#  pragma clang system_header
#elif defined(_CCCL_IMPLICIT_SYSTEM_HEADER_MSVC)
#  pragma system_header
#endif // no system header
#include <thrust/detail/copy.h>
#include <thrust/detail/function.h>
#include <thrust/system/detail/sequential/execution_policy.h>

THRUST_NAMESPACE_BEGIN
namespace system::detail::sequential
{
_CCCL_EXEC_CHECK_DISABLE
template <typename DerivedPolicy,
          typename InputIterator1,
          typename InputIterator2,
          typename OutputIterator,
          typename StrictWeakOrdering>
_CCCL_HOST_DEVICE OutputIterator set_difference(
  sequential::execution_policy<DerivedPolicy>& exec,
  InputIterator1 first1,
  InputIterator1 last1,
  InputIterator2 first2,
  InputIterator2 last2,
  OutputIterator result,
  StrictWeakOrdering comp)
{
  // wrap comp
  thrust::detail::wrapped_function<StrictWeakOrdering, bool> wrapped_comp{comp};

  while (first1 != last1 && first2 != last2)
  {
    if (wrapped_comp(*first1, *first2))
    {
      *result = *first1;
      ++first1;
      ++result;
    } // end if
    else if (wrapped_comp(*first2, *first1))
    {
      ++first2;
    } // end else if
    else
    {
      ++first1;
      ++first2;
    } // end else
  } // end while

  return thrust::copy(exec, first1, last1, result);
} // end set_difference()

_CCCL_EXEC_CHECK_DISABLE
template <typename DerivedPolicy,
          typename InputIterator1,
          typename InputIterator2,
          typename OutputIterator,
          typename StrictWeakOrdering>
_CCCL_HOST_DEVICE OutputIterator set_intersection(
  sequential::execution_policy<DerivedPolicy>&,
  InputIterator1 first1,
  InputIterator1 last1,
  InputIterator2 first2,
  InputIterator2 last2,
  OutputIterator result,
  StrictWeakOrdering comp)
{
  // wrap comp
  thrust::detail::wrapped_function<StrictWeakOrdering, bool> wrapped_comp{comp};

  while (first1 != last1 && first2 != last2)
  {
    if (wrapped_comp(*first1, *first2))
    {
      ++first1;
    } // end if
    else if (wrapped_comp(*first2, *first1))
    {
      ++first2;
    } // end else if
    else
    {
      *result = *first1;
      ++first1;
      ++first2;
      ++result;
    } // end else
  } // end while

  return result;
} // end set_intersection()

_CCCL_EXEC_CHECK_DISABLE
template <typename DerivedPolicy,
          typename InputIterator1,
          typename InputIterator2,
          typename OutputIterator,
          typename StrictWeakOrdering>
_CCCL_HOST_DEVICE OutputIterator set_symmetric_difference(
  sequential::execution_policy<DerivedPolicy>& exec,
  InputIterator1 first1,
  InputIterator1 last1,
  InputIterator2 first2,
  InputIterator2 last2,
  OutputIterator result,
  StrictWeakOrdering comp)
{
  // wrap comp
  thrust::detail::wrapped_function<StrictWeakOrdering, bool> wrapped_comp{comp};

  while (first1 != last1 && first2 != last2)
  {
    if (wrapped_comp(*first1, *first2))
    {
      *result = *first1;
      ++first1;
      ++result;
    } // end if
    else if (wrapped_comp(*first2, *first1))
    {
      *result = *first2;
      ++first2;
      ++result;
    } // end else if
    else
    {
      ++first1;
      ++first2;
    } // end else
  } // end while

  return thrust::copy(exec, first2, last2, thrust::copy(exec, first1, last1, result));
} // end set_symmetric_difference()

_CCCL_EXEC_CHECK_DISABLE
template <typename DerivedPolicy,
          typename InputIterator1,
          typename InputIterator2,
          typename OutputIterator,
          typename StrictWeakOrdering>
_CCCL_HOST_DEVICE OutputIterator set_union(
  sequential::execution_policy<DerivedPolicy>& exec,
  InputIterator1 first1,
  InputIterator1 last1,
  InputIterator2 first2,
  InputIterator2 last2,
  OutputIterator result,
  StrictWeakOrdering comp)
{
  // wrap comp
  thrust::detail::wrapped_function<StrictWeakOrdering, bool> wrapped_comp{comp};

  while (first1 != last1 && first2 != last2)
  {
    if (wrapped_comp(*first1, *first2))
    {
      *result = *first1;
      ++first1;
    } // end if
    else if (wrapped_comp(*first2, *first1))
    {
      *result = *first2;
      ++first2;
    } // end else if
    else
    {
      *result = *first1;
      ++first1;
      ++first2;
    } // end else

    ++result;
  } // end while

  return thrust::copy(exec, first2, last2, thrust::copy(exec, first1, last1, result));
} // end set_union()
} // namespace system::detail::sequential
THRUST_NAMESPACE_END
