﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeconnections/CodeConnectionsRequest.h>
#include <aws/codeconnections/CodeConnections_EXPORTS.h>
#include <aws/codeconnections/model/SyncConfigurationType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeConnections {
namespace Model {

/**
 */
class GetRepositorySyncStatusRequest : public CodeConnectionsRequest {
 public:
  AWS_CODECONNECTIONS_API GetRepositorySyncStatusRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetRepositorySyncStatus"; }

  AWS_CODECONNECTIONS_API Aws::String SerializePayload() const override;

  AWS_CODECONNECTIONS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The branch of the repository link for the requested repository sync
   * status.</p>
   */
  inline const Aws::String& GetBranch() const { return m_branch; }
  inline bool BranchHasBeenSet() const { return m_branchHasBeenSet; }
  template <typename BranchT = Aws::String>
  void SetBranch(BranchT&& value) {
    m_branchHasBeenSet = true;
    m_branch = std::forward<BranchT>(value);
  }
  template <typename BranchT = Aws::String>
  GetRepositorySyncStatusRequest& WithBranch(BranchT&& value) {
    SetBranch(std::forward<BranchT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The repository link ID for the requested repository sync status.</p>
   */
  inline const Aws::String& GetRepositoryLinkId() const { return m_repositoryLinkId; }
  inline bool RepositoryLinkIdHasBeenSet() const { return m_repositoryLinkIdHasBeenSet; }
  template <typename RepositoryLinkIdT = Aws::String>
  void SetRepositoryLinkId(RepositoryLinkIdT&& value) {
    m_repositoryLinkIdHasBeenSet = true;
    m_repositoryLinkId = std::forward<RepositoryLinkIdT>(value);
  }
  template <typename RepositoryLinkIdT = Aws::String>
  GetRepositorySyncStatusRequest& WithRepositoryLinkId(RepositoryLinkIdT&& value) {
    SetRepositoryLinkId(std::forward<RepositoryLinkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sync type of the requested sync status.</p>
   */
  inline SyncConfigurationType GetSyncType() const { return m_syncType; }
  inline bool SyncTypeHasBeenSet() const { return m_syncTypeHasBeenSet; }
  inline void SetSyncType(SyncConfigurationType value) {
    m_syncTypeHasBeenSet = true;
    m_syncType = value;
  }
  inline GetRepositorySyncStatusRequest& WithSyncType(SyncConfigurationType value) {
    SetSyncType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_branch;

  Aws::String m_repositoryLinkId;

  SyncConfigurationType m_syncType{SyncConfigurationType::NOT_SET};
  bool m_branchHasBeenSet = false;
  bool m_repositoryLinkIdHasBeenSet = false;
  bool m_syncTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeConnections
}  // namespace Aws
