
__copyright__ = """
Copyright (C) 2005, Catalin Marinas <catalin.marinas@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License version 2 as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
"""

import sys, os
from optparse import OptionParser, make_option

from stgit.commands.common import *
from stgit.utils import *
from stgit.out import *
from stgit import stack, git
from stgit.config import config


help = 'generate a new commit for the current patch'
usage = """%prog [options] [<files or dirs>]

Include the latest tree changes in the current patch. This command
generates a new GIT commit object with the patch details, the previous
one no longer being visible. The patch attributes like author,
committer and description can be changed with the command line
options. The '--force' option is useful when a commit object was
created with a different tool but the changes need to be included in
the current patch."""

directory = DirectoryHasRepository()
options = [make_option('-f', '--force',
                       help = 'force the refresh even if HEAD and '\
                       'top differ',
                       action = 'store_true'),
           make_option('--update',
                       help = 'only update the current patch files',
                       action = 'store_true'),
           make_option('--undo',
                       help = 'revert the commit generated by the last refresh',
                       action = 'store_true'),
           make_option('-a', '--annotate', metavar = 'NOTE',
                       help = 'annotate the patch log entry'),
           make_option('-p', '--patch',
                       help = 'refresh (applied) PATCH instead of the top one')
           ]

def func(parser, options, args):
    """Generate a new commit for the current or given patch.
    """
    args = git.ls_files(args)
    directory.cd_to_topdir()

    autoresolved = config.get('stgit.autoresolved')
    if autoresolved != 'yes':
        check_conflicts()

    if options.patch:
        if args or options.update:
            raise CmdException, \
                  'Only full refresh is available with the --patch option'
        patch = options.patch
        if not crt_series.patch_applied(patch):
            raise CmdException, 'Patches "%s" not applied' % patch
    else:
        patch = crt_series.get_current()
        if not patch:
            raise CmdException, 'No patches applied'

    if not options.force:
        check_head_top_equal(crt_series)

    if options.undo:
        out.start('Undoing the refresh of "%s"' % patch)
        crt_series.undo_refresh()
        out.done()
        return

    files = [path for (stat, path) in git.tree_status(files = args, verbose = True)]

    if files or not crt_series.head_top_equal():
        if options.patch:
            applied = crt_series.get_applied()
            between = applied[:applied.index(patch):-1]
            pop_patches(crt_series, between, keep = True)
        elif options.update:
            rev1 = git_id(crt_series, '//bottom')
            rev2 = git_id(crt_series, '//top')
            patch_files = git.barefiles(rev1, rev2).split('\n')
            files = [f for f in files if f in patch_files]
            if not files:
                out.info('No modified files for updating patch "%s"' % patch)
                return

        out.start('Refreshing patch "%s"' % patch)

        if autoresolved == 'yes':
            resolved_all()
        crt_series.refresh_patch(files = files,
                                 backup = True, notes = options.annotate)

        if crt_series.empty_patch(patch):
            out.done('empty patch')
        else:
            out.done()

        if options.patch:
            between.reverse()
            push_patches(crt_series, between)
    elif options.annotate:
        # only annotate the top log entry as there is no need to
        # refresh the patch and generate a full commit
        crt_series.log_patch(crt_series.get_patch(patch), None,
                             notes = options.annotate)
    else:
        out.info('Patch "%s" is already up to date' % patch)
