########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Common/ClAuthenticate.py,v 1.9 2005/05/20 20:21:44 jkloth Exp $
"""
Repository client (4ss, 4ss_manager) common login routines

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import os, sys, sha, getpass, urllib

from Ft import GetConfigVar
from Ft.Server import FtServerBaseException
import MessageSource

__all__ = [ 'FtServerAuthenticationException', 'Error',
            'GetUserName', 'GetPass', 'GetHostName', 'GetPort',
            'HashPasswd',
            'PasswordFileManager',
            'UserAuthenticate',
           ]


class FtServerAuthenticationException(FtServerBaseException):
    MessageSource = MessageSource

Error = MessageSource.Error


def _raw_input(prompt=''):
    # _raw_input() from getpass: like raw_input(), but doesn't save
    # the string in the GNU readline history. Modified to use stderr.
    prompt = str(prompt)
    if prompt:
        sys.stderr.write(prompt)
        sys.stderr.flush()
    line = sys.stdin.readline()
    if not line:
        raise EOFError
    if line[-1] == '\n':
        line = line[:-1]
    return line


def GetUserName(prompt='Username: ', emptyOK=True):
    """
    Prompts for a username on stderr.
    Keeps prompting if emptyOK is false and no username is entered.
    """
    if emptyOK:
        return _raw_input(prompt)
    else:
        username = ''
        while not username:
            username = _raw_input(prompt)
        return username


def GetPass(prompt='Password: '):
    """
    Wrapper for getpass.getpass(), with the only difference being that
    on Unix, the prompt will be sent to stderr rather than stdout.
    """
    if os.name == 'posix':
        try:
            stdout = sys.stdout
            sys.stdout = sys.stderr
            passwd = getpass.getpass(prompt)
        finally:
            sys.stdout = stdout
    else:
        passwd = getpass.getpass(prompt)
    return passwd


def GetHostName(prompt='Host: '):
    """
    Prompts for a hostname on stderr.
    """
    return _raw_input(prompt)


def GetPort(prompt='Port: '):
    """
    Prompts for a port on stderr.
    """
    return _raw_input(prompt)


def HashPasswd(passwd):
    """
    Returns a hash of the given password string.
    """
    return sha.new(passwd).hexdigest()


class PasswordFileManager:
    """
    A collection of functions to make reading and writing from
    4Suite Repository password files more convenient.

    A password file maps a username to a password hash, hostname,
    and port. The location of the file is determined by the
    FTSS_PASSWORD_FILE environment variable. If not set, then the
    location is determined from the following:
      - $HOME/.4Suite/<basename>_pass (POSIX systems)
      - %APPDATA%\\4Suite\\<basename>_pass.conf (Windows 2K/XP/2K3)
      - %USERPROFILE%\\4Suite\\<basename>_pass.conf (Windows NT)
      - %WINDIR%\\4Suite\\<basename>_pass.conf (Windows 9x)
    If none of the variables are set, an exception is raised.
    If the password file does not exist, it will be created when the
    first entry is stored. The location must be writable, of course.
    """

    fileBaseName = None
    envName = None

    def __init__(self):
        # environment var is preferred
        if self.envName is not None and self.envName in os.environ:
            filename = os.environ[self.envName]
        else:
            # use a filename constructed from the user's home directory
            if os.name == 'posix':
                dir = os.environ['HOME']
                subdir = '.' + GetConfigVar('NAME')
                base = self.fileBaseName + '_pass'
            elif os.name == 'nt':
                if 'APPDATA' in os.environ:
                    # Windows 2K/XP/2K3
                    # C:\Documents and Settings\<username>\Application Data
                    dir = os.environ['APPDATA']
                elif 'USERPROFILE' in os.environ:
                    # Windows NT (or 95/98 with user profiles)
                    # C:\Winnt\Profiles\<username>
                    dir = os.environ['USERPROFILE']
                else:
                    # Windows 9x (single user)
                    # C:\Windows
                    dir = os.environ['WINDIR']
                subdir = GetConfigVar('NAME')
                base = self.fileBaseName + '_pass.conf'
            else:
                raise FtServerAuthenticationException(Error.PASSWDFILE_UNDEFINED)    
            filename = os.path.join(dir, subdir, base)
        self._filename = filename
        return

    def getFilename(self):
        if os.name == 'posix':
            filename = self.fileBaseName + '_pass'
        else:
            filename = self.fileBaseName + '_pass.conf'
        return filename

    def readEntries(self):
        if not os.path.exists(self._filename):
            return []
        
        f = open(self._filename)
        lines = [ line.strip() for line in f ]
        f.close()

        entries = [ line.split(':') for line in lines ]
        return [ [ urllib.unquote(x) for x in entry ] for entry in entries ]
        
    def writeEntries(self, entries):
        if not os.path.exists(os.path.dirname(self._filename)):
            os.makedirs(os.path.dirname(self._filename))
        entries = [ [ urllib.quote(x) for x in entry ] for entry in entries ]
        lines = [ ':'.join(entry) for entry in entries ]

        f = open(self._filename, 'w')
        f.writelines([ line + '\n' for line in lines ])
        f.close()
        return


def UserAuthenticate(options=None, prompt="User", promptOnly=0):
    """
    Obtains a username and password hash, prompting for them, if necessary.
    Returns a tuple (username, password), which can be (None, None).

    Username sources normally checked are, in order:
    1. the options dictionary;
    2. the agent referenced in the FTSS_AGENT environment variable;
    3. the FTSS_USERNAME environment variable;
    4. a prompt for user input.

    Password sources normally checked are, in order:
    1. the password file referenced in environment var. FTSS_PASSWORD_FILE;
    2. the password file $HOME/.4ss.passwd or %WINDIR%\.4ss.passwd;
    3. the password file referenced in the deprecated environment variable
       FTSERVER_PASSWORD_FILE;
    4. a prompt for user input.

    Optional arguments:

    options - a dictionary of options parsed from the command line
    (see FancyGetOpt). The key 'anonymous' is taken to indicate that no
    authentication is necessary and (None, None) should be returned.
    If the key 'username' is present, the associated value is returned as
    the username and no other potential sources for username are checked.

    prompt - if the user is to be prompted for their username, this is the
    string that indicates the type of user to be mentioned in the prompt.
    It is 'User' or 'Manager', typically.

    promptOnly - if set, this flag forces the username and password to be
    obtained by prompting the user; other sources are not checked.
    """
    options = options or {}

    #See if there is an overide on the command line
    if options.has_key('anonymous') and not promptOnly:
        return (None, None)

    username = options.get('username')

    #See if it is in an agent
    if not username and os.environ.has_key('FTSS_AGENT') and not promptOnly:
        return eval(os.environ['FTSS_AGENT'])

    #See if it is in the environment
    if not username and not promptOnly:
        username = os.environ.get('FTSS_USERNAME')

    #Nope, ask for it
    if not username:
        username = GetUserName("4SS %s Name: " % prompt)

    #Now look for it in the passwd file
    passwordHash = None

    if not promptOnly:
        pm = PasswordFileManager()
        passwordHash, t, t = pm.getUserEntry(username)

    if not passwordHash:
        password = GetPass("Password for %s: " % username)
        passwordHash = HashPasswd(password)

    return (username, passwordHash)
