/*-------------------------------------------------------------------------
 * Copyright (c) 2000 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * mainwin
 *
 * Synopsis:
 *   The main MathWars window
 *
 * $Id: mainwin.c,v 1.3 2000/11/17 11:52:03 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "about_dlg.h"
#include "comp_guess_dlg.h"
#include "game_engine.h"
#include "mainwin.h"
#include "operators.h"
#include "prop_dlg.h"
#include "support.h"

#define MAINWIN        "main_window"
#define CARD_TABLE     "card_table"
#define CARD_ONE       "card_one"
#define CARD_TWO       "card_two"
#define OPERATOR       "operator"
#define EQUAL_SIGN     "equal_sign"
#define TIMER_DIAL     "timer_dial"
#define GAME_ENGINE    "game_eng"
#define ANSWER_ENTRY   "answer_entry"
#define SUBMIT_BUTTON  "submit_button"
#define PLAYER1_LBL    "player1_lbl"
#define PLAYER1_SCORE  "player1_score"
#define PLAYER2_LBL    "player2_lbl"
#define PLAYER2_SCORE  "player2_score"
#define ROUND_NUM      "round_num"

#define CARD_ROWS    3
#define CARD_COLS    4
#define CARD_ONE_X1  0
#define CARD_ONE_X2  1
#define OPER_X1      1
#define OPER_X2      2
#define CARD_TWO_X1  2
#define CARD_TWO_X2  3
#define EQUAL_X1     3
#define EQUAL_X2     4
#define CARD_ROW_TOP 1
#define CARD_ROW_BOT 2

#define ENTER_KEY          65293
#define KEYPAD_ENTER_KEY   65421

static void
on_new_game_activate (GtkMenuItem  *menuitem,
                      gpointer      user_data)
{
   GtkWidget   *mainwin;
   game_engine *game;

   mainwin = lookup_widget (GTK_WIDGET (menuitem), MAINWIN);
   g_assert (mainwin);
   game = gtk_object_get_data (GTK_OBJECT (mainwin), GAME_ENGINE);
   g_assert (game);
   begin_game (game);
   begin_round (game);
}


static void
on_exit_activate (GtkMenuItem  *menuitem,
                  gpointer      user_data)
{
   GtkWidget  *mainwin;

   mainwin = lookup_widget (GTK_WIDGET (menuitem), MAINWIN);
   g_assert (mainwin);

   gtk_widget_destroy (GTK_WIDGET (mainwin));
}


static void
on_preferences1_activate (GtkMenuItem  *menuitem,
                          gpointer      user_data)
{
   show_prop_dlg();
}


static void
on_about_activate (GtkMenuItem  *menuitem,
                   gpointer      user_data)
{
   show_about_dlg();
}


static void
on_mainwin_show (GtkWidget     *widget,
                 gpointer       user_data)
{
   game_engine *game;

   game = create_game_engine (widget);
   gtk_object_set_data (GTK_OBJECT (widget), GAME_ENGINE, game);
}


static void
on_mainwin_destroy (GtkWidget  *widget,
                    gpointer    user_data)
{
   game_engine *game;

   game = gtk_object_get_data (GTK_OBJECT (widget), GAME_ENGINE);
   g_assert (game);
   destroy_game_engine (game);

   gtk_main_quit();
}


static gboolean
on_answer_entry_key_press_event (GtkWidget       *widget,
                                 GdkEventKey     *event,
                                 gpointer         user_data)
{
   GtkWidget  *submit_btn;

   if (event->keyval == ENTER_KEY || event->keyval == KEYPAD_ENTER_KEY)
      {
      submit_btn = get_widget (widget, SUBMIT_BUTTON);
      g_assert (submit_btn);

      gtk_signal_emit_by_name (GTK_OBJECT (submit_btn), "clicked");
      }

   return FALSE;
}


static void
on_submit_clicked (GtkWidget  *widget,
                   gpointer    user_data)
{
   game_engine *game;
   GtkWidget   *answer_entry;
   GtkWidget   *mainwin;

   mainwin = lookup_widget (widget, MAINWIN);
   g_assert (mainwin);
   answer_entry = lookup_widget (mainwin, ANSWER_ENTRY);
   g_assert (answer_entry);
   game = gtk_object_get_data (GTK_OBJECT (mainwin), GAME_ENGINE);
   g_assert (game);

   handle_player_answer (game, gtk_entry_get_text (GTK_ENTRY (answer_entry)));
}


static GnomeUIInfo game_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_NEW_ITEM (N_("_New Game"), NULL, 
                             on_new_game_activate, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM (on_exit_activate, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo settings1_menu_uiinfo[] =
{
  GNOMEUIINFO_MENU_PREFERENCES_ITEM (on_preferences1_activate, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo help1_menu_uiinfo[] =
{
  GNOMEUIINFO_HELP("MathWar"),
  GNOMEUIINFO_SEPARATOR,    
  GNOMEUIINFO_MENU_ABOUT_ITEM (on_about_activate, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menubar1_uiinfo[] =
{
  GNOMEUIINFO_MENU_FILE_TREE (game_menu_uiinfo),
  GNOMEUIINFO_MENU_SETTINGS_TREE (settings1_menu_uiinfo),
  GNOMEUIINFO_MENU_HELP_TREE (help1_menu_uiinfo),
  GNOMEUIINFO_END
};


GtkWidget *
create_mainwin (void)
{
  GtkWidget *app1;
  GtkWidget *dock1;
  GtkWidget *vbox1;
  GtkWidget *table1;
  GtkWidget *player1_lbl;
  GtkWidget *comp_lbl;
  GtkWidget *player1_score;
  GtkWidget *computer_score;
  GtkWidget *timer_dial;
  GtkWidget *card_table;
  GtkWidget *pixmap1;
  GtkWidget *pixmap2;
  GtkWidget *pixmap3;
  GtkWidget *pixmap4;
  GtkWidget *hbox1;
  GtkWidget *entry1;
  GtkWidget *submit_btn;
  GtkWidget *appbar1;
  GtkWidget *hsep;
  GtkWidget *round_lbl;
  GtkWidget *round_num;
  GtkWidget *dial_lbl;

  app1 = gnome_app_new (PACKAGE, _(PACKAGE));
  gtk_object_set_data (GTK_OBJECT (app1), MAINWIN, app1);
  gtk_signal_connect (GTK_OBJECT (app1), "show",
                      GTK_SIGNAL_FUNC (on_mainwin_show),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (app1), "destroy",
                      GTK_SIGNAL_FUNC (on_mainwin_destroy),
                      NULL);

  dock1 = GNOME_APP (app1)->dock;
  gtk_widget_ref (dock1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "dock1", dock1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (dock1);

  gnome_app_create_menus (GNOME_APP (app1), menubar1_uiinfo);

  gtk_widget_ref (menubar1_uiinfo[0].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "game",
                            menubar1_uiinfo[0].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (game_menu_uiinfo[0].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "new_game",
                            game_menu_uiinfo[0].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (game_menu_uiinfo[1].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "separator1",
                            game_menu_uiinfo[1].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (game_menu_uiinfo[2].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "exit1",
                            game_menu_uiinfo[2].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (menubar1_uiinfo[1].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "settings1",
                            menubar1_uiinfo[1].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (settings1_menu_uiinfo[0].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "preferences1",
                            settings1_menu_uiinfo[0].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (menubar1_uiinfo[2].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "help1",
                            menubar1_uiinfo[2].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  gtk_widget_ref (help1_menu_uiinfo[2].widget);
  gtk_object_set_data_full (GTK_OBJECT (app1), "about1",
                            help1_menu_uiinfo[2].widget,
                            (GtkDestroyNotify) gtk_widget_unref);

  vbox1 = gtk_vbox_new (FALSE, 10);
  gtk_widget_ref (vbox1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "vbox1", vbox1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (vbox1);
  gnome_app_set_contents (GNOME_APP (app1), vbox1);
  gtk_container_set_border_width (GTK_CONTAINER (vbox1), 10);

  table1 = gtk_table_new (3, 3, FALSE);
  gtk_widget_ref (table1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "table1", table1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (table1);
  gtk_box_pack_start (GTK_BOX (vbox1), table1, FALSE, FALSE, 0);
  gtk_table_set_col_spacings (GTK_TABLE (table1), 5);

  dial_lbl = gtk_label_new (_("Time Left:"));
  gtk_widget_ref (dial_lbl);
  gtk_object_set_data_full (GTK_OBJECT (app1), "dial_lbl", dial_lbl,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (dial_lbl);
  gtk_table_attach (GTK_TABLE (table1), dial_lbl, 0, 1, 0,  1,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);

  timer_dial = gtk_dial_new 
     (GTK_ADJUSTMENT (gtk_adjustment_new (100, 0.0, 100, 0.0, 0.0, 0.0)));
  gtk_widget_ref (timer_dial);
  gtk_widget_set_sensitive (timer_dial, FALSE);
  gtk_object_set_data_full (GTK_OBJECT (app1), TIMER_DIAL, timer_dial,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (timer_dial);
  gtk_dial_set_view_only (GTK_DIAL (timer_dial), TRUE);
  gtk_table_attach (GTK_TABLE (table1), timer_dial, 0, 1, 1, 3,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);

  round_lbl = gtk_label_new (_("Rounds Left:"));
  gtk_widget_ref (round_lbl);
  gtk_object_set_data_full (GTK_OBJECT (app1), "round_lbl", round_lbl,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (round_lbl);
  gtk_table_attach (GTK_TABLE (table1), round_lbl, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (round_lbl), 1.0, 0.5);

  player1_lbl = gtk_label_new (_("Player 1"));
  gtk_widget_ref (player1_lbl);
  gtk_object_set_data_full (GTK_OBJECT (app1), PLAYER1_LBL, player1_lbl,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (player1_lbl);
  gtk_table_attach (GTK_TABLE (table1), player1_lbl, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (player1_lbl), 1.0, 0.5);

  comp_lbl = gtk_label_new (_("Computer"));
  gtk_widget_ref (comp_lbl);
  gtk_object_set_data_full (GTK_OBJECT (app1), PLAYER2_LBL, comp_lbl,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (comp_lbl);
  gtk_table_attach (GTK_TABLE (table1), comp_lbl, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (comp_lbl), 1.0, 0.5);

  round_num = gtk_label_new ("10");
  gtk_widget_ref (round_num);
  gtk_object_set_data_full (GTK_OBJECT (app1), ROUND_NUM, round_num,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (round_num);
  gtk_table_attach (GTK_TABLE (table1), round_num, 2, 3, 0, 1,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);

  player1_score = gtk_label_new ("0");
  gtk_widget_ref (player1_score);
  gtk_object_set_data_full (GTK_OBJECT (app1), PLAYER1_SCORE, player1_score,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (player1_score);
  gtk_table_attach (GTK_TABLE (table1), player1_score, 2, 3, 1, 2,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);

  computer_score = gtk_label_new ("0");
  gtk_widget_ref (computer_score);
  gtk_object_set_data_full (GTK_OBJECT (app1), PLAYER2_SCORE, computer_score,                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (computer_score);
  gtk_table_attach (GTK_TABLE (table1), computer_score, 2, 3, 2, 3,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);

  card_table = gtk_table_new (CARD_ROWS, CARD_COLS, FALSE);
  gtk_widget_ref (card_table);
  gtk_object_set_data_full (GTK_OBJECT (app1), 
                            CARD_TABLE,
                            card_table,
                            (GtkDestroyNotify)gtk_widget_unref);
  gtk_box_pack_start (GTK_BOX (vbox1), card_table, TRUE, TRUE, 0);
  gtk_widget_show (card_table);
  gtk_table_set_row_spacings (GTK_TABLE (card_table), 5);

  hsep = gtk_hseparator_new ();
  gtk_widget_ref (hsep);
  gtk_object_set_data_full (GTK_OBJECT (hsep), "hsep1", hsep,
                            (GtkDestroyNotify)gtk_widget_unref);
  gtk_widget_show (hsep);
  gtk_table_attach (GTK_TABLE (card_table), hsep,
                    0, 4, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
                            
  pixmap1 = create_pixmap (app1, "card_back1.xpm", FALSE);
  gtk_widget_ref (pixmap1);
  gtk_object_set_data_full (GTK_OBJECT (app1), CARD_ONE, pixmap1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (pixmap1);
  gtk_table_attach (GTK_TABLE (card_table), pixmap1, 
                    CARD_ONE_X1, CARD_ONE_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  pixmap2 = create_pixmap (app1, "plus_sign.xpm", FALSE);
  gtk_widget_ref (pixmap2);
  gtk_object_set_data_full (GTK_OBJECT (app1), OPERATOR, pixmap2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (pixmap2);
  gtk_table_attach (GTK_TABLE (card_table), pixmap2, 
                    OPER_X1, OPER_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  pixmap3 = create_pixmap (app1, "card_back2.xpm", FALSE);
  gtk_widget_ref (pixmap3);
  gtk_object_set_data_full (GTK_OBJECT (app1), CARD_TWO, pixmap3,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (pixmap3);
  gtk_table_attach (GTK_TABLE (card_table), pixmap3, 
                    CARD_TWO_X1, CARD_TWO_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  pixmap4 = create_pixmap (app1, "equal_sign.xpm", FALSE);
  gtk_widget_ref (pixmap4);
  gtk_object_set_data_full (GTK_OBJECT (app1), EQUAL_SIGN, pixmap4,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (pixmap4);
  gtk_table_attach (GTK_TABLE (card_table), pixmap4, 
                    EQUAL_X1, EQUAL_X2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  hsep = gtk_hseparator_new ();
  gtk_widget_ref (hsep);
  gtk_object_set_data_full (GTK_OBJECT (hsep), "hsep2", hsep,
                            (GtkDestroyNotify)gtk_widget_unref);
  gtk_widget_show (hsep);
  gtk_table_attach (GTK_TABLE (card_table), hsep,
                    0, 4, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);

  hbox1 = gtk_hbox_new (FALSE, 10);
  gtk_widget_ref (hbox1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "hbox1", hbox1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (hbox1);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 0);

  entry1 = gtk_entry_new ();
  gtk_widget_ref (entry1);
  gtk_object_set_data_full (GTK_OBJECT (app1), ANSWER_ENTRY, entry1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (entry1);
  gtk_box_pack_start (GTK_BOX (hbox1), entry1, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (entry1), "key_press_event",
                      GTK_SIGNAL_FUNC (on_answer_entry_key_press_event),
                      NULL);

  submit_btn = gtk_button_new_with_label (_("Submit"));
  gtk_widget_ref (submit_btn);
  gtk_object_set_data_full (GTK_OBJECT (app1), SUBMIT_BUTTON, submit_btn,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (submit_btn);
  gtk_box_pack_start (GTK_BOX (hbox1), submit_btn, FALSE, FALSE, 0);
  GTK_WIDGET_SET_FLAGS (submit_btn, GTK_CAN_DEFAULT);
  gtk_signal_connect (GTK_OBJECT (submit_btn), "clicked",
                      GTK_SIGNAL_FUNC (on_submit_clicked),
                      NULL);

  appbar1 = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
  gtk_widget_ref (appbar1);
  gtk_object_set_data_full (GTK_OBJECT (app1), "appbar1", appbar1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (appbar1);
  gnome_app_set_statusbar (GNOME_APP (app1), appbar1);

  gnome_app_install_menu_hints (GNOME_APP (app1), menubar1_uiinfo);

  gtk_widget_grab_default (submit_btn);

  return app1;
}


void 
set_timer_pct (GtkWidget *mainwin,
               gfloat     pct)
{
   GtkDial  *dial;

   dial = GTK_DIAL (lookup_widget (mainwin, TIMER_DIAL));
   gtk_dial_set_percentage (dial, pct);

   return;
}


void
display_pause (GtkWidget *mainwin)
{
   GtkWidget  *tmp_pixmap;
   GtkWidget  *card_table;

   card_table = lookup_widget (mainwin, CARD_TABLE);
   g_assert (card_table);

   tmp_pixmap = lookup_widget (mainwin, CARD_ONE);
   g_assert (tmp_pixmap);
   gtk_widget_destroy (tmp_pixmap);
   gtk_object_remove_data (GTK_OBJECT (mainwin), CARD_ONE);
   tmp_pixmap = create_pixmap (mainwin, "card_back1.xpm", FALSE);
   gtk_widget_ref (tmp_pixmap);
   gtk_object_set_data_full (GTK_OBJECT (mainwin),
                             CARD_ONE,
                             tmp_pixmap,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_table_attach (GTK_TABLE (card_table), tmp_pixmap, 
                     CARD_ONE_X1, CARD_ONE_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
   gtk_widget_show (tmp_pixmap);

   tmp_pixmap = lookup_widget (mainwin, OPERATOR);
   g_assert (tmp_pixmap);
   gtk_widget_destroy (tmp_pixmap);
   gtk_object_remove_data (GTK_OBJECT (mainwin), OPERATOR);
   tmp_pixmap = create_pixmap (mainwin, "unknown_oper.xpm", FALSE);
   gtk_widget_ref (tmp_pixmap);
   gtk_object_set_data_full (GTK_OBJECT (mainwin),
                             OPERATOR,
                             tmp_pixmap,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_table_attach (GTK_TABLE (card_table), tmp_pixmap, 
                     OPER_X1, OPER_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
   gtk_widget_show (tmp_pixmap);

   tmp_pixmap = lookup_widget (mainwin, CARD_TWO);
   g_assert (tmp_pixmap);
   gtk_widget_destroy (tmp_pixmap);
   gtk_object_remove_data (GTK_OBJECT (mainwin), CARD_TWO);
   tmp_pixmap = create_pixmap (mainwin, "card_back2.xpm", FALSE);
   gtk_widget_ref (tmp_pixmap);
   gtk_object_set_data_full (GTK_OBJECT (mainwin),
                             CARD_TWO,
                             tmp_pixmap,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_table_attach (GTK_TABLE (card_table), tmp_pixmap, 
                     CARD_TWO_X1, CARD_TWO_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
   gtk_widget_show (tmp_pixmap);

   return;
}

void
display_cards (GtkWidget     *mainwin,
               gint           card_one_val,
               gint           card_two_val,
               operator_type  operator)
{
   GtkWidget  *tmp_pixmap;
   GtkWidget  *card_table;
   GtkWidget  *answer;
   GString    *pixmap_name;

   g_return_if_fail (card_one_val >= 0 && card_one_val <= 20);
   g_return_if_fail (card_two_val >= 0 && card_two_val <= 20);

   pixmap_name = g_string_new ("");

   card_table = lookup_widget (mainwin, CARD_TABLE);
   g_assert (card_table);

   tmp_pixmap = lookup_widget (mainwin, CARD_ONE);
   g_assert (tmp_pixmap);
   gtk_widget_destroy (tmp_pixmap);
   gtk_object_remove_data (GTK_OBJECT (mainwin), CARD_ONE);
   g_string_sprintf (pixmap_name, "card_%d.xpm", card_one_val);
   tmp_pixmap = create_pixmap (mainwin, pixmap_name->str, FALSE);
   gtk_widget_ref (tmp_pixmap);
   gtk_object_set_data_full (GTK_OBJECT (mainwin), 
                             CARD_ONE,
                             tmp_pixmap, 
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_table_attach (GTK_TABLE (card_table), tmp_pixmap, 
                     CARD_ONE_X1, CARD_ONE_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
   gtk_widget_show (tmp_pixmap);

   tmp_pixmap = lookup_widget (mainwin, OPERATOR);
   g_assert (tmp_pixmap);
   gtk_widget_destroy (tmp_pixmap);
   gtk_object_remove_data (GTK_OBJECT (mainwin), OPERATOR);
   switch (operator)
      {
      case SUBTRACTION:
         tmp_pixmap = create_pixmap (mainwin, "minus_sign.xpm", FALSE);
         break;
         
      case ADDITION:
         tmp_pixmap = create_pixmap (mainwin, "plus_sign.xpm", FALSE);
         break;

      case MULTIPLICATION:
         tmp_pixmap = create_pixmap (mainwin, "mult_sign.xpm", FALSE);
         break;

      default:
         tmp_pixmap = create_pixmap (mainwin, "unknown_oper.xpm", FALSE);
         break;
      }
   gtk_widget_ref (tmp_pixmap);
   gtk_object_set_data_full (GTK_OBJECT (mainwin),
                             OPERATOR,
                             tmp_pixmap,
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_table_attach (GTK_TABLE (card_table), tmp_pixmap,
                     OPER_X1, OPER_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
   gtk_widget_show (tmp_pixmap);

   tmp_pixmap = lookup_widget (mainwin, CARD_TWO);
   g_assert (tmp_pixmap);
   gtk_widget_destroy (tmp_pixmap);
   gtk_object_remove_data (GTK_OBJECT (mainwin), CARD_TWO);
   g_string_sprintf (pixmap_name, "card_%d.xpm", card_two_val);
   tmp_pixmap = create_pixmap (mainwin, pixmap_name->str, FALSE);
   gtk_widget_ref (tmp_pixmap);
   gtk_object_set_data_full (GTK_OBJECT (mainwin), 
                             CARD_TWO,
                             tmp_pixmap, 
                             (GtkDestroyNotify) gtk_widget_unref);
   gtk_table_attach (GTK_TABLE (card_table), tmp_pixmap, 
                     CARD_TWO_X1, CARD_TWO_X2, CARD_ROW_TOP, CARD_ROW_BOT,
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
   gtk_widget_show (tmp_pixmap);

   g_string_free (pixmap_name, TRUE);

   /*
    * Now that the cards are displayed, we should clear any old text out
    * of the entry window.
    */
   answer = lookup_widget (mainwin, ANSWER_ENTRY);
   g_assert (answer);
   gtk_entry_set_text (GTK_ENTRY (answer), "");
   gtk_widget_grab_focus (answer);
}


void
display_names (GtkWidget   *mainwin,
               const gchar *name1,
               const gchar *name2)
{
   GtkWidget  *name_lbl;

   name_lbl = lookup_widget (mainwin, PLAYER1_LBL);
   g_return_if_fail (name_lbl);
   gtk_label_set_text (GTK_LABEL (name_lbl), name1);

   name_lbl = lookup_widget (mainwin, PLAYER2_LBL);
   g_return_if_fail (name_lbl);
   gtk_label_set_text (GTK_LABEL (name_lbl), name2);
}


void
display_rounds_left (GtkWidget *mainwin,
                     gint       rounds)
{
   GtkWidget  *round_num_lbl;
   GString    *str;

   round_num_lbl = lookup_widget (mainwin, ROUND_NUM);
   g_return_if_fail (round_num_lbl);

   str = g_string_new ("");
   g_string_sprintf (str, "%d", rounds);
   gtk_label_set_text (GTK_LABEL (round_num_lbl), str->str);
   g_string_free (str, TRUE);
}


void
display_scores (GtkWidget *mainwin,
                gint       score1,
                gint       score2)
{
   GtkWidget  *score_lbl;
   GString    *str;

   str = g_string_new ("");

   score_lbl = lookup_widget (mainwin, PLAYER1_SCORE);
   g_return_if_fail (score_lbl);
   g_string_sprintf (str, "%d", score1);
   gtk_label_set_text (GTK_LABEL (score_lbl), str->str);

   score_lbl = lookup_widget (mainwin, PLAYER2_SCORE);
   g_return_if_fail (score_lbl);
   g_string_sprintf (str, "%d", score2);
   gtk_label_set_text (GTK_LABEL (score_lbl), str->str);
}
