#
# rmails.rb
#
#   Copyright (c) 1998-2001 Minero Aoki <aamine@loveruby.net>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#


module TMail

  class Scanner_R

    Version = '0.10.0'
    Version.freeze

    #
    # regexps
    #

    atomchar  = Regexp.quote( "\#!$%&`'*+{|}~^/=?" ) + '\\-'
    tokenchar = Regexp.quote( "\#!$%&`'*+{|}~^." )   + '\\-'
    jisstr    = '|\e..[^\e]*\e..'

    ATOM = {}
    TOKEN = {}
    {
      'EUC'  => '|(?:[\xa1-\xfe][\xa1-\xfe])+',
      'SJIS' => '|(?:[\x81-\x9f\xe0-\xef][\x40-\x7e\x80-\xfc])+',
      'UTF8' => '|(?:[\xc0-\xdf][\x80-\xbf]|[\xe0-\xef][\x80-\xbf][\x80-\xbf])+',
      'NONE' => ''
    }.each do |k,v|
      ATOM[k]  = /\A[\w#{atomchar}]+#{v}#{jisstr}/n
      TOKEN[k] = /\A[\w#{tokenchar}]+#{v}#{jisstr}/n
    end

    LWSP    = /\A[\n\r\t ]+/n
    DIGIT   = /\A\d+\z/

    QSTR   = /\A[^"\\\e]+#{jisstr}/n
    CSTR   = /\A[^\)\(\\\e]+#{jisstr}/n
    DSTR   = /\A[^\]\\]+#{jisstr}/n


    RECV_TOKEN = {
      'from' => :FROM,
      'by'   => :BY,
      'via'  => :VIA,
      'with' => :WITH,
      'id'   => :ID,
      'for'  => :FOR
    }


    def initialize( str, header, comments )
      init str

      @header = header
      @comments = comments || []

      @atom_mode = :atom
      @word_re = ATOM[$KCODE]
      @recv_mode = false

      case header
      when 'CTypeH', 'CEncodingH', 'CDispositionH'
        @atom_mode = :token
        @word_re = TOKEN[$KCODE]
      when 'RecvH'
        @recv_mode = true
      end

      @debug = false
    end

    attr_accessor :debug


    def scan( &block )
      if @debug then
        scan_main do |arr|
          s, v = arr
          printf "%7d %-10s %s\n",
                 rest_size(),
                 s.respond_to?(:id2name) ? s.id2name : s.inspect,
                 v.inspect
          yield arr
        end
      else
        scan_main &block
      end
    end

    def scan_main
      until eof? do
        if skip LWSP then
          break if eof?
        end

        if s = readstr(@word_re) then
          case @atom_mode
          when :atom
            if DIGIT === s then
              yield :DIGIT, s
            elsif @recv_mode then
              yield RECV_TOKEN[s.downcase] || :ATOM, s
            else
              yield :ATOM, s
            end
          when :token
            yield :TOKEN, s
          else
            raise 'TMail FATAL: atom mode is not atom/token'
          end

        elsif skip( /\A"/ ) then
          yield :QUOTED, quoted()

        elsif skip( /\A\(/ ) then
          @comments.push comment()

        elsif skip( /\A\[/ ) then
          yield :DOMLIT, domlit()

        else
          c = readchar()
          yield c, c
        end
      end

      yield false, '$'
    end


    private


    def quoted
      scan_qstr QSTR, /\A"/, 'quoted-string'
    end

    def domlit
      scan_qstr DSTR, /\A]/, 'domain-literal'
    end

    def scan_qstr( exp, term, type )
      ret = ''
      while true do
        eof? and scan_error! "found unterminated #{type}"
        
        if    s = readstr(exp) then ret << s
        elsif skip term        then break
        elsif skip( /\A\\/ )   then ret << readchar()
        else
          raise "TMail FATAL: not match in #{type}"
        end
      end

      ret
    end


    def comment
      ret = ''
      nest = 1

      while nest > 0 do
        eof? and scan_error! 'found unterminated comment'

        if    s = readstr(CSTR) then ret << s
        elsif skip( /\A\)/ )    then nest -= 1; ret << ')' unless nest == 0
        elsif skip( /\A\(/ )    then nest += 1; ret << '('
        elsif skip( /\A\\/ )    then ret << readchar()
        else
          raise 'TMail FATAL: not match in comment'
        end
      end

      ret
    end


    # string scanner

    def init( str )
      @src = str
    end

    def eof?
      @src.empty?
    end

    def rest_size
      @src.size
    end

    def readstr( re )
      if m = re.match(@src) then
        @src = m.post_match
        m[0]
      else
        nil
      end
    end

    def readchar
      readstr( /\A./ )
    end

    def skip( re )
      if m = re.match(@src) then
        @src = m.post_match
        true
      else
        false
      end
    end


    def scan_error!( msg )
      raise SyntaxError, msg
    end

  end

end   # module TMail
