/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.modelmbean;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.management.Descriptor;
import javax.management.MBeanException;
import javax.management.RuntimeOperationsException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * @version $Revision: 1.28 $
 */
public class DescriptorSupport implements Descriptor
{
   private static final long serialVersionUID = -6292969195866300415L;

   private HashMap descriptor = new HashMap(20);
   private transient HashMap fields = new HashMap(20);

   public DescriptorSupport()
   {
   }

   public DescriptorSupport(int initNumFields) throws MBeanException, RuntimeOperationsException
   {
      if (initNumFields <= 0)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Number of Fields cannot be <= 0"));
      }
      descriptor = new HashMap(initNumFields);
      fields = new HashMap(initNumFields);
   }

   public DescriptorSupport(DescriptorSupport inDescr)
   {
      if (inDescr != null)
      {
         setFields(inDescr.getFieldNames(), inDescr.getFieldValues(inDescr.getFieldNames()));
      }
   }

   public DescriptorSupport(String xml) throws MBeanException, RuntimeOperationsException, XMLParseException
   {
      if (xml == null)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Descriptor XML string is null"));
      }

      NodeList fields = documentFromXML(xml).getElementsByTagName("field");
      for (int i = 0; i < fields.getLength(); i++)
      {
         addFieldFromXML(fields.item(i));
      }
   }

   public DescriptorSupport(String[] pairs)
   {
      if (pairs != null && pairs.length != 0)
      {
         for (int i = 0; i < pairs.length; ++i)
         {
            String pair = pairs[i];
            if (pair == null)
            {
               throw new RuntimeOperationsException(new IllegalArgumentException("Illegal pair: " + pair));
            }
            int equal = pair.indexOf('=');
            if (equal < 1)
            {
               throw new RuntimeOperationsException(new IllegalArgumentException("Illegal pair: " + pair));
            }
            else
            {
               String name = pair.substring(0, equal);
               Object value = null;
               if (equal < pair.length() - 1)
               {
                  value = pair.substring(equal + 1);
               }
               setField(name, value);
            }
         }
      }
   }

   public DescriptorSupport(String[] names, Object[] values)
   {
      setFields(names, values);
   }

   public Object clone() throws RuntimeOperationsException
   {
      return new DescriptorSupport(this);
   }

   public Object getFieldValue(String name) throws RuntimeOperationsException
   {
      if (name == null || name.trim().length() == 0)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Invalid field name"));
      }
      // Field names are case insensitive, retrieve the value from the case-insensitive map
      ValueHolder holder = (ValueHolder)fields.get(name.toLowerCase());
      return holder == null ? null : holder.fieldValue;
   }

   public void setField(String name, Object value) throws RuntimeOperationsException
   {
      checkField(name, value);
      descriptor.put(name, value);
      fields.put(name.toLowerCase(), new ValueHolder(name, value));
   }

   public void removeField(String name)
   {
      if (name != null)
      {
         ValueHolder holder = (ValueHolder)fields.remove(name.toLowerCase());
         if (holder != null) descriptor.remove(holder.fieldName);
      }
   }

   public String[] getFieldNames()
   {
      // Preserve the case of field names
      String[] names = (String[])descriptor.keySet().toArray(new String[0]);
      return names;
   }

   public Object[] getFieldValues(String[] names)
   {
      if (names == null)
      {
         // All values must be returned
         Object[] values = descriptor.values().toArray(new Object[0]);
         return values;
      }

      ArrayList list = new ArrayList();
      for (int i = 0; i < names.length; ++i)
      {
         try
         {
            Object value = getFieldValue(names[i]);
            list.add(value);
         }
         catch (RuntimeOperationsException x)
         {
            list.add(null);
         }
      }
      Object[] values = list.toArray(new Object[list.size()]);
      return values;
   }

   public String[] getFields()
   {
      ArrayList list = new ArrayList();
      StringBuffer buffer = new StringBuffer();
      // Preserve the case of field names
      for (Iterator i = descriptor.entrySet().iterator(); i.hasNext();)
      {
         Map.Entry entry = (Map.Entry)i.next();
         String key = (String)entry.getKey();
         Object value = entry.getValue();
         buffer.setLength(0);
         buffer.append(key);
         buffer.append("=");
         if (value != null)
         {
            if (value instanceof String)
            {
               buffer.append(value.toString());
            }
            else
            {
               buffer.append("(");
               buffer.append(value.toString());
               buffer.append(")");
            }
         }
         list.add(buffer.toString());
      }
      String[] fields = (String[])list.toArray(new String[list.size()]);
      return fields;
   }

   public void setFields(String[] names, Object[] values) throws RuntimeOperationsException
   {
      if (names == null || values == null || names.length != values.length)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Invalid arguments"));
      }

      for (int i = 0; i < names.length; ++i)
      {
         setField(names[i], values[i]);
      }
   }

   public boolean isValid() throws RuntimeOperationsException
   {
      if (getFieldValue("name") == null || getFieldValue("descriptorType") == null) return false;

      try
      {
         for (Iterator i = descriptor.entrySet().iterator(); i.hasNext();)
         {
            Map.Entry entry = (Map.Entry)i.next();
            String name = (String)entry.getKey();
            Object value = entry.getValue();
            checkField(name, value);
         }
         return true;
      }
      catch (RuntimeOperationsException x)
      {
         return false;
      }
   }

   public String toXMLString() throws RuntimeOperationsException
   {
      StringBuffer buf = new StringBuffer(32);
      buf.append("<Descriptor>");

      try
      {
         if (descriptor.size() != 0)
         {
            for (Iterator i = descriptor.entrySet().iterator();
                 i.hasNext();
                    )
            {
               Map.Entry entry = (Map.Entry)i.next();
               Object value = entry.getValue();
               String valstr = toXMLValueString(value);
               buf.append("<field name=\"");
               buf.append(entry.getKey());
               buf.append("\" value=\"");
               buf.append(valstr);
               buf.append("\"></field>");
            }
         }
         buf.append("</Descriptor>");
         return buf.toString();
      }
      catch (RuntimeException x)
      {
         throw new RuntimeOperationsException(x);
      }
   }

   public String toString() throws RuntimeOperationsException
   {
      StringBuffer buf = new StringBuffer();
      try
      {
         if (descriptor.size() != 0)
         {
            for (Iterator i = descriptor.entrySet().iterator(); i.hasNext();)
            {
               Map.Entry entry = (Map.Entry)i.next();
               buf.append(entry.getKey()).append(" ").append(entry.getValue());
               if (i.hasNext())
               {
                  buf.append(",");
               }
            }
         }
         return buf.toString();

      }
      catch (RuntimeOperationsException x)
      {
         return buf.toString();
      }
   }

   private void addFieldFromXML(Node n) throws XMLParseException, DOMException, RuntimeOperationsException
   {
      if (!(n instanceof Element))
      {
         throw new XMLParseException("Invalid XML descriptor entity");
      }
      else
      {
         NamedNodeMap attributes = n.getAttributes();
         if (attributes.getLength() != 2
             && (attributes.getNamedItem("name") == null
                 || attributes.getNamedItem("value") == null))
         {
            throw new XMLParseException("Invalid XML descriptor element");
         }
         else
         {
            String name =
                    attributes.getNamedItem("name").getNodeValue();
            String value =
                    attributes.getNamedItem("value").getNodeValue();
            setField(name, parseValueString(value));
         }
      }
   }

   private void checkField(String name, Object value) throws RuntimeOperationsException
   {
      if (name == null || name.trim().length() == 0)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Illegal field name"));
      }

      boolean isValid = true;

      if (name.equalsIgnoreCase("name"))
      {
         isValid = value != null && value instanceof String && ((String)value).length() != 0;
      }
      else if (name.equalsIgnoreCase("descriptorType"))
      {
         isValid = value != null && (value.toString().equalsIgnoreCase("MBean") ||
                                     value.toString().equalsIgnoreCase("attribute") ||
                                     value.toString().equalsIgnoreCase("operation") ||
                                     value.toString().equalsIgnoreCase("notification"));
      }
      else if (name.equalsIgnoreCase("role"))
      {
         isValid = value != null && (value.equals("constructor") ||
                                     value.equals("operation") ||
                                     value.equals("getter") ||
                                     value.equals("setter"));
      }
      else if (name.equalsIgnoreCase("persistPolicy"))
      {
         isValid = value != null && (value.toString().equalsIgnoreCase("Never") ||
                                     value.toString().equalsIgnoreCase("OnTimer") ||
                                     value.toString().equalsIgnoreCase("OnUpdate") ||
                                     value.toString().equalsIgnoreCase("NoMoreOftenThan") ||
                                     value.toString().equalsIgnoreCase("Always"));
      }
      else if (name.equalsIgnoreCase("persistPeriod"))
      {
         int v = objectToInt(value);
         isValid = v >= -1;
      }
      else if (name.equalsIgnoreCase("currencyTimeLimit"))
      {
         int v = objectToInt(value);
         isValid = v >= -1;
      }
      else if (name.equalsIgnoreCase("visibility"))
      {
         int v = objectToInt(value);
         isValid = v >= 1 && v <= 4;
      }
      else if (name.equalsIgnoreCase("getMethod") || name.equalsIgnoreCase("setMethod"))
      {
         isValid = value != null && value.toString().trim().length() > 0;
      }
      else if (name.equalsIgnoreCase("protocolMap"))
      {
         isValid = value instanceof Descriptor;
      }
      else if (name.equalsIgnoreCase("lastUpdatedTimeStamp"))
      {
         long v = objectToLong(value);
         isValid = v > 0;
      }
      else if (name.equalsIgnoreCase("severity"))
      {
         int v = objectToInt(value);
         isValid = v >= 0 && v <= 6;
      }
      else if (name.equalsIgnoreCase("messageId"))
      {
         isValid = value != null;
      }
      else if (name.equalsIgnoreCase("log"))
      {
         isValid = value != null &&
                   (value instanceof Boolean ||
                    (value instanceof String && (value.toString().equalsIgnoreCase("true") ||
                                                 value.toString().equalsIgnoreCase("false") ||
                                                 value.toString().equalsIgnoreCase("t") ||
                                                 value.toString().equalsIgnoreCase("f"))));
      }

      if (!isValid)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Invalid value '" + value + "' for field " + name));
      }
   }

   private Document documentFromXML(String xml) throws XMLParseException
   {
      try
      {
         DocumentBuilder db =
                 DocumentBuilderFactory.newInstance().newDocumentBuilder();
         Document d = db.parse(new ByteArrayInputStream(xml.getBytes()));
         return d;
      }
      catch (Exception x)
      {
         throw new XMLParseException(x.toString());
      }
   }

   private Class getObjectValueClass(String value) throws XMLParseException
   {
      int eoc = value.indexOf("/");
      if (eoc == -1)
      {
         throw new XMLParseException("Illegal XML descriptor class name");
      }
      String klass = value.substring(1, eoc);
      Class result = null;
      try
      {
         result = Thread.currentThread().getContextClassLoader().loadClass(klass);
      }
      catch (Exception x)
      {
         throw new XMLParseException(x.toString());
      }
      return result;
   }

   private String getObjectValueString(String value) throws XMLParseException
   {
      int bov = value.indexOf("/");
      if (bov == -1)
      {
         throw new XMLParseException("Illegal XML descriptor object value");
      }
      return value.substring(bov + 1, value.length() - 1);
   }

   private String objectClassToID(Class k)
   {
      StringBuffer result = new StringBuffer();
      result.append(k.getName());
      result.append("/");
      return result.toString();
   }

   private int objectToInt(Object obj) throws RuntimeOperationsException
   {
      try
      {
         return Integer.parseInt(obj.toString());
      }
      catch (NumberFormatException x)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Illegal value '" + obj + "' for numeric field"));
      }
      catch (NullPointerException x)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Illegal value '" + obj + "' for numeric field"));
      }
   }

   private long objectToLong(Object obj) throws RuntimeOperationsException
   {
      try
      {
         return Long.parseLong(obj.toString());
      }
      catch (NumberFormatException x)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Illegal value '" + obj + "' for numeric field"));
      }
      catch (NullPointerException x)
      {
         throw new RuntimeOperationsException(new IllegalArgumentException("Illegal value '" + obj + "' for numeric field"));
      }
   }

   private Object parseValueString(String value) throws XMLParseException
   {
      Object result = null;
      if (value.compareToIgnoreCase("(null)") == 0)
      {
         result = null;
      }
      else if (value.charAt(0) != '(')
      {
         result = value;
      }
      else
      {
         result = parseObjectValueString(value);
      }
      return result;
   }

   private Object parseObjectValueString(String value) throws XMLParseException
   {
      if (value.charAt(value.length() - 1) != ')')
      {
         throw new XMLParseException("Invalid XML descriptor value");
      }

      Object result = null;
      Class k = getObjectValueClass(value);
      String s = getObjectValueString(value);
      try
      {
         if (k != Character.class)
         {
            result =
            k.getConstructor(new Class[]{String.class}).newInstance(new Object[]{s});
         }
         else
         {
            result = new Character(s.charAt(0));
         }
      }
      catch (Exception x)
      {
         throw new XMLParseException(x.toString());
      }
      return result;
   }

   private String toXMLValueString(Object value)
   {
      String result;
      if (value == null)
      {
         result = "(null)";
      }
      else
      {
         Class k = value.getClass();
         if (k == String.class && ((String)value).charAt(0) != '(')
         {
            result = (String)value;
         }
         else
         {
            result = toObjectXMLValueString(k, value);
         }
      }
      return result;
   }

   private String toObjectXMLValueString(Class k, Object value)
   {
      StringBuffer result = new StringBuffer();
      result.append("(");
      result.append(objectClassToID(k));
      result.append(value.toString());
      result.append(")");
      return result.toString();
   }

   private void readObject(ObjectInputStream stream) throws IOException, ClassNotFoundException
   {
      stream.defaultReadObject();
      this.fields = new HashMap(descriptor.size());
      for (Iterator i = descriptor.entrySet().iterator(); i.hasNext();)
      {
         Map.Entry entry = (Map.Entry)i.next();
         String name = (String)entry.getKey();
         fields.put(name.toLowerCase(), new ValueHolder(name, entry.getValue()));
      }
   }

   private static class ValueHolder
   {
      private final String fieldName;
      private final Object fieldValue;

      private ValueHolder(String fieldName, Object value)
      {
         this.fieldName = fieldName;
         this.fieldValue = value;
      }
   }
}
