/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/** @file document.h
 *  @brief Header file for cssed documents. 
 *  Those functions are the document's interface for cssed.
 */
 
/** A convenient structure to pass data to callbacks when using ::CssDialogClass functions.  
 *	It's used to pass data to callbacks. You've got no need to use it
 *	if you don't like it.
 */	
typedef struct _CssedCssDialogsData {
	/** A pointer to the main ::CssedWindow object */
	CssedWindow* main_window;
	/** A pointer to a null terminated char array or NULL */
	gchar* previous_string;
	/** A boolean telling us if we must write down the string in ::previous_string */
	gboolean add_property_string;
} CssedCssDialogsData;

/** To add text at caret position.  
 *	This function lets you add text to a document at the caret position.
 *	@param doc - The ::CssedDoc where to write the text.
 *	@param text - A pointer to the null terminated string to add.
 */
void 
document_add_text( CssedDoc* doc, gchar* text );

/** To add text at caret position and a new line character at end.  
 *	This function lets you add text to a document at the caret position but adds also  
 *	new line character to put the caret on the next text line.
 *	@param doc - The ::CssedDoc where to write the text.
 *	@param text - A pointer to the null terminated string to add.
 */
void
document_add_text_line( CssedDoc * doc, gchar* text );

/** Gets the document at a nootebook's page.  
 *	When you need to get the document from a known position on the
 *	document's notebook, you can safely use this function.
 *	@param window - The main ::CssedWindow object pointer.
 *	@param page_number - The document's position on the document, starting with zero.
 *	@return A pointer to the document at such page, or NULL if the page number is out of range.
 */
CssedDoc*
document_get_from_notebook_page_num(CssedWindow* window, gint page_number);

/** Gets a pointer to the currently open document.  
 *	This is a really used function and returns a pointer to the ::CssedDoc being
 *	edited.
 *	@param window - The main ::CssedWindow object pointer.
 *	@return A pointer to the currently edited document.
 *	@note In cssed there's alway a document opened.
 */
CssedDoc*
document_get_current(CssedWindow* window);

/** Tells us if the document is saved or not.  
 *	This returns the "dirty" flag meaning if TRUE the document has changed and  
 *	has not been saved, if FALSE the document has been saved.
 *	@param doc - The ::CssedDoc to close.
 *	@return A boolean with the modification state of the document.
 */
gboolean
document_get_modified( CssedDoc* doc );

/** Closes and frees one document.  
 *	It closes and frees all memory associated with the document. It doesn't ask
 *	to save the document so all changes will be lost.
 *	@param doc - The ::CssedDoc to close.
 */
gboolean
document_dispose( CssedDoc* doc );

/** Returns the character position at start of any line.  
 *	It returns an integer containing the character count at start of any given line.
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line number.
 */
gint
document_get_position_from_line( CssedDoc* doc, gint line);

/* filetype this will change */

/** Sets the file type.  
 *	It sets highlighting and folding for a document based on its file type..
 *	@param doc - A ::CssedDoc pointer.
 *	@param filetype - One of the built-in file types ( One of the enum ::CssedFileType ).
 *	@warning The file types will be managed in the future by loadable modules.
 *	This function will be deprecated in a near future.
 */
void
document_set_filetype( CssedDoc* doc, CssedFileType filetype );

/** Gets the file type identifier.  
 *	It gets the built-in file type. This function performs a pattern matching search based solely on the file name.
 *	@param filename - The file's name or full path.
 *	@warning The file types will be managed in the future by loadable modules.
 *	This function will be deprecated in a near future.
 */
CssedFileType
document_get_filetype_from_filename( gchar* filename );

/** Gets the file type identifier.  
 *	It gets the built-in file type from the document's object.
 *	@param doc - A ::CssedDoc pointer.
 *	@warning The file types will be managed in the future by loadable modules.
 *	This function will be deprecated in a near future.
 */
CssedFileType
document_get_filetype( CssedDoc* doc );

/** Forward searches text in the document.  
 *	It searches and selects the text passed as parameter from the caret position.
 *	@param doc - A ::CssedDoc pointer.
 *	@param text - The text to look for.
 *	@param flags - An integer containing (SCFIND_REGEX | SCFIND_POSIX) | (SCFIND_WHOLEWORD | SCFIND_MATCHCASE | SCFIND_WORDSTART) Those are defined in Scintilla.h
 */
void 
document_search_next( CssedDoc* doc, gchar* text, gint flags );

/** Backwards searches text in the document.  
 *	It searches and selects the text passed as as parameter from the caret position.
 *	@param doc - A ::CssedDoc pointer.
 *	@param text - The text to look for.
 *	@param flags - An integer containing (SCFIND_REGEX | SCFIND_POSIX) | (SCFIND_WHOLEWORD | SCFIND_MATCHCASE | SCFIND_WORDSTART) Those are defined in Scintilla.h
 */
void
document_search_prev( CssedDoc* doc, gchar* text, gint flags );

/** Returns the line number at one character position.  
 *	This function lets you get the line number where a defined character position is.
 *	@param doc - A ::CssedDoc pointer.
 *	@param position - An integer indicating the character position.
 */
gint
document_get_line_from_position( CssedDoc* doc, gint position);

/** Returns the caret's character position.  
 *	This function lets you get the character position where the caret is.
 *	@param doc - A ::CssedDoc pointer.
 */
gint 
document_get_current_position	(CssedDoc* doc );

/** Moves the caret to a defined position.  
 *	This function lets you move caret to one defined position, the view will scroll to the new caret position.
 *	@param doc - A ::CssedDoc pointer.
 *	@param position - An integer indicating the character position.
 */
void
document_set_current_position	(CssedDoc* doc, gint position);

/** Pastes the text to the clipboard.  
 *	This pastes the text to the clipboard at the current document, and also the current caret position.
 *	@param doc - A ::CssedDoc pointer.
 */
void 
document_paste (CssedDoc* doc);

/** Copies the text selected.  
 *	This copies the selected text to the clipboard.
 *	@param doc - A ::CssedDoc pointer.
 */
void 
document_copy (CssedDoc* doc);

/** Cuts the selected text.  
 *	This copies the selected text to the clipboard, and deletes the selection from the document.
 *	@param doc - A ::CssedDoc pointer.
 */
void 
document_cut (CssedDoc* doc);

/** Gets the character position at the end of current selection.  
 *	Gets the character position at the end of current selection, if no selection exists it will return the caret position.
 *	@param doc - A ::CssedDoc pointer.
 */
gint
document_get_selection_end(CssedDoc* doc);

/** Gets the character position at the start of current selection.  
 *	Gets the character position at the start of current selection, if no selection exists it will return the caret position.
 *	@param doc - A ::CssedDoc pointer.
 */
gint
document_get_selection_start(CssedDoc* doc);

/** Sets the start of one selection.  
 *	Sets the character position where to start the current selection.
 *	@param doc - A ::CssedDoc pointer.
 *	@param position - an integer containing the character position.
 */
void
document_set_selection_start(CssedDoc* doc, gint position);

/** Sets the end of one selection.  
 *	Sets the character position where to end the current selection.
 *	@param doc - A ::CssedDoc pointer.
 *	@param position - an integer containing the character position.
 */
void
document_set_selection_end(CssedDoc* doc, gint position);

/** Gets the character position at the end of one line.  
 *	Gets the character position at the end of one line, based on one character position at the document's text.
 *	@param doc - A ::CssedDoc pointer.
 *	@param position - an integer containing the character position.
 */
gint 
document_get_line_end_from_position(CssedDoc* doc, gint position);

/** Selects some text.  
 *	Selects some text starting at character position "start" and ending at character position "end".
 *	@param doc - A ::CssedDoc pointer.
 *	@param start - an integer containing the character position where the selection should start.
 *	@param end - an integer containing the character position where the selection should end.
 */
void 
document_set_selection_range(CssedDoc* doc,gint start, gint end);

/** Replaces the selected text.  
 *	Replaces the selected text with the text passed as parameter, if no selection exists nothing will happen.
 *	@param doc - A ::CssedDoc pointer.
 *	@param text - The text that will replace the selected one.
 */
void
document_replace_sel(CssedDoc* doc, gchar* text);

/** Deletes the selected text.  
 *	Deletes the selected text, if no selection exists nothing will happen. It's the same as document_replace_sel( doc, "" );
 *	@param doc - A ::CssedDoc pointer.
 */
void
document_clear_sel(CssedDoc* doc);

/** Gets the length of the text on the document.  
 *	It does not include any null terminatiion character.
 *	@param doc - A ::CssedDoc pointer.
 */
gint
document_get_length	(CssedDoc* doc);

/** Gets the document's text.  
 *	You must ensure gchar* text is a buffer big enough to contain the document's text.
 *	@param doc - A ::CssedDoc pointer.
 *	@param len - The length of the buffer passed as parameter (so the length of text to be taken).
 *	@param text - A return value where the text will be stored.
 */
void
document_get_text(CssedDoc* doc, gint len, gchar* text);

/** Gets the document's selected text.  
 *	You must ensure gchar* text is a buffer big enough to contain the document's selected text.
 *	@param doc - A ::CssedDoc pointer.
 *	@param text - A return value where the text will be stored.
 */
void
document_get_selected_text(CssedDoc* doc, gchar* text);

/** Reaches a save point for the document.
 *	It sets the save point so the document is unmodified from this point to the next document's text action.
 *	@param doc - A ::CssedDoc pointer.
 */
void
document_set_save_point( CssedDoc* doc );

/** Writes some text to the program's output list.
 *	It writes some text passed as an argument to the program's output. The parameter "color" should contain a color name ( as "red", "brown" ) 
 *	the color will be parsed as Pango markup..
 *	@param doc - A ::CssedDoc pointer.
 *	@param error_message - The text to be written.
 *	@param color - A Pango parseable string, with the text color.
 */
void 
document_write_error_to_program_output(	CssedDoc* doc, 
										gchar* error_message,
										gchar* color);
/** Gets a line of text.  
 *	You must ensure gchar* text is a buffer big enough to contain the line's text.
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line that will be taken.
 *	@param buffer - A return value where the text will be stored.
 */										
void
document_get_text_line(CssedDoc* doc, gint line, gchar* buffer);

/** Returns one line's character length.  
 *	This gets the length of the line passed as parameter.
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line whose length will be returned
 *	@return An integer containing the line's length.
 */		
gint 
document_get_line_length(CssedDoc* doc, gint line);

/** Returns the number of lines on document's text.  
 *	This gets the number of lines that exists on the text of the document passed as an argument.
 *	@param doc - A ::CssedDoc pointer.
  *	@return An integer containing the number of text lines on the document.
*/	
gint
document_get_line_count( CssedDoc* doc );

/** Changes the folding state of a line.  
 *	It folds a line or unfolds it if folding is enabled for the document and the line is a fold point.
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line to fold/unfold.
 */	
void
document_toggle_fold_at_line(CssedDoc* doc, gint line);

/** Returns if a line is visible on screen or if it's hidden.  
 *	When folding is enabled a line of text can be hidden if it's contained between two fold points. This functions returns a boolean indicating if a given line is visible on screen.
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line you want to know if it's visible.
 */	
gboolean
document_get_line_is_visible(CssedDoc* doc, gint line);

/** Makes a line visible on screen if it's hidden.  
 *	When folding is enabled a line of text can be hidden if it's contained between two fold point. This functions ensures that the line is visible on screen.
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line to make visible.
 */	
void
document_ensure_line_is_visible( CssedDoc* doc, gint line);

/** Returns the line's fold level.  
 *	The fold level is how many fold points this line is nested. This function returns the fold level of any given line. 
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line to know its fold level.
 */	
gint
document_get_fold_level( CssedDoc* doc, gint line );

/** Closes all folds on the document.  
 *	It collapses all fold points hidding the text on them. 
 *	@param doc - A ::CssedDoc pointer.
 */	
void
document_fold_all( CssedDoc* doc );

/** Opens all folds on the document.  
 *	It expands all fold points hidding the text on them. 
 *	@param doc - A ::CssedDoc pointer.
 */	
void
document_unfold_all( CssedDoc* doc );

/** Returns if a fold point is expanded at one line.  
 *	This function is used to know if a fold is collapsed or expanded. 
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line number to check.
 *	@return A boolean indicating if the fold point is expanded at the given line.
 */	
gboolean 
document_get_fold_expanded_at_line( CssedDoc* doc, gint line );

/** Shows an arrow marker at the left margin.  
 *	It is useful to catch the user's attention on one line. 
 *	@param doc - A ::CssedDoc pointer.
 *	@param line - The line where to show the arrow.
 */	
void
document_set_validator_arrow_at_line( CssedDoc* doc, gint line );

/** Deletes all arrow markers at the left margin.  
 *	Use it when you wrote an arrow at the left margin and it's not used anymore.
 *	@param doc - A ::CssedDoc pointer.
 */	
void
document_clear_validator_arrow( CssedDoc* doc );

// access to document object's fields 
/** Gets a pointer to the main ::CssedWindow object.  
 *	This is needed to pass it the window functions defined in cssedwindow.h
 *	@param doc - A ::CssedDoc pointer.
 */
CssedWindow*
document_get_window( CssedDoc* doc );

/** Gets the document's file name.  
 *	It gets - ususally - the full path to the file or NULL if the file is unnamed so unsaved.
 *	You must free the return value with g_free when no longer needed. 
 *	@param doc - A ::CssedDoc pointer.
 *	@return A newly allocated buffer containing the document's file name.
 */
gchar*
document_get_filename( CssedDoc* doc );

/** Gets the document's editor control.
 *	It returns a pointer to a Widget containing the cssed's editor control
 *	based on scintilla ( http://scintilla.org ).
 *	@param doc - A ::CssedDoc pointer.
 *	@return A pointer to the document's scintilla Widget.
 */	
GtkWidget*
document_get_scintilla_widget( CssedDoc* doc );


